<?php

namespace backend\components;

use backend\models\db\UserRole;
use Yii;
use yii\web\Controller;

use backend\models\db\AccountBalance;
use backend\models\db\Budget;
use backend\models\db\Conversation;
use backend\models\db\Currency;
use backend\models\db\DateFormat;
use backend\models\db\Eoms;
use backend\models\db\TimeFormat;
use backend\models\db\TimeZone;
use backend\models\db\User;
use backend\models\db\UserMeta;

use backend\components\helpers\Subscription;
use yii\web\ForbiddenHttpException;
use yii\web\IdentityInterface;

class CustomController extends Controller
{
    public $mockBudgetId;
    public $coach;
    public $defaultAccountId;
    public $fileActions = [];
    public $initUserRole; // the original user role of this user
    public $isFilePage = false;
    public $mainUserId;
    public $pageLoadRequest;
    public $userRole;
    /**
     * @var User Currently accessed User object (logged in or overridden)
     */
    public $user;
    /**
     * @var IdentityInterface Currently accessed Identity object (logged in or overridden).
     * Related User object defined in $this->user
     */
    public $userIdentity;
    public $userMeta;
    /**
     * @var Subscription Object for checking the subscription validity
     */
    public $subscription;

    /** @var int The real active budget (can't be accessed by descendant classes) */
    private $_activeBudgetId;


    public function beforeAction($action)
    {
        // useful for load testing
        Yii::$app->response->headers->set('X-CSRF-Name', Yii::$app->request->csrfParam);
        Yii::$app->response->headers->set('X-CSRF-Token', Yii::$app->request->csrfToken);

        /** @var \backend\components\User $user The user that is actually logged in */
        $user = Yii::$app->user;
        if (!$user->isGuest) {
            // admin/coach specific stuff
            if (!Yii::$app->session->get('initUserRole')) {
                Yii::$app->session->set('initUserRole', $user->identity->userRole->name);
            }
            $this->view->params['initUserRole'] = Yii::$app->session->get('initUserRole');

            // coach/superadmin user override
            if (Yii::$app->session->get('adminSelectedUserId')) {
                if (!Yii::$app->session->get('adminSelectedUserIdentity')) {
                    $coachUserId = Yii::$app->user->identity->id;
                    Yii::$app->user->switchIdentity(User::findOne(Yii::$app->session->get('adminSelectedUserId')));
                    Yii::$app->session->set('adminSelectedUserIdentity', Yii::$app->user->identity);
                    Yii::$app->user->switchIdentity(User::findOne($coachUserId));
                }
                $this->userIdentity = Yii::$app->session->get('adminSelectedUserIdentity');
            }
            else {
                $this->userIdentity = $user->identity;
            }

            // store user role to session and get
            if (!Yii::$app->session->get('userRole')) {
                $userRole = Yii::$app->authManager->getRolesByUser($this->userIdentity->id);
                Yii::$app->session->set('userRole', array_pop($userRole)->name);
            }
            $this->userRole = Yii::$app->session->get('userRole');

            // set user
            if (!Yii::$app->session->get('user')) {
                Yii::$app->session->set('user', User::findOne($this->userIdentity->id));
            }
            $this->user = Yii::$app->session->get('user');

            // load main account ID if not set, relevant after "remember me" login
            if (!is_numeric(Yii::$app->session->get('mainUserId'))) {
                Yii::$app->session->set('mainUserId', $this->userIdentity->getMainUserId());
            }
            $this->mainUserId = Yii::$app->session->get('mainUserId');

            // load meta if not set, relevant after "remember me" login
            if (!Yii::$app->session->get('userMeta')) {

                $userMeta = UserMeta::findAllSorted(['user_id' => $this->userIdentity->id]);
                Yii::$app->session->set('userMeta', $userMeta);

            }
            $this->userMeta = Yii::$app->session->get('userMeta');

            // track login based on idle time
            $user->identity->trackLogin(
                Yii::$app->session->get('lastActivityTime'),
                Yii::$app->session->get('initUserRole'),
                $this->user->id
            );
            Yii::$app->session->set('lastActivityTime', time());

            // check subscription
            if (!Yii::$app->session->get('subscription')) {

                $subscription = new Subscription($this->mainUserId);
                Yii::$app->session->set('subscription', $subscription);

            }
            $this->subscription = Yii::$app->session->get('subscription');

            // only clients and sub-clients (except when admin logged in in an "override" mode)
            if (strpos(Yii::$app->request->getPathInfo(), 'site/login') === false &&
                strpos(Yii::$app->request->getPathInfo(), 'subscription') === false &&
                strpos(Yii::$app->request->getPathInfo(), 'users/edit') === false &&
                strpos(Yii::$app->request->getPathInfo(), 'notifications/get-notification-updates-data') === false &&
                strpos(Yii::$app->request->getPathInfo(), 'conversations/get-messaging-updates-data') === false &&
                !in_array($user->identity->userRole->name, ['superadmin', 'coach'])) {

                // check if subscription valid
                if (!$this->subscription->hasValidSubscription()) {
                    // subscription is not valid
                    if ($this->userIdentity->user_role_id == UserRole::SUBCLIENT) {
                        Yii::$app->user->logout();
                        Yii::$app->session->setFlash('login', [['danger', 'There\'s no valid subscription for the parent account.']]);
                        $this->redirect(Yii::$app->user->loginUrl);
                        return false;
                    }
                    else {
                        Yii::$app->session->set('invalidSubscription', true);
                        // redirect to non-valid subscription site
                        if (strpos(Yii::$app->request->getPathInfo(), 'subscription/index') === false) {
                            $this->redirect(['/subscription/index']);
                            return false;
                        }
                    }
                }
                else {
                    Yii::$app->session->set('invalidSubscription', false);
                    Yii::$app->session->set('noUserDetails', false);
                }

            }

            // check user details filled
            if (Yii::$app->session->get('invalidSubscription')) {

                Yii::$app->session->set('noUserDetails', UserMeta::getUserDetailsPopulated($this->mainUserId));

            }

            // get active budget ID (only if the user already has a valid subscription)
            if (empty(Yii::$app->session->get('invalidSubscription')) &&
                !in_array($this->userRole, ['superadmin', 'coach'])) {

                $this->activeBudgetId = $this->userIdentity->getActiveBudgetId($this->mainUserId);

            }

            // get default account ID
            if (!Yii::$app->session->get('defaultAccountId') && $this->activeBudgetId) {

                $accountId = Budget::findOne($this->activeBudgetId)->default_account_id;
                Yii::$app->session->set('defaultAccountId', $accountId);

            }
            $this->defaultAccountId = Yii::$app->session->get('defaultAccountId');

            // get coach object
            if (!Yii::$app->session->get('coach')) {

                $mainUser = User::findOne($this->mainUserId);
                $coach = $this->userIdentity->getCoach();
                Yii::$app->session->set('coach', $coach ? $coach : $mainUser->getCoach());

            }
            $this->coach = Yii::$app->session->get('coach');

            // check if user has completed First Time Setup (only if not on the way to subscription page
            // and only if not the admin logged in in an "override" mode)
            if (empty(Yii::$app->session->get('invalidSubscription')) &&
                !in_array($this->userRole, ['superadmin', 'coach'])) {

                if (!isset($this->userMeta['fts_step'])) {
                    $this->userMeta['fts_step'] = $this->userIdentity->initFTSMeta($this->mainUserId);
                    Yii::$app->session->set('userMeta', $this->userMeta);
                }

                if ($this->userMeta['fts_step'] != 'completed') {
                    $stepId = (int) strtr($this->userMeta['fts_step'], ['d' => '']);
                    if (!$this->_isAllowedFtsUrl($this->userMeta['fts_step'])) {
                        if ($stepId == 0) {
                            $this->redirect(['/first-time-setup/welcome']);
                            return false;
                        }
                        else {
                            $this->redirect(['/first-time-setup/step-' . $stepId]);
                            return false;
                        }
                    }
                }

                // check whether to force EOMS submission (not for mock budgets)
                if ($this->userMeta['fts_step'] == 'completed' && !$this->mockBudgetId) {
                    if (in_array($this->userRole, ['client', 'sub-client'])) {
                        if (Yii::$app->session->get('forceEoms') == null || Yii::$app->session->get('lastCheckDate') != date('Y-m-d')) {
                            // check last EOMS
                            Yii::$app->session->set('forceEoms', Eoms::checkForceEoms($this->activeBudgetId, $this->mainUserId));
                        }
                        if (Yii::$app->session->get('eomsOverdue') == null) {
                            $unsubmittedEoms = Eoms::getLastEoms($this->activeBudgetId);
                            if ($unsubmittedEoms && $unsubmittedEoms->isOverdue()) {
                                Yii::$app->session->set('eomsOverdue', true);
                            }
                            else {
                                Yii::$app->session->set('eomsOverdue', false);
                            }
                        }

                        if (Yii::$app->session->get('eomsOverdue')) {
                            if (strpos(Yii::$app->request->getPathInfo(), 'budget/jar-summary') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'budget/jar-summary-tooltip') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'budget/adjust-jar-funds-remaining') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'budget/end-of-month-summary') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'budget/end-of-month-summary-equation-numbers') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'budget/end-of-month-summary-submit') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'conversations/get-messaging-updates-data') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'notifications/get-notification-updates-data') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'media/media-modal') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'site/logout') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'admin/user/switch-back') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'users/index') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'users/edit-user') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'users/remove-user') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'users/edit-user-basic-info') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'subscription/index') === false &&
                                strpos(Yii::$app->request->getPathInfo(), 'subscription/update-card') === false) {

                                $this->redirect(['/budget/end-of-month-summary']);
                                return false;
                            }
                        }

                    }

                    // check whether to display EOMS warning
                    if (!Yii::$app->session->get('showEomsWarning')) {
                        Yii::$app->session->set('showEomsWarning', AccountBalance::getShowEomsWarning($this->activeBudgetId));
                    }

                    $this->view->params['showEomsWarning'] = Yii::$app->session->get('showEomsWarning');
                }

            }

            // check for set currency
            if (!Yii::$app->session->get('currency')) {
                Yii::$app->session->set('currency', Currency::getCurrency(
                    isset($this->userMeta['currency_id']) ? $this->userMeta['currency_id'] : null
                ));
            }

            // check for set time zone
            if (!Yii::$app->session->get('timeZone')) {
                Yii::$app->session->set('timeZone', TimeZone::getTimeZoneCode(
                    isset($this->userMeta['time_zone_id']) ? $this->userMeta['time_zone_id'] : null
                ));
            }
            Yii::$app->setTimeZone(Yii::$app->session->get('timeZone'));

            // check for set time format
            if (!Yii::$app->session->get('timeFormat')) {
                Yii::$app->session->set('timeFormat', TimeFormat::getTimeFormat(
                    isset($this->userMeta['time_format_id']) ? $this->userMeta['time_format_id'] : null
                ));
            }

            // check for set date format
            if (!Yii::$app->session->get('dateFormat')) {
                Yii::$app->session->set('dateFormat', DateFormat::getDateFormat(
                    isset($this->userMeta['date_format_id']) ? $this->userMeta['date_format_id'] : null
                ));
            }

            // check for set decimal separator
            if (!Yii::$app->session->get('thousandSeparator') || !Yii::$app->session->get('decimalSeparator')) {
                $formatDetails = Currency::getCurrencyFormatDetails(
                    isset($this->userMeta['currency_format']) ? $this->userMeta['currency_format'] : null
                );
                Yii::$app->session->set('thousandSeparator', $formatDetails['thousand_separator']);
                Yii::$app->session->set('decimalSeparator', $formatDetails['decimal_separator']);
            }
            Yii::$app->formatter->thousandSeparator = Yii::$app->session->get('thousandSeparator');
            Yii::$app->formatter->decimalSeparator  = Yii::$app->session->get('decimalSeparator');

            // set repeating view vars
            $this->view->params['userIdentity']       = $this->userIdentity;
            $this->view->params['username']           = $this->userIdentity->username;
            $this->view->params['userRole']           = $this->userRole;
            $this->view->params['userMeta']           = $this->userMeta;
            $this->view->params['notificationsCount'] = $this->user->getNotifications()->where(['read' => 0, 'archived' => 0])->count();
            $this->view->params['messagesCount']      = Conversation::getMessagesCount($this->userIdentity->id);


            // update last check date if necessary
            if (Yii::$app->session->get('lastCheckDate') != date('Y-m-d')) {
                Yii::$app->session->set('lastCheckDate', date('Y-m-d'));
            }

            // check if file page
            $isFilePage = in_array($action->id, $this->fileActions);

            // check if ajax
            if (Yii::$app->request->isAjax || $isFilePage){
                $this->layout = false;
            }

            // non-ajax request dies here
            if (!Yii::$app->request->isAjax && !$user->isGuest && !$isFilePage) {
                /*
                 * avoid the execution of the page (and unnecessary
                 * loading of data) if this isn't an ajax request
                 */
                $this->view->title = 'loading...';
                Yii::$app->response->content = $this->render('@backend/views/budget/loading', [
                    'title' => 'loading...',
                ]);
                Yii::$app->response->statusCode = 200;
                Yii::$app->end();
            }

            // check if this is a page load request - initial page, not via history.pushState()...
            $this->pageLoadRequest = Yii::$app->getRequest()->getQueryParam('pageLoadRequest', 0);
            $this->view->params['pageLoadRequest'] = $this->pageLoadRequest;
        }
        return parent::beforeAction($action);
    }


    public function getUserMeta($key)
    {
        if (isset($this->userMeta[$key])) {
            return $this->userMeta[$key];
        }
        else {
            return false;
        }
    }

    /**
     * Function used in cases where user tried to access
     * a resource they don't have access to.
     * The execution of the application ends with this function.
     * @return \yii\web\Response
     * @throws ForbiddenHttpException
     */
    public function accessError()
    {
        if (Yii::$app->user->isGuest) {
            return $this->redirect(Yii::$app->user->loginUrl);
        }
        else {
            throw new ForbiddenHttpException('Access to this resource is forbidden.');
        }
    }

    public function jsCallbackToView($jsCallback = false)
    {
        if (!$jsCallback) {
            $jsCallback = Yii::$app->getRequest()->getQueryParam('js_callback', false);
        }
        $this->view->params['jsCallback'] = $jsCallback;
    }

    public function getBudgetIdFromUrl()
    {
        $budgetId = Yii::$app->getRequest()->getQueryParam('budget_id', false);
        if (!$budgetId || !is_numeric($budgetId) || !Yii::$app->user->can('accessBudget', ['budgetId' => $budgetId])) {
            $budgetId = $this->activeBudgetId;
        }
        return $budgetId;
    }


    protected function _isAllowedFtsUrl($stepCode) {

        // controller isn't fully instantiated yet, so we have to use primitive URL parsing as fallback
        $urlParts = explode('/', Yii::$app->request->getPathInfo());
        $stepId = (int) strtr($stepCode, ['d' => '']);

        // list allowed URLs for each step of FTS
        $allowedUrls = [
            [
                'controller' => 'first-time-setup',
                'action'=> 'welcome'
            ],
            [
                'controller' => 'first-time-setup',
                'action'=> 'step-1'
            ],
            [
                'controller' => 'users',
                'action'=> 'edit-user'
            ],
            [
                'controller' => 'users',
                'action'=> 'edit-user-basic-info'
            ],
            [
                'controller' => 'users',
                'action'=> 'add-user'
            ],
            [
                'controller' => 'users',
                'action'=> 'remove-user'
            ],
            [
                'controller' => 'notifications',
                'action' => 'get-notification-updates-data'
            ],
            [
                'controller' => 'conversations',
                'action' => 'get-messaging-updates-data'
            ],
            [
                'module' => 'admin',
                'controller' => 'user',
                'action' => 'switch-back'
            ]
        ];

        // URLs for step 2
        if (($stepId >= 2) || ($stepCode == '1d')) {

            $allowedUrls = array_merge($allowedUrls, [
                [
                    'controller' => 'first-time-setup',
                    'action'=> 'step-2'
                ],
                [
                    'controller' => 'accounts',
                    'action'=> 'add-account'
                ],
                [
                    'controller' => 'accounts',
                    'action'=> 'edit-account'
                ],
                [
                    'controller' => 'accounts',
                    'action'=> 'remove-account'
                ]
            ]);

        }

        // URLs for step 3
        if (($stepId >= 3)  || ($stepCode == '2d')) {

            $allowedUrls = array_merge($allowedUrls, [
                [
                    'controller' => 'first-time-setup',
                    'action'=> 'step-3'
                ],
                [
                    'controller' => 'debts',
                    'action'=> 'add-debt'
                ],
                [
                    'controller' => 'debts',
                    'action'=> 'edit-debt'
                ],
                [
                    'controller' => 'debts',
                    'action'=> 'remove-debt'
                ]
            ]);

        }

        // URLs for step 4
        if ($stepId >= 3) {

            $allowedUrls = array_merge($allowedUrls, [
                [
                    'controller' => 'first-time-setup',
                    'action'=> 'step-4'
                ],
                [
                    'controller' => 'incomes',
                    'action'=> 'add-income'
                ],
                [
                    'controller' => 'incomes',
                    'action'=> 'edit-income'
                ],
                [
                    'controller' => 'incomes',
                    'action'=> 'remove-income'
                ]
            ]);

        }

        // URLs for step 5
        if ($stepId >= 4) {

            $allowedUrls = array_merge($allowedUrls, [
                [
                    'controller' => 'first-time-setup',
                    'action'=> 'step-5'
                ],
                [
                    'controller' => 'first-time-setup',
                    'action'=> 'step-5-jar-detail'
                ],
                [
                    'controller' => 'jars',
                    'action'=> 'add-jar'
                ],
                [
                    'controller' => 'jars',
                    'action'=> 'edit-jar'
                ],
                [
                    'controller' => 'jars',
                    'action'=> 'remove-jar'
                ],
                [
                    'controller' => 'expenses',
                    'action'=> 'add-estimated-expense'
                ],
                [
                    'controller' => 'expenses',
                    'action'=> 'add-recurring-expense'
                ],
                [
                    'controller' => 'expenses',
                    'action'=> 'add-onetime-expense'
                ],
                [
                    'controller' => 'expenses',
                    'action'=> 'edit-estimated-expense'
                ],
                [
                    'controller' => 'expenses',
                    'action'=> 'edit-recurring-expense'
                ],
                [
                    'controller' => 'expenses',
                    'action'=> 'edit-onetime-expense'
                ],
                [
                    'controller' => 'expenses',
                    'action'=> 'remove-expense'
                ],
                [
                    'controller' => 'expenses',
                    'action'=> 'estimated-expenses-table'
                ],
                [
                    'controller' => 'expenses',
                    'action'=> 'recurring-expenses-table'
                ],
                [
                    'controller' => 'expenses',
                    'action'=> 'onetime-expenses-table'
                ],
                [
                    'controller' => 'jars',
                    'action'=> 'create-expense-jar'
                ],
                [
                    'controller' => 'jars',
                    'action'=> 'edit-expense-jar'
                ],
                [
                    'controller' => 'jars',
                    'action'=> 'remove-expense-jar'
                ],
                [
                    'controller' => 'jars',
                    'action'=> 'reorder-jars'
                ],
                [
                    'controller' => 'debts',
                    'action'=> 'debts-table'
                ],
                [
                    'controller' => 'debt-payments',
                    'action'=> 'debt-payments-table-full'
                ],
                [
                    'controller' => 'debt-payments',
                    'action'=> 'add-debt-payment'
                ],
                [
                    'controller' => 'debts',
                    'action'=> 'pause-debt'
                ],
                [
                    'controller' => 'debts',
                    'action'=> 'continue-debt'
                ],
                [
                    'controller' => 'first-time-setup',
                    'action'=> 'complete'
                ]
            ]);

        }

        // URLs for step 6
        if ($stepId >= 5) {
            $allowedUrls = array_merge($allowedUrls, [
                [
                    'controller' => 'first-time-setup',
                    'action'=> 'step-6'
                ],
                [
                    'controller' => 'jars',
                    'action' => 'budget-shortfall-calculator'
                ]
            ]);
        }

        // URLs for step 7 & complete
        if ($stepId >= 6) {
            $allowedUrls = array_merge($allowedUrls, [
                [
                    'controller' => 'first-time-setup',
                    'action'=> 'step-7'
                ],
                [
                    'controller' => 'first-time-setup',
                    'action'=> 'complete'
                ]
            ]);
        }

        $isAllowedUrl = false;
        foreach ($allowedUrls as $url) {
            if (count($urlParts) < count($url)) {
                continue;
            }
            elseif (count($urlParts) == 3 && isset($url['module'])) {
                if ($url['module'] == $urlParts[0] && $url['controller'] == $urlParts[1] && $url['action'] == $urlParts[2]) {
                    $isAllowedUrl = true;
                    break;
                }
            }
            else {
                if ($url['controller'] == $urlParts[0] && $url['action'] == $urlParts[1]) {
                    $isAllowedUrl = true;
                    break;
                }
            }
        }

        return $isAllowedUrl;
    }

    public function getActiveBudgetId()
    {
        return $this->mockBudgetId ?
            $this->mockBudgetId : $this->_activeBudgetId;
    }

    public function setActiveBudgetId($budgetId)
    {
        $this->_activeBudgetId = $budgetId;
    }

}
