<?php

namespace backend\components;

class MailHelper
{
    const MODE_MANDRILL = 1;
    const MODE_MAILER = 2;

    public $mode;

    /**
     * Send an email using the selected transport
     * @param $to
     * @param $params
     * @param $template
     * @param string|null $subject
     * @param array $attachments
     * @return bool|null
     */
    public function sendEmail($to, $params, $template, $subject = null, $attachments = array())
    {
        // add prefix to all outgoing emails
        if (!empty(\Yii::$app->params['emailSubjectPrefix'])) {
            $subject = trim(\Yii::$app->params['emailSubjectPrefix']) . ' ' . $subject;
        }
        switch ($this->mode) {
            case self::MODE_MANDRILL:
                return $this->sendMandrillEmail($to, $params, $template, $subject, $attachments);
            case self::MODE_MAILER:
                return $this->sendMailerEmail($to, $params, $template, $subject, $attachments);
        }
        return null;
    }

    protected function sendMandrillEmail($to, $params, $template, $subject, $attachments = array())
    {
        $replyTo = [\Yii::$app->params['replyToEmail'] => \Yii::$app->name . ' support'];
        return MandrillHelper::sendEmail(
            $template,
            $subject,
            $params,
            $to,
            $attachments,
            $replyTo
        );
    }

    protected function sendMailerEmail($to, $params, $template, $subject, $attachments = array())
    {
        if (is_array($to)) {
            $toEmail = key($to);
            $toName = $to[$toEmail];
        }
        else {
            $toEmail = $to;
            $toName = null;
        }
        $mail = \Yii::$app->mailer->compose(
            [
                'html' => 'html/' . $template,
                'text' => 'text/' . $template
            ],
            $params
        )
            ->setFrom([\Yii::$app->params['supportEmail'] => \Yii::$app->name . ' support'])
            ->setReplyTo([\Yii::$app->params['replyToEmail'] => \Yii::$app->name . ' support'])
            ->setTo($toName ? [$toEmail => $toName] : $toEmail)
            ->setSubject($subject);

        if (!empty($attachments)) {
            foreach ($attachments as $name => $content) {
                $mail->attachContent($content, ['fileName' => $name, 'contentType' => self::getMimeType($name)]);
            }
        }

        return $mail->send();
    }

    public static function getMimeType($filename)
    {
        if (isset(\Yii::$app->params['mimeTypes'])) {
            $info = pathinfo($filename);
            if ($info && isset($info['extension']) && strlen($info['extension']) > 0) {
                $ext = substr($info['extension'], 1);
                if (isset(\Yii::$app->params['mimeTypes'][$ext])) {
                    return \Yii::$app->params['mimeTypes'][$ext];
                }
            }
        }
        return 'application/octet-stream';
    }
}