<?php

namespace backend\components\helpers;

use Yii;
use yii\base\Component;
use yii\helpers\ArrayHelper;
use backend\components\helpers\Formatter;

class DataTables extends Component {

    public static function getRow($row, $columns) {
        $responseRow = [];
        foreach ($columns as $column) {
            $responseRow[] = self::getColumn($row, $column);
        }
        $responseRow[] = self::getMenuColumn($row);
        return $responseRow;
    }

    public static function getColumn($row, $column) {
        $isColEditable = isset($column['isEditable']) && $column['isEditable'] == true;
        $val = $column['value'];
        if ($row['isEditable'] && $isColEditable) {
            $key = $column['key'];
            $type = $column['type'];
            switch ($type) {
                case 'text':
                    $postfix = isset($column['postfix']) ? $column['postfix'] : '';
                    $value = "<span data-value>$val</span> $postfix";
                    $editor = "<input type='text' value='".htmlentities($val, ENT_QUOTES)."' /> $postfix";
                    break;
                case 'currency':
                    $currencySymbol = Formatter::currency('');
                    $fmtVal = Formatter::currency($val);
                    $cls = (intval($val) < 0) ? 'gj-is-negative' : '';
                    $value = "<span data-value class='$cls'>$fmtVal</span>";
                    $editor = "$currencySymbol<input type='text' value='$val' />";
                    break;
                case 'select':
                    $value = "<span data-value>$val</span>";
                    $editor = '<select>';
                    foreach ($column['options'] as $option) {
                        $selected = $option['selected'] ? 'selected' : '';
                        $optName = $option['name'];
                        $optVal = $option['value'];
                        $editor .= "<option $selected value='$optVal'>$optName</option>";
                    }
                    $editor .= '</select>';
                    break;
                case 'date':
                    $fmtVal = Formatter::date($val);
                    $value = "<span data-value>$fmtVal</span>";
                    $format = Formatter::phpDateCodeToDatepickerCode(Yii::$app->session->get('dateFormat'));
                    $momentFormat = Formatter::phpDateCodeToMomentDateCode(Yii::$app->session->get('dateFormat'));
                    $editor = "<input data-date-format='$format' data-moment-format='$momentFormat' type='text' value='$fmtVal' />";
                    break;
                default:
                    $value = 'missing value';
                    $editor = 'missing editor';
                break;
            }

            $editable = "<span data-editable data-editable-field='$key' data-editable-type='$type' class='data-$type'>$editor</span>";
            return "$value $editable";
        } elseif ($column['type'] == 'checkbox') {
            $checked = $column['value'] ? 'checked' : '';
            $cType = $column['cType'];
            $cId = $column['cId'];
            $cName = $column['cName'];
            $disabled = $isColEditable ? '' : 'disabled';
            $val = "<div class='checkbox".($isColEditable ? ' bank-statement-check' : '')."' data-operation-type='$cType' data-operation-id='$cId'>";
            $val .= "<input type='hidden' name='Check[$cName]' value='0'>";
            $val .= "<input type='checkbox' id='check-$cName' name='Check[$cName]' value='1' $checked $disabled>";
            $val .= "<label for='check-$cName'></label>";
            $val .= '</div>';
            return $val;
        }
        else {
            if ($column['type'] == 'currency') {
                $fmtVal = Formatter::currency($val);
                $cls = (intval($val) < 0) ? 'gj-is-negative' : '';
                return "<span data-value class='$cls'>$fmtVal</span>";
            }
            return $val;
        }
    }

    public static function getMenuColumn($row) {
        $icon = isset($row['icon']) ? $row['icon'] : '';
        if ($row['isEditable']) {
            $id = $row['id'];
            $type = $row['type'];
            $editUrl = $row['editUrl'];
            $deleteUrl = $row['deleteUrl'];
            $callback = $row['callback'];
            if ($deleteUrl) {
                $menuItemDelete = "<li><a href='#'' class='open-modal' data-modal-url='$deleteUrl'><i class='glyphicon glyphicon-trash'></i>Delete</a></li>";
            } else { $menuItemDelete = ''; }
            if ($editUrl) {
                $menuItemEdit = '<li><a href="#" data-action="edit-transaction"><i class="glyphicon glyphicon-edit"></i>Edit</a></li>';
            } else { $menuItemEdit = ''; }
            $menu = "<a href='#' id='menu1' class='cog-menu' data-toggle='dropdown' aria-haspopup='true' aria-expanded='false'><i class='glyphicon glyphicon-cog gi-2x'></i></a>";
            $menu .= "<ul class='dropdown-menu' aria-labelledby='menu1'>$menuItemDelete $menuItemEdit</ul>";
            if ($menuItemEdit) {
                $menu .= "<ul class='edit-menu'>";
                $menu .= "<a href='#' class='save-editing' data-action-type='$type' data-action-url='$editUrl' data-action-id='$id' data-action-callback='$callback'><i class='glyphicon glyphicon-ok-sign'></i></a>";
                $menu .= "<a href='#' class='cancel-editing' data-action-callback='$callback'><i class='glyphicon glyphicon-remove-sign'></i></a>";
                $menu .= "</ul>";
            }
        } else { $menu = ''; }
        return "<div class='text-center last menu-column'>$icon $menu</div>";
    }


    /**
     * Get data from the params supplied in the query URL.
     * @param array $columns Table columns - this should correspond with db select query.
     * @return array Query data.
     */
    public static function getQueryParams(array $columns)
    {
        $order = Yii::$app->getRequest()->getQueryParam('order');
        $data = [
            'draw' => Yii::$app->getRequest()->getQueryParam('draw', 0),
            'start' => Yii::$app->getRequest()->getQueryParam('start', 0),
            'length' => Yii::$app->getRequest()->getQueryParam('length', 5),
            'orderColumn' => $columns[$order[0]['column']],
            'orderDirection' => $order[0]['dir'],
            'orderColumnIndex' => $order[0]['column']
        ];
        if (isset($order[1]['column'])) {
            $data['orderColumn'] = [
                $data['orderColumn'],
                $columns[$order[1]['column']]
            ];
            $data['orderDirection'] = [
                $data['orderDirection'],
                $order[1]['dir']
            ];
            $data['orderColumnIndex'] = [
                $data['orderColumnIndex'],
                $order[1]['column']
            ];
        }
        $search = Yii::$app->getRequest()->getQueryParam('search');
        if (isset($search['value']) && (strlen(trim($search['value'])) > 0)) {
            $data['searchString'] = trim($search['value']);
        }
        else {
            $data['searchString'] = false;
        }
        return $data;
    }

    /**
     * Get filtered data based on columns and supplied filter objects.
     * @param array $columns The columns.
     * @param object $countFiltered
     * @param object $selectFiltered
     * @return array The resulting data.
     */
    public static function getResponseData(array $columns, $countFiltered, $selectFiltered)
    {
        $columnNames = [];
        foreach ($columns as $c) {
            $columnNames[] = $c['as'];
        }
        $dataTablesData = DataTables::getQueryParams($columnNames);
        $recordsTotal = $countFiltered
            ->count();

        $searchWheres = [];
        if ($dataTablesData['searchString']) {
            $words = explode(' ', $dataTablesData['searchString']);
            foreach ($words as $w) {
                if (strlen($w) > 0) {
                    foreach ($columns as $c) {
                        if ($c['type'] == 'normal') {
                            $searchWheres[] = ['LIKE', $c['select'], $w];
                        }
                    }
                }
            }
        }

        if ($dataTablesData['searchString']) {
            $selectFiltered = $selectFiltered->andWhere(array_merge(['OR'], $searchWheres));
        }

        $items = [];
        $recordsFiltered = $selectFiltered->count();
        if (is_array($dataTablesData['orderColumnIndex'])) {
            $columnIndex = $dataTablesData['orderColumnIndex'][0];
            $orderDir = $dataTablesData['orderDirection'][0];
        }
        else {
            $columnIndex = $dataTablesData['orderColumnIndex'];
            $orderDir = $dataTablesData['orderDirection'];
        }
        if ($columns[$columnIndex]['type'] == 'normal') {
            if ($dataTablesData['length'] > 0) {
                $selectFiltered
                    ->limit($dataTablesData['length'])
                    ->offset($dataTablesData['start']);
            }
            if (is_array($dataTablesData['orderColumn'])) {
                $selectFiltered
                    ->orderBy(
                        $dataTablesData['orderColumn'][0] . ' ' . $dataTablesData['orderDirection'][0] . ', ' .
                        $dataTablesData['orderColumn'][1] . ' ' . $dataTablesData['orderDirection'][1]
                    );
            }
            else {
                $selectFiltered->orderBy($dataTablesData['orderColumn'] . ' ' . $dataTablesData['orderDirection']);
            }
            $items = $selectFiltered->all();
        }
        elseif (($columns[$columnIndex]['type'] == 'method') || ($columns[$columnIndex]['type'] == 'function')){
            $nsItems = $selectFiltered->all();
            $sItems = [];
            $itemsSorted = [];
            foreach ($nsItems as $i) {
                if ($columns[$columnIndex]['type'] == 'method') {
                    $sItems[] = [
                        'item' => $i,
                        'fx' => $i->{$columns[$columnIndex]['name']}()
                    ];
                }
                elseif ($columns[$columnIndex]['type'] == 'function') {
                    $sItems[] = [
                        'item' => $i,
                        'fx' => $columns[$columnIndex]['name']($i)
                    ];
                }
            }

            usort($sItems, function($a, $b) {
                if ($a['fx'] == $b['fx']) {
                    return 0;
                }
                if ($a['fx'] < $b['fx']) {
                    return -1;
                }
                else {
                    return 1;
                }
            });

            if ($orderDir == 'desc') {
                $sItems = array_reverse($sItems);
            }
            foreach ($sItems as $i) {
                $itemsSorted[] = $i['item'];
            }
            if ($dataTablesData['length'] > 0) {
                $items = array_slice($itemsSorted, $dataTablesData['start'], $dataTablesData['length']);
            }
            else {
                $items = $itemsSorted;
            }
        }
        return [
            'items' => $items,
            'response' => [
                'draw' => $dataTablesData['draw'],
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsFiltered,
                'data' => []
            ]
        ];
    }

    public static function getFilteredData($columnNames, $data, $paramsOverride = null)
    {
        $dataTablesData = DataTables::getQueryParams($columnNames);
        if (is_array($paramsOverride)) {
            $dataTablesData = ArrayHelper::merge($dataTablesData, $paramsOverride);
        }
        elseif ($paramsOverride === true) {
            // add ordering by id parameter
            $dataTablesData['orderColumn'] = [
                $dataTablesData['orderColumn'], 'id'
            ];
        }

        $recordsTotal = count($data);

        // search data
        if ($dataTablesData['searchString']) {
            $usItems = [];
            $words = explode(' ', $dataTablesData['searchString']);
            foreach ($data as $d) {
                $found = false;
                foreach ($columnNames as $c) {
                    if ($c) {
                        foreach ($words as $w) {
                            if (strlen($w) > 0) {
                                $column = is_object($d) ? $d->{$c} : $d[$c];
                                if (strpos(strtolower($column), strtolower($w)) !== false) {
                                    $usItems[] = $d;
                                    $found = true;
                                    break;
                                }
                            }
                        }
                        if ($found) {
                            break;
                        }
                    }
                }
            }
        }
        else {
            $usItems = $data;
        }
        $recordsFiltered = count($usItems);
        // sort items
        $sItems = [];
        foreach ($usItems as $i) {
            if (is_array($dataTablesData['orderColumn'])) {
                $fx = [];
                foreach ($dataTablesData['orderColumn'] as $oc) {
                    $fx[] = is_object($i) ? $i->{$oc} : $i[$oc];
                }
                $sItems[] = [
                    'item' => $i,
                    'fx' => $fx
                ];
            }
            else {
                $sItems[] = [
                    'item' => $i,
                    'fx' => is_object($i) ? $i->{$dataTablesData['orderColumn']} : $i[$dataTablesData['orderColumn']]
                ];
            }
        }

        usort($sItems, function($a, $b) use ($dataTablesData) {
            // supports only 2 columns
            if (is_array($a['fx'])) {
                if ($a['fx'][0] == $b['fx'][0]) {
                    if ($a['fx'][1] < $b['fx'][1]) {
                        return $dataTablesData['orderDirection'][1] == 'desc' ? 1 : -1;
                    }
                    elseif ($a['fx'][1] > $b['fx'][1]) {
                        return $dataTablesData['orderDirection'][1] == 'desc' ? -1 : 1;
                    }
                    else {
                        return 0;
                    }
                }
                elseif ($a['fx'][0] < $b['fx'][0]) {
                    return $dataTablesData['orderDirection'][0] == 'desc' ? 1 : -1;
                }
                else {
                    return $dataTablesData['orderDirection'][0] == 'desc' ? -1 : 1;
                }
            }
            else {
                if ($a['fx'] == $b['fx']) {
                    return 0;
                }
                elseif ($a['fx'] < $b['fx']) {
                    return -1;
                }
                else {
                    return 1;
                }
            }
        });

        if (!is_array($dataTablesData['orderDirection']) && $dataTablesData['orderDirection'] == 'desc') {
            $sItems = array_reverse($sItems);
        }

        $itemsSorted = [];
        foreach ($sItems as $i) {
            $itemsSorted[] = $i['item'];
        }
        if ($dataTablesData['length'] > 0) {
            $items = array_slice($itemsSorted, $dataTablesData['start'], $dataTablesData['length']);
        }
        else {
            $items = $itemsSorted;
        }
        return [
            'items' => $items,
            'response' => [
                'draw' => $dataTablesData['draw'],
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsFiltered,
                'data' => []
            ]
        ];
    }

}
