<?php

namespace backend\controllers;

use backend\components\helpers\JsonTools;
use backend\models\db\TimeSlot;
use backend\models\db\TimeSlotBooking;
use backend\models\db\TimeSlotItem;
use backend\models\form\AddTimeSlotBooking;
use Yii;

use backend\models\db\Appointment;
use backend\models\db\AppointmentUser;
use backend\models\db\TimeZone;
use backend\models\db\User;
use backend\models\db\UserMeta;

use yii\filters\AccessControl;
use backend\components\CustomController;
use backend\components\helpers\Formatter;
use yii\web\Response;


/**
 * Coach controller
 */
class CoachController extends CustomController
{

    public $enableCsrfValidation = false;

    public function behaviors()
    {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'bookings',
                            'events',
                            'add-booking',
                            'cancel-booking'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];

    }

    /**
     * Show client's view
     */
    public function actionIndex()
    {
        $user = User::findOne($this->userIdentity->id);
        if (!$this->coach) {
            return $this->render('index-no-coach', []);
        }

        // get the coach meta
        $coachMeta = UserMeta::findAllSorted(['user_id' => $this->coach->id]);

        // coach time
        $coachTime = Formatter::datetime(Formatter::utcDatetimeToLocalDatetime(time(), $this->coach->getTimeZone()->code, 'Y-m-d H:i:s'));

        // get booked appointments
        $nextAppointment = TimeSlotBooking::getNextAppointment($user->id);
        $nextAppointment = $nextAppointment ?
            Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($nextAppointment->start_at, $user->getTimeZone()->code, 'Y-m-d H:i:s')) :
            '-';

        $this->view->title = 'Coach';
        return $this->render('index', [
            'title' => 'Coach',
            'user' => $user,
            'coach' => $this->coach,
            'coachMeta' => $coachMeta,
            'coachTime' => $coachTime,
            'nextAppointment' => $nextAppointment
        ]);
    }

    public function actionBookings()
    {
        $start = date('Y-m-d', strtotime(Yii::$app->request->getQueryParam('start', date('Y-m-d'))));
        $end   = date('Y-m-d', strtotime(Yii::$app->request->getQueryParam('end', date('Y-m-d'))));
        $startTime = $start . ' 00:00:00';
        $endTime   = $end . ' 23:59:59';
        $timezone = $this->user->getTimeZone();

        $result = TimeSlotBooking::getUserBookings($this->user->id, $startTime, $endTime, $timezone);

        Yii::$app->response->format = Response::FORMAT_JSON;
        return $result;
    }

    public function actionEvents()
    {
        $start = date('Y-m-d', strtotime(Yii::$app->request->getQueryParam('start', date('Y-m-d'))));
        $end   = date('Y-m-d', strtotime(Yii::$app->request->getQueryParam('end', date('Y-m-d'))));
        $startTime = $start . ' 00:00:00';
        $endTime   = $end . ' 23:59:59';
        $timezone = $this->user->getTimeZone();
        $coach = $this->user->getCoach();

        $result = [];
        // coach time slots
        $coachEvents = TimeSlotItem::find()
            ->joinWith('timeSlot')
            ->where(['and',
                ['coach_id' => $coach->id],
                ['between', 'start_at', $startTime, $endTime],
                ['between', 'end_at', $startTime, $endTime]
            ])
            ->all();
        /** @var TimeSlotItem $event */
        foreach ($coachEvents as $event) {
            $result[] = [
                'id' => 'c'.$event->id,
                'allDay' => false, // TODO: check for an all-day event
                'start' => Formatter::utcDatetimeToLocalDatetime($event->start_at, $timezone->code, 'c'),
                'end' => Formatter::utcDatetimeToLocalDatetime($event->end_at, $timezone->code, 'c'),
                'rendering' => 'background',
                'color' => '#7f66b3',
                'editable' => false
            ];
        }

        // this user events
        $result = array_merge($result, TimeSlotBooking::getUserBookings(
            $this->user->id,
            $startTime,
            $endTime,
            $timezone,
            true
        ));

        // other users events
        $result = array_merge($result, TimeSlotBooking::getUserBookings(
            $this->user->id,
            $startTime,
            $endTime,
            $timezone,
            true,
            $coach->id
        ));

        Yii::$app->response->format = Response::FORMAT_JSON;
        return $result;
    }

    public function actionAddBooking()
    {
        $id = Yii::$app->request->getQueryParam('id');
        $start = (int)Yii::$app->request->getQueryParam('start') / 1000;
        $booking = new AddTimeSlotBooking();
        $coach = $this->user->getCoach();
        $timezone = $this->user->getTimeZone();

        if ($id) {
            $slotBooking = TimeSlotBooking::findOne($id);
            if (!$slotBooking) {
                echo JsonTools::errorMessage('Incorrect booking ID provided.');
                die();
            }
        }

        if (Yii::$app->request->isPost && $booking->load(Yii::$app->request->post()) && $booking->validate()) {
            $booking->updateEndTime();
            $timeSlot = $booking->getMatchingTimeSlotItem($coach->id);
            if (!$timeSlot) {
                echo JsonTools::errorMessage('The appointment must fit within the availability provided by the coach.');
                die();
            }
            if ($booking->checkIfOverlays($this->user->id, $coach->id)) {
                echo JsonTools::errorMessage('The appointment must fit between other appointments.');
                die();
            }

            if ($id) {
                $slotBooking->setAttributes([
                    'time_slot_id' => $timeSlot->time_slot_id,
                    'time_slot_item_id' => $timeSlot->id,
                    'start_at' => Formatter::localDatetimeToUtcDatetime($booking->startDate . ' ' . $booking->startTime . ':00', 'Y-m-d H:i:s'),
                    'end_at' => Formatter::localDatetimeToUtcDatetime($booking->endDate . ' ' . $booking->endTime . ':00', 'Y-m-d H:i:s')
                ]);
                if ($slotBooking->save()) {
                    echo JsonTools::successMessage('Appointment successfully updated.');
                } else {
                    echo JsonTools::errorMessage('Could not update the appointment.');
                }
                die();
            }
            else {
                if (!$this->user->canBookAppointment()) {
                    echo JsonTools::errorMessage('You already have a booked appointment. You cannot book any new appointments.');
                    die();
                }
                $slotBooking = new TimeSlotBooking();
                $slotBooking->setAttributes([
                    'time_slot_id' => $timeSlot->time_slot_id,
                    'time_slot_item_id' => $timeSlot->id,
                    'user_id' => $this->user->id,
                    'start_at' => Formatter::localDatetimeToUtcDatetime($booking->startDate . ' ' . $booking->startTime . ':00', 'Y-m-d H:i:s'),
                    'end_at' => Formatter::localDatetimeToUtcDatetime($booking->endDate . ' ' . $booking->endTime . ':00', 'Y-m-d H:i:s')
                ]);
                if ($slotBooking->save()) {
                    // sending notification
                    $slotBooking->sendCreatedNotification($coach, $coach->getTimeZone());
                    echo JsonTools::successMessage('Appointment booked successfully.');
                } else {
                    echo JsonTools::errorMessage('Could not book the appointment.');
                }
                die();
            }
        }

        if ($id) {
            $booking->loadFromDb($slotBooking, $timezone);
        }
        else {
            $booking->startDate = Formatter::localDatetimeToUtcDatetime($start, 'Y-m-d');
            $booking->startTime = Formatter::localDatetimeToUtcDatetime($start, 'H:i');
            $booking->updateEndTime();
        }

        $this->view->title = ($id ? 'Edit' : 'Add') . ' Booking';
        return $this->render('add-booking', [
            'edit' => $id,
            'formModel' => $booking,
            'model' => $slotBooking
            //'engine' => $engine
        ]);
    }

    public function actionCancelBooking()
    {
        $id = Yii::$app->request->getQueryParam('id');
        $confirm = Yii::$app->request->getQueryParam('confirm');
        $timezone = $this->user->getTimeZone();
        $coach = $this->user->getCoach();

        $booking = TimeSlotBooking::findOne($id);
        if (!$booking || !$id) {
            echo JsonTools::errorMessage('Incorrect booking ID provided.');
            die();
        }
        if (!$booking->canCancelBooking()) {
            echo JsonTools::errorMessage('Cannot cancel this appointment anymore. Please contact your coach directly.');
            die();
        }

        if ($confirm) {
            // send notifications
            $booking->sendCancelledNotification($coach, $coach->getTimeZone());
            $booking->delete();
            echo JsonTools::successMessage('Selected appointment has been cancelled successfully.');
            die();
        }

        $this->view->title = 'Confirm Cancellation';
        return $this->render('confirm-cancel', [
            'booking' => $booking,
            'bookingDate' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($booking->start_at, $timezone->code,'Y-m-d H:i:s')),
        ]);
    }

}
