<?php

namespace backend\controllers;

use backend\components\helpers\Formatter;
use backend\models\db\UserRole;
use Yii;

use yii\filters\AccessControl;

use yii\helpers\Url;

use backend\components\CustomController;

use backend\models\db\Conversation;
use backend\models\db\ConversationMessage;
use backend\models\db\ConversationMessageSeen;
use backend\models\db\ConversationToUser;
use backend\models\db\User;

use backend\components\helpers\JsonTools;


/**
 * Conversation controller
 */
class ConversationsController extends CustomController {

    public $enableCsrfValidation = false;
    public $messagingUserId;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'add-conversation-message',
                            'get-conversation-messages-html',
                            'get-conversations-list-data',
                            'get-messaging-updates-data',
                            'mark-as-read',
                            'index'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];

    }

    public function beforeAction($action)
    {
        parent::beforeAction($action);

        $this->messagingUserId = $this->userIdentity->id; //Yii::$app->user->identity->id;
        return true;
    }

    public function actionIndex()
    {
        $user = User::findOne($this->messagingUserId);
        $conversation = $user->getMainConversation();
        $messages = $conversation->getMessages($user->id, 25, false, true);
        $conversation->markAsRead($user->id, $messages);
        $messagesData = $conversation->getData();

        $this->view->title = 'Messages';
        return $this->render('index', [
            'userId' => $user->id,
            'timezone' => $user->getTimeZone(),
            'conversation' => $conversation,
            'messages' => $messages,
            'messagesData' => $messagesData,
        ]);
    }

    /**
     * Add a message to a conversation,
     */
    public function  actionAddConversationMessage()
    {
        $conversationId = Yii::$app->getRequest()->post('conversation_id', false);
        $messageText = Yii::$app->getRequest()->post('message', false);
        if (!is_numeric($conversationId) || !$messageText || !strlen($messageText)) {
            echo JsonTools::errorMessage('An error has occured (1).');
            die();
        }

        // look for conversation
        /** @var Conversation $conversation */
        $conversation = $this->_getConversationByIdAndUserId($conversationId, $this->mainUserId);
        if (!$conversation) {
            echo JsonTools::errorMessage('An error has occured (2).');
            die();
        }
        else {
            $message = $conversation->addMessage($this->messagingUserId, $messageText);
            if (isset($message->id)) {
                echo json_encode([
                    'status' => 'success',
                    'id' => $message->id,
                    'avatar' => $message->user->getAvatarUrl(),
                    'name' => $message->user->getFullName(true),
                    'date' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($message->time, $this->user->getTimeZone()->code, 'Y-m-d H:i:s')),
                    'message' => $message->text,
                    'conversationId' => $conversation->id
                ]);
                die();
            }
            else {
                echo JsonTools::errorMessage('An error has occurred while submitting the message.');
                die();
            }
        }
    }


    /**
     * Get the HTML output of messages for a conversation.
     */
    public function actionGetConversationMessagesHtml()
    {
        $conversationId = Yii::$app->getRequest()->post('conversation_id', false);
        $lastMessageId = Yii::$app->getRequest()->post('last_message_id', false);
        $initLoad = Yii::$app->getRequest()->post('init_load', false);

        if (!is_numeric($conversationId) || !$conversationId) {

            echo JsonTools::errorMessage('An error has occured (3).');
            die();

        }

        // look for convo
        $convo = $this->_getConversationByIdAndUserId($conversationId, $this->mainUserId);

        if (!$convo) {

            echo JsonTools::errorMessage('An error has occured.');
            die();

        }
        else {

            // mark last message as read first
            $lastMessage = $convo->getLastReceivedMessage($this->messagingUserId);

            if ($lastMessage) {

                $seen = ConversationMessageSeen::findOne(['user_id' => $this->messagingUserId, 'conversation_message_id' => $lastMessage->id]);

                if (!$seen) {

                    $seen = new ConversationMessageSeen();

                }

                $seen->setAttributes([
                    'user_id' => $this->messagingUserId,
                    'conversation_message_id' => $lastMessage->id
                ]);
                $seen->save();

            }

            echo json_encode([
                'html' => $convo->getConversationMessagesHtml($this->messagingUserId, 25, $lastMessageId, $initLoad),
                'status' => 'success'
            ]);
            die();
        }
    }

    /**
     * Get data for refreshing conversations info.
     */
    public function actionGetConversationsListData()
    {
        if (Yii::$app->user->isCoach || Yii::$app->user->isSuperadmin) {
            $conversationId = Yii::$app->getRequest()->post('conversation_id', false);

            $user = User::findOne($this->userIdentity->id);
            $conversations = $user->getClientConversations($this->messagingUserId);

            $conversationsData = [];
            foreach ($conversations as $c) {
                $conversationsData[$c->id] = [
                    'unreadMessagesCount' => ($c->id != $conversationId) ? $c->getUnreadMessagesCount($this->messagingUserId) : 0
                ];
            }

            echo json_encode([
                'data' => $conversationsData,
                'status' => 'success'
            ]);
            die();
        }
    }

    /**
     * Get data for header messaging notifications and updates.
     */
    public function actionGetMessagingUpdatesData()
    {
        $unreadMessagesCount = 0;
        $data = [];
        // get all conversations for this user
        $user = User::findOne($this->messagingUserId);
        if ($user) {
            $timezone = $user->getTimeZone();
            $c = $user->getMainConversation();
            if ($c) {
                $unreadMessagesCount = $c->getUnreadMessagesCount($this->messagingUserId);
                foreach ($c->getMessages($this->messagingUserId, 3, false, true) as $message) {
                    $muser = $message->user;
                    $data[] = [
                        'text' => $message->text,
                        'name' => $muser->getFullName(true),
                        'date' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($message->time, $timezone->code, 'Y-m-d H:i:s')),
                        'avatar' => $muser->getAvatarUrl(),
                        'conversationId' => $c->id,
                        'messageId' => $message->id,
                        'url' => $this->_getConversationUrl($c->name, $c->id, Yii::$app->user),
                        //'markAsReadUrl' => Url::toRoute(['mark-as-read', 'message_id' => $message->id]),
                        'isBudgetConversation' => (substr($c->name, 0, 7) == 'budget_')
                    ];
                }
            }
        }

        echo json_encode([
            'data' => $data,
            'unread' => (int)$unreadMessagesCount,
            'status' => 'success'
        ]);
        die();
    }

    /**
     * Mark a message as read
     */
    public function actionMarkAsRead()
    {
        $messageId = Yii::$app->getRequest()->getQueryParam('message_id', false);
        if ($messageId) {
            $message = ConversationMessage::findOne($messageId);
            if (!$message) {
                echo JsonTools::errorMessage('An error occured.');
                die();
            }
            $convoToUser = ConversationToUser::findOne(['conversation_id' => $message->conversation_id, 'user_id' => $this->messagingUserId]);
            if ($convoToUser) {
                $seen = new ConversationMessageSeen();
                $seen->setAttributes([
                    'user_id' => $this->messagingUserId,
                    'conversation_message_id' => $messageId
                ]);
                $seen->save();
            }
            echo JsonTools::successMessage('Message marked as read.');
            die();
        }
        else {
            echo JsonTools::errorMessage('An error occured.');
            die();
        }
    }

    /**
     * Get a conversation based on conversation ID and user ID.
     * @param integer $conversationId The ID of the conversatoin.
     * @param integer $userId The ID of the user.
     * @return object Returns the conversation object.
     */
    private function _getConversationByIdAndUserId($conversationId, $userId)
    {
        $user = User::findOne($userId);
        // if it's a coach, make sure it's been added to the conversation
        if ($user->user_role_id == UserRole::COACH) {
            $isAdded = ConversationToUser::find()
                ->where([
                    'conversation_id' => $conversationId,
                    'user_id' => $userId
                ])
                ->exists();
            if (!$isAdded) {
                $ctu = new ConversationToUser();
                $ctu->setAttributes([
                    'conversation_id' => $conversationId,
                    'user_id' => $userId
                ]);
                $ctu->save();
            }
        }
        $convoQuery = Conversation::find()
            ->joinWith('conversationsToUsers')
            ->where(['id' => $conversationId]);

        if ($user->user_role_id != 1) {
            $convoQuery->andWhere(['user_id' => $userId]);
        }
        $convo = $convoQuery->one();
        return $convo;
    }

    /**
     * Return the conversation's URL.
     * @param string $name Name of the conversation logged in DB.
     * @param integer $conversationId ID of the conversation from DB.
     * @param object $user The user object.
     * @return string URL to conversation.
     */
    private function _getConversationUrl($name, $conversationId, $user)
    {
        if ($name == 'main') {
            if (!$user->isSuperadmin && !$user->isCoach) {
                return Url::toRoute('/coach');
            }
            else {
                return Url::toRoute(['/coach', 'messages_user_id' => $conversationId]);
            }
        }
        else {
            // gotta parse the name now
            if (substr($name, 0, 7) == 'budget_') {
                return Url::toRoute(['/mock-budgets/budget-detail', 'id' => substr($name, 7)]);
            }
        }
    }
}
