<?php

namespace backend\controllers;

use backend\components\helpers\Calculator;
use backend\components\helpers\DripHelper;
use backend\models\db\Budget;
use backend\models\db\Income;
use Yii;

use yii\filters\AccessControl;
use yii\helpers\Url;

use yii\helpers\ArrayHelper;

use backend\components\CustomController;

use backend\models\db\Account;
use backend\models\db\Debt;
use backend\models\db\DebtAmount;
use backend\models\db\DebtEnd;
use backend\models\db\DebtType;
use backend\models\db\DebtPayment;
use backend\models\db\Eoms;
use backend\models\db\Jar;

use backend\models\form\AddDebt;

use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;
use backend\components\helpers\Frequencies;
use backend\components\helpers\JsonTools;

/**
 * Debts controller
 */
class DebtsController extends CustomController {

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'add-debt',
                            'edit-debt',
                            'debt-detail',
                            'debts-table',
                            'debts-table-full',
                            'end-of-month-debt-amounts-table',
                            'debts-table-info',
                            'add-debt-payment',
                            'pause-debt',
                            'continue-debt',
                            'remove-debt'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];

    }

    public function beforeAction($action)
    {
        // for mock budgets navigation
        if ($action->id == 'debt-detail') {
            $debt = Debt::findOne(['id' => Yii::$app->request->getQueryParam('id')]);
            if ($debt && $debt->budget->is_mock) {
                $this->mockBudgetId = $debt->budget_id;
                $this->view->params['mockBudgetId'] = $debt->budget_id;
            }
        }
        elseif ($budgetId = Yii::$app->request->getQueryParam('budget_id')) {
            $budget = Budget::findOne($budgetId);
            if ($budget->is_mock) {
                $this->mockBudgetId = $budget->id;
                $this->view->params['mockBudgetId'] = $budget->id;
            }
        }
        return parent::beforeAction($action);
    }

    /**
     * Main debts page
     */
    public function actionIndex()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        if ($budgetId == $this->mockBudgetId) {
            $this->mockBudgetId = $budgetId;
            $budget = Budget::findOne($budgetId);
        }
        $currentBudget = Budget::findOne($this->activeBudgetId);

        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));

        $debts = $this->_getDebtsMenuList();
        $totalDebtAmount = 0;
        $totalDebtPaid = 0;
        foreach ($debts as $d) {
            $totalDebtAmount += $d->amount;
            $totalDebtPaid += $d->getDebtPaymentsAmount();
        }

        // pass mock budget specific data
        $mockBudgetData = [];
        if (isset($budget)) {
            $jar = $budget->getDebtsJar();
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['budgetId'] = $budget->id;
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['isMockBudget'] = true;
            $firstDebtPaymentDate = $budget->getFirstDebtPaymentDate();
            $lastDebtPaymentDate  = $budget->getLastDebtPaymentDate();
        }
        else {
            $jar = $currentBudget->getDebtsJar();
            $firstDebtPaymentDate = $currentBudget->getFirstDebtPaymentDate();
            $lastDebtPaymentDate  = $currentBudget->getLastDebtPaymentDate();
        }

        $yearsList = Calculator::getYearsListFromTo(
            '/debts/index',
            date('Y', strtotime($firstDebtPaymentDate)),
            date('Y', strtotime($lastDebtPaymentDate)),
            $year
        );
        $monthsList = Calculator::getMonthsListFromTo(
            '/debts/index',
            date('m', strtotime($firstDebtPaymentDate)),
            date('Y', strtotime($firstDebtPaymentDate)),
            date('m', strtotime($lastDebtPaymentDate)),
            date('Y', strtotime($lastDebtPaymentDate)),
            $month,
            $year
        );

        $this->view->title = 'Debts';
        return $this->render('index', $mockBudgetData + [
            'budget' => isset($budget) ? $budget : $currentBudget,
            'jar' => $jar,
            'debts' => $debts,
            'totalDebtAmount' => $totalDebtAmount,
            'totalDebtPaid' => $totalDebtPaid,
            'allowEdit' => true,
            'months' => $monthsList,
            'years'  => $yearsList,
            'year'   => $year,
            'month'  => $month,
            'monthName' => date('F', strtotime('2010-' . $month . '-1'))
        ]);
    }

    /**
     * Add a new debt
     */
    public function actionAddDebt()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        $this->mockBudgetId = $budgetId;

        $debtForm = new AddDebt();
        if (Yii::$app->request->isPost && $debtForm->load(Yii::$app->request->post())) {
            $debtForm->budget_id = $budgetId;
            // set validation attributes
            if ($debtForm->validate()) {
                $formData = $debtForm->getAttributes();
                $accountTypeId = $debtForm->debt_type_id == 1 ? 2 : 3;
                $amountSign = $accountTypeId == 2 ? -1 : 1;

                // need to create a related account
                $debtAccount = new Account();
                $debtAccount->setAttributes([
                    'budget_id' => $debtForm->budget_id,
                    'name' => $debtForm->name,
                    'balance' => $amountSign * $debtForm->amount, // assume that the debt amount is what the user can spend
                    'account_type_id' => $accountTypeId,
                    'archived' => $debtForm->is_debt_active ? 0 : 1
                ]);
                $debtAccount->save();

                $formData['active_debt_account_id'] = $debtAccount->id;
                $formData['archived'] = 0;
                if (!$formData['has_interest_free_period']) {
                    unset($formData['interest_free_end_date']);
                }
                Debt::createDebtFromForm($formData, $this->userMeta['fts_step'] != 'completed');

                DripHelper::sendCustomEvent('Debt created', $this->user->email, [
                    'name' => $formData['name']
                ]);

                if (Yii::$app->request->isAjax) {
                    if ($debtForm->is_debt_active) {
                        echo JsonTools::successMessage(
                            ($debtForm->debt_type_id == 1 ? 'Credit card' : 'Debt') . ' and related account successfully created.'
                        );
                    }
                    else {
                        echo JsonTools::successMessage('Debt and related account successfully created.');
                    }
                    die();
                }

            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($debtForm);
                die();
            }
        }
        else {
            // get potential JS callback info from query
            $frequencies = Frequencies::getAll();
            $this->view->title = 'Add Debt';
            $this->jsCallbackToView();

            return $this->render('add-debt', [
                'budgetId' => $budgetId,
                'formModel' => $debtForm,
                'frequencies' => $frequencies,
                'accounts' => Account::getAccountsListForSelect($budgetId),
                'selectedAccount' => $this->defaultAccountId,
                'debtTypes' => DebtType::getListForSelect()
            ]);
        }
    }


    /**
     * Edit a debt
     */
    public function actionEditDebt()
    {
        $debtId = Yii::$app->getRequest()->getQueryParam('id', false);
        if (!is_numeric($debtId)) {
            return $this->accessError();
        }
        // get the debt
        $debt = Debt::findOne(['id' => $debtId]);
        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $debt->budget_id])) {
            return $this->accessError();
        }
        // check archived
        if ($debt->archived) {
            echo JsonTools::errorMessage('This debt can no longer be edited.');
            die();
        }

        $debtForm = new AddDebt();
        if ($debt->getActiveDebtAccount()->exists() && !$debt->getActiveDebtAccount()->one()->archived) {
            $debtForm->is_debt_active = 1;
        }
        if (Yii::$app->request->isPost && $debtForm->load(Yii::$app->request->post())) {
            $debtForm->budget_id = $debt->budget_id;
            $debtForm->debt_id = $debt->id;

            // set validation attributes
            if ($debtForm->validate()) {
                $accountTypeId = $debtForm->debt_type_id == 1 ? 2 : 3;
                $debtAccount = $debt->getActiveDebtAccount()->where([
                    'account_type_id' => [2, 3]
                ])->one();
                // always need to have the associated debt account
                if (!$debtAccount) {
                    $amountSign = $accountTypeId == 2 ? -1 : 1;
                    $debtAccount = new Account();
                    $debtAccount->setAttributes([
                        'budget_id' => $debtForm->budget_id,
                        'name' => $debtForm->name,
                        'balance' => $amountSign * $debtForm->amount, // set the balance only when creating the account record
                        'account_type_id' => $accountTypeId,
                        'archived' => 0
                    ]);
                    $debtAccount->save();
                }
                // update the related account record
                $debtAccount->setAttributes([
                    'budget_id' => $debtForm->budget_id,
                    'name' => $debtForm->name,
                    'account_type_id' => $accountTypeId,
                    'archived' => $debtForm->is_debt_active ? 0 : 1
                ]);
                $debtAccount->save();
                $debt->active_debt_account_id = $debtAccount->id;

                // update debt record
                $formData = $debtForm->getAttributes();
                if (!$formData['has_interest_free_period']) {
                    $formData['interest_free_end_date'] = null;
                }
                $debt->setAttributes($formData);
                $debt->save();

                if (Yii::$app->request->isAjax) {
                    if ($debtForm->is_debt_active) {
                        echo JsonTools::successMessage(
                            ($debtForm->debt_type_id == 1 ? 'Credit card' : 'Debt') . ' successfully activated.'
                        );
                    }
                    else {
                        echo JsonTools::successMessage('Debt successfully edited.');
                    }
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($debtForm);
                die();
            }
        }
        else {
            $debtForm->setAttributes($debt->getAttributes());
            if ($debt->interest_free_end_date) {
                $debtForm->has_interest_free_period = 1;
            }
            $debtForm->next_due_date = $debt->getNextDue();
            $frequencies = Frequencies::getAll();
            $this->view->title = 'Edit Debt';
            $this->jsCallbackToView();

            return $this->render('edit-debt', [
                'formModel' => $debtForm,
                'debtId' => $debt->id,
                'frequencies' => $frequencies,
                'accounts' => ArrayHelper::map(Account::findAll(['budget_id' => $debt->budget_id, 'archived' => 0]), 'id', 'name'),
                'debtTypes' => DebtType::getListForSelect()
            ]);
        }
    }


    /**
     * Debt detail page
     */
    public function actionDebtDetail()
    {
        $debtId   = Yii::$app->getRequest()->getQueryParam('id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content
        if (!is_numeric($debtId)) {
            return $this->accessError();
        }
        $debt = Debt::findOne($debtId);
        if ($debt->archived) {
            return $this->redirect(['/debts/index']);
        }
        if ($debt->budget_id != $this->activeBudgetId) {
            return $this->accessError();
        }

        $totalDebtAmount = 0;
        $totalDebtPaid = 0;
        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));

        $this->view->title = $debt->name . ' debt';
        if ($this->pageLoadRequest || $fullLoad) {
            // pass mock budget specific data
            $mockBudgetData = [];
            if ($debt->budget->is_mock) {
                $budget = $debt->getBudget()->one();
                $mockBudgetData['budgetName'] = $budget->name;
                $mockBudgetData['budgetId'] = $budget->id;
                $mockBudgetData['budgetName'] = $budget->name;
                $mockBudgetData['isMockBudget'] = true;
                $firstDebtPaymentDate = $budget->getFirstDebtPaymentDate();
                $lastDebtPaymentDate = $budget->getLastDebtPaymentDate();
                $interestPayments = $budget->getThisMonthsDebtInterestPayments();
            }
            else {
                $currentBudget = Budget::findOne($this->activeBudgetId);
                $firstDebtPaymentDate = $currentBudget->getFirstDebtPaymentDate();
                $lastDebtPaymentDate = $currentBudget->getLastDebtPaymentDate();
                $interestPayments = $currentBudget->getThisMonthsDebtInterestPayments();
            }

            $debts = $this->_getDebtsMenuList();
            foreach ($debts as $d) {
                $totalDebtAmount += $d->amount;
                $totalDebtPaid += $d->getDebtPaymentsAmount();
            }

            $yearsList = Calculator::getYearsListFromTo(
                '/debts/index',
                date('Y', strtotime($firstDebtPaymentDate)),
                date('Y', strtotime($lastDebtPaymentDate)),
                $year
            );
            $monthsList = Calculator::getMonthsListFromTo(
                '/debts/index',
                date('m', strtotime($firstDebtPaymentDate)),
                date('Y', strtotime($firstDebtPaymentDate)),
                date('m', strtotime($lastDebtPaymentDate)),
                date('Y', strtotime($lastDebtPaymentDate)),
                $month,
                $year
            );

            return $this->render('debt-detail', $mockBudgetData + [
                'budget' => isset($budget) ? $budget : $currentBudget,
                'debt' => $debt,
                'debts' => $debts,
                'interestPayments' => $interestPayments,
                'allowEdit' => true,
                'fullLoad' => $fullLoad,
                'totalDebtAmount' => $totalDebtAmount,
                'totalDebtPaid' => $totalDebtPaid,
                'months' => $monthsList,
                'years'  => $yearsList,
                'year'   => $year,
                'month'  => $month,
                'monthName' => date('F', strtotime('2010-' . $month . '-1'))
            ]);
        }
        else {
            return $this->render('debt-detail-content', [
                'debt' => $debt,
                'allowEdit' => true,
                'fullLoad' => $fullLoad,
                'totalDebtAmount' => $totalDebtAmount,
                'totalDebtPaid' => $totalDebtPaid
            ]);
        }
    }


    /**
     * Table of all debts
     */
    public function actionDebtsTable()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        $mockBudgetMode = ($budgetId != $this->activeBudgetId);

        $jsCallbackAdd = Yii::$app->getRequest()->getQueryParam('js_callback_add', false);
        $jsCallbackPause = Yii::$app->getRequest()->getQueryParam('js_callback_pause', false);

        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => 'name'
            ],
            [
                'type' => 'method',
                'as' => 'debtTypeName',
                'name' => 'getDebtTypeName'
            ],
            [
                'type' => 'normal',
                'as' => 'interest',
                'select' => 'interest'
            ],
            [
                'type' => 'normal',
                'as' => 'interest_free_end_date',
                'select' => 'interest_free_end_date'
            ],
            [
                'type' => 'normal',
                'as' => 'payment',
                'select' => 'payment'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => 'amount'
            ],
            [
                'type' => 'method',
                'as' => 'debtPaidAmount',
                'name' => 'getDebtPaymentsAmount'
            ]
        ];

        $countFiltered = Debt::find()
            ->where(['budget_id' => $budgetId, 'archived' => '0']);

        $selectFiltered = Debt::find()
            ->select([
                Debt::tableName() . '.*',
                'end_date' =>  DebtEnd::tableName() . '.date'
            ])
            ->joinWith('debtEnd')
            ->where(['budget_id' => $budgetId, 'archived' => '0']);

        $responseData = DataTables::getResponseData($tableColumns, $countFiltered, $selectFiltered);

        /** @var Debt $i */
        foreach ($responseData['items'] as $i) {
            $pauseOrCont = ($i->end_date != null) ?
                '<a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['debts/continue-debt', 'id' => $i->id, 'js_callback' => (isset($jsCallbackPause)  && $jsCallbackPause) ? $jsCallbackPause : false]) . '"><i class="glyphicon glyphicon-play"></i>Resume</a>' :
                '<a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['debts/pause-debt', 'id' => $i->id, 'js_callback' => (isset($jsCallbackPause)  && $jsCallbackPause) ? $jsCallbackPause : false]) . '"><i class="glyphicon glyphicon-pause"></i>Pause</a>';

            if ($i->amount <= 0) {
                $bgSpan = '<span class="row-bg-green"></span>';
            }
            elseif ($i->end_date != null) {
                $bgSpan = '<span class="row-bg-red"></span>';
            }
            else {
                $bgSpan = '';
            }

            // show options based on mode
            if (!$mockBudgetMode) {
                $options = '
                <li>
                    <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['debt-payments/add-debt-payment', 'debt_id' => $i->id, 'js_callback' => (isset($jsCallbackAdd)  && $jsCallbackAdd) ? $jsCallbackAdd : false]) . '"><i class="glyphicon glyphicon-plus"></i>Add Debt Payment</a>
                </li>
                <li>
                    ' . $pauseOrCont . '
                </li>';
            }
            else {

                $options = '
                <li>
                    ' . $pauseOrCont . '
                </li>
                <li>
                    <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['debts/remove-debt', 'id' => $i->id, 'js_callback' => (isset($jsCallbackAdd)  && $jsCallbackAdd) ? $jsCallbackAdd : false]) . '"><i class="glyphicon glyphicon-trash"></i>Remove</a>
                </li>';

            }

            $responseData['response']['data'][] = [
                $i->name . $bgSpan,
                $i->getDebtType()->one()->name,
                (($i->interest) ? $i->interest . '%' : '0%'),
                (($i->interest_free_end_date) ? Formatter::date($i->interest_free_end_date) : '-'),
                Formatter::currency($i->getAverageMonthlyAmount()),
                Formatter::currency($i->amount),
                Formatter::currency($i->getDebtPaymentsAmount()),
                '<div class="text-center">
                    <a href="#" id="menu1" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="glyphicon glyphicon-cog gi-2x"></i></a>
                    <ul class="dropdown-menu" aria-labelledby="menu1">
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['debts/edit-debt', 'id' => $i->id, 'js_callback' => (isset($jsCallbackAdd)  && $jsCallbackAdd) ? $jsCallbackAdd : false]) . '"><i class="glyphicon glyphicon-pencil"></i>Edit</a>
                        </li>
                        ' . $options . '
                    </ul>
                </div>'
            ];
        }

        echo json_encode($responseData['response']);
        die();
    }

    /**
     * Debts table info
     */
    public function actionDebtsTableInfo() {

        $budgetId = $this->getBudgetIdFromUrl();

        $debts = $this->_getDebtsMenuList($budgetId);
        $totalDebtAmount = 0;
        $totalDebtPaid = 0;

        foreach ($debts as $d) {

            $totalDebtAmount += $d->amount;
            $totalDebtPaid += $d->getDebtPaymentsAmount();

        }

        echo json_encode([
            'total-debt-amount' => Formatter::currency($totalDebtAmount),
            'total-debt-amount-paid' => Formatter::currency($totalDebtPaid)
        ]);
        die();

    }


    /**
     * EoM Debt Amounts table
     */
    public function actionEndOfMonthDebtAmountsTable() {

        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        $year = Yii::$app->getRequest()->getQueryParam('year', false);

        if (is_numeric($month) && ($month > 0) && is_numeric($year)) {

            $debts = DebtAmount::find()
                ->select(DebtAmount::tableName() . '.amount, debt_id, ' . Debt::tableName() . '.name, ' . Eoms::tableName() . '.year, ' . Eoms::tableName() . '.month')
                ->joinWith('eoms')
                ->joinWith('debt')
                ->joinWith('debt.budget')
                ->leftJoin(DebtType::tableName(), Debt::tableName() . '.debt_type_id = ' . DebtType::tableName() . '.id')
                ->where([
                    'AND',
                    [Eoms::tableName() . '.budget_id' => $this->activeBudgetId],
                    ['not', ['like', DebtType::tableName() . '.name', 'loan']],
                    ['year' => $year],
                    ['month' => $month],
                    ['is_mock' => 0]
                ])
                ->all();

        }
        else {

            $debts = DebtAmount::find()
                ->select(DebtAmount::tableName() . '.amount, debt_id, ' . Debt::tableName() . '.name, ' . Eoms::tableName() . '.year, ' . Eoms::tableName() . '.month')
                ->joinWith('eoms')
                ->joinWith('debt')
                ->joinWith('debt.budget')
                ->leftJoin(DebtType::tableName(), Debt::tableName() . '.debt_type_id = ' . DebtType::tableName() . '.id')
                ->where([
                    'AND',
                    [Eoms::tableName() . '.budget_id' => $this->activeBudgetId],
                    ['not', ['like', DebtType::tableName() . '.name', 'loan']],
                    ['is_mock' => 0]
                ])
                ->all();

        }

        foreach ($debts as $i => $d) {
            $debts[$i]->amount -= $d->debt->getDebtPaymentsAmountUpUntil(date('Y-m-t', strtotime("{$d->year}-{$d->month}-1")));
        }

        $columnNames = [
            'name',
            'amount',
            'year',
            'month'
        ];

        $responseData = DataTables::getFilteredData($columnNames, $debts);

        foreach ($responseData['items'] as $i) {

            $responseData['response']['data'][] = [
                $i->name,
                Formatter::currency($i->amount),
                $i->year,
                date("F", strtotime("2010-" . $i->month . "-1"))
            ];

        }

        echo json_encode($responseData['response']);
        die();

    }


    /**
     * Pause a debt
     */
    public function actionPauseDebt() {

        $debtId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        // get the debt
        $debt = Debt::findOne(['id' => $debtId]);

        if (!is_numeric($debtId) || !Yii::$app->user->can('accessBudget', ['budgetId' => $debt->budget_id])) {

            return $this->accessError();

        }

        if ($confirm) {

            $debtEnd = new DebtEnd();
            $debtEnd->debt_id = $debt->id;
            $debtEnd->date = date('Y-m-d');
            $debtEnd->save();

            echo JsonTools::successMessage('Debt successfully paused.');
            die();

        }

        $this->view->title = 'Pause Debt';
        $this->jsCallbackToView();

        return $this->render('pause-debt', [
            'debtId' => $debt->id
        ]);

    }


    /**
     * Continue a debt
     */
    public function actionContinueDebt() {

        $debtId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        // get the debt
        $debt = Debt::findOne(['id' => $debtId]);

        if (!is_numeric($debtId) || !Yii::$app->user->can('accessBudget', ['budgetId' => $debt->budget_id])) {

            return $this->accessError();

        }

        if ($confirm) {

            $debtEnd = DebtEnd::findOne(['debt_id' => $debt->id]);
            $debtEnd->delete();

            echo JsonTools::successMessage('Debt successfully resumed.');
            die();

        }

        $this->view->title = 'Continue Debt';
        $this->jsCallbackToView();

        return $this->render('continue-debt', [
            'debtId' => $debt->id
        ]);

    }

    /**
     * Remove a debt
     */
    public function actionRemoveDebt()
    {
        $debtId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);
        $hardDelete = Yii::$app->getRequest()->getQueryParam('hard', false);
        if (!is_numeric($debtId)) {
            return $this->accessError();
        }
        // get the debt
        $debt = Debt::findOne(['id' => $debtId]);
        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $debt->budget_id])) {
            return $this->accessError();
        }
        $hardDeletable = Eoms::isEntryHardDeletable($this->activeBudgetId, $debt);

        if ($confirm) {
            // remove debt payments for one-time debt
            if ($debt->frequency == 'one-time') {
                DebtPayment::deleteAll([
                    'debt_id' => $debt->id,
                    'Year(date)' => date('Y'),
                    'Month(date)' => date('m')
                ]);

                // refund old debt payments as income
                $debtPayments = DebtPayment::findAll([
                    'debt_id' => $debt->id
                ]);

                $amount = 0;
                foreach ($debtPayments as $d) {
                    $amount += $d->amount;
                }

                $budget = $debt->getBudget()->one();
                $incomeData = [
                    'account_id' => $budget->default_account_id,
                    'budget_id' => $debt->budget_id,
                    'amount' => $amount,
                    'name' => 'Deleted debt "' . $debt->name . '" debt payments refund',
                    'date' => date('Y-m-d'),
                    'frequency' => 'one-time',
                    'archived' => 0
                ];

                $income = new Income();
                $income->setAttributes($incomeData);
                $income->save();
            }
            // need to delete the associated account as well
            $needToDeleteAccount = false;
            if ($debt->active_debt_account_id) {
                $debtAccount = Account::findOne($debt->active_debt_account_id);
                if ($debtAccount && !$debtAccount->archived) {
                    // redirect to remove account modal
                    $needToDeleteAccount = $debt->active_debt_account_id;
                }
            }
            if ($needToDeleteAccount) {
                echo JsonTools::redirectMessage(
                    Url::to(['/accounts/account-detail', 'id' => $needToDeleteAccount, 'delete' => 1]),
                    'Please now confirm deletion of the associated account first.'
                );
            }
            else {
                if (Yii::$app->user->hasFinishedFts()) {
                    if ($hardDelete && $hardDeletable) {
                        $debt->deleteCompletely();
                    }
                    else {
                        $debt->archived = 1;
                        $debt->save();
                    }
                }
                else {
                    $debt->deleteCompletely();
                }
                // if possible, just delete the account
                if ($debtAccount && $debtAccount->archived && $hardDelete && $hardDeletable) {
                    $debtAccount->deleteCompletely();
                }
                echo JsonTools::successMessage('Debt successfully removed.');
            }
            die();
        }

        $this->view->title = 'Remove Debt';
        $this->jsCallbackToView();
        return $this->render('remove-debt', [
            'debtId' => $debt->id,
            'hardDeletable' => $hardDeletable
        ]);
    }

    private function _getDebtsMenuList($budgetId = false)
    {
        if (!$budgetId) {
            $budgetId = $this->activeBudgetId;
        }
        return Debt::find()
            ->where(['budget_id' => $budgetId, 'archived' => '0'])
            ->orderBy('name ASC')
            ->all();
    }

}
