<?php

namespace backend\controllers;

use backend\models\db\Eoms;
use Yii;

use yii\caching\TagDependency;
use yii\filters\AccessControl;

use yii\helpers\ArrayHelper;
use yii\helpers\Url;

use backend\components\CustomController;

use backend\models\db\Account;
use backend\models\db\Budget;
use backend\models\db\CcTransaction;
use backend\models\db\Debt;
use backend\models\db\Expense;
use backend\models\db\ExpenseChange;
use backend\models\db\ExpenseDebt;
use backend\models\db\ExpenseEnd;
use backend\models\db\ExpenseStart;
use backend\models\db\Income;
use backend\models\db\IncomeTransaction;
use backend\models\db\Jar;
use backend\models\db\Transaction;

use backend\models\form\AddExpense;
use backend\models\form\RemoveEvent;

use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;
use backend\components\helpers\Frequencies;
use backend\components\helpers\JsonTools;


/**
 * Expense controller
 */
class ExpensesController extends CustomController {

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'add-recurring-expense',
                            'add-estimated-expense',
                            'add-onetime-expense',
                            'add-onetime-np-expense',
                            'edit-recurring-expense',
                            'edit-estimated-expense',
                            'edit-onetime-expense',
                            'pause-recurring-expense',
                            'continue-recurring-expense',
                            'recurring-expenses-table',
                            'estimated-expenses-table',
                            'onetime-expenses-table',
                            'expense-changes-table',
                            'expense-jar-recurring-expenses-table-info',
                            'expense-jar-estimated-expenses-table-info',
                            'expense-jar-onetime-expenses-table-info',
                            'remove-expense',
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];

    }


    /**
     * Expenses
     */
    public function actionIndex()
    {
        $type = Yii::$app->getRequest()->getQueryParam('type', 'one-time');
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }
        $jar = Jar::findOne($jarId);
        $this->mockBudgetId = $jar->getMockBudgetId();
        $expenses = ($type == 'one-time') ? $jar->getThisMonthsOnetimeExpenses() : $jar->getThisMonthsRecurringExpenses();

        $this->view->title = 'Expenses';
        if ($type == 'one-time') {
            $totalMonthlyAvg = $jar->getThisMonthsOnetimeExpensesAmount();
            // get other numbers for one time expenses
            $totalOneTimeAmount = 0;
            foreach ($expenses as $e) {
                $totalOneTimeAmount += $e->amount;
            }

            $totalOneTimeSavedAndPaid = 0;
            foreach ($expenses as $e) {
                $totalOneTimeSavedAndPaid += $e->getAmountAlreadySaved();
            }

            return $this->render('_onetime-expenses-table.php', [
                'expenses' => $expenses,
                'totalMonthlyAvg' => $totalMonthlyAvg,
                'totalOneTimeAmount' => $totalOneTimeAmount,
                'totalOneTimeSavedAndPaid' => $totalOneTimeSavedAndPaid,
                'allowEdit' => !$jar->archived
            ]);
        }
        else {
            $totalMonthlyAvg = $jar->getThisMonthsRecurringExpensesAmount();
            return $this->render('_recurring-expenses-table.php', [
                'expenses' => $expenses,
                'totalMonthlyAvg' => $totalMonthlyAvg,
                'allowEdit' => !$jar->archived
            ]);
        }
    }

    /**
     * Add recurring expense
     */
    public function actionAddRecurringExpense()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }
        $jar = Jar::findOne($jarId);
        $this->mockBudgetId = $jar->getMockBudgetId();

        $expenseForm = new AddExpense(); // this one's for validation, the others are separate since there are 2 forms
        if (Yii::$app->request->isPost && $expenseForm->load(Yii::$app->request->post())) {
            // set validation attributes
            $expenseForm->jar_id = $jarId;
            $expenseForm->budget_id = $this->activeBudgetId;
            $expenseForm['generate_transactions'] = 1;

            if ($expenseForm->validate() && !$jar->archived) {
                $expense = new Expense();
                $expense->setAttributes($expenseForm->getAttributes());
                $expense->debt_id = $expenseForm->debt_id;
                $expense->save();
                TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jar->id);

                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Expense successfully created.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($expenseForm);
                die();
            }
        }
        else {
            $this->view->title = 'Add Recurring Expense';
            $this->jsCallbackToView();

            $frequencies = Frequencies::getAll();
            unset($frequencies['one-time']);

            return $this->render('add-recurring-expense', [
                'formModel' => $expenseForm,
                'frequencies' => $frequencies,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'selectedAccount' => $this->defaultAccountId,
                'jarId' => $jarId
            ]);
        }
    }


    /**
     * Add estimated expense
     */
    public function actionAddEstimatedExpense()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }
        $jar = Jar::findOne($jarId);
        $this->mockBudgetId = $jar->getMockBudgetId();

        $expenseForm = new AddExpense();
        if (Yii::$app->request->isPost && $expenseForm->load(Yii::$app->request->post())) {
            // set validation attributes
            $expenseForm->jar_id = $jarId;
            $expenseForm->budget_id = $this->activeBudgetId;
            $expenseForm['generate_transactions'] = 0;

            if ($expenseForm->validate() && !$jar->archived) {
                $expense = new Expense();
                $expense->setAttributes($expenseForm->getAttributes());
                $expense->debt_id = $expenseForm->debt_id;
                // setting the correct date
                if (empty($expenseForm->date) || $expenseForm->date == '1970-01-01') {
                    $expense->date = date('Y-m-d');
                }
                $expense->save();
                TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jar->id);

                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Expense successfully created.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($expenseForm);
                die();
            }
        }
        else {
            $this->view->title = 'Add Regular Expense';
            $this->jsCallbackToView();

            $frequencies = Frequencies::getAll();
            unset($frequencies['one-time']);

            return $this->render('add-estimated-expense', [
                'formModel' => $expenseForm,
                'frequencies' => $frequencies,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'selectedAccount' => $this->defaultAccountId,
                'jarId' => $jarId
            ]);
        }
    }

    /**
     * Add one-time expense
     */
    public function actionAddOnetimeExpense()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }
        $jar = Jar::findOne($jarId);
        $this->mockBudgetId = $jar->getMockBudgetId();

        $expenseForm = new AddExpense();
        if (Yii::$app->request->isPost && $expenseForm->load(Yii::$app->request->post())) {
            // set validation attributes
            $expenseForm->jar_id = $jarId;
            $expenseForm->budget_id = $this->activeBudgetId;
            if ($expenseForm->validate() && !$jar->archived) {
                $formData = $expenseForm->getAttributes();
                $this->_saveNewOnetimeExpense($formData);
                TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jar->id);

                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Expense successfully created.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($expenseForm);
                die();
            }
        }
        else {
            $expenseForm['date'] = Formatter::date(time());
            $this->view->title = 'Add One-time Expense';
            $this->jsCallbackToView();

            return $this->render('add-onetime-expense', [
                'formModel' => $expenseForm,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'selectedAccount' => $this->defaultAccountId,
                'jarId' => $jarId
            ]);
        }
    }

    /**
     * Add one-time, non-partial expense
     */
    public function actionAddOnetimeNpExpense()
    {
        $expenseForm = new AddExpense();
        if (Yii::$app->request->isPost && $expenseForm->load(Yii::$app->request->post())) {
            // set validation attributes
            $expenseForm->budget_id = $this->activeBudgetId;

            if ($expenseForm->validate()) {
                $formData = $expenseForm->getAttributes();
                $expense = $this->_saveNewOnetimeExpense($formData);

                if (Yii::$app->request->isAjax) {
                    if (strtotime($formData['date']) > strtotime(date('Y-m-d'))) {
                        $jar = $expense->getJar()->one();
                        $link = '<a href="' . Url::toRoute(['jars/expense-jar-detail', 'id' => $jar->id]) . '" class="system-link">here</a>';
                        echo JsonTools::successMessage("You expense has been successfully created. Click $link to view this expense within the jar.");
                    }
                    else {
                        echo JsonTools::successMessage('Expense successfully created.');
                        die();
                    }
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($expenseForm);
                die();
            }
        }
        else {
            $this->view->title = 'Add One-time Expense';
            $this->jsCallbackToView();

            return $this->render('add-onetime-np-expense', [
                'formModel' => $expenseForm,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'jars' => Jar::getJarsForSelect($this->activeBudgetId)
            ]);
        }
    }

    /**
     * Edit recurring expense
     */
    public function actionEditRecurringExpense()
    {
        $expId = Yii::$app->getRequest()->getQueryParam('id', false);
        if (!is_numeric($expId)) {
            return $this->accessError();
        }
        $expense = Expense::findOne(['id' => $expId]);
        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $expense->jar_id])) {
            return $this->accessError();
        }
        $jar = Jar::findOne($expense->jar_id);
        $this->mockBudgetId = $jar->getMockBudgetId();

        $expenseForm = new AddExpense();
        if (Yii::$app->request->isPost && $expenseForm->load(Yii::$app->request->post())) {
            if ($jar->archived) {
                echo JsonTools::errorMessage("This expense can't be edited any longer.");
                die();
            }
            else {
                // set validation attributes
                $expenseForm->expense_id = $expense->id;
                $expenseForm->budget_id = $this->activeBudgetId;
                if ($expenseForm->validate()) {
                    $oldDate = $expense->date;
                    $expense->setAttributes($expenseForm->getAttributes());
                    $expense->debt_id = $expenseForm->debt_id;
                    // to not to overwrite old value with an empty date
                    if ($expenseForm->date == '1970-01-01' || empty($expenseForm->date)) {
                        $expense->date = $oldDate;
                    }
                    $expense->save();
                    TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jar->id);

                    if (Yii::$app->request->isAjax) {
                        echo JsonTools::successMessage('Expense successfully edited.');
                        die();
                    }
                }
                elseif (Yii::$app->request->isAjax) {
                    echo JsonTools::formErrorMessage($expenseForm);
                    die();
                }
            }
        }
        else {
            $expenseForm->setAttributes($expense->getAttributes());
            $expenseForm->account_id = ($expense->account_id != null) ? $expense->account_id : null;
            // next due date so the form can be validated correctly
            $expenseForm->date = $expense->getNextDue();

            $this->view->title = 'Edit Expense';
            $this->jsCallbackToView();

            $frequencies = Frequencies::getAll();
            unset($frequencies['one-time']);

            return $this->render('edit-recurring-expense', [
                'formModel' => $expenseForm,
                'frequencies' => $frequencies,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'expId' => $expense->id,
                'jars' => Jar::getJarsForSelect($this->activeBudgetId)
            ]);
        }
    }

    /**
     * Edit recurring expense
     */
    public function actionEditEstimatedExpense()
    {
        $expId = Yii::$app->getRequest()->getQueryParam('id', false);
        if (!is_numeric($expId)) {
            return $this->accessError();
        }
        $expense = Expense::findOne(['id' => $expId]);
        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $expense->jar_id])) {
            return $this->accessError();
        }
        $jar = Jar::findOne($expense->jar_id);
        $this->mockBudgetId = $jar->getMockBudgetId();

        $expenseForm = new AddExpense();
        if (Yii::$app->request->isPost && $expenseForm->load(Yii::$app->request->post())) {
            // set validation attributes
            $expenseForm->expense_id = $expense->id;
            $expenseForm->budget_id = $this->activeBudgetId;

            if ($jar->archived) {
                echo JsonTools::errorMessage("This expense can't be edited any longer.");
                die();
            }
            else {
                if ($expenseForm->validate()) {
                    $oldDate = $expense->date;
                    $expense->setAttributes($expenseForm->getAttributes());
                    $expense->debt_id = $expenseForm->debt_id;
                    // to not to overwrite old value with an empty date
                    if (empty($expense->date) || $expense->date == '1970-01-01') {
                        $expense->date = $oldDate;
                    }
                    $expense->save();
                    TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jar->id);

                    if (Yii::$app->request->isAjax) {
                        echo JsonTools::successMessage('Expense successfully edited.');
                        die();
                    }
                } elseif (Yii::$app->request->isAjax) {
                    echo JsonTools::formErrorMessage($expenseForm);
                    die();
                }
            }
        }
        else {
            $expenseForm->setAttributes($expense->getAttributes());
            $expenseForm->account_id = ($expense->account_id != null) ? $expense->account_id : null;
            // next due date so the form can be validated correctly
            $expenseForm->date = $expense->getNextDue();

            $this->view->title = 'Edit Expense';
            $this->jsCallbackToView();

            $frequencies = Frequencies::getAll();
            unset($frequencies['one-time']);

            return $this->render('edit-estimated-expense', [
                'formModel' => $expenseForm,
                'frequencies' => $frequencies,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'expId' => $expense->id,
                'jars' => Jar::getJarsForSelect($this->activeBudgetId)
            ]);
        }
    }

    /**
     * Edit one-time expense
     */
    public function actionEditOnetimeExpense()
    {
        $expId = Yii::$app->getRequest()->getQueryParam('id', false);
        if (!is_numeric($expId)) {
            return $this->accessError();
        }
        $expense = Expense::findOne(['id' => $expId]);
        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $expense->jar_id])) {
            return $this->accessError();
        }
        $jar = Jar::findOne($expense->jar_id);
        $this->mockBudgetId = $jar->getMockBudgetId();
        $info = Yii::$app->getRequest()->getQueryParam('info', false);

        $expenseForm = new AddExpense();
        if (Yii::$app->request->isPost && $expenseForm->load(Yii::$app->request->post())) {
            // set validation attributes
            $expenseForm->expense_id = $expense->id;
            $expenseForm->budget_id = $this->activeBudgetId;

            if ($jar->archived) {
                echo JsonTools::errorMessage("This expense can't be edited any longer.");
                die();
            }
            else {
                if ($expenseForm->validate()) {
                    $oldDate = $expense->date;
                    $expense->setAttributes($expenseForm->getAttributes());
                    $expense->frequency = 'one-time';
                    $expense->debt_id = $expenseForm->debt_id;
                    // to not to overwrite old value with an empty date
                    if ($expenseForm->date == '1970-01-01' || empty($expenseForm->date)) {
                        $expense->date = $oldDate;
                    }
                    $expense->save();
                    TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jar->id);

                    if (Yii::$app->request->isAjax) {
                        echo JsonTools::successMessage('Expense successfully edited.');
                        die();
                    }
                } elseif (Yii::$app->request->isAjax) {
                    echo JsonTools::formErrorMessage($expenseForm);
                    die();
                }
            }
        }
        else {
            $expenseForm->setAttributes($expense->getAttributes());
            $expenseForm->account_id = ($expense->account_id != null) ? $expense->account_id : null;

            $this->view->title = 'Edit Expense';
            $this->jsCallbackToView();
            return $this->render('edit-onetime-expense', [
                'info' => $info ? true : false,
                'formModel' => $expenseForm,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'expId' => $expense->id,
                'jars' => Jar::getJarsForSelect($this->activeBudgetId)
            ]);
        }
    }

    /**
     * Recurring expenses table data
     */
    public function actionRecurringExpensesTable()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        $allowEdit = Yii::$app->getRequest()->getQueryParam('allow_edit', false);
        $enhanced = Yii::$app->getRequest()->getQueryParam('enhanced', true);
        $jsCallback = Yii::$app->getRequest()->getQueryParam('js_callback', 'reloadExpenseJarRecurringExpenses');

        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }

        $expenseTableName = Expense::tableName();
        $year = date('Y');
        $month = date('m');

        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => $expenseTableName . '.name'
            ],
            [
                'type' => 'normal',
                'as' => 'frequency',
                'select' => $expenseTableName . '.frequency'
            ],
            [
                'type' => 'method',
                'as' => 'nextPayDay',
                'name' => 'getNextDue'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => $expenseTableName . '.amount'
            ],
            [
                'type' => 'method',
                'as' => 'monthlyAverage',
                'name' => 'getAverageMonthlyAmount'
            ]
        ];

        $jar = Jar::findOne($jarId);
        $countFiltered = $jar->getMonthsRecurringExpensesQuery($month, $year, true, false);
        $selectFiltered = $jar->getMonthsRecurringExpensesQuery($month, $year, true, false);

        $responseData = DataTables::getResponseData($tableColumns, $countFiltered, $selectFiltered);

        /** @var Expense $i */
        foreach ($responseData['items'] as $i) {
            if ($allowEdit && $enhanced) {
                $pauseOrCont = $i->removal_time ?
                    '<a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/continue-recurring-expense', 'id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-play"></i>Resume</a>' :
                    '<a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/pause-recurring-expense', 'id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-pause"></i>Pause</a>';

                $controlsHtml = '<div class="text-center">
                    <a href="#" id="menu1" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="glyphicon glyphicon-cog gi-2x"></i></a>
                    <ul class="dropdown-menu" aria-labelledby="menu1">
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/edit-recurring-expense', 'id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-pencil"></i>Edit</a>
                        </li>
                        <li>
                            ' . $pauseOrCont . '
                        </li>
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/remove-expense', 'id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-trash"></i>Remove</a>
                        </li>
                    </ul>
                </div>';
            }
            elseif ($allowEdit && !$enhanced) {
                $controlsHtml = '<div class="text-center">
                    <a href="#" id="menu1" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="glyphicon glyphicon-cog gi-2x"></i></a>
                    <ul class="dropdown-menu" aria-labelledby="menu1">
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/edit-recurring-expense', 'id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-pencil"></i>Edit</a>
                        </li>
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/remove-expense', 'id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-trash"></i>Remove</a>
                        </li>
                    </ul>
                </div>';
            }
            else {
                $controlsHtml = '';
            }
            $responseData['response']['data'][] = [
                $i->name . ($i->removal_time ? '<span class="row-bg-red"></span>' : ''),
                Formatter::currency($i->amount),
                Frequencies::getLabel($i->frequency),
                Formatter::currency($i->getAverageMonthlyAmount()),
                Formatter::date($i->getNextDue()),
                $controlsHtml
            ];
        }
        echo json_encode($responseData['response']);
        die();
    }


    /**
     * Estimated expenses table data
     */
    public function actionEstimatedExpensesTable() {

        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        $allowEdit = Yii::$app->getRequest()->getQueryParam('allow_edit', false);
        $enhanced = Yii::$app->getRequest()->getQueryParam('enhanced', true);
        $jsCallback = Yii::$app->getRequest()->getQueryParam('js_callback', 'reloadExpenseJarEstimatedExpenses');
        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }

        $expenseTableName = Expense::tableName();
        $year = date('Y');
        $month = date('m');

        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => $expenseTableName . '.name'
            ],
            [
                'type' => 'normal',
                'as' => 'frequency',
                'select' => $expenseTableName . '.frequency'
            ],
            [
                'type' => 'method',
                'as' => 'nextPayDay',
                'name' => 'getNextDue'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => $expenseTableName . '.amount'
            ],
            [
                'type' => 'method',
                'as' => 'monthlyAverage',
                'name' => 'getAverageMonthlyAmount'
            ]
        ];

        $jar = Jar::findOne($jarId);
        $countFiltered = $jar->getMonthsEstimatedExpensesQuery($month, $year, true, false);
        $selectFiltered = $jar->getMonthsEstimatedExpensesQuery($month, $year, true, false);

        $responseData = DataTables::getResponseData($tableColumns, $countFiltered, $selectFiltered);

        /** @var Expense $i */
        foreach ($responseData['items'] as $i) {
            if ($allowEdit && $enhanced) {
                $controlsHtml = '<div class="text-center">
                    <a href="#" id="menu1" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="glyphicon glyphicon-cog gi-2x"></i></a>
                    <ul class="dropdown-menu" aria-labelledby="menu1">
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['transactions/add-transaction', 'expense_id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-plus"></i>Add transaction</a>
                        </li>
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/edit-estimated-expense', 'id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-pencil"></i>Edit</a>
                        </li>
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/remove-expense', 'id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-trash"></i>Remove</a>
                        </li>
                    </ul>
                </div>';
            }
            elseif ($allowEdit && !$enhanced) {
                $controlsHtml = '<div class="text-center">
                    <a href="#" id="menu1" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="glyphicon glyphicon-cog gi-2x"></i></a>
                    <ul class="dropdown-menu" aria-labelledby="menu1">
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/edit-estimated-expense', 'id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-pencil"></i>Edit</a>
                        </li>
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/remove-expense', 'id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-trash"></i>Remove</a>
                        </li>
                    </ul>
                </div>';
            }
            else {
                $controlsHtml = '';
            }

            $responseData['response']['data'][] = [
                $i->name,
                Formatter::currency($i->amount),
                Frequencies::getLabel($i->frequency),
                Formatter::currency($i->getAverageMonthlyAmount()),
                Formatter::date($i->getNextDue()),
                $controlsHtml
            ];

        }

        echo json_encode($responseData['response']);
        die();
    }


    /**
     * One-time expenses table data
     */
    public function actionOnetimeExpensesTable()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        $allowEdit = Yii::$app->getRequest()->getQueryParam('allow_edit', false);
        $jsCallback = Yii::$app->getRequest()->getQueryParam('js_callback', 'reloadExpenseJarOnetimeExpenses');
        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }

        $expenseTableName = Expense::tableName();
        $year = date('Y');
        $month = date('m');

        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => $expenseTableName . '.name'
            ],
            [
                'type' => 'normal',
                'as' => 'date',
                'select' => $expenseTableName . '.date'
            ],
            [
                'type' => 'method',
                'as' => 'nextPaymentDate',
                'name' => 'getNextDue'
            ],
            [
                'type' => 'method',
                'as' => 'paymentsLeftUntil',
                'name' => 'getPaymentsLeftUntil'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => $expenseTableName . '.amount'
            ],
            [
                'type' => 'method',
                'as' => 'supposedlySavedSoFar',
                'name' => 'getSupposedlySavedSoFar'
            ],
            [
                'type' => 'method',
                'as' => 'monthlyAverage',
                'name' => 'getAverageMonthlyAmount'
            ]
        ];

        $today = date('Y-m-d');
        $jar = Jar::findOne($jarId);
        $filtered = $jar->getMonthsOnetimeExpensesQuery($month, $year, true, false);

        $responseData = DataTables::getResponseData($tableColumns, $filtered, $filtered);
        /** @var Expense $i */
        foreach ($responseData['items'] as $i) {
            if ($allowEdit) {
                if ($i->frequency == 'one-time' && $i->date < $today) {
                    $controlsHtml =
                        '<div class="text-center">
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/edit-onetime-expense', 'id' => $i->id, 'js_callback' => $jsCallback, 'info' => 1]) . '">
                                <i class="glyphicon glyphicon-ok-sign"></i>
                            </a>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/remove-expense', 'id' => $i->id, 'js_callback' => $jsCallback, 'info' => 1]) . '">
                                <i class="glyphicon glyphicon-remove-sign"></i>
                            </a>
                        </div>';
                }
                else {
                    $controlsHtml =
                        '<div class="text-center">
                            <a href="#" id="menu1" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="glyphicon glyphicon-cog gi-2x"></i></a>
                            <ul class="dropdown-menu" aria-labelledby="menu1">
                                <li>
                                    <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/edit-onetime-expense', 'id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-pencil"></i>Edit</a>
                                </li>
                                <li>
                                    <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['expenses/remove-expense', 'id' => $i->id, 'js_callback' => $jsCallback]) . '"><i class="glyphicon glyphicon-trash"></i>Remove</a>
                                </li>
                            </ul>
                        </div>';
                }
            }
            else {
                $controlsHtml = '';
            }

            if ($i->generate_transactions) {
                $responseData['response']['data'][] = [
                    $i->name . ($i->date < $today ? '<span class="row-bg-grey"></span>' : ''),
                    Formatter::currency($i->amount),
                    Formatter::date($i->date),
                    $i->getPaymentsLeftUntil(),
                    Formatter::currency($i->getSupposedlySavedSoFar()),
                    Formatter::currency($i->getAverageMonthlyAmount()),
                    $controlsHtml
                ];
            }
            else {
                $responseData['response']['data'][] = [
                    $i->name . ($i->date < $today ? '<span class="row-bg-grey"></span>' : ''),
                    Formatter::currency($i->amount),
                    Formatter::date($i->date),
                    '-',
                    '-',
                    '-',
                    $controlsHtml
                ];
            }
        }

        echo json_encode($responseData['response']);
        die();
    }


    /**
     * History of changes made to expenses table
     */
    public function actionExpenseChangesTable()
    {
        $budget = Budget::findOne($this->activeBudgetId);
        $changesForTable = $budget->getExpenseChangesForTable();
        $columnNames = [
            'time',
            'name',
            'parameter',
            'from',
            'to'
        ];
        $responseData = DataTables::getFilteredData($columnNames, $changesForTable);
        foreach ($responseData['items'] as $i) {
            switch (strtolower($i['parameter'])) {
                case 'jar_id':
                    $from = Jar::findOne($i['from'])->name;
                    $to = Jar::findOne($i['to'])->name;
                    break;
                default:
                    $from = $i['from'];
                    $to = $i['to'];
                    break;
            }
            $responseData['response']['data'][] = [
                Formatter::datetime($i['time']),
                $i['name'],
                $i['parameter'],
                $from,
                $to
            ];
        }
        echo json_encode($responseData['response']);
        die();
    }


    /**
     * Pause a expense
     */
    public function actionPauseRecurringExpense() {

        $expId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if (!is_numeric($expId)) {

            return $this->accessError();

        }

        $expense = Expense::findOne(['id' => $expId]);

        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $expense->jar_id])) {

            return $this->accessError();

        }

        if ($confirm && ($expense->frequency != 'one-time') && ($expense->generate_transactions == 1)) {

            $expenseEnd = new ExpenseEnd();
            $expenseEnd->expense_id = $expense->id;
            $expenseEnd->date = date('Y-m-d');
            $expenseEnd->save();

            echo JsonTools::successMessage('Expense successfully paused.');
            die();

        }

        $this->view->title = 'Pause Expense';

        return $this->render('pause-recurring-expense', [
            'expenseId' => $expense->id
        ]);

    }

    /**
     * Continue a expense
     */
    public function actionContinueRecurringExpense()
    {
        $expId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);
        if (!is_numeric($expId)) {
            return $this->accessError();
        }
        $expense = Expense::findOne(['id' => $expId]);
        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $expense->jar_id])) {
            return $this->accessError();
        }

        if ($confirm && ($expense->frequency != 'one-time') && ($expense->generate_transactions == 1)) {
            $expenseEnd = ExpenseEnd::findOne(['expense_id' => $expense->id]);
            $expenseEnd->delete();
            echo JsonTools::successMessage('Expense successfully resumed.');
            die();
        }

        $this->view->title = 'Continue Expense';
        return $this->render('continue-recurring-expense', [
            'expenseId' => $expense->id
        ]);
    }

    /**
     * Return expense jar recurring expenses table info
     */
    public function actionExpenseJarRecurringExpensesTableInfo() {

        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);

        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {

            return $this->accessError();

        }

        // get the jar
        $jar = Jar::findOne($jarId);

        echo json_encode([
            'this-months-recurring-exps-average' => Formatter::currency($jar->getThisMonthsRecurringExpensesAmount())
        ]);
        die();

    }


    /**
     * Return expense jar recurring expenses table info
     */
    public function actionExpenseJarEstimatedExpensesTableInfo() {

        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);

        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {

            return $this->accessError();

        }

        // get the jar
        $jar = Jar::findOne($jarId);

        echo json_encode([
            'this-months-estimated-exps-average' => Formatter::currency($jar->getThisMonthsEstimatedExpensesAmount())
        ]);
        die();

    }

    /**
     * Return expense jar one-time expenses table info
     */
    public function actionExpenseJarOnetimeExpensesTableInfo() {

        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);

        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {

            return $this->accessError();

        }

        // get the jar
        $jar = Jar::findOne($jarId);

        // total amount of one-time expenses
        $oneTimeExps = $jar->getThisMonthsOnetimeExpenses();
        $totalOneTimeAmount = 0;
        foreach ($oneTimeExps as $e) {

            $totalOneTimeAmount += $e->amount;

        }

        // total saved / paid amount
        $totalOneTimeSavedAndPaid = 0;
        foreach ($oneTimeExps as $e) {

            $totalOneTimeSavedAndPaid += $e->getAmountAlreadySaved();

        }

        echo json_encode([
            'this-months-onetime-exps-average' => Formatter::currency($jar->getThisMonthsOnetimeExpensesAmount()),
            'total-onetime-exps-amount' => Formatter::currency($totalOneTimeAmount),
            'total-onetime-saved-paid-amount' => Formatter::currency($totalOneTimeSavedAndPaid)
        ]);

        die();

    }


    /**
     * Remove an expense
     */
    public function actionRemoveExpense()
    {
        $expenseId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);
        $hardDelete = Yii::$app->getRequest()->getQueryParam('hard', false);
        $info = Yii::$app->getRequest()->getQueryParam('info', false);
        if (!is_numeric($expenseId)) {
            return $this->accessError();
        }
        // get jar
        $expense = Expense::findOne(['id' => $expenseId]);
        if (!$expense) {
            return $this->accessError();
        }
        $jar = $expense->getJar()->one();
        $this->mockBudgetId = $jar->getMockBudgetId();

        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
            return $this->accessError();
        }
        $hardDeletable = Eoms::isEntryHardDeletable($this->activeBudgetId, $expense);

        $removeForm = new RemoveEvent();
        if ($confirm) {
            $jarId = $expense->jar_id;
            if (Yii::$app->user->hasFinishedFts()) {
                if ($hardDelete && $hardDeletable) {
                    $expense->deleteCompletely();
                }
                else {
                    $expense->archived = 1;
                    $expense->save();
                }
            }
            else {
                $expense->deleteCompletely();
            }
            if ($jarId) {
                TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jarId);
            }
            echo JsonTools::successMessage('Expense successfully removed.');
            die();
        }
        else {
            $this->view->title = 'Remove Expense';
            $this->jsCallbackToView();
            return $this->render('remove-expense', [
                'info' => $info ? true : false,
                'expenseId' => $expense->id,
                'formModel' => $removeForm,
                'frequency' => $expense->frequency,
                'isRecurring' => $expense->isRecurring(),
                'hardDeletable' => $hardDeletable
            ]);
        }
    }

    /**
     * Save a new one-time expense.
     * @param array $formData Data from form.
     * @param bool $unbudgeted
     * @return Expense|bool false = error, else returns ID of entered expense
     */
    private function _saveNewOnetimeExpense($formData, $unbudgeted = false)
    {
        $expense = new Expense();
        $expense->setAttributes($formData);
        $expense->frequency = 'one-time';
        $expense->debt_id = $formData['debt_id'];
        $expense->is_unbudgeted = $unbudgeted ? 1 : 0;
        $saved = $expense->save();

        return $saved ? $expense : false;
    }

}
