<?php

namespace backend\controllers;

use backend\components\helpers\Calculator;
use backend\widgets\BudgetShortfallCalculator;
use Yii;

use yii\filters\AccessControl;

use backend\components\CustomController;

use backend\models\db\Budget;
use backend\models\db\Debt;
use backend\models\db\Jar;

use backend\models\form\CreateJar;
use backend\models\form\EditJar;

use yii\helpers\ArrayHelper;
use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;
use backend\components\helpers\JsonTools;
use yii\helpers\Url;


/**
 * Jars controller
 */
class JarsController extends CustomController
{
    public $enableCsrfValidation = false;

    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'create-expense-jar',
                            'edit-expense-jar',
                            'expense-jar-detail',
                            'expense-jar-header-info',
                            'debt-jar-detail',
                            'debt-jar-header-info',
                            'interest-jar-detail',
                            'jars-name-total-table',
                            'remove-expense-jar',
                            'restore-expense-jar',
                            'deleted-jars',
                            'deleted-expense-jar-detail',
                            'deleted-debt-jar-detail',
                            'reorder-jars',
                            'budget-shortfall-calculator'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];
    }

    public function beforeAction($action)
    {
        // for mock budgets navigation
        if (in_array($action->id, ['expense-jar-detail', 'interest-jar-detail', 'debt-jar-detail',
            'deleted-expense-jar-detail', 'deleted-debt-jar-detail'])) {
            $jar = Jar::findOne(['id' => Yii::$app->request->getQueryParam('id')]);
            if ($jar && $jar->budget->is_mock) {
                $this->mockBudgetId = $jar->budget_id;
                $this->view->params['mockBudgetId'] = $jar->budget_id;
            }
        }
        elseif ($budgetId = Yii::$app->request->getQueryParam('budget_id')) {
            $budget = Budget::findOne($budgetId);
            if ($budget->is_mock) {
                $this->mockBudgetId = $budget->id;
                $this->view->params['mockBudgetId'] = $budget->id;
            }
        }
        return parent::beforeAction($action);
    }

    public function actionBudgetShortfallCalculator()
    {
        $request = Yii::$app->request->post();

        return BudgetShortfallCalculator::widget([
            'budgetId' => $request['budgetId'],
            'jarId' => $request['jarId'],
            'callbackUrl' => Url::to(['/jars/budget-shortfall-calculator']),
            'amounts' => isset($request['amounts']) ? $request['amounts'] : []
        ]);
    }

    /**
     * Jars screen
     */
    public function actionIndex()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        $budget = Budget::findOne($budgetId);

        $jars = Jar::find()
            ->where(['budget_id' => $budgetId, 'archived' => 0])
            ->orderBy('order')
            ->all();
        $this->view->title = 'My Jars';

        // pass mock budget specific data
        $mockBudgetData = [];
        if ($budgetId == $this->mockBudgetId) {
            $this->mockBudgetId = $budgetId;
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['budgetId'] = $budget->id;
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['isMockBudget'] = true;
        }

        return $this->render('index', $mockBudgetData + [
            'budget' => $budget,
            'jars' => $jars,
            'allowEdit' => true
        ]);
    }

    /**
     * Create a new expense jar
     */
    public function actionCreateExpenseJar()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        $jarForm = new CreateJar();
        $this->mockBudgetId = $budgetId;

        if ($jarForm->load(Yii::$app->request->post())) {
            // set validation attributes
            $userMeta = Yii::$app->session->get('userMeta');
            $jarForm->budget_id = $budgetId;

            if ($jarForm->validate()) {

                $jar = new Jar();
                $jar->setAttributes([
                    'name' => $jarForm->name,
                    'budget_id' => $budgetId
                ]);
                $jar->save();

                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Jar successfully created.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($jarForm);
                die();
            }
        }
        else {
            $this->view->title = 'Create a Jar';
            $this->jsCallbackToView();

            return $this->render('create-expense-jar', [
                'budgetId' => $budgetId,
                'formModel' => $jarForm
            ]);
        }
    }

    /**
     * Edit an expense jar
     */
    public function actionEditExpenseJar()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('id', false);
        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }

        $jarForm = new EditJar();
        $jar = Jar::findOne(['id' => $jarId]);

        // check if not "Everyday" jar
        /*if ($jar->name == 'Everyday') {

            echo JsonTools::errorMessage('The "Everyday" jar can not be edited.');
            die();

        }*/

        if ($jarForm->load(Yii::$app->request->post())) {
            // set validation attributes
            $jarForm->budget_id = $this->userMeta['active_budget_id'];
            $jarForm->jar_id = $jarId;

            if ($jarForm->validate()) {
                // save jar
                $jar->name = $jarForm->name;
                $jar->save();
                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Jar successfully saved.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($jarForm);
                die();
            }
        }
        else {
            $jarForm->name = $jar->name;
        }

        $this->view->title = 'Edit Jar';
        $this->jsCallbackToView();


        return $this->render('edit-expense-jar', [
            'title' => $this->view->title,
            'formModel' => $jarForm,
            'jarId' => $jar->id
        ]);

    }


    /**
     * Show the detail of an expense jar
     */
    public function actionExpenseJarDetail()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content
        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }

        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));
        // get the jar
        $jar = Jar::findOne(['id' => $jarId]);

        // get one-time expenses
        $oneTimeExps = $jar->getThisMonthsOnetimeExpenses(true, false);

        // get total average monthly amounts
        $totalMonthlyEstimAvg = $jar->getEstimatedExpensesAmount(null, null, true, false);
        $totalMonthlyRecurrAvg = $jar->getRecurringExpensesAmount(null, null, true, false);
        $totalMonthlyOneTimeAvg = $jar->getOnetimeExpensesAmount(null, null, true, false);

        // get other numbers for one time expenses
        $totalOneTimeAmount = 0;
        foreach ($oneTimeExps as $e) {
            $totalOneTimeAmount += $e->amount;
        }

        $totalOneTimeSavedAndPaid = 0;
        foreach ($oneTimeExps as $e) {
            $totalOneTimeSavedAndPaid += $e->getAmountAlreadySaved();
        }

        // get all jars for sidebar
        $this->view->title = $jar->name . ' jar';

        // pass mock budget specific data
        $mockBudgetData = [];
        if ($jar->budget->is_mock) {
            $budget = $jar->getBudget()->one();
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['budgetId'] = $budget->id;
            $mockBudgetData['isMockBudget'] = true;
        }

        //$budget = Budget::findOne($this->activeBudgetId);
        $firstMoneyOperationDate = $jar->getFirstMoneyOperationDate();
        $lastMoneyOperationDate = $jar->getLastMoneyOperationDate();

        $yearsList = Calculator::getYearsListFromTo(
            $jar->jar_type_id == 1 ?
                ['/jars/expense-jar-detail', 'id' => $jarId] :
                ['/jars/debt-jar-detail', 'id' => $jarId],
            date('Y', strtotime($firstMoneyOperationDate)),
            date('Y', strtotime($lastMoneyOperationDate)),
            $year
        );
        $monthsList = Calculator::getMonthsListFromTo(
            $jar->jar_type_id == 1 ?
                ['/jars/expense-jar-detail', 'id' => $jarId] :
                ['/jars/debt-jar-detail', 'id' => $jarId],
            date('m', strtotime($firstMoneyOperationDate)),
            date('Y', strtotime($firstMoneyOperationDate)),
            date('m', strtotime($lastMoneyOperationDate)),
            date('Y', strtotime($lastMoneyOperationDate)),
            $month,
            $year
        );

        // switch view based on whether this is an initial page load
        if ($this->pageLoadRequest || $fullLoad) {
            $jars = Jar::find()
                ->where(['budget_id' => $jar->budget_id, 'archived' => $jar->archived])
                ->orderBy('order')
                ->all();

            $budget = $jar->getBudget()->one();
            return $this->render('expense-jar-detail', $mockBudgetData + [
                'budget' => $budget,
                'enhanced' => ($jar->budget_id == $this->activeBudgetId),
                'jar' => $jar,
                'jars' => $jars,
                'totalMonthlyRecurrAvg' => $totalMonthlyRecurrAvg,
                'totalMonthlyEstimAvg' => $totalMonthlyEstimAvg,
                'totalMonthlyOneTimeAvg' => $totalMonthlyOneTimeAvg,
                'totalOneTimeAmount' => $totalOneTimeAmount,
                'totalOneTimeSavedAndPaid' => $totalOneTimeSavedAndPaid,
                'allowEdit' => !$jar->archived,
                'months' => $monthsList,
                'years'  => $yearsList,
                'year'   => $year,
                'month'  => $month,
                'monthName' => date('F', strtotime('2010-' . $month . '-1'))
            ]);
        }
        else {
            return $this->render('expense-jar-detail-content', $mockBudgetData + [
                'enhanced' => ($jar->budget_id == $this->activeBudgetId),
                'jar' => $jar,
                'totalMonthlyRecurrAvg' => $totalMonthlyRecurrAvg,
                'totalMonthlyEstimAvg' => $totalMonthlyEstimAvg,
                'totalMonthlyOneTimeAvg' => $totalMonthlyOneTimeAvg,
                'totalOneTimeAmount' => $totalOneTimeAmount,
                'totalOneTimeSavedAndPaid' => $totalOneTimeSavedAndPaid,
                'allowEdit' => !$jar->archived,
                'months' => $monthsList,
                'years'  => $yearsList,
                'year'   => $year,
                'month'  => $month,
                'monthName' => date('F', strtotime('2010-' . $month . '-1'))
            ]);
        }
    }

    public function actionDeletedExpenseJarDetail()
    {
        return $this->actionExpenseJarDetail();
    }

    public function actionDeletedDebtJarDetail()
    {
        return $this->actionDebtJarDetail();
    }

    /**
     * Return expense jar header info
     */
    public function actionExpenseJarHeaderInfo()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('id', false);

        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }

        // get the jar
        $jar = Jar::findOne($jarId);

        echo json_encode([
            'jar-amount' => Formatter::currency($jar->getThisMonthsFundsBroughtForwardAmount()),
            'this-months-expenses-amount' => Formatter::currency($jar->getMonthlyBudget()),
            'this-months-transactions-amount' => Formatter::currency($jar->getThisMonthsTransactionsAmount()),
            'this-months-money-in' => Formatter::currency($jar->getThisMonthsMoneyInAmount()),
            'this-months-funds-remaining' => Formatter::currency($jar->getThisMonthsFundsRemainingAmount())
        ]);
        die();
    }


    /**
     * Restore an expense jar that has been removed
     */
    public function actionRestoreExpenseJar() {

        $jarId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {

            return $this->accessError();

        }

        // get the jar
        $jar = Jar::findOne(['id' => $jarId]);

        if ($confirm) {

            $jar->archived = 0;
            $jar->save();

            echo JsonTools::successMessage('Jar successfully restored.');
            die();

        }

        $this->view->title = 'Restore Jar';

        return $this->render('restore-expense-jar', [
            'jarId' => $jar->id
        ]);

    }


    public function actionJarsNameTotalTable() {

        $budgetId = $this->getBudgetIdFromUrl();
        $budget = Budget::findOne($budgetId);

        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'order',
                'select' => 'order'
            ],
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => 'name'
            ],
            [
                'type' => 'method',
                'as' => 'monthly_budget',
                'select' => 'getMonthlyBudget'
            ]

        ];


        $filtered = Jar::find()
            ->where(['budget_id' => $budgetId, 'archived' => 0]);

        $responseData = DataTables::getResponseData($tableColumns, $filtered, $filtered);

        foreach ($responseData['items'] as $i) {

            /** @var Jar $i  */
            $responseData['response']['data'][] = [
                $i->name,
                Formatter::currency($i->getMonthlyBudget()),
            ];

        }

        echo json_encode($responseData['response']);
        die();

    }


    /**
     * Remove an expense jar
     */
    public function actionRemoveExpenseJar() {

        $jarId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {

            return $this->accessError();

        }

        // get the jar
        $jar = Jar::findOne(['id' => $jarId]);

        // check if not "Everyday" jar
        if ($jar->type == Jar::EVERYDAY_JAR_TYPE) {

            echo JsonTools::errorMessage('The "'.Jar::EVERYDAY_JAR_NAME.'" jar can not be removed.');
            die();

        }

        if ($confirm) {

            $jar->archived = 1;
            $jar->save();

            echo JsonTools::successMessage('Jar successfully removed.');
            die();

        }

        $this->view->title = 'Remove Jar';
        $this->jsCallbackToView();

        return $this->render('remove-expense-jar', [
            'jarId' => $jar->id
        ]);

    }

    /**
     * Deleted jars screen
     */
    public function actionDeletedJars()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        $budget = Budget::findOne($budgetId);

        $jars = Jar::findAll(['budget_id' => $budgetId, 'archived' => 1]);
        $this->view->title = 'Deleted Jars';

        // pass mock budget specific data
        $mockBudgetData = [];
        if ($budgetId == $this->mockBudgetId) {
            $this->mockBudgetId = $budgetId;
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['budgetId'] = $budget->id;
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['isMockBudget'] = true;
        }

        return $this->render('index', $mockBudgetData + [
            'title' => $this->view->title,
            'jars' => $jars,
            'budget' => $budget,
            'allowEdit' => false
        ]);
    }

    /**
     * Detail of debt jar
     */
    public function actionDebtJarDetail()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content
        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }
        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month = Yii::$app->getRequest()->getQueryParam('month', date('m'));

        // get the jar & budget
        $jar = Jar::findOne(['id' => $jarId]);
        $this->mockBudgetId = $jar->getMockBudgetId();

        $budget = Budget::findOne($jar->budget_id);
        $debts = Debt::findAll(['budget_id' => $jar->budget_id, 'archived' => 0]);
        $debtIds = [];
        foreach ($debts as $d) {
            $debtIds[] = $d->id;
        }

        $totalDebtAmount = 0;
        $totalDebtPaid = 0;
        foreach ($debts as $d) {
            $totalDebtAmount += $d->amount;
            $totalDebtPaid += $d->getDebtPaymentsAmount();
        }

        $debtPayments = $budget->getThisMonthsDebtPayments();
        // get all jars for sidebar
        $this->view->title = $jar->name . ' jar';

        // pass mock budget specific data
        $mockBudgetData = [];
        if ($jar->budget->is_mock) {
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['budgetId'] = $budget->id;
            $mockBudgetData['isMockBudget'] = true;
        }

        $firstDebtPaymentDate = $budget->getFirstDebtPaymentDate();
        $lastDebtPaymentDate  = $budget->getLastDebtPaymentDate();
        $yearsList = Calculator::getYearsListFromTo(
            ['/jars/debt-jar-detail', 'id' => $jarId],
            date('Y', strtotime($firstDebtPaymentDate)),
            date('Y', strtotime($lastDebtPaymentDate)),
            $year
        );
        $monthsList = Calculator::getMonthsListFromTo(
            ['/jars/debt-jar-detail', 'id' => $jarId],
            date('m', strtotime($firstDebtPaymentDate)),
            date('Y', strtotime($firstDebtPaymentDate)),
            date('m', strtotime($lastDebtPaymentDate)),
            date('Y', strtotime($lastDebtPaymentDate)),
            $month,
            $year
        );

        // switch view based on whether this is an initial page load
        if ($this->pageLoadRequest || $fullLoad) {
            $jars = Jar::find()
                ->where(['budget_id' => $jar->budget_id, 'archived' => $jar->archived])
                ->orderBy('order')
                ->all();

            return $this->render('debt-jar-detail', $mockBudgetData + [
                'budget' => $budget,
                'jar' => $jar,
                'jars' => $jars,
                'debts' => $debts,
                'totalDebtAmount' => $totalDebtAmount,
                'totalDebtPaid' => $totalDebtPaid,
                'debtPayments' => $debtPayments,
                'debtNames' => ArrayHelper::map(Debt::findAll(['budget_id' => $jar->budget_id]), 'id', 'name'),
                'allowEdit' => !$jar->archived,
                'months' => $monthsList,
                'years'  => $yearsList,
                'year'   => $year,
                'month'  => $month,
                'monthName' => date('F', strtotime('2010-' . $month . '-1'))
            ]);
        }
        else {
            return $this->render('debt-jar-detail-content', $mockBudgetData + [
                'budget' => $budget,
                'jar' => $jar,
                'debts' => $debts,
                'totalDebtAmount' => $totalDebtAmount,
                'totalDebtPaid' => $totalDebtPaid,
                'debtPayments' => $debtPayments,
                'debtNames' => ArrayHelper::map(Debt::findAll(['budget_id' => $jar->budget_id]), 'id', 'name'),
                'allowEdit' => !$jar->archived,
                'months' => $monthsList,
                'years'  => $yearsList,
                'year'   => $year,
                'month'  => $month,
                'monthName' => date('F', strtotime('2010-' . $month . '-1'))
            ]);
        }
    }

    /**
     * Show debt jar header info
     */
    public function actionDebtJarHeaderInfo()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('id', false);
        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }
        // get the jar
        $jar = Jar::findOne($jarId);
        echo json_encode([
            'jar-amount' => Formatter::currency($jar->getThisMonthsFundsBroughtForwardAmount()),
            'this-months-expenses-amount' => Formatter::currency($jar->getMonthlyBudget()),
            'this-months-transactions-amount' => Formatter::currency($jar->getThisMonthsTransactionsAmount()),
            'this-months-funds-remaining' => Formatter::currency($jar->getThisMonthsFundsRemainingAmount())
        ]);
        die();
    }

    /**
     * Detail of interest jar
     */
    public function actionInterestJarDetail() {

        $jarId = Yii::$app->getRequest()->getQueryParam('id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content
        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }

        // get the jar & budget
        $jar = Jar::findOne(['id' => $jarId]);
        $budget = Budget::findOne($jar->budget_id);

        $debts = Debt::findAll(['budget_id' => $jar->budget_id, 'archived' => 0]);
        $debtIds = [];
        foreach ($debts as $d) {
            $debtIds[] = $d->id;
        }

        $totalDebtAmount = 0;
        $totalDebtPaid = 0;

        foreach ($debts as $d) {
            $totalDebtAmount += $d->amount;
            $totalDebtPaid += $d->getDebtPaymentsAmount();
        }

        $interestPayments = $budget->getThisMonthsDebtInterestPayments();

        // get all jars for sidebar
        $this->view->title = $jar->name . ' jar';

        // pass mock budget specific data
        $mockBudgetData = [];
        if ($jar->budget->is_mock) {
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['budgetId'] = $budget->id;
            $mockBudgetData['isMockBudget'] = true;
        }

        // switch view based on whether this is an initial page load
        if ($this->pageLoadRequest || $fullLoad) {

            $jars = Jar::find()
                ->where(['budget_id' => $jar->budget_id, 'archived' => 0])
                ->orderBy('order')
                ->all();

            return $this->render('interest-jar-detail', $mockBudgetData + [
                'budget' => $budget,
                'jar' => $jar,
                'jars' => $jars,
                'debts' => $debts,
                'totalDebtAmount' => $totalDebtAmount,
                'totalDebtPaid' => $totalDebtPaid,
                'interestPayments' => $interestPayments,
                'debtNames' => ArrayHelper::map(Debt::findAll(['budget_id' => $jar->budget_id]), 'id', 'name'),
                'allowEdit' => false
            ]);

        }
        else {

            return $this->render('interest-jar-detail-content', $mockBudgetData + [
                'budget' => $budget,
                'jar' => $jar,
                'debts' => $debts,
                'totalDebtAmount' => $totalDebtAmount,
                'totalDebtPaid' => $totalDebtPaid,
                'interestPayments' => $interestPayments,
                'debtNames' => ArrayHelper::map(Debt::findAll(['budget_id' => $jar->budget_id]), 'id', 'name'),
                'allowEdit' => false
            ]);

        }

    }

    /**
     * Reorder Jars API
     */
    public function actionReorderJars()
    {
        $request = Yii::$app->request->post();
        if (!isset($request['jarOrder'])) {
           echo JsonTools::errorMessage('Jar order update failed. Please try again later.');
           die();
        }
        if (!empty($request['budgetId'])) {
            $this->mockBudgetId = (int)$request['budgetId'];
        }
        $jars = Jar::find()
            ->where(['archived' => 0, 'budget_id' => $this->activeBudgetId])
            ->all();

        foreach ($jars as $jar) {
            $newOrder = array_search($jar->id, $request['jarOrder']);
            $jar->order = $newOrder;
            $jar->save();
        }
        echo JsonTools::successMessage('Jar order saved.');
        die();
    }

}
