<?php

namespace backend\controllers;

use backend\models\db\Eoms;
use Yii;

use yii\caching\TagDependency;
use yii\filters\AccessControl;

use yii\helpers\ArrayHelper;
use yii\helpers\Url;

use backend\components\CustomController;

use backend\models\db\Account;
use backend\models\db\Budget;
use backend\models\db\CcTransaction;
use backend\models\db\Debt;
use backend\models\db\DebtPayment;
use backend\models\db\Expense;
use backend\models\db\Income;
use backend\models\db\IncomeTransaction;
use backend\models\db\Jar;
use backend\models\db\Transaction;
use backend\models\db\Transfer;
use backend\models\db\TransferEvent;

use backend\components\helpers\Calculator;
use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;
use backend\components\helpers\JsonTools;

/**
 * Transactions controller
 */
class MoneyOperationsController extends CustomController {

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'add-money-operation',
                            'money-operations-table',
                            'money-operations-table-info',
                            'account-expense-transactions-table',
                            'account-expense-transactions-table-info',
                            'check-money-operation',
                            'edit-transaction',
                            'delete-transaction'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];

    }


    /**
     * Show transactions
     */
    public function actionIndex()
    {
        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));
        $budget = Budget::findOne($this->activeBudgetId);
        $moneyOperationsAmount = $budget->getMonthsMoneyOperationsAmount($month, $year);
        $firstMoneyOperationDate = $budget->getFirstMoneyOperationDate();
        $lastMoneyOperationDate = $budget->getLastMoneyOperationDate();

        $yearsList = Calculator::getYearsListFromTo(
            '/money-operations/index',
            date('Y', strtotime($firstMoneyOperationDate)),
            date('Y', strtotime($lastMoneyOperationDate)),
            $year
        );
        $monthsList = Calculator::getMonthsListFromTo(
            '/money-operations/index',
            date('m', strtotime($firstMoneyOperationDate)),
            date('Y', strtotime($firstMoneyOperationDate)),
            date('m', strtotime($lastMoneyOperationDate)),
            date('Y', strtotime($lastMoneyOperationDate)),
            $month,
            $year
        );

        $this->view->title = 'Transactions';
        return $this->render('index', [
            'moneyOperationsAmount' => $moneyOperationsAmount,
            'months' => $monthsList,
            'years'  => $yearsList,
            'year'   => $year,
            'month'  => $month,
            'monthName' => date('F', strtotime('2010-' . $month . '-1'))
        ]);

    }


    /**
     * Add a new money operation
     */
    public function actionAddMoneyOperation() {

        $this->view->title = "Add transaction";

        return $this->render('add-money-operation');

    }


    /**
     * Data for  money operations table
     */
    public function actionMoneyOperationsTable()
    {
        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));

        $budget = Budget::findOne($this->activeBudgetId);
        $moneyOperations = $budget->getMonthsMoneyOperations($month, $year, true);
        $columnNames = [
            'date',
            'name',
            'description',
            'account_name',
            'amount',
            'jar_name',
            'checked'
        ];
        $responseData = DataTables::getFilteredData($columnNames, $moneyOperations);
        $accounts = Account::getAccountsList($budget->id);

        foreach ($responseData['items'] as $i) {
            $url = null;
            $actionClass = get_class($i);
            if (get_class($i) == 'backend\models\db\DebtPayment') {
                $icon = '<i class="glyphicon glyphicon-triangle-bottom ico-spend"></i>';
                $amount = -($i->amount);
            } elseif (get_class($i) == 'backend\models\db\Transaction') {
                $url = '/transactions/edit-es-transaction';
                $icon = '<i class="glyphicon glyphicon-triangle-bottom ico-spend"></i>';
                $amount = -($i->amount);
            } elseif (get_class($i) == 'backend\models\db\CcTransaction') {
                $icon = '<i class="glyphicon glyphicon-triangle-bottom ico-transfer"></i>';
                $amount = -($i->amount);
            } elseif (get_class($i) == 'backend\models\db\TransferEvent') {
                $icon = '<i class="glyphicon glyphicon-triangle-right ico-transfer"></i>';
                $amount = $i->amount;
            } else {
                $url = '/incomes/edit-onetime-income';
                $icon = '<i class="glyphicon glyphicon-triangle-top ico-receive"></i>';
                $amount = $i->amount;
            }
            $hasJar = strlen($i->jar_name);

            $accountOptions = [];
            foreach ($accounts as $account) {
                $accountOptions[] = [
                    'name' => $account->name,
                    'value' => $account->id,
                    'selected' => $i->account_name == $account->name
                ];
            }

            $isEditable = Eoms::isTransactionChangeable($this->activeBudgetId, $i->date);
            $row = [
                'isEditable' => false
            ];

            if (get_class($i) == 'backend\models\db\IncomeTransaction' ||
                get_class($i) == 'backend\models\db\Transaction' ||
                get_class($i) == 'backend\models\db\DebtPayment') {
                $name = $i->name;
            }
            else {
                $name = '-';
            }
            if (get_class($i) == 'backend\models\db\Transaction' && $i->expense->is_adjustment == 0) {
                $nameColumn = [
                    'key' => 'expense_id',
                    'type' => 'select',
                    'value' => $name
                ];
            }
            elseif (get_class($i) == 'backend\models\db\IncomeTransaction' && $i->income->is_adjustment == 0) {
                $nameColumn = [
                    'key' => 'income_id',
                    'type' => 'select',
                    'value' => $name
                ];
            }
            else {
                $nameColumn = [
                    'key' => 'name',
                    'value' => $name,
                ];
            }

            if (isset($i->account_id)) {
                $accountColumn = [
                    'key' => 'account_id',
                    'type' => 'select',
                    'value' => $i->account_name
                ];
            }
            else {
                $accountColumn = [
                    'key' => 'account_name',
                    'type' => 'text',
                    'value' => $i->account_name
                ];
            }

            $bankStatType = '';
            if (get_class($i) == 'backend\models\db\IncomeTransaction') {
                $bankStatType = "it";
            }
            elseif (get_class($i) == 'backend\models\db\DebtPayment') {
                $bankStatType = "dp";
            }
            elseif (get_class($i) == 'backend\models\db\Transaction') {
                $bankStatType = "t";
            }
            elseif (get_class($i) == 'backend\models\db\TransferEvent') {
                $bankStatType = "te";
            }
            $bankStatName = $bankStatType . $i->id;
            $canCheck = !empty($bankStatType);

            $columns = [
                [
                    'key' => 'date',
                    'type' => 'date',
                    'value' => $i->date
                ],
                $nameColumn,
                [
                    'key' => 'description',
                    'type' => 'text',
                    'value' => $i->description
                ],
                $accountColumn,
                [
                    'key' => 'amount',
                    'type' => 'currency',
                    'value' => $amount
                ], [
                    'key' => 'jar_id',
                    'type' => 'select',
                    'value' => ($hasJar ? $i->jar_name : '-')
                ],
                [
                    'key' => 'checked',
                    'type' => 'checkbox',
                    'value' => $i->checked,
                    'cId' => $i->id,
                    'cType' => $bankStatType,
                    'cName' => $bankStatName,
                    'isEditable' => $canCheck
                ]
            ];



            $rowData = DataTables::getRow($row, $columns);
            array_pop($rowData);
            if ($isEditable) {
                $rowData[] = '<div class="text-center">
                    '.$icon.'
                    <a href="#" id="menu1" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="glyphicon glyphicon-cog gi-2x"></i></a>
                    <ul class="dropdown-menu" aria-labelledby="menu1">'.
                        ($url ? '<li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute([
                                 $url,
                                'id' => $i->id,
                                'js_callback' => 'reloadMoneyOperationsTable'
                            ]) . '"><i class="glyphicon glyphicon-pencil"></i>Edit</a>
                        </li>' : '').'
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute([
                                '/money-operations/delete-transaction',
                                'transaction_id' => $i->id,
                                'transaction_type' => $actionClass,
                                'js_callback' => 'reloadMoneyOperationsTable']) . '">
                                <i class="glyphicon glyphicon-trash"></i>Remove
                            </a>
                        </li>
                    </ul>
                </div>';
            }
            else {
                $rowData[] = '<div class="text-center">' . $icon . '</div>';
            }
            $responseData['response']['data'][] = $rowData;
        }

        echo json_encode($responseData['response']);
        die();
    }

    public function actionEditTransaction()
    {
        $class   = Yii::$app->request->post('type', false);
        $transId = Yii::$app->request->post('id', false);
        $fields  = Yii::$app->request->post('fields', false);

        if (!is_numeric($transId) || !$class || empty($fields)) {
            echo JsonTools::errorMessage('Access to the resource denied.');
            die();
        }

        $jar = null;
        switch ($class) {
            case 'backend\models\db\DebtPayment':
                $transaction = DebtPayment::findOne($transId);
                $parent = Debt::findOne($transaction->debt_id);
                // check if user can edit budget
                if (!Yii::$app->user->can('accessBudget', ['budgetId' => $parent->budget_id])) {
                    echo JsonTools::errorMessage('Access to the resource denied.');
                    die();
                }
                break;

            case 'backend\models\db\Transaction':
                $transaction = Transaction::findOne($transId);
                $parent = Expense::findOne($transaction->expense_id);
                $jar = Jar::findOne($parent->jar_id);
                // check if user can edit jar
                if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
                    echo JsonTools::errorMessage('Access to the resource denied.');
                    die();
                }
                break;

            case 'backend\models\db\CcTransaction':
                $transaction = CcTransaction::findOne($transId);
                $parent = Expense::findOne($transaction->expense_id);
                $jar = Jar::findOne($parent->jar_id);
                // check if user can edit jar
                if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
                    echo JsonTools::errorMessage('Access to the resource denied.');
                    die();
                }
                break;

            case 'backend\models\db\TransferEvent':
                $transaction = TransferEvent::findOne($transId);
                $parent = Transfer::findOne($transaction->transfer_id);
                $account = Account::findOne($parent->account_from_id);
                // check if user can edit budget
                if (!$account->budget_id == $this->activeBudgetId) {
                    echo JsonTools::errorMessage('Access to the resource denied.');
                    die();
                }
                break;

            case 'backend\models\db\IncomeTransaction':
                $transaction = IncomeTransaction::findOne($transId);
                $parent = Income::findOne($transaction->income_id);
                // check if user can edit budget
                if (!Yii::$app->user->can('accessBudget', ['budgetId' => $parent->budget_id])) {
                    echo JsonTools::errorMessage('Access to the resource denied.');
                    die();
                }
                break;

            default:
                echo JsonTools::errorMessage('Incorrect transaction type provided.');
                die();
        }

        // check date
        $isEditable = Eoms::isTransactionChangeable($this->activeBudgetId, $transaction->date);
        if (!$isEditable) {
            echo JsonTools::errorMessage('This transaction can no longer be edited.');
            die();
        }

        $saveChild = false;
        $saveParent = false;
        foreach ($fields as $field) {
            if (array_key_exists($field['type'], $transaction->attributes)) {
                $transaction->{$field['type']} = $field['value'];
                $saveChild = true;
            }
            elseif (array_key_exists($field['type'], $parent->attributes)) {
                $parent->{$field['type']} = $field['value'];
                $saveParent = true;
            }
        }
        if ($saveChild) {
            if (!$transaction->validate()) {
                echo JsonTools::errorMessage('There was an error when saving the transaction: ' . implode(', ', $transaction->errors));
                $saveChild = false;
            } else {
                $transaction->save();
            }
        }
        if ($saveParent) {
            if (!$parent->validate()) {
                echo JsonTools::errorMessage('There was an error when saving the transaction parent: ' . implode(', ', $parent->errors));
                $saveParent = false;
            }
            else {
                $parent->save();
            }
        }
        if ($saveChild || $saveParent) {
            if ($jar) {
                TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jar->id);
            }
            echo JsonTools::successMessage('Transaction updated successfully.');
        }
        die();
    }

    public function actionDeleteTransaction()
    {
        $class    = Yii::$app->request->get('transaction_type', false);
        $transId  = Yii::$app->request->get('transaction_id', false);
        $confirm  = Yii::$app->request->get('confirm', false);
        $callback = Yii::$app->request->get('js_callback', false);

        if (!is_numeric($transId) || !$class) {
            echo JsonTools::errorMessage('Access to the resource denied.');
            die();
        }

        $jar = null;
        switch ($class) {
            case 'backend\models\db\DebtPayment':
                $transaction = DebtPayment::findOne($transId);
                $parent = Debt::findOne(['id' => $transaction->debt_id]);
                if (!Yii::$app->user->can('accessBudget', ['budgetId' => $parent->budget_id])) {
                    echo JsonTools::errorMessage('Access to the resource denied.');
                    die();
                }
                break;

            case 'backend\models\db\Transaction':
                $transaction = Transaction::findOne($transId);
                // check if user can edit jar
                $jar = Jar::findOne($transaction->expense->jar_id);
                if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
                    echo JsonTools::errorMessage('Access to the resource denied.');
                    die();
                }
                break;

            case 'backend\models\db\CcTransaction':
                $transaction = CcTransaction::findOne($transId);
                $parent = Expense::findOne($transaction->expense_id);
                $jar = Jar::findOne($parent->jar_id);
                if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
                    echo JsonTools::errorMessage('Access to the resource denied.');
                    die();
                }
                break;

            case 'backend\models\db\TransferEvent':
                $transaction = TransferEvent::findOne($transId);
                $parent = Transfer::findOne($transaction->transfer_id);
                $account = Account::findOne($parent->account_from_id);
                if ($account->budget_id != $this->activeBudgetId) {
                    echo JsonTools::errorMessage('Access to the resource denied.');
                    die();
                }
                break;

            case 'backend\models\db\IncomeTransaction':
                $transaction = IncomeTransaction::findOne($transId);
                $parent = Income::findOne(['id' => $transaction->income_id]);
                if (!Yii::$app->user->can('accessBudget', ['budgetId' => $parent->budget_id])) {
                    echo JsonTools::errorMessage('Access to the resource denied.');
                    die();
                }
                break;

            default:
                echo JsonTools::errorMessage('Incorrect transaction type provided.');
                die();
        }

        // check date
        $isEditable = Eoms::isTransactionChangeable($this->activeBudgetId, $transaction->date);
        if (!$isEditable) {
            echo JsonTools::errorMessage('This transaction can no longer be removed.');
            die();
        }
        elseif ($confirm) {
            $transaction->delete();
            if ($jar) {
                TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jar->id);
            }
            echo JsonTools::successMessage('Transaction successfully deleted.');
            die();
        }

        switch ($class) {
            case 'backend\models\db\DebtPayment':
                $title = 'Delete Debt Payment';
                $prompt = 'Do you really wish to remove this debt payment?';
                break;
            case 'backend\models\db\Transaction':
                $title = 'Delete Transaction';
                $prompt = 'Do you really wish to delete this transaction?';
                break;
            case 'backend\models\db\CcTransaction':
                $title = 'Delete Credit Card Transaction';
                $prompt = 'Do you really wish to delete this credit card transaction?';
                break;
            case 'backend\models\db\TransferEvent':
                $title = 'Delete Transfer Event';
                $prompt = 'Do you really wish to delete this transfer event?';
                break;
            case 'backend\models\db\IncomeTransaction':
                $title = 'Delete Income Transaction';
                $prompt = 'Do you really wish to delete this income transaction?';
                break;
        }

        return $this->render('/money-operations/delete-transaction', [
            'title' => $title,
            'prompt' => $prompt,
            'transactionId' => $transId,
            'transactionType' => $class,
            'jsCallback' => $callback
        ]);

    }

    public function actionMoneyOperationsTableInfo()
    {
        $budget = Budget::findOne($this->activeBudgetId);
        $moneyOperationsAmount = $budget->getMonthsMoneyOperationsAmount(date('m'), date('Y'));

        echo json_encode([
            'this-months-money-operations-amount' => Formatter::currency($moneyOperationsAmount)
        ]);
        die();
    }


    public function actionAccountExpenseTransactionsTable()
    {
        $accountId = Yii::$app->getRequest()->getQueryParam('account_id', false);
        if (!is_numeric($accountId)) {
            return $this->accessError();
        }
        $account = Account::findOne($accountId);
        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $account->budget_id])) {
            return $this->accessError();
        }

        $expenses = $account->getMonthsExpenseTransactions(date('m'), date('Y'));
        $columnNames = [
            //'checked', // non-sortable
            'date',
            'expense_name',
            'name',
            'amount',
            'account_name',
            'jar_name'
        ];
        $responseData = DataTables::getFilteredData($columnNames, $expenses);
        foreach ($responseData['items'] as $i) {
            $actionClass = get_class($i);
            $isEditable = $i->date >= date('Y-m-01') && $i->date <= date('Y-m-t');

            $row = [
                'isEditable' => $isEditable
            ];
            if ($isEditable) {
                $row['id'] = $i->id;
                $row['type'] = $actionClass;
                $row['editUrl'] = '/money-operations/edit-transaction';
                $row['deleteUrl'] = Url::toRoute(['/money-operations/delete-transaction', 'transaction_id' => $i->id, 'transaction_type' => $actionClass, 'js_callback' => 'reloadAccountsAfterExpenseTransactionDelete']);
                $row['callback'] = 'reloadAccountsAfterExpenseTransactionDelete';
            }

            if ($actionClass == 'backend\models\db\DebtPayment') {
                $bankStatType = "dp";
            }
            elseif ($actionClass == 'backend\models\db\Transaction') {
                $bankStatType = "t";
            }
            $bankStatName = $bankStatType . $i->id;
            $hasJar = strlen($i->jar_name);
            $jarOptions = [];
            if ($hasJar) {
                $budget = Budget::findOne($this->activeBudgetId);
                $jars = $budget->getActiveJars();
                foreach ($jars as $jar) {
                    $jarOptions[] = [
                        'name' => $jar->name,
                        'value' => $jar->id,
                        'selected' => ($i->jar_name == $jar->name)
                    ];
                }
            }
            $columns = [
                /*[
                    'key' => 'checked',
                    'type' => 'checkbox',
                    'value' => $i->checked,
                    'cId' => $i->id,
                    'cType' => $bankStatType,
                    'cName' => $bankStatName,
                    'isEditable' => false
                ],*/ [
                    'key' => 'date',
                    'type' => 'date',
                    'value' => $i->date,
                    'isEditable' => true
                ], [
                    'key' => 'expense_name',
                    'type' => 'text',
                    'value' => isset($i->expense_name) ? $i->expense_name : '-',
                    'isEditable' => false
                ], [
                    'key' => 'name',
                    'type' => 'text',
                    'value' => $i->name,
                    'isEditable' => true
                ], [
                    'key' => 'amount',
                    'type' => 'currency',
                    'value' => -($i->amount),
                    'isEditable' => true
                ], [
                    'key' => 'jar_id',
                    'type' => 'select',
                    'value' => ($hasJar ? $i->jar_name : '-'),
                    'options' => $jarOptions,
                    'isEditable' => $hasJar
                ]
            ];
            $responseData['response']['data'][] = DataTables::getRow($row, $columns);
        }
        echo json_encode($responseData['response']);
        die();
    }


    public function actionAccountExpenseTransactionsTableInfo() {

        $accountId = Yii::$app->getRequest()->getQueryParam('account_id', false);

        if (!is_numeric($accountId)) {

            return $this->accessError();

        }

        $account = Account::findOne($accountId);

        if ($account->budget_id != $this->activeBudgetId) {

            return $this->accessError();

        }

        echo json_encode([
            'this-months-expense-transactions-amount' => Formatter::currency($account->getMonthsExpenseTransactionsAmount(date('m'), date('Y')))
        ]);
        die();

    }


    public function actionCheckMoneyOperation() {

        $operationId = Yii::$app->getRequest()->getQueryParam('operation_id', false);
        $operationType = Yii::$app->getRequest()->getQueryParam('operation_type', false);
        $checked = Yii::$app->getRequest()->getQueryParam('checked', 0);

        if (!in_array($checked, [0, 1])) {

            echo JsonTools::errorMessage('Checking operation failed.');
            die();

        }

        if ($operationType == 't' || $operationType == 'tr') {

            $transaction = Transaction::find()
                ->joinWith('expense.jar')
                ->where([
                    'AND',
                    [Transaction::tableName() . '.id' => $operationId],
                    ['budget_id' => $this->activeBudgetId]
                ])
                ->one();

            if ($transaction) {

                $transaction->checked = $checked;
                $transaction->save();
                echo JsonTools::successMessage('Operation checked.');
                die();

            }
            else {


            }

        }
        elseif ($operationType == 'dp') {

            $debtPayment = DebtPayment::find()
                ->joinWith('debt')
                ->where([
                    'AND',
                    [DebtPayment::tableName() . '.id' => $operationId],
                    ['budget_id' => $this->activeBudgetId]
                ])
                ->one();

            if ($debtPayment) {

                $debtPayment->checked = $checked;
                $debtPayment->save();
                echo JsonTools::successMessage('Operation checked.');
                die();

            }
            else {

                echo JsonTools::errorMessage('Checking operation failed.');
                die();

            }

        }
        elseif ($operationType == 'it') {

            $incomeTransaction = IncomeTransaction::find()
                ->joinWith('income')
                ->where([
                    'AND',
                    [IncomeTransaction::tableName() . '.id' => $operationId],
                    ['budget_id' => $this->activeBudgetId]
                ])
                ->one();

            if ($incomeTransaction) {

                $incomeTransaction->checked = $checked;
                $incomeTransaction->save();
                echo JsonTools::successMessage('Operation checked.');
                die();

            }
            else {

                echo JsonTools::errorMessage('Checking operation failed.');
                die();

            }

        }
        elseif ($operationType == 'te') {

            $transferEvent = TransferEvent::find()
                ->joinWith('transfer')
                ->where([
                    'AND',
                    [TransferEvent::tableName() . '.id' => $operationId],
                    ['budget_id' => $this->activeBudgetId]
                ])
                ->one();

            if ($transferEvent) {

                $transferEvent->checked = $checked;
                $transferEvent->save();
                echo JsonTools::successMessage('Operation checked.');
                die();

            }
            else {

                echo JsonTools::errorMessage('Checking operation failed.');
                die();

            }

        }

    }

}

