<?php

namespace backend\controllers;

use backend\components\helpers\Formatter;
use backend\models\db\User;
use Yii;

use yii\data\ActiveDataProvider;
use yii\filters\AccessControl;

use backend\components\CustomController;

use backend\models\db\Notification;
use backend\models\db\NotificationReminder;

use backend\models\form\SnoozeNotification;

use yii\helpers\Url;
use backend\components\helpers\JsonTools;

class NotificationsController extends CustomController {

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'get-notification-updates-data',
                            'mark-as-read',
                            'mark-all-as-read',
                            'snooze-notification',
                            'delete',
                            'delete-all'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];

    }


    /**
     * Notifications home page with notifications listings
     */
    public function actionIndex()
    {
        $user = User::findOne($this->userIdentity->id);
        // get ID of first unread notification
        $unreadNotifications = Notification::find()
            ->where([
                'read' => 0,
                'archived' => 0,
                'user_id' => $this->userIdentity->id
            ])
            ->all();
        $unreadNotificationIds = [];
        foreach ($unreadNotifications as $u) {
            $unreadNotificationIds[] = $u->id;
        }
        $newNotificationsQuery = Notification::find()
            ->where([
                'user_id' => $this->userIdentity->id,
                'id' => $unreadNotificationIds
            ])->orderBy('id DESC');
        $newNotificationsProvider = new ActiveDataProvider([
            'query' => $newNotificationsQuery,
            'pagination' => [
                'pageSize' => false,
            ],
        ]);

        $oldNotificationsQuery = Notification::find()
            ->where([
                'and',
                ['archived' => 0],
                ['user_id' => $this->userIdentity->id],
                ['NOT IN', 'id', $unreadNotificationIds]
            ])->orderBy('id DESC');
        $oldNotificationsProvider = new ActiveDataProvider([
            'query' => $oldNotificationsQuery,
            'pagination' => [
                'pageSize' => 15,
            ],
        ]);

        // set new notifications as read
        /*if (Yii::$app->user->identity->id == $this->userIdentity->id) {
            Notification::updateAll([
                'read' => 1
            ], [
                'read' => 0,
                'archived' => 0,
                'user_id' => $this->userIdentity->id
            ]);
        }*/

        $this->view->title = 'Notifications';
        return $this->render('index', [
            'timezone' => $user->getTimeZone(),
            'newNotificationsProvider' => $newNotificationsProvider,
            'oldNotificationsProvider' => $oldNotificationsProvider
        ]);
    }


    /**
     * Archive a notification
     */
    public function actionMarkAsRead()
    {
        $notificationId = Yii::$app->getRequest()->getQueryParam('notification_id', false);

        $notification = Notification::findOne([
            'id' => $notificationId,
            'user_id' => $this->userIdentity->id
        ]);

        if (!$notification) {
            echo JsonTools::errorMessage('Invalid notification ID.');
            die();
        } else {
            $notification->read = 1;
            $notification->save();
            echo JsonTools::successMessage('Notification marked as read.');
            die();
        }
    }

    public function actionMarkAllAsRead()
    {
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if ($confirm) {
            $updated = Notification::updateAll([
                'read' => 1
            ], [
                'read' => 0,
                'user_id' => $this->userIdentity->id
            ]);

            echo JsonTools::successMessage("{$updated} notifications marked as read.");
            die();
        }
        else {
            $this->view->title = 'Mark All Notifications as Read';
            return $this->render('mark-all-as-read');
        }
    }

    public function actionDelete()
    {
        $notificationId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if ($confirm) {
            $notification = Notification::findOne([
                'id' => $notificationId,
                'user_id' => $this->userIdentity->id
            ]);

            if (!$notification) {
                echo JsonTools::errorMessage('Invalid notification ID.');
                die();
            } else {
                $notification->archived = 1;
                $notification->save();
                echo JsonTools::successMessage('Notification deleted.');
                die();
            }
        }
        else {
            $this->view->title = 'Delete Notification';
            return $this->render('delete', [
                'id' => $notificationId
            ]);
        }
    }

    public function actionDeleteAll()
    {
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if ($confirm) {
            $updated = Notification::updateAll([
                'archived' => 1
            ], [
                'read' => 1,
                'user_id' => $this->userIdentity->id
            ]);

            echo JsonTools::successMessage("{$updated} notifications deleted.");
            die();
        }
        else {
            $this->view->title = 'Delete all Notifications';
            return $this->render('delete-all');
        }
    }

    /**
     * Get data for notification updates.
     */
    public function actionGetNotificationUpdatesData()
    {
        $unreadNotifications = $this->user
            ->getNotifications()
            ->where([
                'read' => 0,
                'archived' => 0
            ])
            ->all();

        $data = [];
        $viewUrl = Url::toRoute(['/notifications']);
        foreach ($unreadNotifications as $u) {
            $data[] = [
                'iconClass' => $u->notificationTemplate->getTemplateGlyphicon(),
                'categoryId' => $u->notificationTemplate->notification_category_id,
                'notificationId' => $u->id,
                'notificationDate' => Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($u->created_at, $this->user->getTimeZone()->code, 'Y-m-d H:i:s')),
                'content' => $u->content,
                'markAsReadUrl' => Url::toRoute(['mark-as-read', 'message_id' => $u->id]),
                'viewUrl' => $viewUrl
            ];
        }

        echo json_encode([
            'data' => $data,
            'status' => 'success'
        ]);
        die();
    }

    /**
     * Remind the user after a period of time
     */
    public function actionSnoozeNotification()
    {
        $notificationId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        $notification = Notification::findOne([
            'id' => $notificationId,
            'user_id' => Yii::$app->user->identity->id
        ]);

        if (!$notification) {
            echo JsonTools::errorMessage('Invalid notification ID.');
            die();
        }

        $snoozeForm = new SnoozeNotification();
//        if ($snoozeForm->load(Yii::$app->request->post())) {
        if ($confirm) {
//            if ($snoozeForm->validate()) {
//                $remindAt = date('Y-m-d H:i', time() + $snoozeForm->hours * 3600);
                $remindAt = date('Y-m-d H:i');
                $reminder = new NotificationReminder();
                $reminder->setAttributes([
                    'notification_id' => $notification->id,
                    'remind_at' => $remindAt
                ]);
                $saved = $reminder->save();

                if ($saved) {
                    echo JsonTools::successMessage('Reminder successfully created.');
                }
                else {
                    echo JsonTools::errorMessage('An error has occured.');
                }
                die();
//            }
//            elseif (Yii::$app->request->isAjax) {
//                echo JsonTools::formErrorMessage($snoozeForm);
//                die();
//            }
        }
        else {
            $snoozeForm->period_id = 0;
            $this->view->title = 'Remind me later';
            return $this->render('snooze-notification', [
                'notificationId' => $notification->id,
                'formModel' => $snoozeForm
            ]);
        }
    }
}
