<?php

namespace backend\controllers;

use backend\components\helpers\DripHelper;
use backend\components\membership\SiteConnector;
use backend\models\db\UserRole;
use chsergey\rest\Model;
use common\models\form\Register;
use common\models\form\ResetPassword;
use Yii;
use yii\filters\AccessControl;
use yii\web\Controller;
use yii\helpers\Url;

use backend\models\db\User;
use common\models\form\Login;
use common\models\form\ForgottenPassword;
use backend\components\helpers\Password;
use backend\components\membership\User as MembershipUser;

/**
 * Site controller
 */
class SiteController extends Controller
{
    public function init()
    {
        // set layout for this whole controller
        $this->layout = 'pre-login';
    }

    public function beforeAction($action)
    {
        // just for load tests, to bypass the login page
        if (Yii::$app->request->headers->get('X-Login-As') == 'democlient-cEP$4estutr7yewUPaTUcrAsw!6?w_f3') {
            $user = User::findOne(['username' => 'democlient']);
            Yii::$app->user->login($user, 0);
            return $this->redirect(['budget/dashboard']);
        }

        return parent::beforeAction($action);
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'login',
                            'forgotten-password',
                            'reset-password',
                            'error'
                        ],
                        'allow' => true,
                    ],
                    [
                        'actions' => ['logout', 'index'],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ],
        ];
    }


    /**
     * The dashboard.
     */
    public function actionIndex()
    {
        return $this->redirect(['/budget/dashboard']);
    }

    /**
     * Forgotten password.
     */
    public function actionForgottenPassword()
    {
        if (!\Yii::$app->user->isGuest) {
            return $this->redirect('/site/index');
        }
        $forgottenPassForm = new ForgottenPassword();

        if ($forgottenPassForm->load(Yii::$app->request->post()) && $forgottenPassForm->validate()) {
            $password = new Password();
            $password->generateRandom();

            $source = Yii::$app->request->get('source');
            $parentUser = null;
            $requests = [];
            if (empty($forgottenPassForm->email)) {
                // reset password for a single user
                $user = User::findOne(['username' => $forgottenPassForm->username]);
                if ($user) {
                    $requests = [
                        $user->createNewPasswordResetRequest($source)
                    ];
                    $parentUser = $user;
                }
            }
            else {
                // reset password for all users matching the email address
                $users = User::findAll(['email' => $forgottenPassForm->email]);
                foreach ($users as $user) {
                    $requests[] = $user->createNewPasswordResetRequest($source);
                    if ($user->user_role_id == UserRole::CLIENT || !$parentUser) {
                        $parentUser = $user;
                    }
                }
            }

            $loginUrl = Url::toRoute(Yii::$app->user->loginUrl, true);
            if (count($requests) > 0) {
                if (count($requests) > 1) {
                    foreach ($requests as $request) {
                        // send the email
                        DripHelper::sendEmailEvent(
                            'forgotten-password-single',
                            $parentUser->email,
                            'Forgotten Password',
                            [
                                'parent_username' => $parentUser->getFullName(),
                                'login_url' => $loginUrl,
                                'reset_url' => $request->getPasswordResetUrl(),
                                'username' => $request->getUser()->one()->username
                            ]
                        );
                    }
                }
                else {
                    // send the email
                    DripHelper::sendEmailEvent(
                        'forgotten-password-single',
                        $parentUser->email,
                        'Forgotten Password',
                        [
                            'parent_username' => $parentUser->getFullName(),
                            'login_url' => $loginUrl,
                            'reset_url' => $requests[0]->getPasswordResetUrl(),
                            'username' => $requests[0]->getUser()->one()->username
                        ]
                    );
                }
            }

            Yii::$app->session->setFlash('login', [['success', 'Your unique reset password link has been successfully sent! Please check your email.']]);
            return $this->redirect('login');
        }

        $this->view->title = 'Forgotten Password';
        return $this->render('forgotten-password', [
            'title' => $this->view->title,
            'formModel' => $forgottenPassForm
        ]);
    }

    public function actionResetPassword()
    {
        if (!\Yii::$app->user->isGuest) {
            return $this->redirect('/site/index');
        }
        $user = User::findByPasswordResetToken(Yii::$app->request->get('token'));
        if (!$user) {
            Yii::$app->session->setFlash('login', [['danger', 'Incorrect or expired password reset token.']]);
            return $this->redirect('login');
        }
        $isSetup = !empty($user->click_funnels_data) && $user->password == 'no-password';

        $resetPassword = new ResetPassword();
        if ($isSetup) {
            $resetPassword->scenario = 'setup';
            $uniqueEmail = $user->uniqueEmailInDatabase();
            if (!$uniqueEmail) {
                $resetPassword->scenario .= '-not-unique';
            }
        }
        else {
            $uniqueEmail = true;
        }

        if ($resetPassword->load(Yii::$app->request->post()) && $resetPassword->validate()) {
            $password = new Password($resetPassword->password);
            $user->password = $password->getHash();
            $user->password_salt = $password->getSalt();
            if ($isSetup) {
                $user->username = $resetPassword->username;
                if (!$uniqueEmail) {
                    $user->email = $resetPassword->email;
                }
            }
            if ($user->save()) {
                $request = $user->getUserPasswordResetRequests()->one();
                $request->markUsed();
                if ($isSetup) {
                    // finish setting up the account
                    Register::setupAccount($user, $resetPassword->password);
                    Yii::$app->session->setFlash('login', [['success', 'Your account has been set up successfully.']]);
                    return $this->redirect('login');
                }
                else {
                    // update membership password
                    $user->updateMembershipUser($resetPassword->password);
                    if ($request->isFromMembershipPage) {
                        return $this->redirect(Yii::$app->params['membershipSiteUrl']);
                    } else {
                        Yii::$app->session->setFlash('login', [['success', 'Password changed successfully.']]);
                        return $this->redirect('login');
                    }
                }
            }
        }

        $this->view->title = $isSetup ? 'Set Up User Account' : 'Reset Password';
        return $this->render('reset-password', [
            'title' => $this->view->title,
            'formModel' => $resetPassword,
            'user' => $user,
            'isSetup' => $isSetup,
            'uniqueEmail' => $uniqueEmail
        ]);
    }

    /**
     * User account login.
     */
    public function actionLogin()
    {
        if (!\Yii::$app->user->isGuest) {
            return $this->redirect('/site/index');
        }
        $loginForm = new Login();
        if ($loginForm->load(Yii::$app->request->post()) && $loginForm->validate()) {
            $result = User::authenticate($loginForm->username, $loginForm->password, $loginForm->rememberMe);
            if (is_object($result)) {
                DripHelper::sendCustomEvent('User logged in', $result->email);
                if (Yii::$app->user->isSuperAdmin) {
                    return $this->redirect(['/admin/default/admin-dashboard']);
                }
                elseif (Yii::$app->user->isCoach) {
                    return $this->redirect(['/admin/default/coach-dashboard']);
                }
                else {
                    return $this->redirect(['budget/dashboard']);
                }
            }
            else {
                switch ($result) {
                    case User::RESULT_INCORRECT:
                        Yii::$app->session->setFlash('login', [['danger', 'Incorrect username or password.']]);
                        break;
                    case User::RESULT_INACTIVE:
                        $resendUrl = Url::toRoute('register/resend');
                        Yii::$app->session->setFlash('login', [['danger', 'Account has not been activated yet. <a href="' . $resendUrl . '">Resend activation email?</a>']]);
                        break;
                    case User::RESULT_FTS:
                        Yii::$app->session->setFlash('login', [['danger', 'The main user for this account needs to finish the First Time Setup before you can log in.']]);
                        break;
                    case User::RESULT_ERROR:
                        Yii::$app->session->setFlash('login', [['danger', 'An error occured, please contact support.']]);
                        break;
                }
            }
        }

        $this->view->title = 'Login';
        return $this->render('login', [
            'formModel' => $loginForm
        ]);
    }


    /**
     * User account logout.
     */
    public function actionLogout()
    {

        // go back to the admin account
        if (Yii::$app->session->get('adminSelectedUserId')) {
            return $this->redirect(['/admin/user/switch-back']);
        }
        // log out in a normal way
        else {
            Yii::$app->getSession()->destroy();

            Yii::$app->user->logout();
        //     var_dump(Yii::$app->user->identity->username);
        //     var_dump(Yii::$app->session->isActive);

        //     setcookie("PHPSESSID", "logout", time()-3600, '/');
        // setcookie("_identity", "logout", time()-3600, '/');
        // setcookie("_backend_csrf", "logout", time()-3600, '/');
        // setcookie("test", "logout", time()-3600, '/');

        // $cookies = Yii::$app->response->cookies;

        // $cookies->add(new \yii\web\Cookie([
        //     'name' => 'abc',
        //     'value' => 'xyz',
        //     'expire' => time() + 86400 * 365,
        // ]));
        //     return '';
            return $this->redirect(Yii::$app->user->loginUrl);
        }
    }

    /**
     * Error handling
     */
    public function actionError() {

        $exception = Yii::$app->errorHandler->exception;

        if ($exception !== null) {

            $statusCode = $exception->statusCode;
            $name = method_exists($exception, 'getName') ? $exception->getName() : 'Error';
            $message = method_exists($exception, 'getMessage') ? $exception->getMessage() : '';

            $this->layout = false;
            $this->view->title = $name;

            return $this->render('error', [
                'exception' => $exception,
                'statusCode' => $statusCode,
                'name' => $name,
                'message' => $message
            ]);

        }

    }

}
