<?php

namespace backend\controllers;

use backend\components\helpers\DripHelper;
use backend\models\db\DebtPayment;
use backend\models\db\Eoms;
use backend\models\db\Income;
use backend\models\db\IncomeTransaction;
use Yii;

use yii\caching\TagDependency;
use yii\filters\AccessControl;

use yii\helpers\ArrayHelper;
use yii\helpers\Url;

use backend\components\CustomController;

use backend\models\db\Account;
use backend\models\db\Budget;
use backend\models\db\CcTransaction;
use backend\models\db\Debt;
use backend\models\db\Expense;
use backend\models\db\ExpenseChange;
use backend\models\db\ExpenseChangeMeta;
use backend\models\db\Jar;
use backend\models\db\Transaction;

use backend\models\form\AddTransaction;

use backend\components\helpers\Calculator;
use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;
use backend\components\helpers\JsonTools;


/**
 * Transactions controller
 */
class TransactionsController extends CustomController {

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'add-transaction',
                            'add-es-transaction',
                            'edit-es-transaction',
                            'transactions-table',
                            'jar-transactions-table',
                            'jar-transactions-table-footer',
                            'transactions-by-expense-for-month-table',
                            'transaction-totals-by-expense-for-year-graph',
                            'transaction-totals-by-expense-for-year-with-previous-years-graph',
                            'transaction-totals-by-expense-for-year-with-planned-graph',
                            'transaction-totals-by-jar-for-year-graph',
                            'transaction-totals-by-jar-for-year-with-previous-years-graph',
                            'transaction-totals-by-jar-for-year-with-planned-graph',
                            'transaction-totals-by-jar-for-year-chart',
                            'expense-jar-transactions-table-info',
                            'transactions-totals-for-budget-with-planned-graph',
                            'transactions-totals-for-budget-by-jar-chart',
                            'tax-deductible-transactions-for-month-table',
                            'transactions-by-expense-for-year-chart'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];

    }


    public function actionIndex() {



    }


    /**
     * Add a new transaction
     */
    public function actionAddTransaction()
    {
        $expenseId = Yii::$app->getRequest()->getQueryParam('expense_id', false);
        if (!is_numeric($expenseId)) {
            return $this->accessError();
        }

        $expense = Expense::findOne($expenseId);
        $jar = Jar::findOne($expense->jar_id);

        if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
            return $this->accessError();
        }

        $transactionForm = new AddTransaction();
        if (Yii::$app->request->isPost && $transactionForm->load(Yii::$app->request->post())) {
            $transactionForm->jar_id = $jar->id;
            $transactionForm->budget_id = $this->activeBudgetId;
            $transactionForm->expense_id = $expense->id;

            // set validation attributes
            if ($transactionForm->validate()) {
                $formData = $transactionForm->getAttributes();
                $transaction = $this->_saveNewTransaction($formData);

                if ($transaction && Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Transaction successfully created.');
                    die();
                } else {
                    echo JsonTools::errorMessage('Error creating transaction');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($transactionForm);
                die();
            }
        }
        else {
            $transactionForm->account_id = ($expense->account_id != null) ? 'a' . $expense->account_id : 'd' . $expense->getExpenseDebt()->one()->debt_id;
            $transactionForm['date'] = Formatter::date(time());

            $this->view->title = 'Add Transaction';
            return $this->render('add-transaction', [
                'formModel' => $transactionForm,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'selectedAccount' => $this->defaultAccountId,
                'expenseId' => $expense->id
            ]);
        }
    }


    /**
     * Add a new transaction with expense select
     */
    public function actionAddEsTransaction()
    {
        $transactionForm = new AddTransaction();

        if (Yii::$app->request->isPost && $transactionForm->load(Yii::$app->request->post())) {
            $transactionForm->budget_id = $this->activeBudgetId;

            // set validation attributes
            if ($transactionForm->validate()) {
                $formData = $transactionForm->getAttributes();
                // one-off debt payment?
                if (!empty($formData['debt_id'])) {
                    $payment = new DebtPayment();
                    $payment->setAttributes([
                        'debt_id' => $formData['debt_id'],
                        'debt_payment_type_id' => 1,
                        'account_id' => $formData['account_id'],
                        'amount' => $formData['amount'],
                        'date' => $formData['date'],
                        'description' => $formData['description'],
                        'is_unbudgeted' => 1
                    ]);
                    if ($payment->save()) {
                        $budget = Budget::findOne($this->activeBudgetId);
                        $debtsJar = $budget->getDebtsJar();
                        if ($debtsJar) {
                            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $debtsJar->id);
                        }
                        DripHelper::sendCustomEvent('Transaction added', $this->user->email, [
                            'type' => 'unbudgeted debt payment',
                            'description' => $payment->description
                        ]);
                        echo JsonTools::successMessage('Unbudgeted debt payment successfully created.');
                        die();
                    }
                    echo JsonTools::errorMessage('Error creating debt payment.');
                    die();
                }
                else {
                    // adding unbudgeted expense?
                    if (empty($formData['expense_id'])) {
                        $expense = $this->_saveUnbudgetedExpense($formData);
                        if ($expense) {
                            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $expense->jar_id);
                            $formData['expense_id'] = $expense->id;
                            if ($this->_saveNewTransaction($formData)) {
                                DripHelper::sendCustomEvent('Transaction added', $this->user->email, [
                                    'type' => 'unbudgeted expense',
                                    'description' => $formData['description']
                                ]);
                                echo JsonTools::successMessage('Unbudgeted expense successfully created.');
                                die();
                            }
                        }
                        echo JsonTools::errorMessage('Error creating expense.');
                        die();
                    } else {
                        $transaction = $this->_saveNewTransaction($formData);
                        if ($transaction && Yii::$app->request->isAjax) {
                            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $transaction->expense->jar_id);

                            DripHelper::sendCustomEvent('Transaction added', $this->user->email, [
                                'type' => 'expense',
                                'description' => $transaction->description
                            ]);

                            if (strtotime($formData['date']) > strtotime(date('Y-m-d'))) {
                                $expense = $transaction->getExpense()->one();
                                $jar = $expense->getJar()->one();

                                $link = '<a href="' . Url::toRoute(['jars/expense-jar-detail', 'id' => $jar->id]) . '" class="system-link">here</a>';
                                echo JsonTools::successMessage("Your transaction has been successfully created. Click $link to view this transaction within the jar if it occured this month.");
                                die();
                            } else {
                                echo JsonTools::successMessage('Transaction successfully created.');
                                die();
                            }
                        } else {
                            echo JsonTools::errorMessage('Error creating transaction.');
                            die();
                        }
                    }
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($transactionForm);
                die();
            }
        }
        else {
            $transactionForm['date'] = Formatter::date(time());

            // select expenses
            $jars = Jar::findAll(['budget_id' => $this->activeBudgetId, 'archived' => 0]);
            $jarsOptions = [];
            foreach($jars as $jar) {
                $jarsOptions[$jar->id] = [
                    'data-debt' => $jar->jar_type_id == 1 ? 0 : 1
                ];
            }
            $jarId = Yii::$app->getRequest()->getQueryParam('jar', false);
            $jarIds = $jarId ? [$jarId] : array_values(ArrayHelper::map($jars, 'id', 'id'));
            $expenses = Expense::getExpensesForTransaction($jarIds, false, true);
            $expensesOptions = [];
            foreach ($expenses as $expense) {
                $expensesOptions[$expense->id] = [
                    'data-default-account' => $expense->account_id,
                    'data-recurring' => $expense->isRecurring ? 1 : 0
                ];
            }
            $budget = Budget::findOne($this->activeBudgetId);
            $debts = $budget->getMonthsDebts(date('m'), date('Y'), false, false);

            $this->view->title = 'Add Budgeted Expense';
            $rendered = $this->render('add-es-transaction', [
                'formModel' => $transactionForm,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'selectedAccount' => $this->defaultAccountId,
                'expenses' => ArrayHelper::map($expenses, 'id', 'name'),
                'debts' => ArrayHelper::map($debts, 'id', 'name'),
                'expensesOptions' => $expensesOptions,
                'jars' => Jar::getJarsForSelect($this->activeBudgetId, false),
                'jarsOptions' => $jarsOptions
            ]);

            if (Yii::$app->request->isAjax) {
                echo $rendered;
                die();
            }
            return $rendered;
        }
    }

    /**
     * Add a new transaction with expense select
     */
    public function actionEditEsTransaction()
    {
        $transactionId = Yii::$app->getRequest()->getQueryParam('id', false);
        $jsCallback = Yii::$app->getRequest()->getQueryParam('js_callback', false);
        if (!is_numeric($transactionId)) {
            return $this->accessError();
        }
        $transaction = Transaction::findOne(['id' => $transactionId]);
        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $transaction->expense->jar_id])) {
            return $this->accessError();
        }
        $jar = Jar::findOne($transaction->expense->jar_id);
        $this->mockBudgetId = $jar->getMockBudgetId();

        $transactionForm = new AddTransaction(['scenario' => 'update']);
        $transactionForm->budget_id = $this->activeBudgetId;
        if (Yii::$app->request->isPost && $transactionForm->load(Yii::$app->request->post())) {
            // set validation attributes
            if ($transactionForm->validate()) {
                $oldJarId = $transaction->expense->jar_id;
                // change to unbudgeted expense
                if ($transaction->expense_id != $transactionForm->expense_id && !$transactionForm->expense_id) {
                    $expense = $this->_saveUnbudgetedExpense($transactionForm->getAttributes());
                    if ($expense) {
                        $transactionForm->expense_id = $expense->id;
                    }
                    else {
                        echo JsonTools::errorMessage('Error updating transaction.');
                        die();
                    }
                }
                $transaction->attributes = $transactionForm->getAttributes();
                if ($transaction->save()) {
                    TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $transactionForm->jar_id);
                    if ($oldJarId != $transactionForm->jar_id) {
                        TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $oldJarId);
                    }
                    echo JsonTools::successMessage('Transaction successfully updated.');
                    die();
                } else {
                    echo JsonTools::errorMessage('Error updating transaction.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($transactionForm);
                die();
            }
        }
        else {
            $transactionForm->attributes = $transaction->attributes;
            $transactionForm->jar_id = $jar->id;
            $transactionForm->budget_id = $this->activeBudgetId;

            // select expenses
            $budget = Budget::findOne($transactionForm->budget_id);
            $jars = $budget->getActiveJars();
            $jarsOptions = [];
            foreach($jars as $item) {
                $jarsOptions[$item->id] = [
                    'data-debt' => $item->jar_type_id == 1 ? 0 : 1
                ];
            }
            $expenses = Expense::getExpensesForTransaction([$jar->id], false, true);
            $expensesOptions = [];
            foreach ($expenses as $expense) {
                $expensesOptions[$expense->id] = [
                    'data-default-account' => $expense->account_id,
                    'data-recurring' => $expense->isRecurring ? 1 : 0
                ];
            }
            $debts = $budget->getMonthsDebts(date('m'), date('Y'), false, false);

            $this->view->title = 'Edit Transaction';
            $rendered = $this->render('add-es-transaction', [
                'id' => $transactionId,
                'jsCallback' => $jsCallback,
                'formModel' => $transactionForm,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'selectedAccount' => $transactionForm->account_id ? $transactionForm->account_id : $this->defaultAccountId,
                'expenses' => ArrayHelper::map($expenses, 'id', 'name'),
                'debts' => ArrayHelper::map($debts, 'id', 'name'),
                'expensesOptions' => $expensesOptions,
                'jars' => Jar::getJarsForSelect($this->activeBudgetId),
                'jarsOptions' => $jarsOptions
            ]);

            if (Yii::$app->request->isAjax) {
                echo $rendered;
                die();
            }
            return $rendered;
        }
    }

    /**
     * Jar page transactions table data
     */
    public function actionJarTransactionsTable()
    {
        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        $allowEdit = Yii::$app->getRequest()->getQueryParam('allow_edit', false);

        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }

        $jar = Jar::findOne($jarId);
        $expenses = Expense::findAll(['jar_id' => $jar->id]);
        $expenseIds = [];
        foreach ($expenses as $e) {
            $expenseIds[] = $e->id;
        }

        $transactions = Transaction::find()
            ->select(Transaction::tableName() . '.* ,' . Account::tableName() . '.name as account_name,'
                . Expense::tableName() . '.name as parent_name')
            ->joinWith('account')
            ->joinWith('expense')
            ->where([
                'and',
                ['IN', Transaction::tableName() . '.expense_id', $expenseIds],
                ['Month(' . Transaction::tableName() . '.date)' => $month],
                ['Year(' . Transaction::tableName() . '.date)' => $year]
            ])->all();

        $ccTransactions = CcTransaction::find()
            ->select(CcTransaction::tableName() . '.*, ' . Debt::tableName() . '.name as debt_name, ' . Debt::tableName() . '.name as account_name,'
                . Expense::tableName() . '.name as parent_name')
            ->joinWith('debt')
            ->joinWith('expense')
            ->where([
                'and',
                ['IN', CcTransaction::tableName() . '.expense_id', $expenseIds],
                ['Month(' . CcTransaction::tableName() . '.date)' => $month],
                ['Year(' . CcTransaction::tableName() . '.date)' => $year],
            ])->all();

        $incomeTransactions = IncomeTransaction::find()
            ->select(IncomeTransaction::tableName() . '.* ,' . Account::tableName() . '.name as account_name,'
                . Income::tableName() . '.name as parent_name')
            ->joinWith('account')
            ->joinWith('income')
            ->where([
                'and',
                [Income::tableName() . '.jar_id' => $jar->id],
                ['Month(' . IncomeTransaction::tableName() . '.date)' => $month],
                ['Year(' . IncomeTransaction::tableName() . '.date)' => $year]
            ])->all();

        $allTransactions = array_merge($transactions, $ccTransactions, $incomeTransactions);

        $columnNames = [
            'date',
            'parent_name',
            'description',
            'amount',
            'account_name'
        ];

        $responseData = DataTables::getFilteredData($columnNames, $allTransactions);

        foreach ($responseData['items'] as $i) {
            $actionClass = get_class($i);
            $isEditable = Eoms::isTransactionChangeable($this->activeBudgetId, $i->date);
            $row = ['isEditable' => false];

            $isCC = $actionClass == 'backend\models\db\CcTransaction';
            $postfix = $isCC ? '(CC)' : '';

            $columns = [
                [
                    'key' => 'date',
                    'type' => 'date',
                    'value' => Formatter::date($i->date)
                ], [
                    'key' => 'name',
                    'type' => 'text',
                    'value' => $i->parent_name,
                    'postfix' => $postfix
                ], [
                    'key' => 'description',
                    'type' => 'text',
                    'value' => $i->description
                ], [
                    'key' => 'amount',
                    'type' => 'currency',
                    'value' => get_class($i) == 'backend\models\db\Transaction' ? -1 * $i->amount : $i->amount
                ], [
                    'key' => 'account',
                    'type' => 'text',
                    'value' => $i->account_name
                ]
            ];

            $rowData = DataTables::getRow($row, $columns);
            if ($isEditable) {
                array_pop($rowData);
                $rowData[] = '<div class="text-center">
                    <a href="#" id="menu1" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="glyphicon glyphicon-cog gi-2x"></i></a>
                    <ul class="dropdown-menu" aria-labelledby="menu1">
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute([
                                get_class($i) == 'backend\models\db\Transaction' ?
                                    '/transactions/edit-es-transaction' : '/incomes/edit-onetime-income',
                                'id' => $i->id,
                                'js_callback' => 'reloadExpenseJarExpenses'
                            ]) . '"><i class="glyphicon glyphicon-pencil"></i>Edit</a>
                        </li>
                        <li>
                            <a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['/money-operations/delete-transaction', 'transaction_id' => $i->id, 'transaction_type' => $actionClass, 'js_callback' => 'reloadExpenseJarTransactions']) . '"><i class="glyphicon glyphicon-trash"></i>Remove</a>
                        </li>
                    </ul>
                </div>';
            }

            $responseData['response']['data'][] = $rowData;
        }
        echo json_encode($responseData['response']);
        die();
    }

    public function actionJarTransactionsTableFooter()
    {
        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }
        $jar = Jar::findOne($jarId);

        echo Formatter::currency($jar->getMonthsMoneyInAmount($month, $year) - $jar->getMonthsTransactionsAmount($month, $year));
        die();
    }


    /**
     * transaction by expense for month, used for report
     */
    public function actionTransactionsByExpenseForMonthTable()
    {
        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        $year = Yii::$app->getRequest()->getQueryParam('year', false);
        $expenseId = Yii::$app->getRequest()->getQueryParam('expense_id', false);

        $expense = Expense::findOne($expenseId);
        $jar = $expense->jar;

        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
            return $this->accessError();
        }

        $whereQuery = ['and',
            [Transaction::tableName() . '.expense_id' => $expenseId],
            ['Year(' . Transaction::tableName() . '.date)' => $year]
        ];
        if ($month) {
            $whereQuery[] = [
                'Month(' . Transaction::tableName() . '.date)' => $month
            ];
        }
        else {
            $whereQuery[] = [
                'BETWEEN', 'Month(' . Transaction::tableName() . '.date)', '1', '12'
            ];
        }

        $countFiltered = Transaction::find()
            ->select(Transaction::tableName() . '.*')
            ->where($whereQuery);

        $selectFiltered = Transaction::find()
            ->select(Transaction::tableName() . '.* ,' . Account::tableName() . '.name as account_name')
            ->joinWith('account')
            ->where($whereQuery);

        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'date',
                'select' => 'date'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'name' => 'amount'
            ],
            [
                'type' => 'normal',
                'as' => 'description',
                'name' => 'description'
            ],
            [
                'type' => 'normal',
                'as' => 'account_name',
                'name' => 'account_name'
            ]
        ];

        $responseData = DataTables::getResponseData($tableColumns, $countFiltered, $selectFiltered);
        foreach ($responseData['items'] as $i) {
            $responseData['response']['data'][] = [
                Formatter::date($i->date),
                Formatter::currency($i->amount),
                empty($i->description) ? '-' : $i->description,
                $i->account_name,
            ];
        }

        echo json_encode($responseData['response']);
        die();
    }


    /**
     * graph data for transaction totals for year graph
     */
    public function actionTransactionTotalsByExpenseForYearGraph() {

        $year = Yii::$app->getRequest()->getQueryParam('year', false);
        $expenseId = Yii::$app->getRequest()->getQueryParam('expenseId', false);

        $expense = Expense::findOne($expenseId);
        $jar = $expense->jar;

        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {

            return $this->accessError();

        }

        if ($year && $expenseId) {

            $transactions = Transaction::find()
                ->select(Transaction::tableName() . '.*')
                ->where([
                    'and',
                    ['expense_id' => $expenseId],
                    ['Year(' . Transaction::tableName() . '.date)' => $year]
                ])
                ->all();

            $transactionsOrdered = [];

            for ($i = 1; $i <= 12; $i++) {

                $transactionsOrdered[$i] = 0;

            }

            foreach ($transactions as $t) {

                $transactionsOrdered[date('n',strtotime($t->date))] += $t->amount;

            }

            $series = [];
            $series[] = [
                'name' => $expense->name,
                'data' => array_values($transactionsOrdered)
            ];

            $response = [
                'title' => $year,
                'year' => $year,
                'series' => $series,
                'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']
            ];

        }
        else {

            $response = false;

        }

        echo json_encode($response);
        die();

    }


    /**
     * graph data for transaction totals for year including comparison with previous years graph
     */
    public function actionTransactionTotalsByExpenseForYearWithPreviousYearsGraph() {

        $expenseId = Yii::$app->getRequest()->getQueryParam('expenseId', false);

        $expense = Expense::findOne($expenseId);
        $jar = $expense->jar;

        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {

            return $this->accessError();

        }

        if ($expenseId) {

            $transactions = Transaction::find()
                ->select(Transaction::tableName() . '.*')
                ->where(['expense_id' => $expenseId])
                ->orderBy('date asc')
                ->all();

            if (count($transactions)) {

                $firstYear = date('Y', strtotime(reset($transactions)->date));
                $lastYear = date('Y', strtotime(end($transactions)->date));

            }
            else {

                $firstYear = $lastYear = date('Y');

            }

            $transactionsOrdered = [];
            for ($y = $lastYear; $y >= $firstYear; $y--) {

                for ($i = 1; $i <= 12; $i++) {

                    $transactionsOrdered[$y][$i] = 0;

                }

            }

            foreach ($transactions as $t) {

                $transactionsOrdered[date('Y',strtotime($t->date))][date('n',strtotime($t->date))] += $t->amount;

            }

            $series = [];
            foreach ($transactionsOrdered as $year => $t) {

                $series[] = [
                    'name' => $year,
                    'data' => array_values($t)
                ];

            }

            $response = [
                'title' => $expense->name,
                'year' => '',
                'series' => $series,
                'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']
            ];

        }
        else {

            $response = false;

        }

        echo json_encode($response);
        die();

    }


    /**
     * graph data for transaction totals for year with planned amount comparison graph
     */
    public function actionTransactionTotalsByExpenseForYearWithPlannedGraph() {

        $year = Yii::$app->getRequest()->getQueryParam('year', false);
        $expenseId = Yii::$app->getRequest()->getQueryParam('expenseId', false);

        $expense = Expense::findOne($expenseId);
        $jar = $expense->jar;

        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {

            return $this->accessError();

        }

        if ($year && $expenseId) {

            $changes = [];

            // transactions serie data
            $transactions = Transaction::find()
                ->select(Transaction::tableName() . '.*')
                ->where([
                    'and',
                    ['expense_id' => $expenseId],
                    ['Year(' . Transaction::tableName() . '.date)' => $year]
                ])
                ->with('expense')
                ->all();

            $transactionsOrdered = [];

            for ($i = 1; $i <= 12; $i++) {

                $transactionsOrdered[$i] = 0;
                $changes[$i] = [
                    'amount' => $expense->amount,
                    'frequency' => $expense->frequency
                ];

            }

            foreach ($transactions as $t) {

                $transactionsOrdered[date('n',strtotime($t->date))] += $t->amount;

            }

            // planned amount serie data
            $changeMeta = ExpenseChangeMeta::find()
                ->select(ExpenseChangeMeta::tableName() . '.*, MONTH(' . ExpenseChange::tableName() . '.time) as month')
                ->joinWith('expenseChange')
                ->where([
                    'and',
                    ['expense_id' => $expenseId],
                    ['YEAR(time)' => $year],
                    ['type' => 'E'],
                    ['key' => ['amount', 'frequency']]
                ])
                ->orderBy('time desc')
                ->all();

            foreach ($changeMeta as $c) {

                for ($i = $c->month; $i>= 1; $i--) {

                    $changes[$i][$c->key] = $c->value;

                }

            }

            $expenseStartDate = ExpenseChange::find()
                ->select('time')
                ->where([
                    'and',
                    ['expense_id' => $expenseId],
                    ['type' => 'C'],
                ])
                ->scalar();

            $expenseEndDate = ExpenseChange::find()
                ->select('time')
                ->where([
                    'and',
                    ['expense_id' => $expenseId],
                    ['type' => 'D'],
                ])
                ->scalar();

            if (!$expenseEndDate) {

                $expenseEndDate = date('Y-m-d');

            }

            $changesOrdered = [];
            foreach ($changes as $month => $c) {
                $changeStartTime = strtotime(date("Y-m-t", strtotime("$year-$month-20")));
                $changeEndTime = strtotime(date("Y-m-1", strtotime("$year-$month-20")));
                if (($changeStartTime >= strtotime($expenseStartDate)) && ($changeEndTime <= strtotime($expenseEndDate))) {
                    $changesOrdered[] = Calculator::monthlyAverageAmount($c['amount'], $c['frequency']);
                }
                else {
                    $changesOrdered[] = null;
                }
            }

            $differences = [];
            $transactionsOrderedValues = array_values($transactionsOrdered);
            $changesOrderedValues = array_values($changesOrdered);

            foreach ($transactionsOrderedValues as $key => $t) {

                $differences[] = $changesOrderedValues[$key] - $t;

            }

            $series = [];
            $series[] = [
                'name' => 'Spent',
                'data' => $transactionsOrderedValues
            ];
            $series[] = [
                'name' => 'Planned',
                'data' => $changesOrderedValues
            ];
            $series[] = [
                'name' => 'Saved',
                'data' => $differences
            ];

            $response = [
                'title' => $year,
                'year' => $year,
                'series' => $series,
                'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']
            ];

        }
        else {

            $response = false;

        }

        echo json_encode($response);
        die();

    }


    /**
     * graph data for transaction totals for year graph by jar
     */
    public function actionTransactionTotalsByJarForYearGraph()
    {
        $year = Yii::$app->getRequest()->getQueryParam('year', false);
        $jarId = Yii::$app->getRequest()->getQueryParam('jarId', false);
        $jar = Jar::findOne($jarId);
        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
            return $this->accessError();
        }

        if ($year && $jarId) {
            $transactions = Transaction::find()
                ->select(Transaction::tableName() . '.*')
                ->joinWith('expense')
                ->where([
                    'and',
                    ['jar_id' => $jarId],
                    ['Year(' . Transaction::tableName() . '.date)' => $year]
                ])
                ->all();

            $transactionsOrdered = [];
            for ($i = 1; $i <= 12; $i++) {
                $transactionsOrdered[$i] = 0;
            }
            foreach ($transactions as $t) {
                $transactionsOrdered[date('n',strtotime($t->date))] += $t->amount;
            }
            $series = [];
            $series[] = [
                'name' => $jar->name,
                'data' => array_values($transactionsOrdered)
            ];
            $response = [
                'title' => $year,
                'year' => $year,
                'series' => $series,
                'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']
            ];
        }
        else {
            $response = false;
        }
        echo json_encode($response);
        die();
    }


    /**
     * graph data for transaction totals for year by jar including comparison with previous years graph
     */
    public function actionTransactionTotalsByJarForYearWithPreviousYearsGraph()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('jarId', false);
        $jar = Jar::findOne($jarId);

        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
            return $this->accessError();
        }
        if ($jarId) {
            $transactions = Transaction::find()
                ->select(Transaction::tableName() . '.*')
                ->joinWith('expense')
                ->where(['jar_id' => $jarId])
                ->orderBy(Transaction::tableName() . '.date asc')
                ->all();

            if (count($transactions)) {
                $firstYear = date('Y', strtotime(reset($transactions)->date));
                $lastYear = date('Y', strtotime(end($transactions)->date));
            }
            else {
                $firstYear = $lastYear = date('Y');
            }

            $transactionsOrdered = [];
            for ($y = $lastYear; $y >= $firstYear; $y--) {
                for ($i = 1; $i <= 12; $i++) {
                    $transactionsOrdered[$y][$i] = 0;
                }
            }
            foreach ($transactions as $t) {
                $transactionsOrdered[date('Y',strtotime($t->date))][date('n',strtotime($t->date))] += $t->amount;
            }
            $series = [];
            foreach ($transactionsOrdered as $year => $t) {
                $series[] = [
                    'name' => $year,
                    'data' => array_values($t)
                ];
            }
            $response = [
                'title' => $jar->name,
                'year' => '',
                'series' => $series,
                'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']
            ];
        }
        else {
            $response = false;
        }
        echo json_encode($response);
        die();
    }

    /**
     * graph data for transaction totals for year and for jar with planned amount comparison graph
     */
    public function actionTransactionTotalsByJarForYearWithPlannedGraph()
    {
        $year = Yii::$app->getRequest()->getQueryParam('year', false);
        $jarId = Yii::$app->getRequest()->getQueryParam('jarId', false);
        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }

        if ($year && $jarId) {
            $jar = Jar::findOne($jarId);
            // transactions series data
            $transactions = Transaction::find()
                ->select(Transaction::tableName() . '.*')
                ->joinWith('expense')
                ->where([
                    'and',
                    ['jar_id' => $jarId],
                    ['Year(' . Transaction::tableName() . '.date)' => $year]
                ])
                ->with('expense')
                ->all();
            $expenseStartDate = ExpenseChange::find()
                ->select('time')
                ->joinWith('expense')
                ->where([
                    'and',
                    ['jar_id' => $jar->id],
                    ['type' => 'C']
                ])
                ->scalar();
            $expenseEndDate = ExpenseChange::find()
                ->select('time')
                ->joinWith('expense')
                ->where([
                    'and',
                    ['jar_id' => $jar->id],
                    ['type' => 'D']
                ])
                ->scalar();

            if (!$expenseEndDate) {
                $expenseEndDate = date('Y-m-d');
            }
            $transactionsOrdered = [];
            $changesOrdered = [];

            for ($i = 1; $i <= 12; $i++) {
                $transactionsOrdered[$i] = 0;
                $changeStartTime = strtotime(date("$year-$i-t"));
                $changeEndTime = strtotime(date("$year-$i-1"));

                if (($changeStartTime >= strtotime($expenseStartDate)) && ($changeEndTime <= strtotime($expenseEndDate))) {
                    $changesOrdered[$i] = $jar->getMonthsExpensesAmount($i, $year);
                }
                else {
                    $changesOrdered[$i] = 0;
                }
            }

            foreach ($transactions as $t) {
                $transactionsOrdered[date('n',strtotime($t->date))] += $t->amount;
            }
            $differences = [];
            $transactionsOrderedValues = array_values($transactionsOrdered);
            $changesOrderedValues = array_values($changesOrdered);
            foreach ($transactionsOrderedValues as $key => $t) {
                $differences[] = $changesOrderedValues[$key] - $t;
            }
            $series = [];
            $series[] = [
                'name' => 'Spent',
                'data' => $transactionsOrderedValues
            ];
            $series[] = [
                'name' => 'Planned',
                'data' => $changesOrderedValues
            ];
            $series[] = [
                'name' => 'Saved',
                'data' => $differences
            ];
            $response = [
                'title' => $year,
                'year' => $year,
                'series' => $series,
                'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']
            ];
        }
        else {
            $response = false;
        }
        echo json_encode($response);
        die();
    }

    /**
     * graph data for transaction totals for year graph by jar
     */
    public function actionTransactionTotalsByJarForYearChart()
    {
        $year = Yii::$app->getRequest()->getQueryParam('year', false);
        $jarId = Yii::$app->getRequest()->getQueryParam('jarId', false);

        // check if user can edit jar
        if (!Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }

        if ($year) {
            $jar = Jar::findOne($jarId);
            // get transactions
            $transactions = Transaction::find()
                ->select(Transaction::tableName() . '.*, ' . Expense::tableName() . '.name')
                ->joinWith('expense')
                ->where([
                    'AND',
                    ['jar_id' => $jarId],
                    ['Year(' . Transaction::tableName() . '.date)' => $year]
                ])
                ->orderBy(Transaction::tableName() . '.date asc')
                ->all();

            if (count($transactions)) {
                $transactionsOrdered = [];
                $expenseNames = []; // [expenseId => exepenseName]

                foreach ($transactions as $t) {
                    if (!isset($transactionsOrdered[$t->expense_id])) {
                        $transactionsOrdered[$t->expense_id] = 0;
                        $expenseNames[$t->expense_id] = $t->name;
                    }
                    $transactionsOrdered[$t->expense_id] += $t->amount;
                }
                $response = [
                    'title' => 'Breakdown for ' . $year,
                    'year' => $year,
                    'series' => array_values($transactionsOrdered),
                    'labels' => array_values($expenseNames)
                ];
            }
            else {
                $response = false;
            }
        }
        else {
            $response = false;
        }
        echo json_encode($response);
        die();
    }

    /**
     * Total transactions amount for budget vs planned amount
     */
    public function actionTransactionsTotalsForBudgetWithPlannedGraph() {

        $year = Yii::$app->getRequest()->getQueryParam('year', false);

        if ($year) {

            $budget = Budget::findOne($this->activeBudgetId);

            $budgetStartDate = date('Y-m-d', strtotime($budget->created_time));
            $budgetEndDate = date('Y-m-d');

            $transactionsOrdered = [];
            $plannedOrdered = [];

            for ($i = 1; $i <= 12; $i++) {

                $plannedStartTime = strtotime(date("$year-$i-t"));
                $plannedEndTime = strtotime(date("$year-$i-1"));

                if (($plannedStartTime >= strtotime($budgetStartDate)) && ($plannedEndTime <= strtotime($budgetEndDate))) {

                    $plannedOrdered[$i] = $budget->getMonthsExpensesAmount($i, $year);
                    $transactionsOrdered[$i] = $budget->getMonthsTransactionsAmount($i, $year);

                }
                else {

                    $plannedOrdered[$i] = 0;
                    $transactionsOrdered[$i] = 0;

                }

            }

            $differences = [];
            $transactionsOrderedValues = array_values($transactionsOrdered);
            $plannedOrderedValues = array_values($plannedOrdered);

            foreach ($transactionsOrderedValues as $key => $t) {

                $differences[] = $plannedOrderedValues[$key] - $t;

            }

            $series = [];
            $series[] = [
                'name' => 'Spent',
                'data' => $transactionsOrderedValues
            ];
            $series[] = [
                'name' => 'Planned',
                'data' => $plannedOrderedValues
            ];
            $series[] = [
                'name' => 'Saved',
                'data' => $differences
            ];

            $response = [
                'title' => $year,
                'year' => $year,
                'series' => $series,
                'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']
            ];

        }
        else {

            $response = false;

        }

        echo json_encode($response);
        die();

    }


    /**
     * Transaction totals for budget chart
     */
    public function actionTransactionsTotalsForBudgetByJarChart() {

        $year = Yii::$app->getRequest()->getQueryParam('year', false);

        if ($year) {

            $budget = Budget::findOne($this->activeBudgetId);
            $jars = $budget->jars;
            $jarIds = ArrayHelper::map($jars, 'id', 'id');
            $jarNamesAll = ArrayHelper::map($jars, 'id', 'name');

            $budgetStartDate = date('Y-m-d', strtotime($budget->created_time));
            $budgetEndDate = date('Y-m-d');

            // get transactions
            $transactions = Transaction::find()
                ->select(Transaction::tableName() . '.*, ' . Expense::tableName() . '.name, jar_id')
                ->joinWith('expense')
                ->where([
                    'AND',
                    ['jar_id' => $jarIds],
                    ['Year(' . Transaction::tableName() . '.date)' => $year]
                ])
                ->orderBy(Transaction::tableName() . '.date asc')
                ->all();

            if (count($transactions)) {

                $transactionsOrdered = [];
                $jarNames = []; // [expenseId => exepenseName]

                foreach ($transactions as $t) {

                    if (!isset($transactionsOrdered[$t->jar_id])) {

                        $transactionsOrdered[$t->jar_id] = 0;
                        $jarNames[$t->jar_id] = $jarNamesAll[$t->jar_id];

                    }

                    $transactionsOrdered[$t->jar_id] += $t->amount;

                }

                $response = [
                    'title' => 'Breakdown for ' . $year,
                    'year' => $year,
                    'series' => array_values($transactionsOrdered),
                    'labels' => array_values($jarNames)
                ];

            }
            else {

                $response = false;

            }

        }
        else {

            $response = false;

        }

        echo json_encode($response);
        die();

    }


    /**
     * Table of tax deductible transactions for month and year
     */
    public function actionTaxDeductibleTransactionsForMonthTable() {

        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        $year = Yii::$app->getRequest()->getQueryParam('year', false);

        $budget = Budget::findOne($this->activeBudgetId);

        // get jar IDs
        $jars = $budget->jars;
        $jarIds = ArrayHelper::map($jars, 'id', 'id');

        $filtered = Transaction::find()
            ->select(Transaction::tableName() . '.*, ' . Expense::tableName() . '.name as expense_name, ' . Account::tableName() . '.name as account_name')
            ->joinWith('expense')
            ->joinWith('account')
            ->where([
                'and',
                ['jar_id' => $jarIds],
                ['Month(' . Transaction::tableName() . '.date)' => $month],
                ['Year(' . Transaction::tableName() . '.date)' => $year],
                [Transaction::tableName() . '.tax_deductible' => 1]
            ]);

        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'date',
                'select' => 'date'
            ],
            [
                'type' => 'normal',
                'as' => 'expense_name',
                'name' => 'expense_name'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'name' => 'amount'
            ],
            [
                'type' => 'normal',
                'as' => 'account_name',
                'name' => 'account_name'
            ]
        ];

        $responseData = DataTables::getResponseData($tableColumns, $filtered, $filtered);

        foreach ($responseData['items'] as $i) {

            $responseData['response']['data'][] = [
                Formatter::date($i->date),
                $i->expense_name,
                Formatter::currency($i->amount),
                $i->account_name,
            ];

        }

        echo json_encode($responseData['response']);
        die();

    }

    /**
     * Data for year's transactions pie chart
     */
    public function actionTransactionsByExpenseForYearChart()
    {
        $year = Yii::$app->getRequest()->getQueryParam('year', false);
        if ($year) {
            $budget = Budget::findOne($this->activeBudgetId);
            $jarId = Yii::$app->getRequest()->getQueryParam('jarId', false);
            $jarIds = ArrayHelper::map($budget->jars, 'id', 'id');

            $budgetStartDate = date('Y-m-d', strtotime($budget->created_time));
            $budgetEndDate = date('Y-m-d');

            // get transactions
            $transactions = Transaction::find()
                ->select(Transaction::tableName() . '.*, ' . Expense::tableName() . '.name')
                ->joinWith('expense')
                ->join('LEFT OUTER JOIN', Jar::tableName(), Jar::tableName() . '.id = ' . Expense::tableName() . '.jar_id')
                ->where([
                    'AND',
                    ['jar_id' => $jarId ? $jarId : $jarIds],
                    ['Year(' . Transaction::tableName() . '.date)' => $year]
                ])
                ->orderBy([
                    Jar::tableName() . '.name' => 'ASC',
                    Transaction::tableName() . '.date' => 'asc'
                ])
                ->all();

            if (count($transactions)) {
                $transactionsOrdered = [];
                $expenseNames = []; // [expenseId => expenseName]
                foreach ($transactions as $t) {
                    if (!isset($transactionsOrdered[$t->expense_id])) {
                        $transactionsOrdered[$t->expense_id] = [
                            'meta' => $t->name,
                            'value' => 0
                        ];
                        $expenseNames[$t->expense_id] = $t->name;
                    }
                    $transactionsOrdered[$t->expense_id]['value'] += $t->amount;
                }

                $response = [
                    'title' => 'Breakdown for ' . $year . ($jarId ? ' for ' . Jar::findOne($jarId)->name : ''),
                    'year' => $year,
                    'series' => array_values($transactionsOrdered),
                    'labels' => array_values($expenseNames)
                ];
            }
            else {
                $response = false;
            }
        }
        else {
            $response = false;
        }
        echo json_encode($response);
        die();
    }

    /**
     * Info for jar transactions table
     */
    public function actionExpenseJarTransactionsTableInfo()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }
        // get the jar
        $jar = Jar::findOne($jarId);
        echo json_encode([
            'this-months-jar-transactions-amount' => Formatter::currency($jar->getThisMonthsMoneyInAmount() - $jar->getThisMonthsTransactionsAmount())
        ]);
        die();
    }

    public function _saveNewTransaction($formData)
    {
        $transaction = new Transaction();
        $transaction->setAttributes($formData);
        $saved = $transaction->save();
        return $saved ? $transaction : false;
    }

    protected function _saveUnbudgetedExpense(array $formData)
    {
        $expense = new Expense();
        $expense->setAttributes([
            'jar_id' => $formData['jar_id'],
            'name' => Expense::OTHER_UNBUDGETED_EXPENSE_NAME,
            'account_id' => $formData['account_id'],
            'amount' => $formData['amount'],
            'date' => $formData['date'],
            'frequency' => 'one-time',
            'is_unbudgeted' => 1,
            'is_adjustment' => 0,
            'tax_deductible' => 0,
            'generate_transactions' => 0
        ]);
        if ($expense->save()) {
            return $expense;
        }
        return null;
    }

}
