<?php

namespace backend\controllers;

use backend\components\helpers\Calculator;
use backend\components\helpers\DripHelper;
use backend\models\db\Budget;
use backend\models\db\Eoms;
use Yii;

use yii\filters\AccessControl;

use yii\helpers\ArrayHelper;
use yii\helpers\Url;

use backend\components\CustomController;

use backend\models\db\Account;
use backend\models\db\Transfer;
use backend\models\db\TransferEvent;
use backend\models\db\TransferEnd;

use backend\models\form\AddTransfer;

use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;
use backend\components\helpers\Frequencies;
use backend\components\helpers\JsonTools;


/**
 * Transactions controller
 */
class TransfersController extends CustomController {

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'add-transfer',
                            'add-onetime-transfer',
                            'edit-transfer',
                            'pause-transfer',
                            'continue-transfer',
                            'transfer-detail',
                            'transfers-table',
                            'remove-transfer'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];

    }

    /**
     * Transfers screen
     */
    public function actionIndex()
    {
        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));
        $budget = Budget::findOne($this->activeBudgetId);
        $firstTransferDate = $budget->getFirstTransferDate();
        $lastTransferDate  = $budget->getLastTransferDate();

        $yearsList = Calculator::getYearsListFromTo(
            '/transfers/index',
            date('Y', strtotime($firstTransferDate)),
            date('Y', strtotime($lastTransferDate)),
            $year
        );
        $monthsList = Calculator::getMonthsListFromTo(
            '/transfers/index',
            date('m', strtotime($firstTransferDate)),
            date('Y', strtotime($firstTransferDate)),
            date('m', strtotime($lastTransferDate)),
            date('Y', strtotime($lastTransferDate)),
            $month,
            $year
        );

        $transfers = $this->_getTransfersMenuList();
        $this->view->title = 'Transfers';
        return $this->render('index', [
            'transfers' => $transfers,
            'allowEdit' => true,
            'months' => $monthsList,
            'years'  => $yearsList,
            'year'   => $year,
            'month'  => $month,
            'monthName' => date('F', strtotime('2010-' . $month . '-1'))
        ]);
    }

    /**
     * Add a new transfer
     */
    public function actionAddTransfer()
    {
        $transferForm = new AddTransfer();
        if (Yii::$app->request->isPost && $transferForm->load(Yii::$app->request->post())) {
            $transferForm->budget_id = $this->activeBudgetId;
            // set validation attributes
            if ($transferForm->validate()) {
                $formData = $transferForm->getAttributes();
                $this->_saveNewTransfer($formData);

                DripHelper::sendCustomEvent('Transfer created', $this->user->email, [
                    'name' => $formData['name']
                ]);

                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Transfer successfully created.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($transferForm);
                die();
            }
        }
        else {
            $transferForm['date'] = Formatter::date(time());
            $this->jsCallbackToView();
            $this->view->title = 'Add Transfer';
            return $this->render('add-transfer', [
                'formModel' => $transferForm,
                'frequencies' => Frequencies::getAll(),
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId)
            ]);
        }
    }

    /**
     * Add one-time transfer
     */
    public function actionAddOnetimeTransfer()
    {
        $transferForm = new AddTransfer();
        if (Yii::$app->request->isPost && $transferForm->load(Yii::$app->request->post())) {
            $transferForm->budget_id = $this->activeBudgetId;
            $transferForm['frequency'] = 'one-time';
            // set validation attributes
            if ($transferForm->validate()) {
                $formData = $transferForm->getAttributes();
                $transfer = $this->_saveNewTransfer($formData);
                DripHelper::sendCustomEvent('Transaction added', $this->user->email, [
                    'type' => 'transfer',
                    'description' => $transfer->name
                ]);
                if (Yii::$app->request->isAjax) {
                    if (strtotime($formData['date']) > strtotime(date('Y-m-d'))) {
                        $link = '<a href="' . Url::toRoute(['transfers/transfer-detail', 'id' => $transfer->id]) . '" class="system-link">here</a>';
                        echo JsonTools::successMessage("You transfer has been successfully created. Click $link to view this transfer's detail page.");
                    }
                    else {
                        echo JsonTools::successMessage('Transfer successfully created.');
                    }
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($transferForm);
                die();
            }
        }
        else {
            $transferForm['date'] = Formatter::date(time());
            $this->view->title = 'Add One-time Transfer';
            $this->jsCallbackToView();

            return $this->render('add-onetime-transfer', [
                'formModel' => $transferForm,
                'accounts' => ArrayHelper::map(Account::findAll(['budget_id' => $this->activeBudgetId, 'archived' => 0]), 'id', 'name')
            ]);
        }
    }

    /**
     * Edit a transfer
     */
    public function actionEditTransfer()
    {
        $transferId = Yii::$app->getRequest()->getQueryParam('id', false);
        if (!is_numeric($transferId)) {
            return $this->accessError();
        }
        $transfer = Transfer::findOne($transferId);
        $account = Account::findOne($transfer->account_from_id);
        if ($account->budget_id != $this->activeBudgetId) {
            return $this->accessError();
        }

        $transferForm = new AddTransfer();
        if (Yii::$app->request->isPost && $transferForm->load(Yii::$app->request->post())) {
            $transferForm->budget_id = $this->activeBudgetId;
            $transferForm->transfer_id = $transfer->id;
            // set validation attributes
            if ($transferForm->validate()) {
                $formData = $transferForm->getAttributes();
                $transfer = Transfer::findOne($transferId);
                $transfer->setAttributes($formData);
                $transfer->save();
                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Transfer successfully edited.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($transferForm);
                die();
            }
        }
        else {
            $transferForm->setAttributes($transfer->getAttributes());
            $transferForm->date = $transfer->getNextDue();
            $frequencies = Frequencies::getAll();

            $this->view->title = 'Edit Transfer';
            return $this->render('edit-transfer', [
                'formModel' => $transferForm,
                'transferId' => $transfer->id,
                'frequencies' => $frequencies,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId)
            ]);
        }
    }

    /**
     * Transfer detail page
     */
    public function actionTransferDetail()
    {
        $transferId = Yii::$app->getRequest()->getQueryParam('id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content
        if (!is_numeric($transferId)) {
            return $this->accessError();
        }
        $transfer = Transfer::findOne($transferId);
        if ($transfer->archived) {
            return $this->redirect(['/transfers/index']);
        }
        $account = Account::findOne($transfer->account_from_id);
        if ($account->budget_id != $this->activeBudgetId) {
            return $this->accessError();
        }

        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));

        $this->view->title = $transfer->name . ' transfer';
        if ($this->pageLoadRequest || $fullLoad) {
            $transfers = $this->_getTransfersMenuList();

            $budget = Budget::findOne($this->activeBudgetId);
            $firstIncomeDate = $budget->getFirstIncomeDate();
            $lastIncomeDate = $budget->getLastIncomeDate();
            $yearsList = Calculator::getYearsListFromTo(
                '/transfers/index',
                date('Y', strtotime($firstIncomeDate)),
                date('Y', strtotime($lastIncomeDate)),
                $year
            );
            $monthsList = Calculator::getMonthsListFromTo(
                '/transfers/index',
                date('m', strtotime($firstIncomeDate)),
                date('Y', strtotime($firstIncomeDate)),
                date('m', strtotime($lastIncomeDate)),
                date('Y', strtotime($lastIncomeDate)),
                $month,
                $year
            );

            return $this->render('transfer-detail', [
                'transfer' => $transfer,
                'transfers' => $transfers,
                'allowEdit' => true,
                'fullLoad' => $fullLoad,
                'months' => $monthsList,
                'years'  => $yearsList,
                'year'   => $year,
                'month'  => $month,
                'monthName' => date('F', strtotime('2010-' . $month . '-1'))
            ]);
        }
        else {
            return $this->render('transfer-detail-content', [
                'transfer' => $transfer,
                'allowEdit' => true,
                'fullLoad' => $fullLoad
            ]);
        }
    }

    public function actionTransfersTable()
    {
        $jsCallbackPause = Yii::$app->getRequest()->getQueryParam('js_callback_pause', false);
        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => Transfer::tableName() .'name'
            ],
            [
                'type' => 'normal',
                'as' => 'frequency',
                'select' => Transfer::tableName() .'frequency'
            ],
            [
                'type' => 'normal',
                'as' => 'account_from_name',
                'select' => 'a_from.name'
            ],
            [
                'type' => 'normal',
                'as' => 'account_to_name',
                'select' => 'a_to.name'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => Transfer::tableName() .'amount'
            ]
        ];

        $countFiltered = Transfer::find()
            ->leftJoin(Account::tableName(), Account::tableName() . ".id = " . Transfer::tableName() . ".account_from_id")
            ->where([
                'and',
                [Account::tableName() . '.budget_id' => $this->activeBudgetId],
                [Transfer::tableName() . '.archived' => 0],
                ['!=', 'frequency', 'one-time']
            ]);

        $selectFiltered = Transfer::find()
            ->select([
                Transfer::tableName() . '.*',
                'account_from_name' => 'a_from.name',
                'account_to_name' => 'a_to.name',
            ])
            ->leftJoin(Account::tableName() . ' as a_from', "a_from.id = " . Transfer::tableName() . ".account_from_id")
            ->leftJoin(Account::tableName() . ' as a_to', "a_to.id = " . Transfer::tableName() . ".account_to_id")
            ->where([
                'and',
                ['a_from.budget_id' => $this->activeBudgetId],
                [Transfer::tableName() . '.archived' => 0],
                ['!=', 'frequency', 'one-time']
            ]);

        $responseData = DataTables::getResponseData($tableColumns, $countFiltered, $selectFiltered);

        /** @var Transfer $i */
        foreach ($responseData['items'] as $i) {
            // pause or continue
            $transferEnd = $i->getTransferEnd()->one();

            $pauseOrCont = $transferEnd ?
                '<a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['transfers/continue-transfer', 'id' => $i->id, 'js_callback' => (isset($jsCallbackPause)  && $jsCallbackPause) ? $jsCallbackPause : false]) . '"><i class="glyphicon glyphicon-play"></i>Resume</a>' :
                '<a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['transfers/pause-transfer', 'id' => $i->id, 'js_callback' => (isset($jsCallbackPause)  && $jsCallbackPause) ? $jsCallbackPause : false]) . '"><i class="glyphicon glyphicon-pause"></i>Pause</a>';
            $editTransferUrl = Url::toRoute(['/transfers/edit-transfer', 'id' => $i->id, 'js_callback' => 'reloadTransfersAfterEdit']);

            $responseData['response']['data'][] = [
                $i->name . ($transferEnd ? '<span class="row-bg-red"></span>' : ''),
                Frequencies::getLabel($i->frequency),
                $transferEnd ? '-' : $i->getNextDue(),
                $i->account_from_name,
                $i->account_to_name,
                Formatter::currency($i->amount),
                '<div class="text-center">
                    <a href="#" id="menu1" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="glyphicon glyphicon-cog gi-2x"></i></a>
                    <ul class="dropdown-menu" aria-labelledby="menu1">
                        <li>
                            ' . $pauseOrCont . '
                            <a href="#" class="open-modal" data-modal-url="'.$editTransferUrl.'"><i class="glyphicon glyphicon-edit"></i>Edit</a>
                        </li>
                    </ul>
                </div>'
            ];
        }

        echo json_encode($responseData['response']);
        die();
    }

    /**
     * Pause a transfer
     */
    public function actionPauseTransfer() {

        $transferId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        // get the transfer
        $transfer = Transfer::findOne($transferId);
        $account = Account::findOne($transfer->account_from_id);

        if ($account->budget_id != $this->activeBudgetId) {

            return $this->accessError();

        }

        if ($confirm) {

            $transferEnd = new TransferEnd();
            $transferEnd->transfer_id = $transfer->id;
            $transferEnd->date = date('Y-m-d');
            $transferEnd->save();

            echo JsonTools::successMessage('Transfer successfully paused.');
            die();

        }

        $this->view->title = 'Pause Transfer';
        $this->jsCallbackToView();

        return $this->render('pause-transfer', [
            'transferId' => $transfer->id
        ]);

    }


    /**
     * Continue a transfer
     */
    public function actionContinueTransfer()
    {
        $transferId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);
        // get the transfer
        $transfer = Transfer::findOne($transferId);
        $account = Account::findOne($transfer->account_from_id);
        if ($account->budget_id != $this->activeBudgetId) {
            return $this->accessError();
        }

        if ($confirm) {
            $transferEnd = TransferEnd::findOne(['transfer_id' => $transfer->id]);
            $transferEnd->delete();
            echo JsonTools::successMessage('Transfer successfully resumed.');
            die();
        }
        $this->view->title = 'Continue Transfer';
        $this->jsCallbackToView();
        return $this->render('continue-transfer', [
            'transferId' => $transfer->id
        ]);
    }

    /**
     * Remove a transfer
     */
    public function actionRemoveTransfer()
    {
        $transferId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);
        $hardDelete = Yii::$app->getRequest()->getQueryParam('hard', false);
        // get the transfer
        $transfer = Transfer::findOne($transferId);
        $account = Account::findOne($transfer->account_from_id);
        if ($account->budget_id != $this->activeBudgetId) {
            return $this->accessError();
        }
        $hardDeletable = Eoms::isEntryHardDeletable($this->activeBudgetId, $transfer);

        if ($confirm) {
            if (Yii::$app->user->hasFinishedFts()) {
                if ($hardDelete && $hardDeletable) {
                    $transfer->deleteCompletely();
                }
                else {
                    $transfer->archived = 1;
                    $transfer->save();
                    // remove transfer events for one-time transfer
                    if ($transfer->frequency == 'one-time') {
                        TransferEvent::deleteAll(['transfer_id' => $transfer->id]);
                    }
                }
            }
            else {
                $transfer->deleteCompletely();
            }
            echo JsonTools::successMessage('Transfer successfully removed.');
            die();
        }
        $this->view->title = 'Remove Transfer';
        return $this->render('remove-transfer', [
            'transferId' => $transfer->id,
            'frequency' => $transfer->frequency,
            'hardDeletable' => $hardDeletable
        ]);
    }

    private function _getTransfersMenuList()
    {
        return Transfer::find()
            ->leftJoin(Account::tableName(), Account::tableName() . ".id = " . Transfer::tableName() . ".account_from_id")
            ->where([
                'and',
                [Account::tableName() . '.budget_id' => $this->activeBudgetId, Transfer::tableName() . '.archived' => 0],
                ['!=', Transfer::tableName() . '.frequency', 'one-time']
            ])
            ->orderBy(Transfer::tableName() . '.name ASC')
            ->all();
    }

    /**
     * Save a newly created transfer.
     * @param array $formData Data from form.
     * @return Transfer|bool
     */
    private function _saveNewTransfer($formData)
    {
        $transfer = new Transfer();
        $transfer->setAttributes($formData);
        $saved = $transfer->save();
        return $saved ? $transfer : false;
    }

}
