<?php

namespace backend\controllers;

use Yii;
use yii\filters\AccessControl;
use yii\helpers\ArrayHelper;
use yii\web\UploadedFile;

use backend\components\CustomController;

use backend\models\db\Budget;
use backend\models\db\UserParent;
use backend\models\db\Country;
use backend\models\db\Currency;
use backend\models\db\DateFormat;
use backend\models\db\TimeFormat;
use backend\models\db\TimeZone;
use backend\models\db\User;
use backend\models\db\UserLogin;
use backend\models\db\UserMeta;

use backend\models\form\AddUser;
use backend\models\form\EditUserBasicInfo;

use backend\components\helpers\DataTables;
use backend\components\helpers\Password;
use backend\components\helpers\Formatter;
use backend\components\helpers\JsonTools;


/**
 * User controller
 */
class UsersController extends CustomController {

    public $months;
    public $years;
    public $days;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'edit-user',
                            'user-detail',
                            'usage-history-table',
                            'user-profile'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                    [
                        'actions' => [
                            'add-user',
                            'manage-subscription',
                            'edit-user-basic-info',
                            'remove-user'
                        ],
                        'allow' => true,
                        'roles' => ['client', 'coach', 'superadmin'],
                    ],
                ],
            ]
        ];

    }


    public function init() {

        $this->months = [
            '1'  => 'January',
            '2'  => 'February',
            '3'  => 'March',
            '4'  => 'April',
            '5'  => 'May',
            '6'  => 'June',
            '7'  => 'July',
            '8'  => 'August',
            '9'  => 'September',
            '10' => 'October',
            '11' => 'November',
            '12' => 'December'
        ];

        $this->years = [];
        for ($i = date('Y'); $i >= date('Y') - 100; $i--) {

            $this->years[$i] = $i;

        }

        $this->days = [];
        for ($i = 1; $i <= 31; $i++) {

            $this->days[$i] = $i;

        }

    }


    /**
     * Common user dashboard
     */
    public function actionIndex() {

        $users = User::getAllUsersForUserId($this->mainUserId);

        $basicInfoForm = new EditUserBasicInfo();
        $basicInfoForm->setAttributes([
            'currency_id' => $this->getUserMeta('currency_id'),
            'time_zone_id' => $this->getUserMeta('time_zone_id'),
            'time_format_id' => $this->getUserMeta('time_format_id'),
            'date_format_id' => $this->getUserMeta('date_format_id'),
            'currency_format' => $this->getUserMeta('currency_format')
        ]);

        $this->view->title = 'Account Settings';

        return $this->render('index', [
            'currencies' => Currency::getList(),
            'timeZones' => ArrayHelper::map(TimeZone::find()->all(), 'id', 'name'),
            'timeFormats' => ArrayHelper::map(TimeFormat::find()->all(), 'id', 'name'),
            'dateFormats' => ArrayHelper::map(DateFormat::find()->all(), 'id', 'name'),
            'currencyFormats' => Currency::CURRENCY_FORMATS,
            'users' => $users,
            'formModel' => $basicInfoForm
        ]);

    }


    /**
     * Add user
     */
    public function actionAddUser()
    {
        if (!Yii::$app->user->can('addSubClient', ['userId' => $this->mainUserId])) {
            return $this->accessError();
        }
        if (count(User::getAllUsersForUserId($this->mainUserId)) >= 5) {
            echo JsonTools::errorMessage('Only 5 users are allowed per account.');
            die();
        }

        $settingsForm = new AddUser();
        if (Yii::$app->request->isPost) {
            $settingsForm->load(Yii::$app->request->post());
            $settingsForm->avatar = UploadedFile::getInstance($settingsForm, 'avatar');
            $settingsForm->user_role_id = Yii::$app->params['subClientUserRoleId'];
            $settingsForm->user_parent_id = $this->mainUserId;

            if ($settingsForm->validate()) {
                $settingsForm->setDerivedAttributes();
                // save user
                $user = new User();
                $user->username = $settingsForm->username;
                $user->email = $settingsForm->email;
                $user->user_role_id = Yii::$app->params['subClientUserRoleId'];

                $passwordPlain = $settingsForm->password;
                // check password
                if (strlen($settingsForm->password) > 0) {
                    $password = new Password($settingsForm->password);
                    $user->password = $password->getHash();
                    $user->password_salt = $password->getSalt();
                    $settingsForm->password = '';
                    $settingsForm->password_repeat = '';
                }
                $user->save();

                // we need this here so we can unset avatar if none given
                $formValues = $settingsForm->getAttributes();

                // save file
                if (($formValues['avatar']) && (strlen($formValues['avatar']) > 0)){
                    $avatarObj = UploadedFile::getInstance($settingsForm, 'avatar');
                    $avatarName = $user->id . '-' . time() . '.' . $avatarObj->extension;
                    // don't know how to turn off borders by looking at docs, so we're just cropping the image
                    $image = $user->saveAvatar($avatarObj, $avatarName);

                    if ($image) {
                        $formValues['avatar'] = $avatarName;
                    }
                }
                else {
                    unset($formValues['avatar']);
                }

                // save meta
                $metaKeys = UserMeta::getMetaKeys();

                foreach ($formValues as $key => $value) {
                    if (in_array($key, $metaKeys)) {
                        $userMetaEntry = new UserMeta();
                        $userMetaEntry->user_id = $user->id;
                        $userMetaEntry->key = $key;
                        $userMetaEntry->value = $value;

                        if (isset($settingsForm->{$key . '_visible'})) {
                            $userMetaEntry->visible = $settingsForm->{$key . '_visible'};
                        }
                        $userMetaEntry->save();
                    }
                }

                // set parent-child relationship
                $userParent = new UserParent();
                $userParent->user_id = $user->id;
                $userParent->user_parent_id = $this->mainUserId;
                $userParent->save();

                // send email with the info
                $parentUser = User::findOne($this->mainUserId);
                $parentUser->sendNewUserEmail($user->username, $passwordPlain);

                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('User successfully created.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($settingsForm);
                die();
            }
        }

        // get address settings for main user copy
        $mainUser = User::findOne($this->mainUserId);
        $meta = $mainUser->getUserMetaSorted();

        $this->view->title = 'Add User';

        return $this->render('add-user', [
            'formModel' => $settingsForm,
            'countries' => ArrayHelper::map(Country::find()->all(), 'id', 'name'),
            'days' => $this->days,
            'months' => $this->months,
            'years' => $this->years,
            'mainUserAdrLine1' => $meta['address_line_1'],
            'mainUserAdrLine2' => $meta['address_line_2'],
            'mainUserCountryId' => $meta['country_id']
        ]);
    }


    /**
     * Edit user
     */
    public function actionEditUser()
    {
        $settingsForm = new AddUser();

        $noBasicInfo = Yii::$app->getRequest()->getQueryParam('user-details', false);
        if ($noBasicInfo) {
            $settingsForm->scenario = 'subscription-details';
        }

        // check if editing a sub-client, or coach editing client
        $urlUserId = Yii::$app->getRequest()->getQueryParam('id', false);

        if ($urlUserId && (Yii::$app->user->can('addSubClient', ['userId' => $this->mainUserId]) ||
            Yii::$app->user->isCoach || Yii::$app->user->isSuperAdmin)) {
            $userId = $urlUserId;
        }
        else {
            $userId = Yii::$app->user->identity->id;
        }

        // check if subclient is editing his own details
        if (Yii::$app->user->isSubClient) {
            if ($urlUserId != Yii::$app->user->id) {
                return $this->accessError();
            }
        }

        if (Yii::$app->request->isPost) {
            $settingsForm->load(Yii::$app->request->post());
            $settingsForm->avatar = UploadedFile::getInstance($settingsForm, 'avatar');
            $settingsForm->user_role_id = Yii::$app->params['subClientUserRoleId'];
            $settingsForm->user_parent_id = $this->mainUserId;

            if ($settingsForm->validate()) {
                $settingsForm->setDerivedAttributes();

                // save user
                $user = User::findOne(['id' => $userId]);
                $user->username = $settingsForm->username;
                $user->email = $settingsForm->email;

                // check password
                $newPassword = null;
                if (strlen($settingsForm->password) > 0) {
                    $newPassword = $settingsForm->password;
                    $password = new Password($settingsForm->password);
                    $user->password = $password->getHash();
                    $user->password_salt = $password->getSalt();
                    $settingsForm->password = '';
                    $settingsForm->password_repeat = '';
                }
                $user->save();

                // save changes to current identity
                Yii::$app->user->identity->username = $user->username;
                Yii::$app->user->identity->email = $user->email;

                // we need this here so we can unset avatar if none given
                $formValues = $settingsForm->getAttributes();

                // save file
                if (($formValues['avatar']) && (strlen($formValues['avatar']) > 0)) {
                    $avatarObj = UploadedFile::getInstance($settingsForm, 'avatar');
                    $avatarName = $user->id . '-' . time() . '.' . $avatarObj->extension;
                    // don't know how to turn off borders by looking at docs, so we're just cropping the image
                    $image = $user->saveAvatar($avatarObj, $avatarName);
                    if ($image) {
                        $formValues['avatar'] = $avatarName;
                    }
                }
                else {
                    unset($formValues['avatar']);
                }

                // save meta
                $metaKeys = UserMeta::getMetaKeys();
                $userMetaSorted = UserMeta::findAllSorted(['user_id' => $userId]);
                $userMetaNew = Yii::$app->session->get('userMeta');

                foreach ($formValues as $key => $value) {
                    if (in_array($key, $metaKeys)) {
                        if (in_array($key, array_keys($userMetaSorted))){
                            $userMetaEntry = UserMeta::findOne([
                                'user_id' => $userId,
                                'key' => $key
                            ]);
                        }
                        else {
                            $userMetaEntry = new UserMeta();
                            $userMetaEntry->user_id = $userId;
                            $userMetaEntry->key = $key;
                        }

                        $userMetaEntry->value = $value;
                        if (isset($settingsForm->{$key . '_visible'})) {
                            $userMetaEntry->visible = $settingsForm->{$key . '_visible'};
                        }
                        $userMetaEntry->save();
                        $userMetaNew[$key] = $value;
                    }
                }

                // update membership user (if exists)
                $user->updateMembershipUser(
                    $newPassword,
                    !empty($formValues['first_name']) ? $formValues['first_name'] : null,
                    !empty($formValues['last_name']) ? $formValues['last_name'] : null
                );

                // save meta for session
                Yii::$app->session->set('userMeta', $userMetaNew);

                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Settings successfully saved.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($settingsForm);
                die();
            }
        }
        else {
            $noClose = Yii::$app->getRequest()->getQueryParam('no_close', false);
            $user = User::findOne(['id' => $userId]);
            $settingsForm->username = $user->username;
            $settingsForm->email = $user->email;

            // user meta
            $userMeta = UserMeta::findAll(['user_id' => $userId]);
            $userMetaSorted = ArrayHelper::map($userMeta, 'key', 'value');
            $settingsForm->setAttributes($userMetaSorted, false);
            $userMetaVisible = ArrayHelper::map($userMeta, 'key', 'visible');

            foreach ($userMeta as $m) {
                if (isset($settingsForm->{$m->key . '_visible'})) {
                    $settingsForm->{$m->key . '_visible'} = $userMetaVisible[$m->key];
                }
            }

            if (!isset($userMetaVisible['address_line_1']) && !isset($userMetaVisible['address_line_2']) && !isset($userMetaVisible['address_line_3'])) {
                $settingsForm->address_visible = 1;
            }
            else {
                $settingsForm->address_visible = $userMetaVisible['address_line_1'];
            }

            if (isset($userMetaSorted['birth_date']) && (strlen($userMetaSorted['birth_date']) > 0)) {
                $settingsForm->birth_date_day = (int) date('d', strtotime($userMetaSorted['birth_date']));
                $settingsForm->birth_date_month = (int) date('n', strtotime($userMetaSorted['birth_date']));
                $settingsForm->birth_date_year = (int) date('Y', strtotime($userMetaSorted['birth_date']));
            }

            if (is_numeric($settingsForm->birth_date_month) && is_numeric($settingsForm->birth_date_year)) {
                $monthDate = $settingsForm->birth_date_year . '-' . $settingsForm->birth_date_month . '-1';
                $daysInMonth = date('t', strtotime($monthDate));
                for ($i = $daysInMonth + 1; $i <= 31; $i++) {
                    unset($this->days[$i]);
                }
            }
            $settingsForm->avatar = '';

            // get address settings for main user copy
            $mainUser = User::findOne($this->mainUserId);
            $mainUserMeta = $mainUser->getUserMetaSorted();

            $this->view->title = 'Edit User';
            $this->jsCallbackToView();

            return $this->render('edit-user', [
                'formModel' => $settingsForm,
                'countries' => ArrayHelper::map(Country::find()->all(), 'id', 'name'),
                'days' => $this->days,
                'months' => $this->months,
                'years' => $this->years,
                'userId' => $userId,
                'noClose' => $noClose,
                'noDependant' => ($this->mainUserId == $userId),
                'noBasicInfo' => $noBasicInfo,
                'user' => $user,
                'mainUserAdrLine1' => $mainUserMeta['address_line_1'],
                'mainUserAdrLine2' => $mainUserMeta['address_line_2'],
                'mainUserCountryId' => $mainUserMeta['country_id']
            ]);
        }
    }

    /**
     * Edit user's basic info
     */
    public function actionEditUserBasicInfo()
    {
        $basicInfoForm = new EditUserBasicInfo();
        if (Yii::$app->request->isPost) {
            $basicInfoForm->load(Yii::$app->request->post());
            if ($basicInfoForm->validate()) {
                // save meta
                $metaKeys = UserMeta::getMetaKeys();
                $userMetaSorted = UserMeta::findAllSorted(['user_id' => $this->mainUserId]);
                $userMetaNew = Yii::$app->session->get('userMeta');
                foreach ($basicInfoForm->getAttributes() as $key => $value) {
                    if (in_array($key, $metaKeys)) {
                        if (in_array($key, array_keys($userMetaSorted))){
                            $userMetaEntry = UserMeta::findOne([
                                'user_id' => $this->mainUserId,
                                'key' => $key
                            ]);
                        }
                        else {
                            $userMetaEntry = new UserMeta();
                            $userMetaEntry->user_id = $this->mainUserId;
                            $userMetaEntry->key = $key;
                        }
                        $userMetaEntry->value = $value;
                        $userMetaEntry->save();
                        $userMetaNew[$key] = $value;
                    }
                }

                // reset relevant session variables
                Yii::$app->session->remove('timeFormat');
                Yii::$app->session->remove('currency');
                Yii::$app->session->remove('dateFormat');
                Yii::$app->session->remove('timeZone');
                Yii::$app->session->remove('thousandSeparator');
                Yii::$app->session->remove('decimalSeparator');

                // save meta for session
                Yii::$app->session->set('userMeta', $userMetaNew);
                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Settings successfully saved.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($basicInfoForm);
                die();
            }
        }
    }

    public function actionUsageHistoryTable() {

        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month = Yii::$app->getRequest()->getQueryParam('month', date('m'));

        $filtered = UserLogin::find()
            ->where([
                'and',
                ['user_id' => $this->userIdentity->id],
                ['MONTH(time)' => $month],
                ['YEAR(time)' => $year],
                ['>', 'duration', 0]
            ]);

        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'time',
                'select' => 'time'
            ],
            [
                'type' => 'normal',
                'as' => 'duration',
                'name' => 'duration'
            ]
        ];

        $responseData = DataTables::getResponseData($tableColumns, $filtered, $filtered);

        foreach ($responseData['items'] as $i) {

            $hours = floor($i->duration / 3600);
            $hours = ($hours > 9) ? $hours : '0' . $hours;
            $minutes = floor(($i->duration - $hours * 3600) / 60);
            $minutes = ($minutes > 9) ? $minutes : '0' . $minutes;
            $seconds = $i->duration % 60;
            $seconds = ($seconds > 9) ? $seconds : '0' . $seconds;
            $duration = "$hours:$minutes:$seconds";

            $responseData['response']['data'][] = [
                Formatter::datetime($i->time),
                $duration
            ];

        }

        echo json_encode($responseData['response']);
        die();

    }


    public function actionUserProfile() {

        $userId = $this->user->id;
        $user = User::findOne($userId);

        $meta = $user->getUserMetaSorted();
        $registration = $user->getUserRegistrations()->one();

        if ($user->user_role_id == 3) {

            $userPlan = $user->getUserPlans()
                ->with('userPlanType')
                ->orderBy('id desc')
                ->one();

        }
        else {

            $userPlan = $user->userParent
                ->userParent
                ->getUserPlans()
                ->with('userPlanType')
                ->orderBy('id desc')
                ->one();

        }

        // login data
        $userLogins = UserLogin::find()
            ->where([
                'and',
                ['user_id' => $user->id],
                ['MONTH(time)' => date('m')],
                ['YEAR(time)' => date('Y')]
            ])
            ->all();

        $loginDurationRaw = 0;
        $loginCount = 0;
        foreach ($userLogins as $u) {

            $loginCount++;
            $loginDurationRaw += $u->duration;

        }

        $hours = floor($loginDurationRaw / 3600);
        $hours = ($hours > 9) ? $hours : '0' . $hours;
        $minutes = floor(($loginDurationRaw - $hours * 3600) / 60);
        $minutes = ($minutes > 9) ? $minutes : '0' . $minutes;
        $seconds = $loginDurationRaw % 60;
        $seconds = ($seconds > 9) ? $seconds : '0' . $seconds;
        $loginDuration = "$hours:$minutes:$seconds";

        // expenses changes
        $budget = Budget::findOne($this->activeBudgetId);
        $expenseChanges = $budget->getExpenseChangesForTable(date('Y'), date('m'), $user->id);

        // overspending jars
        $overspendingJars = [];
        $budget = Budget::findOne($this->activeBudgetId);
        $jars = $budget->getActiveJars();
        foreach ($jars as $j) {
            $balance = $j->getThisMonthsFundsRemainingAmount();
            if ($balance < 0) {
                $overspendingJars[] = [
                    'name' => $j->name,
                    'balance' => $balance
                ];
            }
        }

        // notes
        $notes = $user->getClientNotes()
            ->where(['user_id' => Yii::$app->user->id])
            ->orderBy('time_edited desc')
            ->all();

        $this->view->title = 'Consolidated report for ' . $user->getFullName(true);

        return $this->render('user-profile', [
            'user' => $user,
            'meta' => $meta,
            'registration' => $registration,
            'userPlan' => $userPlan,
            'loginCount' => $loginCount,
            'loginDuration' => $loginDuration,
            'expenseChanges' => $expenseChanges,
            'overspendingJars' => $overspendingJars,
            'notes' => $notes
        ]);

    }


    /**
     * Remove an user
     */
    public function actionRemoveUser() {

        $userId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        // get the user
        $user = User::findOne(['id' => $userId]);
        if (!is_numeric($userId) || (!is_object($user->userParent)) || ($user->userParent->user_parent_id != $this->mainUserId)) {

            echo JsonTools::errorMessage('You are not allowed to remove this user.');
            die();

        }

        if ($confirm) {

            $user->archived = 1;
            $user->save();

            echo JsonTools::successMessage('User successfully removed.');
            die();

        }

        $this->view->title = 'Remove User';

        return $this->render('remove-user', [
            'userId' => $user->id
        ]);

    }

}
