<?php

namespace backend\models\db;

use backend\components\helpers\Formatter;
use Yii;
use backend\models\ChangeMetaModel;
use yii\helpers\ArrayHelper;

/**
 * This is the model class for table "accounts".
 *
 * @property integer $id
 * @property integer $budget_id
 * @property string $name
 * @property double $balance
 * @property integer $account_type_id
 * @property string $created_time
 * @property integer $archived
 *
 * @property Budget $budget
 * @property Expense[] $expenses
 * @property Income[] $incomes
 * @property Transaction[] $transactions
 * @property Transfer[] $transfers
 * @property Transfer[] $transfers0
 */
class Account extends ChangeMetaModel
{

    private $_currentBalance;
    private $_thisMonthsMoneyOperations;
    private $_lastMonthsAccountBalance;

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'accounts';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['budget_id', 'name', 'account_type_id'], 'required'],
            [['budget_id', 'account_type_id', 'archived'], 'integer'],
            [['balance'], 'number'],
            [['created_time'], 'safe'],
            [['name'], 'string', 'max' => 60]
        ];
    }

    public function fields()
    {
        $fields = parent::fields();
        unset($fields['account_type_id']);
        $fields['account_type'] = function() {
            return $this->getAccountType()->one()->name;
        };
        return $fields;
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'budget_id' => 'Budget ID',
            'name' => 'Name',
            'balance' => 'Balance',
            'account_type_id' => 'Account Type ID',
            'created_time' => 'Created Time',
            'archived' => 'Archived',
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBudget()
    {
        return $this->hasOne(Budget::className(), ['id' => 'budget_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getExpenses()
    {
        return $this->hasMany(Expense::className(), ['account_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getIncomes()
    {
        return $this->hasMany(Income::className(), ['account_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTransactions()
    {
        return $this->hasMany(Transaction::className(), ['account_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTransfersFrom()
    {
        return $this->hasMany(Transfer::className(), ['account_from_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTransfersTo()
    {
        return $this->hasMany(Transfer::className(), ['account_to_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccountChanges()
    {
        return $this->hasMany(AccountChange::className(), ['account_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccountType()
    {
        return $this->hasOne(AccountType::className(), ['id' => 'account_type_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccountDebts()
    {
        return $this->hasMany(Debt::className(), ['active_debt_account_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccountBalances()
    {
        return $this->hasMany(AccountBalance::className(), ['account_id' => 'id']);
    }

    /**
     * @inheritdoc
     */
    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);
        if ($this->scenario == self::SCENARIO_CLONE) {
            return;
        }

        // account created
        if ($insert) {
            $accountChange = new AccountChange();
            $accountChange->setAttributes([
                'type'       => 'C',
                'account_id' => $this->id,
                'user_id'    => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $accountChange->save();
        }
        // account details changed
        else {
            // check if and what changed
            $changes = [];
            $accountAttrs = $this->getAttributes();

            foreach ($changedAttributes as $key => $o) {
                if ($accountAttrs[$key] != $o) {
                    $changes[$key] = $o;
                }
            }

            if (count($changes) > 0) {
                $accountChange = new AccountChange();
                $accountChange->setAttributes([
                    'type'       => 'E',
                    'account_id' => $this->id,
                    'user_id'    => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                        Yii::$app->user->identity->id : null
                ]);
                $accountChange->save();

                foreach($changes as $key => $value) {
                    $accountChangeMeta = new AccountChangeMeta();
                    $accountChangeMeta->setAttributes([
                        'account_change_id' => $accountChange->id,
                        'key'               => $key,
                        'value'             => (string)$value
                    ]);
                    $accountChangeMeta->save();
                }
            }
        }
        // account archived
        if (isset($changedAttributes['archived']) && $this->archived == 1) {
            $accountChange = new AccountChange();
            $accountChange->setAttributes([
                'type'       => 'D',
                'account_id' => $this->id,
                'user_id'    => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $accountChange->save();
        }
    }


    /**
     * Get adjustment amount for the account for the given month
     * @param string|null $month
     * @param string|null $year
     * @return float
     */
    public function getAdjustmentAmount($month = null, $year = null)
    {
        if (!$month) {
            $month = date('n');
        }
        if (!$year) {
            $year = date('Y');
        }
        $transactions = $this->getMonthsAdjustmentsTransactions($month, $year);
        $amount = 0;
        foreach ($transactions as $t) {
            if (get_class($t) == 'backend\models\db\IncomeTransaction') {
                $amount += $t->amount;
            }
            else {
                $amount -= $t->amount;
            }
        }
        return $amount;
    }

    /**
     *
     * @param string $month
     * @param string $year
     * @return array
     */
    public function getMonthsAdjustmentsTransactions($month, $year)
    {
        $from = date('Y-m-01', strtotime("$year-$month-20"));
        $to   = date('Y-m-t', strtotime("$year-$month-20"));

        $incomes = IncomeTransaction::find()
            ->joinWith('income')
            ->where([
                'and',
                ['between', IncomeTransaction::tableName() . '.date', $from, $to],
                [IncomeTransaction::tableName() . '.account_id' => $this->id],
                ['is_adjustment' => Income::INCOME_ADJUSTMENT]
            ])
            ->all();

        $transactions = Transaction::find()
            ->joinWith('expense')
            ->where([
                'and',
                ['between', Transaction::tableName() . '.date', $from, $to],
                [Transaction::tableName() . '.account_id' => $this->id],
                ['is_adjustment' => Expense::EXPENSE_ADJUSTMENT]
            ])
            ->all();

        return array_merge($incomes, $transactions);
    }

    /**
     * Delete time, if the account was deleted
     * @return null|string
     */
    public function getDeletedTime()
    {
        $deleted = AccountChange::findOne([
            'account_id' => $this->id,
            'type' => 'D'
        ]);
        return $deleted ? $deleted->time : null;
    }

    /**
     * Gets the first money operation date for the account
     * @return string
     */
    public function getFirstMoneyOperationDate()
    {
        return $this->getMoneyOperationDate(true);
    }

    /**
     * Gets the last money operation date for the account
     * @return string
     */
    public function getLastMoneyOperationDate()
    {
        return $this->getMoneyOperationDate(false);
    }

    /**
     * Finds the date of the first or the last money operation for the account
     * @param bool $first
     * @return string
     */
    public function getMoneyOperationDate($first = true)
    {
        $transaction = Transaction::find()
            ->joinWith('expense')
            ->where([
                Transaction::tableName() . '.account_id' => $this->id,
                Expense::tableName() . '.archived' => 0
            ])
            ->orderBy(Transaction::tableName() . '.date '.($first ? 'ASC' : 'DESC'))
            ->one();

        $debtPayment = DebtPayment::find()
            ->joinWith('debt')
            ->where([
                DebtPayment::tableName() . '.account_id' => $this->id,
                Debt::tableName() . '.archived' => 0
            ])
            ->orderBy(DebtPayment::tableName() . '.date '.($first ? 'ASC' : 'DESC'))
            ->one();

        $transfer = TransferEvent::find()
            ->joinWith('transfer')
            ->where([
                Transfer::tableName() . '.archived' => 0
            ])
            ->andWhere(['or',
                [Transfer::tableName() . '.account_from_id' => $this->id],
                [Transfer::tableName() . '.account_to_id' => $this->id]
            ])
            ->orderBy(TransferEvent::tableName() . '.date '.($first ? 'ASC' : 'DESC'))
            ->one();

        $income = IncomeTransaction::find()
            ->joinWith('income')
            ->where([
                IncomeTransaction::tableName() . '.account_id' => $this->id,
                Income::tableName() . '.archived' => 0
            ])
            ->orderBy(IncomeTransaction::tableName() . '.date '.($first ? 'ASC' : 'DESC'))
            ->one();

        $date = date('Y-m-d');
        if ($transaction &&
            $first && $transaction->date < $date || !$first && $transaction->date > $date) {
            $date = $transaction->date;
        }
        if ($debtPayment &&
            $first && $debtPayment->date < $date || !$first && $debtPayment->date > $date) {
            $date = $debtPayment->date;
        }
        if ($transfer &&
            $first && $transfer->date < $date || !$first && $transfer->date > $date) {
            $date = $transfer->date;
        }
        if ($income &&
            $first && $income->date < $date || !$first && $income->date > $date) {
            $date = $income->date;
        }

        return $date;
    }

    /**
     * Get the account balance as it was when the account was created
     * @return float
     */
    public function getInitialBalance()
    {
        // try to get the last value before the first transaction
        // the change meta holds the date when the change happened
        // and since the balance change means *Last Month End-Balance*
        // it will be valid for the entire month when first transaction happened
        $firstTransactionDate = $this->getFirstMoneyOperationDate();
        $edit = AccountChangeMeta::find()
            ->joinWith('accountChange')
            ->where([
                'account_id' => $this->id,
                'type' => 'E',
                'key' => 'balance'
            ])
            ->andWhere([
                '>', 'time', date('Y-m-t 23:59:59', strtotime($firstTransactionDate))
            ])
            ->orderBy('time ASC')
            ->one();
        if ($edit) {
            return $edit->value;
        }
        // just get the balance
        return $this->balance;
    }

    /**
     * Get the balance as it was at the end of the given month/year. If month/year not provided
     * the balance is as of today.
     * @param null $month
     * @param null $year
     * @param bool $includeBalanceAdjustments
     * @return bool|float
     */
    public function getCurrentBalance($month = null, $year = null, $includeBalanceAdjustments = true)
    {
        if (!$month) {
            $month = date('n');
        }
        if (!$year) {
            $year = date('Y');
        }
        $timeNow = strtotime("$year-$month-20");
        $currentBalance = $this->getMonthsAccountBalance(
            date('n', strtotime('-1 month', $timeNow)),
            date('Y', strtotime('-1 month', $timeNow))
        );
        if (!$currentBalance) {
            // try with EoMS from 2 months ago
            $currentBalance = $this->getMonthsAccountBalance(
                date('n', strtotime('-2 months', $timeNow)),
                date('Y', strtotime('-2 months', $timeNow))
            );
            if ($currentBalance) {
                $startTime = strtotime(date('Y-m-01', strtotime('-1 month', $timeNow)));
            }
            else {
                // need to calculate all activity since the account inception
                $currentBalance = $this->getInitialBalance();
                $startTime = strtotime($this->created_time);
            }
            $payments = $this->getTimePeriodIncomeTransactions(date('Y-m-01', $startTime), date('Y-m-t', $timeNow), $includeBalanceAdjustments);
            $expenses = $this->getTimePeriodExpenseTransactions(date('Y-m-01', $startTime), date('Y-m-t', $timeNow), $includeBalanceAdjustments);
            if ($this->account_type_id == 1 || $this->account_type_id == 2) {
                foreach ($payments as $payment) {
                    $currentBalance += $payment->amount;
                }
                foreach ($expenses as $expense) {
                    $currentBalance -= $expense->amount;
                }
            }
            else {
                foreach ($payments as $payment) {
                    $currentBalance -= $payment->amount;
                }
                foreach ($expenses as $expense) {
                    $currentBalance += $expense->amount;
                }
            }
            return $currentBalance;
        }
        else {
            $payments = $this->getMonthsIncomeTransactionsAmount($month, $year, $includeBalanceAdjustments);
            $expenses = $this->getMonthsExpenseTransactionsAmount($month, $year, $includeBalanceAdjustments);
            // the balance for debt/cc accounts is treated as a negative value for the account
            return $this->account_type_id == 1 || $this->account_type_id == 2 ?
                $currentBalance + $payments - $expenses :
                $currentBalance - $payments + $expenses;
        }
    }

    /**
     * Calculates and returns the resulting balance
     * after applying the $operation to the $currentBalance
     * @param double $currentBalance
     * @param $operation
     * @param bool $reverseSign Necessary when calculating the balance from the previous months
     * @return float
     */
    public function getBalanceAfterOperation($currentBalance, $operation, $reverseSign = false)
    {
        $partialBalance = $this->getAmountWithSign($operation);
        return $reverseSign ?
            $currentBalance - $partialBalance :
            $currentBalance + $partialBalance;
    }

    /**
     * Returns negative amount if the operation was expense, positive otherwise
     * @param $operation
     * @return mixed
     */
    public function getAmountWithSign($operation)
    {
        if ((get_class($operation) == 'backend\models\db\DebtPayment') || (get_class($operation) == 'backend\models\db\Transaction')) {
            return -$operation->amount;
        }
        elseif (get_class($operation) == 'backend\models\db\TransferEvent') {
            if ($operation->account_from_id == $this->id) {
                return -$operation->amount;
            }
            else {
                return $operation->amount;
            }
        }
        else {
            return $operation->amount;
        }
    }

    /**
     * Get unique account balance operation code basing on the operation type
     * @param $operation
     * @return string
     */
    public static function getOperationType($operation)
    {
        switch (get_class($operation)) {
            case 'backend\models\db\DebtPayment':
                $k = 'dp';
                break;
            case 'backend\models\db\Transaction':
                $k = 'tr';
                break;
            case 'backend\models\db\TransferEvent':
                $k = 'te';
                break;
            case 'backend\models\db\IncomeTransaction':
                $k = 'it';
                break;
            default:
                $k = 'ot';
                break;
        }
        return $k;
    }

    /**
     * Get unique operation key basing on its type and ID
     * @param $operation
     * @return string
     */
    public static function getOperationKey($operation)
    {
        return self::getOperationType($operation) . $operation->id;
    }

    /**
     * @param $type
     * @param $id
     * @return null|\yii\db\ActiveRecord
     */
    public static function getOperationByType($type, $id)
    {
        $operation = null;
        switch ($type) {
            case 'dp':
                $operation = DebtPayment::findOne($id);
                break;
            case 'tr':
                $operation = Transaction::findOne($id);
                break;
            case 'te':
                $operation = TransferEvent::findOne($id);
                break;
            case 'it':
                $operation = IncomeTransaction::findOne($id);
                break;
        }
        return $operation;
    }


    /**
     * Get a list of transactions for a certain month and year.
     * @param int $month Month.
     * @param int $year Year.
     * @return array List of transactions.
     */
    public function getMonthsMoneyOperations($month, $year)
    {
        return $this->getTimePeriodMoneyOperations(
            date("$year-$month-1"),
            date("$year-$month-" . date('t', strtotime("$year-$month-1")))
        );
    }

    /**
     * Get a list of transactions for the given date span.
     * @param $from From date
     * @param $to To date
     * @param bool $includeBalanceAdjustments
     * @return array
     */
    public function getTimePeriodMoneyOperations($from, $to, $includeBalanceAdjustments = true)
    {
        $from = date('Y-m-d', strtotime($from));
        $to   = date('Y-m-d', strtotime($to));

        $expenses = $this->getTimePeriodExpenseTransactions($from, $to, $includeBalanceAdjustments);
        $incomes  = $this->getTimePeriodIncomeTransactions($from, $to, $includeBalanceAdjustments);

        $moneyOperations = array_merge($expenses, $incomes);

        return $moneyOperations;

    }


    /**
     * The transactions for this month
     * @return array
     */
    public function getThisMonthsMoneyOperations()
    {
        if ($this->_thisMonthsMoneyOperations == null) {
            $this->_thisMonthsMoneyOperations = $this->getMonthsMoneyOperations(date('m'), date('Y'));
        }
        return $this->_thisMonthsMoneyOperations;
    }


    /**
     * Get the resulting balance of all money operations in this month.
     * @param int $month Month.
     * @param int $year Year.
     * @param int $accountId ID of filtered account.
     * @return float Resulting sum of all money operations in this month.
     */
    public function getMonthsMoneyOperationsAmount($month, $year)
    {
        $moneyOperations = $this->getMonthsMoneyOperations($month, $year);

        $amount = 0;
        foreach ($moneyOperations as $m) {
            if (get_class($m) == 'backend\models\db\DebtPayment') {
                $amount -= $m->amount;
            }
            elseif (get_class($m) == 'backend\models\db\Transaction') {
                $amount -= $m->amount;
            }
            elseif (get_class($m) == 'backend\models\db\IncomeTransaction') {
                $amount += $m->amount;
            }
        }

        return $amount;
    }


    /**
     * Get a list of expenses for a certain month and year.
     * @param int $month Month.
     * @param int $year Year.
     * @return array List of transactions.
     */
    public function getMonthsExpenseTransactions($month, $year, $includeAdjustments = true, $includeTransfers = true)
    {
        return $this->getTimePeriodExpenseTransactions(
            date("$year-$month-1"),
            date("$year-$month-" . date('t', strtotime("$year-$month-1"))),
            $includeAdjustments,
            $includeTransfers
        );
    }

    /**
     * Get a list of expenses for the given date span.
     * @param $from From date
     * @param $to To date
     * @param bool $includeBalanceAdjustments
     * @param bool $includeTransfers
     * @return array
     */
    public function getTimePeriodExpenseTransactions($from, $to, $includeBalanceAdjustments = true, $includeTransfers = true)
    {
        $from = date('Y-m-d', strtotime($from));
        $to   = date('Y-m-d', strtotime($to));
        if ($from > $to) {
            return [];
        }

        if ($includeTransfers) {
            $transfers = TransferEvent::find()
                ->select([
                    'id' => TransferEvent::tableName() . '.id',
                    'date' => TransferEvent::tableName() . '.date',
                    'name' => Transfer::tableName() . '.name',
                    'amount' => TransferEvent::tableName() . '.amount',
                    'account_from_id' => 'a_from.id',
                    'account_from_name' => 'a_from.name',
                    'account_to_id' => 'a_to.id',
                    'account_to_name' => 'a_to.name',
                    'account_name' => 'CONCAT(a_from.name, " => ", a_to.name)',
                    'jar_name' => 'CONCAT("")',
                    'checked' => TransferEvent::tableName() . '.checked',
                    'is_adjustment' => Transfer::tableName() . '.is_adjustment'
                ])
                ->joinWith('transfer')
                ->leftJoin(Account::tableName() . ' as a_from', "a_from.id = " . Transfer::tableName() . ".account_from_id")
                ->leftJoin(Account::tableName() . ' as a_to', "a_to.id = " . Transfer::tableName() . ".account_to_id")
                ->where(['and',
                    ['between', TransferEvent::tableName() . '.date', $from, $to],
                    ['a_from.id' => $this->id]
                ]);
            if (!$includeBalanceAdjustments) {
                $transfers->andWhere([
                    'is_adjustment' => 0
                ]);
            }
            $transfers = $transfers->all();
        }
        else {
            $transfers = [];
        }

        $transactions = Transaction::find()
            ->select([
                'id' => Transaction::tableName() . '.id',
                'date' => Transaction::tableName() . '.date',
                'expense_name' => Expense::tableName() . '.name',
                'name' => Transaction::tableName() . '.description',
                'amount' => Transaction::tableName() . '.amount',
                'account_name' => Account::tableName() . '.name',
                'jar_name' => Jar::tableName() . '.name',
                'checked' => Transaction::tableName() . '.checked',
                'is_adjustment' => Expense::tableName() . '.is_adjustment'
            ])
            ->joinWith('expense')
            ->joinWith('account')
            ->leftJoin(Jar::tableName(), Jar::tableName() . ".id = " . Expense::tableName() . ".jar_id")
            ->where([
                'and',
                ['between', Transaction::tableName() . '.date', $from, $to],
                [Account::tableName() . '.id' => $this->id]
            ]);
        if ($includeBalanceAdjustments) {
            $transactions->andWhere([
                'or',
                ['is_adjustment' => 0],
                ['is_adjustment' => Expense::EXPENSE_ADJUSTMENT]
            ]);
        }
        else {
            $transactions->andWhere([
                'is_adjustment' => 0
            ]);
        }
        $transactions = $transactions->all();

        $debtPayments = DebtPayment::find()
            ->select([
                'id' => DebtPayment::tableName() . '.id',
                'date' => DebtPayment::tableName() . '.date',
                'name' => Debt::tableName() . '.name' ,
                'amount' => DebtPayment::tableName() . '.amount' ,
                'account_name' => Account::tableName() . '.name',
                'jar_name' => DebtPaymentType::tableName() . '.name',
                'checked' => DebtPayment::tableName() . '.checked',
                'is_adjustment' => DebtPayment::tableName() . '.is_adjustment'
            ])
            ->joinWith('account')
            ->joinWith('debt')
            ->joinWith('debtPaymentType')
            ->where([
                'and',
                ['between', DebtPayment::tableName() . '.date', $from, $to],
                [Account::tableName() . '.id' => $this->id]
            ]);
        if ($includeBalanceAdjustments) {
            $debtPayments->andWhere([
                'or',
                ['is_adjustment' => 0],
                ['is_adjustment' => DebtPayment::DEBT_PAYMENT_ADJUSTMENT]
            ]);
        }
        else {
            $debtPayments->andWhere([
                'is_adjustment' => 0
            ]);
        }
        $debtPayments = $debtPayments->all();

        return array_merge($transfers, $transactions, $debtPayments);
    }

    public function getTimePeriodIncomeTransactions($from, $to, $includeBalanceAdjustments = true, $includeTransfers = true)
    {
        $from = date('Y-m-d', strtotime($from));
        $to   = date('Y-m-d', strtotime($to));
        if ($from > $to) {
            return [];
        }

        if ($includeTransfers) {
            $transfers = TransferEvent::find()
                ->select([
                    'id' => TransferEvent::tableName() . '.id',
                    'date' => TransferEvent::tableName() . '.date',
                    'name' => Transfer::tableName() . '.name',
                    'amount' => TransferEvent::tableName() . '.amount',
                    'account_from_id' => 'a_from.id',
                    'account_from_name' => 'a_from.name',
                    'account_to_id' => 'a_to.id',
                    'account_to_name' => 'a_to.name',
                    'account_name' => 'CONCAT(a_from.name, " => ", a_to.name)',
                    'jar_name' => 'CONCAT("")',
                    'checked' => TransferEvent::tableName() . '.checked',
                    'is_adjustment' => Transfer::tableName() . '.is_adjustment'
                ])
                ->joinWith('transfer')
                ->leftJoin(Account::tableName() . ' as a_from', "a_from.id = " . Transfer::tableName() . ".account_from_id")
                ->leftJoin(Account::tableName() . ' as a_to', "a_to.id = " . Transfer::tableName() . ".account_to_id")
                ->where(['and',
                    ['between', TransferEvent::tableName() . '.date', $from, $to],
                    ['a_to.id' => $this->id]
                ]);
            if (!$includeBalanceAdjustments) {
                $transfers->andWhere([
                    'is_adjustment' => 0
                ]);
            }
            $transfers = $transfers->all();
        }
        else {
            $transfers = [];
        }

        $incomes = IncomeTransaction::find()
            ->select([
                'id' => IncomeTransaction::tableName() . '.id',
                'date' => IncomeTransaction::tableName() . '.date',
                'name' => IncomeTransaction::tableName() . '.description',
                'amount' => IncomeTransaction::tableName() . '.amount',
                'account_name' => Account::tableName() . '.name',
                'jar_name' => Jar::tableName() . '.name',
                'checked' => IncomeTransaction::tableName() . '.checked',
                'is_adjustment' => Income::tableName() . '.is_adjustment'
            ])
            ->joinWith('account')
            ->joinWith('income')
            ->leftJoin(Jar::tableName(), Jar::tableName() . ".id = " . Income::tableName() . ".jar_id")
            ->where([
                'and',
                ['between', IncomeTransaction::tableName() . '.date', $from, $to],
                [Account::tableName() . '.id' => $this->id]
            ]);
        if ($includeBalanceAdjustments) {
            $incomes->andWhere([
                'or',
                ['is_adjustment' => 0],
                ['is_adjustment' => Income::INCOME_ADJUSTMENT]
            ]);
        }
        else {
            $incomes->andWhere([
                'is_adjustment' => 0
            ]);
        }
        $incomes = $incomes->all();

        return array_merge($transfers, $incomes);
    }

    /**
     * Get total amount of expenses for month & year.
     * @param int $month Month.
     * @param int $year Year.
     * @return float Amount.
     */
    public function getMonthsExpenseTransactionsAmount($month, $year, $includeAdjustments = true, $includeTransfers = true)
    {
        $expenses = $this->getMonthsExpenseTransactions($month, $year, $includeAdjustments, $includeTransfers);
        $amount = 0;
        foreach ($expenses as $e) {
            $amount += $e->amount;
        }
        return $amount;
    }

    /**
     * Get a list of expenses for a certain month and year.
     * @param int $month Month.
     * @param int $year Year.
     * @return array List of transactions.
     */
    public function getMonthsIncomeTransactions($month, $year, $includeAdjustments = true, $includeTransfers = true)
    {
        return $this->getTimePeriodIncomeTransactions(
            date("$year-$month-1"),
            date("$year-$month-" . date('t', strtotime("$year-$month-20"))),
            $includeAdjustments,
            $includeTransfers
        );
    }

    /**
     * Get total amount of income transactions for month & year.
     * @param int $month Month.
     * @param int $year Year.
     * @return float Amount.
     */
    public function getMonthsIncomeTransactionsAmount($month, $year, $includeAdjustments = true, $includeTransfers = true)
    {
        $transactions = $this->getMonthsIncomeTransactions($month, $year, $includeAdjustments, $includeTransfers);
        $amount = 0;
        foreach ($transactions as $t) {
            $amount += $t->amount;
        }
        return $amount;
    }

    /**
     * Get the account balance as it was submitted in the EoMS for the given month
     * @param $month
     * @param $year
     * @return bool|float
     */
    public function getMonthsAccountBalance($month, $year)
    {
        $accountBalance = AccountBalance::find()
            ->joinWith('eoms')
            ->joinWith('account.budget')
            ->where([
                'account_id' => $this->id,
                'month' => $month,
                'year' => $year,
                'is_mock' => 0,
                'submitted' => 1
            ])
            ->one();
        return $accountBalance ? $accountBalance->balance : false;
    }

    /**
     * Get the latest submitted to EoMS account balance
     * @return bool|float
     */
    public function getLatestSubmittedAccountBalance()
    {
        $accountBalance = AccountBalance::find()
            ->joinWith('eoms')
            ->joinWith('account.budget')
            ->where([
                'account_id' => $this->id,
                'is_mock' => 0,
                'submitted' => 1
            ])
            ->orderBy('year DESC, month DESC')
            ->one();
        return $accountBalance ? $accountBalance->balance : false;
    }

    /**
     * Get last month's logged account balance
     * @return bool|float
     */
    public function getLastMonthsAccountBalance()
    {
        if ($this->_lastMonthsAccountBalance == null) {
            $timeNow = strtotime(date('Y-m-20'));
            $this->_lastMonthsAccountBalance = $this->getMonthsAccountBalance(
                date('n', strtotime('-1 month', $timeNow)),
                date('Y', strtotime('-1 month', $timeNow))
            );
        }
        return $this->_lastMonthsAccountBalance;
    }

    /**
     * Saves adjustment records to reflect the adjustment in real transactions
     * @param Eoms $eoms
     * @param $oldAmount
     * @param $newAmount
     */
    public function saveBalanceAdjustmentRecord(Eoms $eoms, $oldAmount, $newAmount, $isCcAdjustment = true)
    {
        $absBalance      = Formatter::float(abs($oldAmount - $newAmount));
        $transactionDate = date('Y-m-t', strtotime($eoms->year . '-' . $eoms->month . '-20'));
        $operationName   = ($isCcAdjustment ? 'Credit Card' : 'Account') .
            ' Amount Adjustment for ' . date('M Y', strtotime("{$eoms->year}-{$eoms->month}-20"));

        if ($isCcAdjustment) {
            $debtJar = Jar::find()
                ->where([
                    'type' => 'debts',
                    'jar_type_id' => 2,
                    'budget_id' => $eoms->budget_id
                ])
                ->one();
            $jarId = $debtJar->id;
        }
        else {
            $jarId = null;
        }

        Income::deleteAll([
            'account_id'    => $this->id,
            'jar_id'        => $jarId,
            'date'          => $transactionDate,
            'is_adjustment' => Income::INCOME_ADJUSTMENT
        ]);
        Expense::deleteAll([
            'account_id'    => $this->id,
            'jar_id'        => $jarId,
            'date'          => $transactionDate,
            'is_adjustment' => Expense::EXPENSE_ADJUSTMENT
        ]);

        if ($absBalance > 0) {
            // it's an expense
            if ($newAmount < $oldAmount) {
                $expense = new Expense();
                $expense->setAttributes([
                    'jar_id'                => $jarId,
                    'name'                  => $operationName,
                    'account_id'            => $this->id,
                    'amount'                => $absBalance,
                    'date'                  => $transactionDate,
                    'frequency'             => 'one-time',
                    'is_unbudgeted'         => 1,
                    'is_adjustment'         => Expense::EXPENSE_ADJUSTMENT,
                    'generate_transactions' => 0
                ]);
                $expense->save();

                $transaction = $expense->getTransactions()->one();
                if (!$transaction) {
                    $transaction = new Transaction();
                    $transaction->setAttributes([
                        'account_id'     => $this->id,
                        'expense_id'     => $expense->id,
                        'date'           => $expense->date,
                        'description'    => $expense->name,
                        'amount'         => $expense->amount,
                        'tax_deductible' => $expense->tax_deductible ? 1 : 0
                    ]);
                    $transaction->save();
                }
            }
            // it's an income
            elseif ($newAmount > $oldAmount) {
                $income = new Income();
                $income->setAttributes([
                    'budget_id'     => $eoms->budget_id,
                    'account_id'    => $this->id,
                    'jar_id'        => $jarId,
                    'name'          => $operationName,
                    'amount'        => $absBalance,
                    'date'          => $transactionDate,
                    'frequency'     => 'one-time',
                    'is_adjustment' => Income::INCOME_ADJUSTMENT
                ]);
                $income->save();

                $transaction = $income->getIncomeTransactions()->one();
                if (!$transaction) {
                    $transaction = new IncomeTransaction();
                    $transaction->setAttributes([
                        'income_id'   => $income->id,
                        'account_id'  => $this->id,
                        'amount'      => $income->amount,
                        'date'        => $income->date,
                        'description' => $income->name
                    ]);
                    $transaction->save();
                }
            }
        }
    }

    /**
     * @return int|null
     */
    public function getMockBudgetId()
    {
        if ($this->budget->is_mock) {
            return $this->budget_id;
        }
        return null;
    }

    public function deleteCompletely()
    {
        foreach ($this->getAccountChanges() as $change) {
            AccountChangeMeta::deleteAll(['account_change_id' => $change->id]);
        }
        AccountChange::deleteAll(['account_id' => $this->id]);
        AccountAdjustment::deleteAll(['account_id' => $this->id]);
        AccountBalance::deleteAll(['account_id' => $this->id]);
        AccountEomsBalance::deleteAll(['account_id' => $this->id]);
        $this->delete();
    }

    public static function getAccountsList($budgetId, $noCcAccounts = false)
    {
        $accounts = Account::find()
            ->where([
                'budget_id' => $budgetId,
                'archived' => 0
            ])
            ->orderBy('name ASC');
        if ($noCcAccounts) {
            $accounts->andWhere([
                'account_type_id' => 1
            ]);
        }
        return $accounts->all();
    }

    /**
     * Get a list of accounts for select input element.
     * @param $budgetId
     * @param bool $noCcAccounts
     * @return array List of accounts [ID => name]
     */
    public static function getAccountsListForSelect($budgetId, $noCcAccounts = false)
    {
        return ArrayHelper::map(self::getAccountsList($budgetId, $noCcAccounts), 'id', 'name');
    }

}
