<?php

namespace backend\models\db;

use backend\models\form\AdjustJarFundsRemaining;
use Yii;

use backend\models\ChangeMetaModel;
use yii\helpers\ArrayHelper;
use backend\components\helpers\Formatter;
use backend\components\helpers\Frequencies;


/**
 * This is the model class for table "budgets".
 *
 * @property integer $id
 * @property integer $user_id
 * @property string $name
 * @property string $created_time
 * @property integer $is_mock
 * @property integer $archived
 *
 * @property Account[] $accounts
 * @property BudgetChats[] $budgetChats
 * @property User $user
 * @property Jar[] $jars
 */
class Budget extends \yii\db\ActiveRecord {


    private $_activeJars;
    private $_totalMonthlyBudget;
    private $_minimumMonthlyIncomeAmount;
    private $_averageMonthlyIncomeAmount;
    private $_thisMonthsExpensesAmount;
    private $_thisMonthsIncomes;
    private $_thisMonthsMoneyOperations;
    private $_thisMonthsDebtPayments;
    private $_thisMonthsDebtPaymentsAmount = [];
    private $_thisMonthsDebtInterestPayments;
    private $_thisMonthsDebtInterestPaymentsAmount;
    private $_thisMonthsDebtInterestAmount;
    private $_thisMonthsDebtsAmount;
    private $_thisMonthsDebts;
    private $_thisMonthsIncomeTransactions;
    private $_thisMonthsIncomeTransactionsAmount;
    private $_thisMonthsTransactions;
    private $_thisMonthsTransactionsAmount;
    private $_thisMonthsBudgetBalance;

    /**
     * @inheritdoc
     */
    public static function tableName() {
        return 'budgets';
    }

    /**
     * @inheritdoc
     */
    public function rules() {
        return [
            [['user_id', 'default_account_id', 'archived'], 'integer'],
            [['name'], 'string', 'max' => 200],
            [['created_time', 'is_mock'], 'safe'],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels() {
        return [
            'id' => 'ID',
            'user_id' => 'User ID',
            'name' => 'Name',
            'default_account_id' => 'Default Account ID',
            'is_mock' => 'Is Mock',
            'archived' => 'Archived'
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccounts() {
        return $this->hasMany(Account::className(), ['budget_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBudgetChats() {
        return $this->hasMany(BudgetChat::className(), ['budget_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUser() {
        return $this->hasOne(User::className(), ['id' => 'user_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getEoms() {
        return $this->hasMany(Eoms::className(), ['budget_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getIncomes() {
        return $this->hasMany(Income::className(), ['budget_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getJars() {
        return $this->hasMany(Jar::className(), ['budget_id' => 'id']);
    }


    /**
     * @return \yii\db\ActiveQuery
     */
    public function getDebts() {
        return $this->hasMany(Debt::className(), ['budget_id' => 'id']);
    }


    public function setUpInitialEntities()
    {
        // set up the central account
        $account = new Account();
        $account->setAttributes([
            'name' => 'Central Account',
            'budget_id' => $this->id,
            'balance' => 0,
            'account_type_id' => 1
        ]);
        $account->save();

        // let's save the account as the deafult account
        $this->default_account_id = $account->id;
        $this->save();

        // set up default jars
        foreach (Yii::$app->params['defaultRegistrationJars'] as $order => $j) {
            $jar = new Jar();
            $jar->name = $j['name'];
            $jar->amount = 0;
            $jar->budget_id = $this->id;
            $jar->type = $j['type'];
            $jar->order = $order;
            $jar->save();
        }
        return $this;
    }

    /**
     * Clone a budget to a new one.
     *
     * This method used to use single SQL queries to insert new data. The issue
     * with that approach was that eventually it would have required joins accross
     * 8 or possibly even more tables for that approach to work, so I went for
     * an (Lord, forgive me) approach with inserts in a loop. I decided for this
     * mainly for maintanability and easier debugging.
     *
     * Note: the execution time was approximately the same for both approaches
     * after running for 10 tables and with an increasing number of records,
     * it can be assumed that the method currently in use will be more efficient.
     *
     * @param string $name The name of the newly cloned budget.
     * @param bool $isMock Whether the cloned budget should be a mock budget
     * @return boolean Success of the cloning process.
     */
    public function cloneBudget($name, $isMock = false) {

        // new budget
        // update other data later after everything else goes well
        $budget = new self;
        $budget->setAttributes([
            'name' => 'temp_budget_' . time(),
            'user_id' => $this->user_id,
            'archived' => 1
        ]);
        $budget->save();

        $activateBudget = true; // we'll check this to see whether all went fine and the budget can be activated
        $connection = Yii::$app->db;

        // copy accounts
        $accounts = Account::findAll(['budget_id' => $this->id]);
        $accountIdsMap = []; // old => new
        foreach ($accounts as $a) {

            $newAccount = new Account(['scenario' => ChangeMetaModel::SCENARIO_CLONE]);
            $newAccount->setAttributes($a->getAttributes());
            $newAccount->budget_id = $budget->id;
            $saved = $newAccount->save();

            if ($saved) {

                $accountIdsMap[$a->id] = $newAccount->id;

            }
            else {

                $activateBudget = false;

            }

        }

        // copy account changes
        if ($activateBudget) {

            $accountChangeIdsMap = [];

            $accountChanges = AccountChange::find()
                ->where(['IN', 'account_id', array_keys($accountIdsMap)])
                ->all();

            foreach ($accountChanges as $ac) {

                $newAccountChange = new AccountChange();
                $newAccountChange->setAttributes($ac->getAttributes());
                $newAccountChange->account_id = $accountIdsMap[$ac->account_id];
                $saved = $newAccountChange->save();

                if ($saved) {

                    $accountChangeIdsMap[$ac->id] = $newAccountChange->id;

                }
                else {

                    $activateBudget = false;

                }

            }

        }

        // copy account changes meta
        if ($activateBudget) {

            $accountChangeMeta = AccountChangeMeta::find()
                ->where(['IN', 'account_change_id', array_keys($accountChangeIdsMap)])
                ->all();

            foreach ($accountChangeMeta as $acm) {

                $newAccountChangeMeta = new AccountChangeMeta();
                $newAccountChangeMeta->setAttributes($acm->getAttributes());
                $newAccountChangeMeta->account_change_id = $accountChangeIdsMap[$acm->account_change_id];
                $saved = $newAccountChangeMeta->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy account balances
        if ($activateBudget) {

            $accountBalances = AccountBalance::find()
                ->where(['IN', 'account_id', array_keys($accountIdsMap)])
                ->all();

            foreach ($accountBalances as $ab) {

                $newAccountBalance = new AccountBalance();
                $newAccountBalance->setAttributes($ab->getAttributes());
                $newAccountBalance->account_id = $accountIdsMap[$ab->account_id];
                $saved = $newAccountBalance->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy debts
        if ($activateBudget) {

            $debts = Debt::findAll(['budget_id' => $this->id]);
            $debtIdsMap = []; // old => new
            foreach ($debts as $d) {

                $newDebt = new Debt(['scenario' => ChangeMetaModel::SCENARIO_CLONE]);
                $newDebt->setAttributes($d->getAttributes());
                $newDebt->budget_id = $budget->id;
                $newDebt->account_id = $accountIdsMap[$d->account_id];
                $saved = $newDebt->save();

                if ($saved) {

                    $debtIdsMap[$d->id] = $newDebt->id;

                }
                else {

                    $activateBudget = false;

                }

            }

        }

        // copy debt changes
        if ($activateBudget) {

            $debtChangeIdsMap = [];

            $debtChanges = DebtChange::find()
                ->where(['IN', 'debt_id', array_keys($debtIdsMap)])
                ->all();

            foreach ($debtChanges as $dc) {

                $newDebtChange = new DebtChange();
                $newDebtChange->setAttributes($dc->getAttributes());
                $newDebtChange->debt_id = $debtIdsMap[$dc->debt_id];
                $saved = $newDebtChange->save();

                if ($saved) {

                    $debtChangeIdsMap[$dc->id] = $newDebtChange->id;

                }
                else {

                    $activateBudget = false;

                }

            }

        }

        // copy debt changes meta
        if ($activateBudget) {

            $debtChangeMeta = DebtChangeMeta::find()
                ->where(['IN', 'debt_change_id', array_keys($debtChangeIdsMap)])
                ->all();

            foreach ($debtChangeMeta as $dcm) {

                $newDebtChangeMeta = new DebtChangeMeta();
                $newDebtChangeMeta->setAttributes($dcm->getAttributes());
                $newDebtChangeMeta->debt_change_id = $debtChangeIdsMap[$dcm->debt_change_id];
                $saved = $newDebtChangeMeta->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy debt ends
        // (this has to go before copying debt changes because
        // saving DebtEnd models triggers saving to DebtChange table)
        if ($activateBudget) {

            $debtEnds = DebtEnd::find()
                ->where(['IN', 'debt_id', array_keys($debtIdsMap)])
                ->all();

            foreach ($debtEnds as $de) {

                $newDebtEnd = new DebtEnd(['scenario' => ChangeMetaModel::SCENARIO_CLONE]);
                $newDebtEnd->setAttributes($de->getAttributes());
                $newDebtEnd->debt_id = $debtIdsMap[$de->debt_id];
                $saved = $newDebtEnd->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy debt payments
        if ($activateBudget) {

            $debtPayments = DebtPayment::find()
                ->where(['IN', 'debt_id', array_keys($debtIdsMap)])
                ->all();

            foreach ($debtPayments as $dp) {

                $newDebtPayment = new DebtPayment();
                $newDebtPayment->setAttributes($dp->getAttributes());
                $newDebtPayment->debt_id = $debtIdsMap[$dp->debt_id];
                $newDebtPayment->account_id = $accountIdsMap[$dp->account_id];
                $saved = $newDebtPayment->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy jars
        if ($activateBudget) {

            $jars = Jar::findAll(['budget_id' => $this->id]);
            $jarIdsMap = []; // old => new
            foreach ($jars as $j) {

                $newJar = new Jar(['scenario' => ChangeMetaModel::SCENARIO_CLONE]);
                $newJar->setAttributes($j->getAttributes());
                $newJar->budget_id = $budget->id;
                $saved = $newJar->save();

                if ($saved) {

                    $jarIdsMap[$j->id] = $newJar->id;

                }
                else {

                    $activateBudget = false;

                }

            }

        }

        // copy jar changes
        if ($activateBudget) {

            $jarChangeIdsMap = [];

            $jarChanges = JarChange::find()
                ->where(['IN', 'jar_id', array_keys($jarIdsMap)])
                ->all();

            foreach ($jarChanges as $jc) {

                $newJarChange = new JarChange();
                $newJarChange->setAttributes($jc->getAttributes());
                $newJarChange->jar_id = $jarIdsMap[$jc->jar_id];
                $saved = $newJarChange->save();

                if ($saved) {

                    $jarChangeIdsMap[$jc->id] = $newJarChange->id;

                }
                else {

                    $activateBudget = false;

                }

            }

        }

        // copy jar changes meta
        if ($activateBudget) {

            $jarChangeMeta = JarChangeMeta::find()
                ->where(['IN', 'jar_change_id', array_keys($jarChangeIdsMap)])
                ->all();

            foreach ($jarChangeMeta as $jcm) {

                $newJarChangeMeta = new JarChangeMeta();
                $newJarChangeMeta->setAttributes($jcm->getAttributes());
                $newJarChangeMeta->jar_change_id = $jarChangeIdsMap[$jcm->jar_change_id];
                $saved = $newJarChangeMeta->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy expenses
        if ($activateBudget) {

            $expenseIdsMap = [];

            $expenses = Expense::find()
                ->where(['IN', 'jar_id', array_keys($jarIdsMap)])
                ->all();

            foreach ($expenses as $e) {

                $newExpense = new Expense(['scenario' => ChangeMetaModel::SCENARIO_CLONE]);
                $newExpense->setAttributes($e->getAttributes());
                $newExpense->jar_id = $jarIdsMap[$e->jar_id];
                $newExpense->account_id = $accountIdsMap[$e->account_id];
                $saved = $newExpense->save();

                if ($saved) {

                    $expenseIdsMap[$e->id] = $newExpense->id;

                }
                else {

                    $activateBudget = false;

                }

            }

        }

        // copy expense changes
        if ($activateBudget) {

            $expenseChangeIdsMap = [];

            $expenseChanges = ExpenseChange::find()
                ->where(['IN', 'expense_id', array_keys($expenseIdsMap)])
                ->all();

            foreach ($expenseChanges as $ec) {

                $newExpenseChange = new ExpenseChange();
                $newExpenseChange->setAttributes($ec->getAttributes());
                $newExpenseChange->expense_id = $expenseIdsMap[$ec->expense_id];
                $saved = $newExpenseChange->save();

                if ($saved) {

                    $expenseChangeIdsMap[$ec->id] = $newExpenseChange->id;

                }
                else {

                    $activateBudget = false;

                }

            }

        }

        // copy expense changes meta
        if ($activateBudget) {

            $expenseChangeMeta = ExpenseChangeMeta::find()
                ->where(['IN', 'expense_change_id', array_keys($expenseChangeIdsMap)])
                ->all();

            foreach ($expenseChangeMeta as $ecm) {

                $newExpenseChangeMeta = new ExpenseChangeMeta();
                $newExpenseChangeMeta->setAttributes($ecm->getAttributes());
                $newExpenseChangeMeta->expense_change_id = $expenseChangeIdsMap[$ecm->expense_change_id];
                $saved = $newExpenseChangeMeta->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy expense debts
        if ($activateBudget) {

            $expenseDebts = ExpenseDebt::find()
                ->where(['IN', 'expense_id', array_keys($expenseIdsMap)])
                ->all();

            foreach ($expenseDebts as $ed) {

                $newExpenseDebt = new ExpenseDebt();
                $newExpenseDebt->expense_id = $expenseIdsMap[$ed->expense_id];
                $newExpenseDebt->debt_id = $debtIdsMap[$ed->debt_id];
                $saved = $newExpenseDebt->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy expense ends
        if ($activateBudget) {

            $expenseEnds = ExpenseEnd::find()
                ->where(['IN', 'expense_id', array_keys($expenseIdsMap)])
                ->all();

            foreach ($expenseEnds as $ee) {

                $newExpenseEnd = new ExpenseEnd(['scenario' => ChangeMetaModel::SCENARIO_CLONE]);
                $newExpenseEnd->setAttributes($ee->getAttributes());
                $newExpenseEnd->expense_id = $expenseIdsMap[$ee->expense_id];
                $saved = $newExpenseEnd->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy expense starts
        if ($activateBudget) {

            $expenseStarts = ExpenseStart::find()
                ->where(['IN', 'expense_id', array_keys($expenseIdsMap)])
                ->all();

            foreach ($expenseStarts as $es) {

                $newExpenseStart = new ExpenseStart();
                $newExpenseStart->setAttributes($es->getAttributes());
                $newExpenseStart->expense_id = $expenseIdsMap[$es->expense_id];
                $saved = $newExpenseStart->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy credit card transactions
        if ($activateBudget) {

            $ccTransactions = CcTransaction::find()
                ->where(['IN', 'expense_id', array_keys($expenseIdsMap)])
                ->all();

            foreach ($ccTransactions as $ct) {

                $newCcTransaction = new CcTransaction();
                $newCcTransaction->setAttributes($ct->getAttributes());
                $newCcTransaction->expense_id = $expenseIdsMap[$ct->expense_id];
                $newCcTransaction->debt_id = $debtIdsMap[$ct->debt_id];
                $saved = $newCcTransaction->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy incomes
        if ($activateBudget) {

            $incomeIdsMap = [];

            $incomes = Income::find()
                ->where(['budget_id' => $this->id])
                ->all();

            foreach ($incomes as $i) {

                $newIncome = new Income(['scenario' => ChangeMetaModel::SCENARIO_CLONE]);
                $newIncome->setAttributes($i->getAttributes());
                $newIncome->budget_id = $budget->id;
                $newIncome->account_id = $accountIdsMap[$i->account_id];
                $saved = $newIncome->save();

                if ($saved) {

                    $incomeIdsMap[$i->id] = $newIncome->id;

                }
                else {

                    $activateBudget = false;

                }

            }

        }

        // copy income changes
        if ($activateBudget) {

            $incomeChangeIdsMap = [];

            $incomeChanges = IncomeChange::find()
                ->where(['IN', 'income_id', array_keys($incomeIdsMap)])
                ->all();

            foreach ($incomeChanges as $ic) {

                $newIncomeChange = new IncomeChange();
                $newIncomeChange->setAttributes($ic->getAttributes());
                $newIncomeChange->income_id = $incomeIdsMap[$ic->income_id];
                $saved = $newIncomeChange->save();

                if ($saved) {

                    $incomeChangeIdsMap[$ic->id] = $newIncomeChange->id;

                }
                else {

                    $activateBudget = false;

                }

            }

        }

        // copy income changes meta
        if ($activateBudget) {

            $incomeChangeMeta = IncomeChangeMeta::find()
                ->where(['IN', 'income_change_id', array_keys($incomeChangeIdsMap)])
                ->all();

            foreach ($incomeChangeMeta as $icm) {

                $newIncomeChangeMeta = new IncomeChangeMeta();
                $newIncomeChangeMeta->setAttributes($icm->getAttributes());
                $newIncomeChangeMeta->income_change_id = $incomeChangeIdsMap[$icm->income_change_id];
                $saved = $newIncomeChangeMeta->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy income ends
        if ($activateBudget) {

            $incomeEnds = IncomeEnd::find()
                ->where(['IN', 'income_id', array_keys($incomeIdsMap)])
                ->all();

            foreach ($incomeEnds as $ie) {

                $newIncomeEnd = new IncomeEnd(['scenario' => ChangeMetaModel::SCENARIO_CLONE]);
                $newIncomeEnd->setAttributes($ie->getAttributes());
                $newIncomeEnd->income_id = $incomeIdsMap[$ie->income_id];
                $saved = $newIncomeEnd->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy income transactions
        if ($activateBudget) {

            $incomeTransactions = IncomeTransaction::find()
                ->where(['IN', 'income_id', array_keys($incomeIdsMap)])
                ->all();

            foreach ($incomeTransactions as $it) {

                $newIncomeTransaction = new IncomeTransaction();
                $newIncomeTransaction->setAttributes($it->getAttributes());
                $newIncomeTransaction->income_id = $incomeIdsMap[$it->income_id];
                $newIncomeTransaction->account_id = $accountIdsMap[$it->account_id];
                $saved = $newIncomeTransaction->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy transactions
        if ($activateBudget) {

            $transactions = Transaction::find()
                ->where(['IN', 'expense_id', array_keys($expenseIdsMap)])
                ->all();

            foreach ($transactions as $t) {

                $newTransaction = new Transaction();
                $newTransaction->setAttributes($t->getAttributes());
                $newTransaction->expense_id = $expenseIdsMap[$t->expense_id];
                $newTransaction->account_id = $accountIdsMap[$t->account_id];
                $saved = $newTransaction->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy transfers
        if ($activateBudget) {

            $transferIdsMap = [];

            $transfers = Transfer::find()
                ->where(['budget_id' => $this->id])
                ->all();

            foreach ($transfers as $t) {

                $newTransfer = new Transfer(['scenario' => ChangeMetaModel::SCENARIO_CLONE]);
                $newTransfer->setAttributes($t->getAttributes());
                $newTransfer->budget_id = $budget->id;
                $newTransfer->account_from_id = $accountIdsMap[$t->account_from_id];
                $newTransfer->account_to_id = $accountIdsMap[$t->account_to_id];
                $saved = $newTransfer->save();

                if ($saved) {

                    $transferIdsMap[$t->id] = $newTransfer->id;

                }
                else {

                    $activateBudget = false;

                }

            }

        }

        // copy transfer changes
        if ($activateBudget) {

            $transferChangeIdsMap = [];

            $transferChanges = TransferChange::find()
                ->where(['IN', 'transfer_id', array_keys($transferIdsMap)])
                ->all();

            foreach ($transferChanges as $tc) {

                $newTransferChange = new TransferChange();
                $newTransferChange->setAttributes($tc->getAttributes());
                $newTransferChange->transfer_id = $transferIdsMap[$tc->transfer_id];
                $saved = $newTransferChange->save();

                if ($saved) {

                    $transferChangeIdsMap[$tc->id] = $newTransferChange->id;

                }
                else {

                    $activateBudget = false;

                }

            }

        }

        // copy transfer changes meta
        if ($activateBudget) {

            $transferChangeMeta = TransferChangeMeta::find()
                ->where(['IN', 'transfer_change_id', array_keys($transferChangeIdsMap)])
                ->all();

            foreach ($transferChangeMeta as $tcm) {

                $newTransferChangeMeta = new TransferChangeMeta();
                $newTransferChangeMeta->setAttributes($tcm->getAttributes());
                $newTransferChangeMeta->transfer_change_id = $transferChangeIdsMap[$tcm->transfer_change_id];
                $saved = $newTransferChangeMeta->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy transfer ends
        if ($activateBudget) {

            $transferEnds = TransferEnd::find()
                ->where(['IN', 'transfer_id', array_keys($transferIdsMap)])
                ->all();

            foreach ($transferEnds as $te) {

                $newTransferEnd = new TransferEnd(['scenario' => ChangeMetaModel::SCENARIO_CLONE]);
                $newTransferEnd->setAttributes($te->getAttributes());
                $newTransferEnd->transfer_id = $transferIdsMap[$te->transfer_id];
                $saved = $newTransferEnd->save();

                if (!$saved) {

                    $activateBudget = false;

                }

            }

        }

        // copy transfer events
        if ($activateBudget) {
            $transferEvents = TransferEvent::find()
                ->where(['IN', 'transfer_id', array_keys($transferIdsMap)])
                ->all();
            foreach ($transferEvents as $te) {
                $newTransferEvent = new TransferEvent();
                $newTransferEvent->setAttributes($te->getAttributes());
                $newTransferEvent->transfer_id = $transferIdsMap[$te->transfer_id];
                $saved = $newTransferEvent->save();
                if (!$saved) {
                    $activateBudget = false;
                }
            }
        }

        if ($activateBudget) {
            $budget->setAttributes([
                'archived' => 0,
                'default_account_id' => $accountIdsMap[$this->default_account_id],
                'name' => $name,
                'is_mock' => $isMock ? 1 : 0
            ]);
            $budget->save();
        }

        return $activateBudget;

    }


    /**
     * Get a list of currently active jars.
     * @return array List of jars.
     */
    public function getActiveJars()
    {
        if ($this->_activeJars == null) {
            $this->_activeJars = Jar::find()
                ->where(['budget_id' => $this->id, 'archived' => 0])
                ->orderBy('order')
                ->all();
        }
        return $this->_activeJars;
    }


    /**
     * Get a list of jars which were active during a month
     * @param integer $month Month.
     * @param integer $year Year.
     * @return array List of jars.
     */
    public function getMonthsActiveJars($month, $year) {

        $jars = Jar::find()
            ->leftJoin(JarChange::tableName() . ' as jcc', "jcc.jar_id = " . Jar::tableName() . '.id AND jcc.type = "C"')
            ->leftJoin(JarChange::tableName() . ' as jcd', "jcd.jar_id = " . Jar::tableName() . '.id AND jcd.type = "D"')
            ->where([
                'AND',
                ['budget_id' => $this->id],
                ['<=', 'DATE(jcc.time)', date('Y-m-t', strtotime(date("$year-$month-1")))],
                [
                    'OR',
                    ['>=', 'DATE(jcd.time)', "$year-$month-1"],
                    ['jcd.time' => null]
                ]
            ])
            ->orderBy('name asc')
            ->all();

        return $jars;

    }

    /**
     * Gets the total funds remaining amount for the budget for the month,
     * as seen at the bottom of EoMS, in the Jar Funds Remaining field
     * @param $month
     * @param $year
     * @return float
     */
    public function getMonthsFundsRemainingAmount($month, $year)
    {
        $amount = 0;
        /** @var Jar $jar */
        foreach ($this->getActiveJars() as $jar) {
            $remain = $jar->getMonthsFundsRemainingAmount($month, $year);
            $amount += $remain > 0 ? $remain : 0;
        }
        return $amount;
    }

    /**
     * Get the expenses total amount for a month
     * @param integer $month Month.
     * @param integer $year Year.
     * @return float Total amount.
     */
    public function getMonthsExpensesAmount($month, $year)
    {
        $jars = $this->getMonthsActiveJars($month, $year);
        $amount = 0;
        foreach ($jars as $j) {
            $amount += $j->getMonthsExpensesAmount($month, $year);
        }
        return $amount;
    }

    public function getTotalMonthlyBudget()
    {
        if ($this->_totalMonthlyBudget == null) {
            $jars = $this->getActiveJars();
            $this->_totalMonthlyBudget = 0;
            /** @var Jar $j */
            foreach ($jars as $j) {
                $this->_totalMonthlyBudget += $j->getMonthlyBudget();
            }
        }
        return $this->_totalMonthlyBudget;
    }

    /**
     * Gets the total balance of all normal accounts
     * @param null $month
     * @param null $year
     * @return float
     */
    public function getTotalAccountsBalance($month = null, $year = null)
    {
        $total = 0;
        $accounts = $this->getAccounts()->where(['archived' => 0, 'account_type_id' => 1])->all();
        /** @var Account $account */
        foreach ($accounts as $account) {
            $total += $account->getCurrentBalance($month, $year);
        }
        return $total;
    }

    /**
     * Gets the total balance (amount to pay off) of all debts
     * @param null $month
     * @param null $year
     * @return float
     */
    public function getTotalDebtsBalance($month = null, $year = null)
    {
        $total = 0;
        $debts = $this->getDebts()->where(['archived' => 0])->all();
        /** @var Debt $debt */
        foreach ($debts as $debt) {
            $total += !$month && !$year || (int)$month == date('n') && $year == date('Y') ?
                $debt->amount :
                $debt->getDebtLeftAmount($month, $year);
        }
        // FIXME: debt can't be negative
        return $total < 0 ? 0 : -1 * $total;
    }

    public function getThisMonthsExpensesAmount()
    {
        if ($this->_thisMonthsExpensesAmount == null) {
            $jars = $this->getActiveJars();
            $this->_thisMonthsExpensesAmount = 0;
            /** @var Jar $j */
            foreach ($jars as $j) {
                $this->_thisMonthsExpensesAmount += $j->getThisMonthsExpensesAmount();
            }
        }
        return $this->_thisMonthsExpensesAmount;
    }

    public function getThisMonthsIncomes()
    {
        if ($this->_thisMonthsIncomes == null) {
            $this->_thisMonthsIncomes = Income::find()
                ->select([
                    Income::tableName() . '.*',
                ])
                ->leftJoin(IncomeEnd::tableName(), IncomeEnd::tableName() . '.income_id = ' . Income::tableName() . '.id')
                ->where([
                    'and',
                    ['budget_id' => $this->id],
                    ['archived' => '0'],
                    ['!=', 'frequency', 'one-time'],
                    [IncomeEnd::tableName() . '.date' => null]
                ])
                ->all();
        }
        return $this->_thisMonthsIncomes;
    }


    public function getMinimumMonthlyIncomeAmount()
    {
        if ($this->_minimumMonthlyIncomeAmount == null) {
            $incomes = $this->getThisMonthsIncomes();
            $this->_minimumMonthlyIncomeAmount = 0;
            foreach ($incomes as $i) {
                $this->_minimumMonthlyIncomeAmount += $i->getMinimumMonthlyAmount();
            }
        }
        return $this->_minimumMonthlyIncomeAmount;
    }


    public function getAverageMonthlyIncomeAmount()
    {
        if ($this->_averageMonthlyIncomeAmount == null) {
            $incomes = $this->getThisMonthsIncomes();
            $this->_averageMonthlyIncomeAmount = 0;
            foreach ($incomes as $i) {
                $this->_averageMonthlyIncomeAmount += $i->getAverageMonthlyAmount();
            }
        }
        return $this->_averageMonthlyIncomeAmount;
    }


    /**
     * Get a list of transactions for a certain month and year.
     * @param int $month Month.
     * @param int $year Year.
     * @return array List of transactions.
     */
    public function getMonthsMoneyOperations($month, $year, $includeCcTrans = false) {

        $expenses = $this->getMonthsExpenseTransactions($month, $year, $includeCcTrans);

        $transfers = TransferEvent::find()
            ->select([
                'id' => TransferEvent::tableName() . '.id',
                'date' => TransferEvent::tableName() . '.date',
                'name' => Transfer::tableName() . '.name',
                'description' => Transfer::tableName() . '.name',
                'amount' => TransferEvent::tableName() . '.amount',
                'account_from_name' => 'a_from.name',
                'account_to_name' => 'a_to.name',
                'account_name' => 'CONCAT(a_from.name, " => ", a_to.name)',
                'jar_name' => 'CONCAT("")',
                'checked' => TransferEvent::tableName() . '.checked'
            ])
            ->joinWith('transfer')
            ->leftJoin(Account::tableName() . ' as a_from', "a_from.id = " . Transfer::tableName() . ".account_from_id")
            ->leftJoin(Account::tableName() . ' as a_to', "a_to.id = " . Transfer::tableName() . ".account_to_id")
            ->where([
                'and',
                ['between', TransferEvent::tableName() . '.date', date("$year-$month-1"), date("$year-$month-" . date('t', strtotime("$year-$month-1")))],
                ['a_from.budget_id' => $this->id]
            ])
            ->all();


        $incomes = IncomeTransaction::find()
            ->select([
                'id' => IncomeTransaction::tableName() . '.id',
                'date' => IncomeTransaction::tableName() . '.date',
                'name' => Income::tableName() . '.name',
                'description' => IncomeTransaction::tableName() . '.description',
                'amount' => IncomeTransaction::tableName() . '.amount',
                'account_id' => Account::tableName() . '.id',
                'account_name' => Account::tableName() . '.name',
                'jar_name' => "IF(".Income::tableName().".jar_id IS NULL, '', ".Jar::tableName().".name)",
                'checked' => IncomeTransaction::tableName() . '.checked'
            ])
            ->joinWith('account')
            ->joinWith('income')
            ->joinWith('income.jar')
            ->where([
                'and',
                ['between', IncomeTransaction::tableName() . '.date', date("$year-$month-1"), date("$year-$month-" . date('t', strtotime("$year-$month-1")))],
                [Income::tableName() . '.budget_id' => $this->id]
            ])
            ->all();

        $moneyOperations = array_merge($expenses, $transfers, $incomes);

        return $moneyOperations;

    }


    /**
     * The transactions for this month
     * @return array
     */
    public function getThisMonthsMoneyOperations($includeCcTrans = false)
    {
        if ($this->_thisMonthsMoneyOperations == null) {
            $this->_thisMonthsMoneyOperations = $this->getMonthsMoneyOperations(date('m'), date('Y'), $includeCcTrans);
        }
        return $this->_thisMonthsMoneyOperations;
    }

    /**
     * Get the resulting balance of all money operations in this month.
     * @param int $month Month.
     * @param int $year Year.
     * @param int $accountId ID of filtered account.
     * @return float Resulting sum of all money operations in this month.
     */
    public function getMonthsMoneyOperationsAmount($month, $year, $includeCcTrans = false)
    {
        $moneyOperations = $this->getMonthsMoneyOperations($month, $year, $includeCcTrans);

        $amount = 0;
        foreach ($moneyOperations as $m) {
            if (get_class($m) == 'backend\models\db\DebtPayment') {
                $amount -= $m->amount;
            }
            elseif (get_class($m) == 'backend\models\db\Transaction') {
                $amount -= $m->amount;
            }
            elseif (get_class($m) == 'backend\models\db\CcTransaction') {
                $amount -= $m->amount;
            }
            elseif (get_class($m) == 'backend\models\db\TransferEvent') {
                $amount += $m->amount;
            }
            elseif (get_class($m) == 'backend\models\db\IncomeTransaction') {
                $amount += $m->amount;
            }
        }
        return $amount;
    }

    /**
     * Finds the date of the first or the last money operation for the budget
     * @param bool $first
     * @return string
     */
    public function getMoneyOperationDate($first = true)
    {
        $transaction = Transaction::find()
            ->select(Transaction::tableName() . '.date')
            ->joinWith('expense')
            ->joinWith('expense.account')
            ->where([
                Account::tableName() . '.budget_id' => $this->id,
                Expense::tableName() . '.archived' => 0
            ])
            ->orderBy(Transaction::tableName() . '.date '.($first ? 'ASC' : 'DESC'))
            ->one();

        $debtPayment = DebtPayment::find()
            ->select(DebtPayment::tableName() . '.date')
            ->joinWith('debt')
            ->where([Debt::tableName() . '.budget_id' => $this->id])
            ->orderBy(DebtPayment::tableName() . '.date '.($first ? 'ASC' : 'DESC'))
            ->one();

        $transfer = TransferEvent::find()
            ->select(TransferEvent::tableName() . '.date')
            ->joinWith('transfer')
            ->leftJoin(Account::tableName() . ' as a_from', "a_from.id = " . Transfer::tableName() . ".account_from_id")
            ->leftJoin(Account::tableName() . ' as a_to', "a_to.id = " . Transfer::tableName() . ".account_to_id")
            ->where([
                'a_from.budget_id' => $this->id,
                'a_to.budget_id' => $this->id,
                Transfer::tableName() . '.archived' => 0
            ])
            ->orderBy(TransferEvent::tableName() . '.date '.($first ? 'ASC' : 'DESC'))
            ->one();

        $income = IncomeTransaction::find()
            ->select(IncomeTransaction::tableName() . '.date')
            ->joinWith('income')
            ->where([
                Income::tableName() . '.budget_id' => $this->id,
                Income::tableName() . '.archived' => 0
            ])
            ->orderBy(IncomeTransaction::tableName() . '.date '.($first ? 'ASC' : 'DESC'))
            ->one();

        $date = date('Y-m-d');
        if ($transaction &&
            ($first && $transaction->date < $date || !$first && $transaction->date > $date)) {
            $date = $transaction->date;
        }
        if ($debtPayment &&
            ($first && $debtPayment->date < $date || !$first && $debtPayment->date > $date)) {
            $date = $debtPayment->date;
        }
        if ($transfer &&
            ($first && $transfer->date < $date || !$first && $transfer->date > $date)) {
            $date = $transfer->date;
        }
        if ($income &&
            ($first && $income->date < $date || !$first && $income->date > $date)) {
            $date = $income->date;
        }

        return $date;
    }

    /**
     * Get the date of the first money operation for the budget
     * @return string
     */
    public function getFirstMoneyOperationDate()
    {
        return $this->getMoneyOperationDate(true);
    }

    /**
     * Get the date of the last money operation for the budget
     * @return string
     */
    public function getLastMoneyOperationDate()
    {
        return $this->getMoneyOperationDate(false);
    }

    public function getIncomeDate($first = true)
    {
        $income = IncomeTransaction::find()
            ->joinWith('income')
            ->where([
                Income::tableName() . '.budget_id' => $this->id,
                'is_adjustment' => 0,
                'archived' => 0
            ])
            ->orderBy('date '.($first ? 'ASC' : 'DESC'))
            ->one();
        return $income ? $income->date : date('Y-m-d');
    }

    public function getFirstIncomeDate()
    {
        return $this->getIncomeDate(true);
    }

    public function getLastIncomeDate()
    {
        return $this->getIncomeDate(false);
    }

    public function getTransferDate($first = true)
    {
        $transfer = TransferEvent::find()
            ->joinWith('transfer')
            ->leftJoin(Account::tableName() . ' as a_from', "a_from.id = " . Transfer::tableName() . ".account_from_id")
            ->leftJoin(Account::tableName() . ' as a_to', "a_to.id = " . Transfer::tableName() . ".account_to_id")
            ->where([
                'a_from.budget_id' => $this->id,
                'a_to.budget_id' => $this->id,
                'is_adjustment' => 0,
                Transfer::tableName() . '.archived' => 0
            ])
            ->orderBy(TransferEvent::tableName() . '.date '.($first ? 'ASC' : 'DESC'))
            ->one();
        return $transfer ? $transfer->date : date('Y-m-d');
    }

    public function getFirstTransferDate()
    {
        return $this->getTransferDate(true);
    }

    public function getLastTransferDate()
    {
        return $this->getTransferDate(false);
    }

    public function getDebtPaymentDate($first = true)
    {
        $debtPayment = DebtPayment::find()
            ->joinWith('debt')
            ->where(['and',
                [Debt::tableName() . '.budget_id' => $this->id],
                ['!=', 'debt_payment_type_id', 2]
            ])
            ->orderBy('date '.($first ? 'ASC' : 'DESC'))
            ->one();
        return $debtPayment ? $debtPayment->date : date('Y-m-d');
    }

    public function getFirstDebtPaymentDate()
    {
        return $this->getDebtPaymentDate(true);
    }

    public function getLastDebtPaymentDate()
    {
        return $this->getDebtPaymentDate(false);
    }

    public function getInterestPaymentDate($first = true)
    {
        $debtPayment = DebtPayment::find()
            ->joinWith('debt')
            ->where([
                Debt::tableName() . '.budget_id' => $this->id,
                'debt_payment_type_id' => 2
            ])
            ->orderBy('date '.($first ? 'ASC' : 'DESC'))
            ->one();
        return $debtPayment ? $debtPayment->date : date('Y-m-d');
    }

    public function getFirstInterestPaymentDate()
    {
        return $this->getInterestPaymentDate(true);
    }

    public function getLastInterestPaymentDate()
    {
        return $this->getInterestPaymentDate(false);
    }

    /**
     * Get a list of expenses for a certain month and year.
     * @param int $month Month.
     * @param int $year Year.
     * @return array List of transactions.
     */
    public function getMonthsExpenseTransactions($month, $year, $includeCcTrans = false)
    {
        return $this->getTimePeriodExpenseTransactions(
            date("$year-$month-1"),
            date("$year-$month-" . date('t', strtotime("$year-$month-1"))),
            $includeCcTrans
        );
    }

    public function getTimePeriodExpenseTransactions($from, $to, $includeCcTrans = false)
    {
        $transactions = Transaction::find()
            ->select([
                'id' => Transaction::tableName() . '.id',
                'date' => Transaction::tableName() . '.date',
                'name' => Expense::tableName() . '.name',
                'expense_id' => Expense::tableName() . '.id',
                'description' => Transaction::tableName() . '.description',
                'amount' => Transaction::tableName() . '.amount',
                'account_id' => Account::tableName() . '.id',
                'account_name' => Account::tableName() . '.name',
                'jar_id' => Jar::tableName() . '.id',
                'jar_name' => Jar::tableName() . '.name',
                'is_adjustment' => Expense::tableName() . '.is_adjustment',
                'checked' => Transaction::tableName() . '.checked'
            ])
            ->joinWith('expense')
            ->joinWith('account')
            ->leftJoin(Jar::tableName(), Jar::tableName() . ".id = " . Expense::tableName() . ".jar_id")
            ->where([
                'and',
                ['between', Transaction::tableName() . '.date', $from, $to],
                [Jar::tableName() . '.budget_id' => $this->id],
            ])
            ->all();

        if ($includeCcTrans) {
            $ccTransactions = CcTransaction::find()
                ->select([
                    'id' => CcTransaction::tableName() . '.id',
                    'date' => CcTransaction::tableName() . '.date',
                    'name' => Expense::tableName() . '.name',
                    'description' => CcTransaction::tableName() . '.description',
                    'amount' => CcTransaction::tableName() . '.amount',
                    'account_name' => 'CONCAT(' . Debt::tableName() . '.name,\' (CC)\')',
                    'jar_name' => Jar::tableName() . '.name'
                ])
                ->joinWith('expense')
                ->joinWith('debt')
                ->leftJoin(Jar::tableName(), Jar::tableName() . ".id = " . Expense::tableName() . ".jar_id")
                ->where([
                    'and',
                    ['between', CcTransaction::tableName() . '.date', $from, $to],
                    [Jar::tableName() . '.budget_id' => $this->id],
                ])
                ->all();
        }
        else {
            $ccTransactions = [];
        }

        $debtPayments = DebtPayment::find()
            ->select([
                'id' => DebtPayment::tableName() . '.id',
                'date' => DebtPayment::tableName() . '.date',
                'name' => Debt::tableName() . '.name' ,
                'debt_id' => Debt::tableName() . '.id' ,
                'description' => DebtPayment::tableName() . '.description',
                'amount' => DebtPayment::tableName() . '.amount' ,
                'account_id' => Account::tableName() . '.id',
                'account_name' => Account::tableName() . '.name',
                'jar_name' => Jar::tableName() . '.name',
                'is_adjustment' => DebtPayment::tableName() . '.is_adjustment',
                'checked' => DebtPayment::tableName() . '.checked'
            ])
            ->joinWith('account')
            ->joinWith('debt')
            ->joinWith('debtPaymentType')
            ->leftJoin(Jar::tableName(), Jar::tableName() . ".budget_id = " . $this->id . " AND " . Jar::tableName() . ".jar_type_id = 2")
            ->where([
                'and',
                ['between', DebtPayment::tableName() . '.date', $from, $to],
                [Debt::tableName() . '.budget_id' => $this->id]
            ])
            ->all();
        if ($includeCcTrans) {
            $moneyOperations = array_merge($transactions, $ccTransactions, $debtPayments);
        }
        else {
            $moneyOperations = array_merge($transactions, $debtPayments);
        }
        return $moneyOperations;
    }

    public function getMonthsExpenseTransacitonsAmount($month, $year, $includeCcTrans = false)
    {
        $expenses = $this->getMonthsExpenses($month, $year, $includeCcTrans = false);
        $amount = 0;
        foreach ($expenses as $e) {
            $amount += $e->amount;
        }
        return $amount;
    }

    public function getTimePeriodTransferEvents($from, $to)
    {
        return TransferEvent::find()
            ->joinWith('transfer')
            ->where(['and',
                [Transfer::tableName() . '.budget_id' => $this->id],
                ['between', TransferEvent::tableName() . '.date', $from, $to]
            ])
            ->all();
    }

    public function getMonthsAccounts($month, $year)
    {
        $firstDay = "$year-$month-1";
        $lastDay = "$year-$month-" . date('t', strtotime("$year-$month-1"));
        return Account::find()
            ->leftJoin(AccountChange::tableName() . ' as acs', Account::tableName() . ".id = acs.account_id AND acs.type = 'C'")
            ->leftJoin(AccountChange::tableName() . ' as aca', Account::tableName() . ".id = aca.account_id AND aca.type = 'E'")
            ->leftJoin(AccountChangeMeta::tableName() . ' as acam', "aca.id = acam.account_change_id AND `key` = 'archived' AND `value` = '1'")
            ->leftJoin(AccountChange::tableName() . ' as ace', Account::tableName() . ".id = ace.account_id AND ace.type = 'D'")
            ->where([
                'and',
                ['!=', 'account_type_id', 2],
                ['budget_id' => $this->id],
                ['<=', 'DATE(' . Account::tableName() . '.created_time)', $lastDay],
                ['<=', 'DATE(acs.time)', $lastDay],
            ])
            ->andWhere([
                'OR',
                [Account::tableName() . '.archived' => 0],
                ['and',
                    [Account::tableName() . '.archived' => 1],
                    ['>=', 'DATE(ace.time)', $firstDay],
                    // don't show accounts that were archived only during the month but are now inactive
                    ['or',
                        ['aca.time' => NULL],
                        ['and',
                            ['<', 'DATE(aca.time)', $firstDay],
                            ['not', 'acam.key IS NULL']
                        ]
                    ]
                ]
            ])
            ->orderBy('name asc')
            ->groupBy(Account::tableName() . '.id')
            ->all();
    }

    /**
     * Get a list of active CC debts which are stored as accounts of CC type.
     * @return array
     */
    public function getMonthsActiveCcDebts($month, $year)
    {
        $lastDay = "$year-$month-" . date('t', strtotime("$year-$month-1"));
        return Account::find()
            ->leftJoin(AccountChange::tableName() . ' as acs', Account::tableName() . ".id = acs.account_id AND acs.type = 'C'")
            ->leftJoin(AccountChange::tableName() . ' as ace', Account::tableName() . ".id = ace.account_id AND ace.type = 'D'")
            ->where([
                'AND',
                ['account_type_id' => 2],
                ['budget_id' => $this->id],
                ['<=', 'DATE(' . Account::tableName() . '.created_time)', $lastDay],
                ['<=', 'DATE(acs.time)', $lastDay],
            ])
            ->andWhere([
                'OR',
                [Account::tableName() . '.archived' => 0],
                ['>=', 'DATE(ace.time)', "$year-$month-1"]
            ])
            ->all();
    }

    /**
     * Get list of debts for a month.
     * @param int $month Month.
     * @param int $year Year.
     * @param bool $debtTypeId
     * @param bool $includeDeleted
     * @return array List of selected debts.
     */
    public function getMonthsDebts($month, $year, $debtTypeId = false, $includeDeleted = true)
    {
        $jar = self::getDebtsJar();
        if (!$jar) {
            return [];
        }
        $debtsQuery = $jar->getMonthsDebtsExpensesQuery($month, $year, $includeDeleted);
        if ($debtTypeId) {
            if (is_numeric($debtTypeId) && ($debtTypeId > 0)) {
                $debtsQuery->andWhere([Debt::tableName() . '.debt_type_id' => $debtTypeId]);
            }
            elseif (is_numeric($debtTypeId) && ($debtTypeId < 0)) {
                $debtsQuery->andWhere(['!=', Debt::tableName() . '.debt_type_id', abs($debtTypeId)]);
            }
            elseif (is_array($debtTypeId)) {
                $debtsQuery->andWhere(['IN', Debt::tableName() . '.debt_type_id', $debtTypeId]);
            }
        }
        return $debtsQuery->all();
    }

    /**
     * Get debts for this month.
     * @param bool $debtTypeId
     * @param bool $includeDeleted
     * @return array List of selected debts.
     */
    public function getThisMonthsDebts($debtTypeId = false, $includeDeleted = true)
    {
        if ($this->_thisMonthsDebts == null) {
            $this->_thisMonthsDebts = $this->getMonthsDebts(date('m'), date('Y'), $debtTypeId, $includeDeleted);
        }
        return $this->_thisMonthsDebts;
    }

    /**
     * Get total amount of this months debt payment budget.
     * @return float Total amount.
     */
    public function getThisMonthsDebtsAmount($debtTypeId = false)
    {
        if ($this->_thisMonthsDebtsAmount == null) {
            $this->_thisMonthsDebtsAmount = $this->getMonthsDebtsAmount(date('m'), date('Y'), $debtTypeId);
        }
        return $this->_thisMonthsDebtsAmount;
    }

    public function getMonthsDebtsAmount($month, $year, $debtTypeId = false)
    {
        $debts = $this->getMonthsDebts($month, $year, $debtTypeId, false);
        $amount = 0;
        /** @var Debt $d */
        foreach ($debts as $d) {
            if (!$d->end_date) {
                $amount += $d->getAverageMonthlyAmount($month, $year);
            }
            else {
                // FIXME ???
                $payments = DebtPayment::find()
                    ->where(['>=', 'date',  date('Y-m-d')])
                    ->all();
                foreach ($payments as $p) {
                    $amount += $p->amount;
                }
            }
        }
        return $amount;
    }

    /**
     * Get list of debt payments for a month.
     * @param int $month Month.
     * @param int $year Year.
     * @param bool $debtTypeId
     * @param int $debtPaymentTypeId
     * @param bool $excludeAdjustments
     * @return array List of selected debt payments.
     */
    public function getMonthsDebtPayments($month, $year, $debtTypeId = false, $debtPaymentTypeId = 1, $excludeBalanceAdjustments = false, $excludeJarAdjustments = false)
    {
        $debtPaymentsQuery = DebtPayment::find()
            ->joinWith('debt')
            ->joinWith('account')
            ->where([
                'and',
                ['or',
                    [Debt::tableName() . '.budget_id' => $this->id],
                    [Account::tableName() . '.budget_id' => $this->id]
                ],
                ['=', 'debt_payment_type_id', $debtPaymentTypeId],
                ['between', DebtPayment::tableName() . '.date', date("$year-$month-1"), date("$year-$month-" . date('t', strtotime("$year-$month-1")))]
            ]);

        if (is_numeric($debtTypeId) && ($debtTypeId > 0)) {
            $debtPaymentsQuery->andWhere([Debt::tableName() . '.debt_type_id' => $debtTypeId]);
        }
        elseif (is_numeric($debtTypeId) && ($debtTypeId < 0)) {
            $debtPaymentsQuery->andWhere(['!=', Debt::tableName() . '.debt_type_id', abs($debtTypeId)]);
        }
        elseif (is_array($debtTypeId)) {
            $debtPaymentsQuery->andWhere(['IN', Debt::tableName() . '.debt_type_id', $debtTypeId]);
        }
        if ($excludeBalanceAdjustments || $excludeJarAdjustments) {
            $exclude = [];
            if ($excludeBalanceAdjustments) {
                $exclude[] = DebtPayment::DEBT_PAYMENT_ADJUSTMENT;
            }
            if ($excludeJarAdjustments) {
                $exclude[] = DebtPayment::JAR_ADJUSTMENT;
            }
            $debtPaymentsQuery->andWhere(['NOT IN', 'is_adjustment', $exclude]);
        }

        return $debtPaymentsQuery->all();
    }

    /**
     * Get the funds remaining amount for debts jar
     * @param float $jarAmount
     * @return float
     */
    public function getThisMonthsDebtsFundsRemainingAmount($jarAmount)
    {
        return $this->getThisMonthsDebtsAmount() + $jarAmount - $this->getThisMonthsDebtPaymentsAmount();
    }

    public function getMonthsDebtsFundsRemainingAmount($month, $year, $jarAmount, $debtPaymentTypeId = 1)
    {
        return $this->getMonthsDebtsAmount($month, $year) + $jarAmount - $this->getMonthsDebtPaymentsAmount($month, $year, false, $debtPaymentTypeId);
    }

    /**
     * The debtPayments for this month
     * @return float Numeric amount.
     */
    public function getThisMonthsDebtPayments($debtTypeId = false, $debtPaymentTypeId = 1)
    {
        if (!isset($this->_thisMonthsDebtPayments[$debtPaymentTypeId])) {
            $this->_thisMonthsDebtPayments[$debtPaymentTypeId] = $this->getMonthsDebtPayments(date('m'), date('Y'), $debtTypeId, $debtPaymentTypeId);
        }
        return $this->_thisMonthsDebtPayments[$debtPaymentTypeId];
    }

    /**
     * The total value of transactions for this month
     * @return float Numeric amount.
     */
    public function getThisMonthsDebtPaymentsAmount($debtTypeId = false, $debtPaymentTypeId = 1)
    {
        if (!isset($this->_thisMonthsDebtPaymentsAmount[$debtPaymentTypeId])) {
            $debtPayments = $this->getThisMonthsDebtPayments($debtTypeId, $debtPaymentTypeId);
            $amount = 0;
            foreach ($debtPayments as $t) {
                $amount += $t->amount;
            }
            $this->_thisMonthsDebtPaymentsAmount[$debtPaymentTypeId] = $amount;
        }
        return $this->_thisMonthsDebtPaymentsAmount[$debtPaymentTypeId];
    }

    /**
     * The total value of transactions
     * @param $month
     * @param $year
     * @param bool $debtTypeId
     * @param int $debtPaymentTypeId
     * @param bool $excludeAdjustments
     * @return float Numeric amount.
     */
    public function getMonthsDebtPaymentsAmount($month, $year, $debtTypeId = false, $debtPaymentTypeId = 1, $excludeBalanceAdjustments = false, $excludeJarAdjustments = false)
    {
        $debtPayments = $this->getMonthsDebtPayments($month, $year, $debtTypeId, $debtPaymentTypeId, $excludeBalanceAdjustments, $excludeJarAdjustments);
        $amount = 0;
        foreach ($debtPayments as $t) {
            $amount += $t->amount;
        }
        return $amount;
    }

    /**
     * Get list of debt payments for a month.
     * @param int $month Month.
     * @param int $year Year.
     * @return array List of selected debt payments.
     */
    public function getMonthsDebtInterestPayments($month, $year)
    {
        $debts = $this->getDebts()->all();
        $debtIds = array();
        foreach ($debts as $d) {
            $debtIds[] = $d->id;
        }
        $interestPayments = DebtPayment::find()
            ->where([
                'and',
                ['in', 'debt_id', $debtIds],
                ['debt_payment_type_id' => '2'],
                ['between', 'date', date("$year-$month-1"), date("$year-$month-" . date('t', strtotime("$year-$month-1")))]
            ])
            ->all();
        return $interestPayments;
    }

    public function getMonthsDebtInterestPaymentsAmount($month, $year)
    {
        $debtPayments = $this->getMonthsDebtInterestPayments($month, $year);
        $amount = 0;
        foreach ($debtPayments as $t) {
            $amount += $t->amount;
        }
        return $amount;
    }

    /**
     * The debtPayments for this month
     * @return array
     */
    public function getThisMonthsDebtInterestPayments()
    {
        if ($this->_thisMonthsDebtInterestPayments == null) {
            $this->_thisMonthsDebtInterestPayments = $this->getMonthsDebtInterestPayments(date('m'), date('Y'));
        }
        return $this->_thisMonthsDebtInterestPayments;
    }

    /**
     * The total value of transactions for this month
     * @return float Numeric amount.
     */
    public function getThisMonthsDebtInterestPaymentsAmount()
    {
        if ($this->_thisMonthsDebtInterestPaymentsAmount == null) {
            $debtPayments = $this->getThisMonthsDebtInterestPayments();
            $amount = 0;
            foreach ($debtPayments as $t) {
                $amount += $t->amount;
            }
            $this->_thisMonthsDebtInterestPaymentsAmount = $amount;
        }
        return $this->_thisMonthsDebtInterestPaymentsAmount;
    }

    /**
     * Get list of debt payments for a month.
     * @param int $month Month.
     * @param int $year Year.
     * @return array List of selected debt payments.
     */
    public function getMonthsIncomeTransactions($month, $year)
    {
        return $this->getTimePeriodIncomeTransactions(
            date("$year-$month-1"),
            date("$year-$month-" . date('t', strtotime("$year-$month-1")))
        );
    }

    public function getTimePeriodIncomeTransactions($from, $to)
    {
        $incomes = $this->getIncomes()->all();
        $incomeIds = array();
        foreach ($incomes as $i) {
            $incomeIds[] = $i->id;
        }
        $incomeTransactions = IncomeTransaction::find()
            ->select([
                'id' => IncomeTransaction::tableName() . '.id',
                'amount' => IncomeTransaction::tableName() . '.amount',
                'date' => IncomeTransaction::tableName() . '.date',
                'name' => Income::tableName() . '.name',
                'income_id' => Income::tableName() . '.id',
                'description' => IncomeTransaction::tableName() . '.description',
                'account_name' => Account::tableName() . '.name',
                'account_id' => Account::tableName() . '.id',
                //'jar_id' => Jar::tableName() . '.id',
                //'jar_name' => Jar::tableName() . '.name'
            ])
            ->joinWith('income')
            ->joinWith('account')
            ->where([
                'and',
                ['in', 'income_id', $incomeIds],
                ['between', IncomeTransaction::tableName() . '.date', date('Y-m-d', strtotime($from)), date('Y-m-d', strtotime($to))]
            ])
            ->all();
        return $incomeTransactions;
    }

    /**
     * The income transactions for this month
     * @return array
     */
    public function getThisMonthsIncomeTransactions()
    {
        if ($this->_thisMonthsIncomeTransactions == null) {
            $this->_thisMonthsIncomeTransactions = $this->getMonthsIncomeTransactions(date('m'), date('Y'));
        }
        return $this->_thisMonthsIncomeTransactions;
    }


    /**
     * The total value of transactions for this month
     * @return float Numeric amount.
     */
    public function getThisMonthsIncomeTransactionsAmount() {

        if ($this->_thisMonthsIncomeTransactionsAmount == null) {

            $incomeTransactions = $this->getThisMonthsIncomeTransactions();

            $amount = 0;

            foreach ($incomeTransactions as $i) {

                $amount += $i->amount;

            }

            $this->_thisMonthsIncomeTransactionsAmount = $amount;

        }

        return $this->_thisMonthsIncomeTransactionsAmount;

    }


    /**
     * Get list of debt payments for a month.
     * @param int $month Month.
     * @param int $year Year.
     * @return array List of selected debt payments.
     */
    public function getMonthsTransactions($month, $year) {

        $expenseIds = array();
        $jars = $this->getJars()->all();
        foreach ($jars as $j) {

            $thisJarExpenses = $j->getExpenses()->all();
            if (is_array($thisJarExpenses)) {

                foreach ($thisJarExpenses as $e) {

                    $expenseIds[] = $e->id;

                }

            }

        }

        $transactions = Transaction::find()
            ->where([
                'and',
                ['in', 'expense_id', $expenseIds],
                ['between', 'date', date("$year-$month-1"), date("$year-$month-" . date('t', strtotime("$year-$month-1")))]
            ])
            ->all();

        return $transactions;

    }


    /**
     * The transactions for this month
     * @return array
     */
    public function getThisMonthsTransactions()
    {
        if ($this->_thisMonthsTransactions == null) {
            $this->_thisMonthsTransactions = $this->getMonthsTransactions(date('m'), date('Y'));
        }
        return $this->_thisMonthsTransactions;
    }

    /**
     * The total value of transactions for month and year.
     * @param integer $month Month.
     * @param integer $year Year.
     * @return float Total amount.
     */
    public function getMonthsTransactionsAmount($month, $year)
    {
        $transactions = $this->getMonthsTransactions($month, $year);
        $amount = 0;
        foreach ($transactions as $t) {
            $amount += $t->amount;
        }
        return $amount;
    }

    /**
     * The total value of transactions for this month
     * @return float Numeric amount.
     */
    public function getThisMonthsTransactionsAmount()
    {
        if ($this->_thisMonthsTransactionsAmount == null) {
            $transactions = $this->getThisMonthsTransactions();
            $amount = 0;
            foreach ($transactions as $t) {
                $amount += $t->amount;
            }
            $this->_thisMonthsTransactionsAmount = $amount;
        }
        return $this->_thisMonthsTransactionsAmount;
    }

    public function getThisMonthsBudgetBalance()
    {
        $thisMonthsBudgetBalance = 0;
        $jars = $this->getActiveJars();
        /** @var Jar $j */
        foreach ($jars as $j) {
            $thisMonthsBudgetBalance += $j->getThisMonthsFundsRemainingAmount();
        }
        return $thisMonthsBudgetBalance;
    }

    /**
     * Get expenses changes to be shown in a table.
     * @param integer $year Year.
     * @param integer $month Month.
     * @return array The changes to be shown in a table.
     */
    public function getExpenseChangesForTable($year = false, $month = false, $userId = false)
    {
        $jars = Jar::findAll(['budget_id' => $this->id]);
        $jarIds = ArrayHelper::map($jars, 'id', 'id');
        $expenses = Expense::findAll(['jar_id' => $jarIds]);
        $expenseIds = ArrayHelper::map($expenses, 'id', 'id');
        $expensesIndexed = [];
        foreach ($expenses as $e) {
            $expensesIndexed[$e->id] = $e;
        }
        $accounts = Account::findAll(['budget_id' => $this->id]);
        $accountsIndexed = [];
        foreach ($accounts as $a) {
            $accountsIndexed[$a->id] = $a;
        }
        $debts = Debt::findAll(['budget_id' => $this->id]);
        $debtsIndexed = [];
        foreach ($debts as $d) {
            $debtsIndexed[$d->id] = $d;
        }

        // get changes
        $changesQuery = ExpenseChangeMeta::find()
            ->select(ExpenseChangeMeta::tableName() . '.*, ' . ExpenseChange::tableName() . '.time, expense_id, ' . Expense::tableName() . '.name')
            ->joinWith('expenseChange')
            ->leftJoin(Expense::tableName(), ExpenseChange::tableName() . '.expense_id = ' . Expense::tableName() . '.id')
            ->where([Expense::tableName() . '.id' => $expenseIds]);

        if ($year) {
            $changesQuery->andWhere(['YEAR(' . ExpenseChange::tableName() . '.time)' => $year]);
        }

        if ($month) {
            $changesQuery->andWhere(['MONTH(' . ExpenseChange::tableName() . '.time)' => $month]);
        }

        if ($userId) {
            $changesQuery->andWhere([ExpenseChange::tableName() . '.user_id' => $userId]);
        }

        $changes = $changesQuery->orderBy('time desc')
            ->all();

        // change records array for table
        $sortedChanges = [];
        foreach ($changes as $c) {
            $sortedChanges[$c->expense_id][$c->key][strtotime($c->time)] = $c->value;
        }

        $changesForTable = [];
        foreach ($changes as $c) {
            // set pointer to current item
            reset($sortedChanges[$c->expense_id][$c->key]);
            while (key($sortedChanges[$c->expense_id][$c->key]) !== strtotime($c->time)) next($sortedChanges[$c->expense_id][$c->key]);
            $newValue = prev($sortedChanges[$c->expense_id][$c->key]);
            if (!$newValue) {
                $newValue = $expensesIndexed[$c->expense_id]->{$c->key};
                if (($newValue == null) && ($c->key == 'account_id')) {
                    $expenseDebt = ExpenseDebt::findOne(['expense_id' => $c->expense_id]);
                    $newValue = 'd' . $expenseDebt->debt_id;
                }
            }

            if ($c->key == 'account_id') {
                $changesForTable[] = [
                    'time' => strtotime($c->time),
                    'name' => $c->name,
                    'parameter' => 'Account',
                    'from' => (substr($c->value, 0, 1) != 'd') ? $accountsIndexed[$c->value]->name : $debtsIndexed[substr($c->value, 1)]->name . ' (debt)',
                    'to' => (substr($newValue, 0, 1) != 'd') ? $accountsIndexed[$newValue]->name : $debtsIndexed[substr($newValue, 1)]->name . ' (debt)',
                ];
            }
            elseif ($c->key == 'tax_deductible') {
                $changesForTable[] = [
                    'time' => strtotime($c->time),
                    'name' => $c->name,
                    'parameter' => 'Tax Deductible',
                    'from' => ($c->value == 1) ? 'Yes' : 'No',
                    'to' =>  ($newValue == 1) ? 'Yes' : 'No'
                ];
            }
            elseif ($c->key == 'generate_transactions') {
                $changesForTable[] = [
                    'time' => strtotime($c->time),
                    'name' => $c->name,
                    'parameter' => 'Generate Transactions',
                    'from' => ($c->value == 1) ? 'Yes' : 'No',
                    'to' =>  ($newValue == 1) ? 'Yes' : 'No'
                ];
            }
            elseif ($c->key == 'frequency') {
                $changesForTable[] = [
                    'time' => strtotime($c->time),
                    'name' => $c->name,
                    'parameter' => ucfirst($c->key),
                    'from' => Frequencies::getLabel($c->value),
                    'to' => Frequencies::getLabel($newValue)
                ];
            }
            elseif ($c->key == 'amount') {
                $changesForTable[] = [
                    'time' => strtotime($c->time),
                    'name' => $c->name,
                    'parameter' => ucfirst($c->key),
                    'from' => Formatter::currency($c->value),
                    'to' => Formatter::currency($newValue)
                ];
            }
            else {
                $changesForTable[] = [
                    'time' => strtotime($c->time),
                    'name' => $c->name,
                    'parameter' => ucfirst($c->key),
                    'from' => $c->value,
                    'to' => $newValue
                ];
            }
        }
        return $changesForTable;
    }

    public function getIncomeChangesForTable()
    {
        $incomes = Income::findAll(['budget_id' => $this->id]);
        $incomeIds = ArrayHelper::map($incomes, 'id', 'id');
        $incomesIndexed = [];
        foreach ($incomes as $i) {
            $incomesIndexed[$i->id] = $i;
        }
        $accounts = Account::findAll(['budget_id' => $this->id]);
        $accountsIndexed = [];
        foreach ($accounts as $a) {
            $accountsIndexed[$a->id] = $a;
        }

        // get changes
        $changes = IncomeChangeMeta::find()
            ->select(IncomeChangeMeta::tableName() . '.*, ' . IncomeChange::tableName() . '.time, income_id, ' . Income::tableName() . '.name')
            ->joinWith('incomeChange')
            ->leftJoin(Income::tableName(), IncomeChange::tableName() . '.income_id = ' . Income::tableName() . '.id')
            ->where([Income::tableName() . '.id' => $incomeIds])
            ->orderBy('time desc')
            ->all();

        // change records array for table
        $sortedChanges = [];
        foreach ($changes as $c) {
            $sortedChanges[$c->income_id][$c->key][strtotime($c->time)] = $c->value;
        }

        $changesForTable = [];
        foreach ($changes as $c) {
            // set pointer to current item
            reset($sortedChanges[$c->income_id][$c->key]);
            while (key($sortedChanges[$c->income_id][$c->key]) !== strtotime($c->time)) next($sortedChanges[$c->income_id][$c->key]);
            $newValue = prev($sortedChanges[$c->income_id][$c->key]);
            if (!$newValue) {
                $newValue = $incomesIndexed[$c->income_id]->{$c->key};
            }

            if ($c->key == 'account_id') {
                $changesForTable[] = [
                    'time' => strtotime($c->time),
                    'name' => $c->name,
                    'parameter' => 'Account',
                    'from' => $accountsIndexed[$c->value]->name,
                    'to' => $accountsIndexed[$newValue]->name
                ];
            }
            elseif ($c->key == 'frequency') {
                $changesForTable[] = [
                    'time' => strtotime($c->time),
                    'name' => $c->name,
                    'parameter' => ucfirst($c->key),
                    'from' => Frequencies::getLabel($c->value),
                    'to' => Frequencies::getLabel($newValue)
                ];
            }
            elseif ($c->key == 'amount') {
                $changesForTable[] = [
                    'time' => strtotime($c->time),
                    'name' => $c->name,
                    'parameter' => ucfirst($c->key),
                    'from' => Formatter::currency($c->value),
                    'to' => Formatter::currency($newValue)
                ];
            }
            else {
                $changesForTable[] = [
                    'time' => strtotime($c->time),
                    'name' => $c->name,
                    'parameter' => ucfirst($c->key),
                    'from' => $c->value,
                    'to' => $newValue
                ];
            }
        }
        return $changesForTable;
    }

    /**
     * @return Jar|null
     */
    public function getDebtsJar()
    {
        return Jar::find()
            ->where([
                'budget_id' => $this->id,
                'jar_type_id' => 2
            ])
            ->one();
    }

}
