<?php

namespace backend\models\db;

use yii\helpers\Url;

use backend\models\db\Users;

use backend\components\helpers\Formatter;

/**
 * This is the model class for table "conversations".
 *
 * @property integer $id
 * @property integer $archived
 *
 * @property ConversationMessage[] $conversationMessages
 * @property ConversationToUser[] $conversationsToUsers
 * @property User[] $users
 */
class Conversation extends \yii\db\ActiveRecord
{

    private $_data;
    private $_unreadMessagesCount;

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'conversations';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['name'], 'string'],
            [['archived'], 'integer']
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'name' => 'Name',
            'archived' => 'Archived',
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getConversationMessages()
    {
        return $this->hasMany(ConversationMessage::className(), ['conversation_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getConversationsToUsers()
    {
        return $this->hasMany(ConversationToUser::className(), ['conversation_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUsers()
    {
        return $this->hasMany(User::className(), ['id' => 'user_id'])->viaTable('conversations_to_users',
            ['conversation_id' => 'id']);
    }

    /**
     * Add a message to the conversation.
     * @param integer $userId The ID of the user adding the message.
     * @param string $text The message text.
     * @return ConversationMessage The conversation message object.
     */
    public function addMessage($userId, $text)
    {
        $message = new ConversationMessage();
        $message->setAttributes([
            'user_id' => $userId,
            'conversation_id' => $this->id,
            'text' => $text
        ]);
        $message->save();
        return isset($message->id) ? ConversationMessage::findOne($message->id) : $message;
    }

    /**
     * Look for last seen message log.
     * @param integer $userId The ID of the user this count is for.
     * @return object|bool Returns last seen message object, or false if there is none.
     */
    public function getLastSeenMessageLog($userId)
    {
        $lastSeen = ConversationMessageSeen::find()
            ->joinWith('conversationMessage')
            ->where([
                'and',
                [ConversationMessageSeen::tableName() . '.user_id' => $userId],
                [ConversationMessage::tableName() . '.conversation_id' => $this->id]
            ])
            ->orderBy(ConversationMessageSeen::tableName() . '.id DESC')
            ->one();
        return ($lastSeen) ? $lastSeen : false;
    }

    /**
     * Get the last message a user has received
     * @param integer $userId The ID of the user receiving the messages.
     * @return Object|boolean Returns message on success, false if no message.
     */
    public function getLastReceivedMessage($userId)
    {
        $message = $this->getConversationMessages()
            ->where(['!=', 'user_id', $userId])
            ->andWhere(['conversation_id' => $this->id])
            ->orderBy('id DESC')
            ->one();
        return ($message) ? $message : false;
    }

    public function getLastMessage()
    {
        $message = $this->getConversationMessages()
            ->where(['conversation_id' => $this->id])
            ->orderBy('id DESC')
            ->one();
        return ($message) ? $message : false;
    }

    /**
     * Get the number of unread conversations for a user.
     * @param integer $userId The user to which this information is related.
     * @return integer Number of unread messages for this conversation.
     */
    public function getUnreadMessagesCount($userId)
    {
        if ($this->_unreadMessagesCount == null) {
            $lastSeenLog = $this->getLastSeenMessageLog($userId);
            $messagesCountQuery = $this->getConversationMessages()
                ->where([
                    'and',
                    ['conversation_id' => $this->id],
                    ['!=', 'user_id', $userId]
                ]);
            if ($lastSeenLog) {
                $messagesCountQuery->andWhere(['>', 'id', $lastSeenLog->conversation_message_id]);
            }
            $this->_unreadMessagesCount = $messagesCountQuery->count();
        }

        return $this->_unreadMessagesCount;
    }

    /**
     * Marks the entire thread as read
     * @param $userId
     * @param array $messages
     */
    public function markAsRead($userId, $messages = array())
    {
        if (empty($messages)) {
            $messages = $this->getConversationMessages()->all();
        }
        foreach ($messages as $message) {
            $seen = ConversationMessageSeen::find()
                ->where([
                    'conversation_message_id' => $message->id,
                    'user_id' => $userId
                ])
                ->exists();
            if (!$seen) {
                $cms = new ConversationMessageSeen();
                $cms->setAttributes([
                    'conversation_message_id' => $message->id,
                    'user_id' => $userId
                ]);
                $cms->save();
            }
        }
    }

    /**
     * Get the current conversation's data.
     * @return array Conversation data.
     */
    public function getData()
    {
        if ($this->_data == null) {
            $users = $this->users;
            $participants = [];

            foreach ($users as $u) {
                $participants[$u->id] = $u;
                if ($u->user_role_id == UserRole::CLIENT) {
                    $children = $u->userChildren;
                    foreach ($children as $c) {
                        $cUser = $c->user;
                        $participants[$cUser->id] = $cUser;
                    }
                }
            }

            // add superadmins to participants
            $sas = User::findAll(['user_role_id' => UserRole::SUPERADMIN]);
            foreach ($sas as $s) {
                $participants[$s->id] = $s;
            }
            $this->_data = [
                'participants' => $participants
            ];
        }

        return $this->_data;
    }

    public function getMessages($userId, $limit = 25, $lastMessageId = false, $includeOwn = false)
    {
        $messagesQuery = $this->getConversationMessages()->orderBy('id DESC')->limit($limit);
        if ($lastMessageId) {
            $messagesQuery->where(['>', 'id', $lastMessageId]);
        }
        if (!$includeOwn) {
            $messagesQuery->andWhere(['!=', 'user_id', $userId]);
        }
        return array_reverse($messagesQuery->all());
    }

    public function getConversationMessagesHtml($userId, $limit = 25, $lastMessageId = false, $includeOwn = false)
    {
        $messages = $this->getMessages($userId, $limit, $lastMessageId, $includeOwn);

        // build the HTML output
        $out = '';
        $data = $this->getData();
        foreach ($messages as $m) {
            if ($m->user_id == $userId) {
                $out .= '<li class="mine" data-message-id="' . $m->id . '"><p>' . $m->text . '</p></li>';
            }
            else {
                $meta = $data['participants'][$m->user_id]->getUserMetaSorted();
                $fullName = $meta['full_name'];
                $username = $data['participants'][$m->user_id]->username;

                $out .= '<li class="their" data-message-id="' . $m->id . '">
                    <p>
                        <img class="avatar" src="' . $data['participants'][$m->user_id]->getAvatarUrl() . '">
                        <span class="name">' . $fullName . (strlen($fullName) ? ' (' . $username . ')' : $username) . '</span>
                        ' . $m->text . '
                    </p>
                </li>';
            }
        }
        return $out;
    }

    /**
     * Returns the number of unread messages for the given user
     * @param integer $userId
     * @return int
     */
    public static function getMessagesCount($userId)
    {
        $count = 0;
        $conversations = Conversation::find()
            ->joinWith('conversationsToUsers')
            ->where(['user_id' => $userId])
            ->all();

        foreach ($conversations as $c) {
            /** @var Conversation $c */
            $count += $c->getUnreadMessagesCount($userId);
        }
        return $count;
    }
}
