<?php

namespace backend\models\db;

use backend\components\helpers\Formatter;
use backend\models\form\AddDebt;
use Yii;

use backend\models\ChangeMetaModel;

use backend\components\helpers\Calculator;
use yii\caching\TagDependency;

/**
 * This is the model class for table "debts".
 *
 * @property integer $id
 * @property integer $account_id
 * @property integer $active_debt_account_id
 * @property integer $jar_id
 * @property integer $debt_type_id
 * @property string $name
 * @property double $interest
 * @property double $monthly_interest
 * @property string $date
 * @property string $next_due_date
 * @property string $frequency
 * @property double $payment
 * @property double $payment_minimum
 * @property double $amount
 * @property double $old_amount
 * @property integer $tax_deductible
 * @property integer $added_with_budget
 * @property integer $archived
 *
 * @property Account $activeDebtAccount
 * @property DebtPayment[] $debtPayments
 * @property DebtType $debtType
 * @property Jar $jar
 * @property Budget $budget
 */
class Debt extends ChangeMetaModel
{

    public $start_date = false;
    public $end_date = false;

    private $_averageMonthlyAmount;
    private $_debtPaymentsAmount;
    private $_oldDebtPaymentsAmount;
    private $_debtLeftAmount;
    private $_oldDebtLeftAmount;
    private $_lastMonthsDebtAmount;
    private $_nextDue;
    private $_thisMonthsCcTransactions;
    private $_thisMonthsCcTransactionsAmount;
    private $_thisMonthsDebtPayments;
    private $_thisMonthsDebtPaymentsAmount;
    private $_thisMonthsDebtInterestPayments;
    private $_thisMonthsDebtInterestPaymentsAmount;

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'debts';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['account_id','active_debt_account_id','budget_id', 'debt_type_id', 'tax_deductible', 'archived'], 'integer'],
            [['interest', 'payment', 'amount', 'old_amount'], 'number'],
            [['date', 'next_due_date', 'interest_free_end_date'], 'safe'],
            [['name'], 'string', 'max' => 200],
            [['frequency'], 'string', 'max' => 20]
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'budget_id' => 'Budget ID',
            'debt_type_id' => 'Debt Type ID',
            'account_id' => 'Account ID',
            'active_debt_account_id' => 'Active Debt Account ID',
            'name' => 'Name',
            //'interest' => 'Interest',
            'interest' => 'Monthly Interest',
            'date' => 'Date',
            'frequency' => 'Frequency',
            'amount' => 'Amount',
            'old_amount' => 'Old Debt Amount',
            'tax_deductible' => 'Tax Deductible',
            'interest_free_end_date' => 'Interest Free End Date',
            'archived' => 'Archived',
            'payment' => 'Payment'
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getDebtPayments()
    {
        return $this->hasMany(DebtPayment::className(), ['debt_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getDebtType()
    {
        return $this->hasOne(DebtType::className(), ['id' => 'debt_type_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getDebtEnd()
    {
        return $this->hasOne(DebtEnd::className(), ['debt_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBudget()
    {
        return $this->hasOne(Budget::className(), ['id' => 'budget_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccount()
    {
        return $this->hasOne(Account::className(), ['id' => 'account_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getActiveDebtAccount()
    {
        return $this->hasOne(Account::className(), ['id' => 'active_debt_account_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getDebtAmounts()
    {
        return $this->hasMany(DebtAmount::className(), ['debt_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getDebtChanges()
    {
        return $this->hasMany(DebtChange::className(), ['debt_id' => 'id']);
    }

    /**
     * @inheritdoc
     */
    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);
        if ($this->scenario == self::SCENARIO_CLONE) {
            return;
        }

        $jar = $this->budget->getDebtsJar();
        if ($jar) {
            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jar->id);
        }

        // new debt record created
        if ($insert) {
            $debtChange = new DebtChange();
            $debtChange->setAttributes([
                'type'    => 'C',
                'debt_id' => $this->id,
                'user_id' => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $debtChange->save();

            $todayDate = date('Y-m-d');
            // generate payment if today or in the past
            if (strtotime($this->date) <= strtotime($todayDate)) {
                if ($this->payment) {
                    $debtPayment = new DebtPayment();
                    $debtPayment->setAttributes([
                        'debt_id' => $this->id,
                        'debt_payment_type_id' => 1,
                        'account_id' => $this->account_id,
                        'amount' => $this->payment,
                        'date' => $this->date
                    ]);
                    $debtPayment->save();
                }
            }
        }
        else {
            // debt amount updated
            if (count($changedAttributes) == 1 &&
                isset($changedAttributes['amount']) &&
                $this->amount != $changedAttributes['amount']) {

                $debtChange = new DebtChange();
                $debtChange->setAttributes([
                    'type'    => 'U',
                    'debt_id' => $this->id,
                    'user_id' => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                        Yii::$app->user->identity->id : null
                ]);
                $debtChange->save();

                $debtChangeMeta = new DebtChangeMeta();
                $debtChangeMeta->setAttributes([
                    'debt_change_id' => $debtChange->id,
                    'key'            => 'amount',
                    'value'          => (string)$changedAttributes['amount']
                ]);
                $debtChangeMeta->save();

            }
            // debt edited
            else {
                // check if and what changed
                $changes = array();
                $debtAttrs = $this->getAttributes();

                foreach ($changedAttributes as $key => $o) {
                    if (isset($debtAttrs[$key]) && ($debtAttrs[$key] != $o)) {
                        $changes[$key] = $o;
                    }
                }

                if (count($changes) > 0) {
                    $debtChange = new DebtChange();
                    $debtChange->setAttributes([
                        'type'    => 'E',
                        'debt_id' => $this->id,
                        'user_id' => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                            Yii::$app->user->identity->id : null
                    ]);
                    $debtChange->save();

                    foreach($changes as $key => $value) {
                        $debtChangeMeta = new DebtChangeMeta();
                        $debtChangeMeta->setAttributes([
                            'debt_change_id' => $debtChange->id,
                            'key'            => $key,
                            'value'          => (string)$value
                        ]);
                        $debtChangeMeta->save();
                    }
                }
            }
            // debt removed
            if (isset($changedAttributes['archived']) && $this->archived == 1) {
                $debtChange = new DebtChange();
                $debtChange->setAttributes([
                    'type'    => 'D',
                    'debt_id' => $this->id,
                    'user_id' => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                        Yii::$app->user->identity->id : null
                ]);
                $debtChange->save();
            }
        }

        // debt created or edited (and not archived) - need to make sure
        // we have credit card account
        if ($insert || (!isset($changedAttributes['archived']) &&
            $this->archived != 1)) {
            $ccAccount = $this->active_debt_account_id ?
                Account::findOne($this->active_debt_account_id) : null;

            if (!$ccAccount) {
                $ccAccount = new Account();
                $ccAccount->setAttributes([
                    'budget_id' => $this->budget_id,
                    'name' => $this->name,
                    'account_type_id' => 2
                ]);
                $ccAccount->save();
                // update the debt account_id
                $this->active_debt_account_id = $ccAccount->id;
                $this->save();
            }
        }
    }

    /**
     * @inheritdoc
     */
    public function afterDelete()
    {
        $jar = $this->budget->getDebtsJar();
        if ($jar) {
            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jar->id);
        }
        parent::afterDelete();
    }


    public function getDebtTypeName()
    {
        return $this->getDebtType()->one()->name;
    }

    public function isLoan()
    {

    }

    /**
     * Delete time, if the debt was deleted
     * @return null|string
     */
    public function getDeletedTime()
    {
        $deleted = DebtChange::findOne([
            'debt_id' => $this->id,
            'type' => 'D'
        ]);
        return $deleted ? $deleted->time : null;
    }

    /**
     * Get adjustment amount for the account for the given month
     * @param string|null $month
     * @param string|null $year
     * @return float
     */
    public function getAdjustmentAmount($month = null, $year = null)
    {
        if (!$month) {
            $month = date('n');
        }
        if (!$year) {
            $year = date('Y');
        }
        $transactions = $this->getMonthsAdjustmentsTransactions($month, $year);
        $amount = 0;
        foreach ($transactions as $t) {
            $amount += $t->amount;
        }
        return $amount;
    }

    /**
     * @param $month
     * @param $year
     * @return array
     */
    public function getMonthsAdjustmentsTransactions($month, $year)
    {
        $from = date('Y-m-01', strtotime("$year-$month-20"));
        $to   = date('Y-m-t', strtotime("$year-$month-20"));

        return DebtPayment::find()
            ->where([
                'and',
                ['between', 'date', $from, $to],
                ['debt_id' => $this->id],
                ['is_adjustment' => DebtPayment::DEBT_PAYMENT_ADJUSTMENT]
            ])
            ->all();
    }

    /**
     * Get Current Payment as it was at the given time in the past
     * @param string $month
     * @param string $year
     * @return float
     */
    public function getCurrentPayment($month, $year)
    {
        // return the latest value for the current month
        if ("{$year}-{$month}" == date('Y-m')) {
            return $this->payment;
        }
        $edit = DebtChangeMeta::find()
            ->joinWith('debtChange')
            ->where([
                'debt_id' => $this->id,
                'type' => 'E',
                'key' => 'payment'
            ])
            ->andWhere([
                '>', 'time', date('Y-m-t 23:59:59', strtotime("{$year}-{$month}-20"))
            ])
            ->orderBy('time ASC')
            ->one();
        if ($edit) {
            return $edit->value;
        }
        // just get the payment
        return $this->payment;
    }

    /**
     * Get Current Frequency as it was at the given time in the past
     * @param string $month
     * @param string $year
     * @return string
     */
    public function getCurrentFrequency($month, $year)
    {
        // return the latest value for the current month
        if ("{$year}-{$month}" == date('Y-m')) {
            return $this->frequency;
        }
        $edit = DebtChangeMeta::find()
            ->joinWith('debtChange')
            ->where([
                'debt_id' => $this->id,
                'type' => 'E',
                'key' => 'frequency'
            ])
            ->andWhere([
                '>', 'time', date('Y-m-t 23:59:59', strtotime("{$year}-{$month}-20"))
            ])
            ->orderBy('time ASC')
            ->one();
        if ($edit) {
            return $edit->value;
        }
        // just get the payment
        return $this->frequency;
    }

    /**
     * Get the monthly average amount for a record.
     * @param string|null $month
     * @param string|null $year
     * @return float Monthly average amount for record.
     */
    public function getAverageMonthlyAmount($month = null, $year = null)
    {
        if (!$month) {
            $month = date('m');
        }
        if (!$year) {
            $year = date('Y');
        }
        return Calculator::monthlyAverageAmount($this->getCurrentPayment($month, $year), $this->getCurrentFrequency($month, $year));
    }

    /**
     * Get the monthly minimum amount
     * @return float
     */
    public function getMinimumMonthlyAmount()
    {
        return Calculator::monthlyMinimumAmount($this->payment, $this->frequency);
    }

    /**
     * Get the value of already made payments for this debt.
     * @param bool $onlyOldDebt Include only payments that apply to the old debt
     * @param bool $onlyAdjusted If true it takes the amount provided in EoMS, otherwise the payments from the recurring expenses
     * @return float Sum of this debt's payments
     */
    public function getDebtPaymentsAmount($onlyOldDebt = false, $onlyAdjusted = true)
    {
        $amount = 0;
        $payments = $onlyOldDebt ?
            $this->getDebtPayments()->where(['debt_payment_type_id' => 3]) :
            $this->getDebtPayments();
        if ($onlyAdjusted) {
            $payments->andWhere([
                'is_adjustment' => DebtPayment::DEBT_PAYMENT_ADJUSTMENT
            ]);
        }
        else {
            $payments->andWhere([
                'is_adjustment' => 0
            ]);
        }
        foreach ($payments->all() as $p) {
            $amount += $p->amount;
        }
        return $amount;
    }

    /**
    * Get the value of already made payments for this debt up until a certain date
    * @param string $date The date
    * @return float Sum of this debt's payments.
    */
    public function getDebtPaymentsAmountUpUntil($date)
    {
        $amount = 0;
        $payments = $this->getDebtPayments()
            ->where(['<=', 'date', $date])
            ->all();
        foreach ($payments as $p) {
            $amount += $p->amount;
        }
        return $amount;
    }

    /**
     * Get the debt amount as it was when the debt was created
     * @return float
     */
    public function getInitialAmount()
    {
        $edit = DebtChangeMeta::find()
            ->joinWith('debtChange')
            ->where([
                'debt_id' => $this->id,
                'key' => 'amount'
            ])
            ->andWhere([
                'or',
                ['type' => 'E'],
                ['type' => 'U']
            ])
            ->orderBy('time ASC')
            ->one();
        if ($edit) {
            return $edit->value;
        }
        return $this->amount;
    }

    /**
     * Get the amount of debt left.
     * @param null $month
     * @param null $year
     * @return float How much debt is left.
     */
    public function getDebtLeftAmount($month = null, $year = null)
    {
        if (!$month) {
            $month = date('m');
        }
        if (!$year) {
            $year = date('Y');
        }
        $timeNow = strtotime("{$year}-{$month}-20");
        $debtAmount = $this->getMonthsDebtAmount(
            date('n', strtotime('-1 month', $timeNow)),
            date('Y', strtotime('-1 month', $timeNow))
        );
        if (!$debtAmount) {
            // need to calculate all activity since the account inception
            return $this->getInitialAmount() - $this->getDebtPaymentsAmount();
        }
        else {
            return $debtAmount - $this->getMonthsDebtPaymentsAmount(date('m', $timeNow), date('Y', $timeNow), null);
        }
    }

    /**
     * Get the amount of old debt (added during FTS) left
     * @return float How much old debt is left
     */
    public function getOldDebtLeftAmount()
    {
        if ($this->_oldDebtLeftAmount == null) {
            $amount = $this->old_amount - $this->getDebtPaymentsAmount(true);
            $this->_oldDebtLeftAmount = $amount;
        }
        return $this->_oldDebtLeftAmount;
    }

    /**
     * Get the next due date of debt payment.
     * @return string Date of next due date of debt payment.
     */
    public function getNextDue($startDate = false, $reset = false, $dateLimit = false)
    {
        if (($this->_nextDue == null) || $reset) {
            $date = $this->next_due_date ? $this->next_due_date : $this->date;
            if ($startDate) {
                $this->_nextDue = Calculator::nextDueDate($date, $this->frequency, $dateLimit, $startDate);
            }
            else {
                $this->_nextDue = Calculator::nextDueDate($date, $this->frequency, $dateLimit);
            }
        }
        return $this->_nextDue;
    }

    /**
     * Get list of debt payments for a month.
     * @param int $month Month.
     * @param int $year Year.
     * @param bool $includeBalanceAdjustments
     * @return array List of selected debt payments
     */
    public function getMonthsDebtPayments($month, $year, $includeBalanceAdjustments = true)
    {
        $debtPayments = DebtPayment::find()
            ->where([
                'and',
                ['debt_id' => $this->id],
                ['debt_payment_type_id' => [1, 3]],
                ['MONTH(date)' => $month],
                ['YEAR(date)' => $year]
            ]);
        if ($includeBalanceAdjustments === null) {
            $debtPayments->andWhere([
                'is_adjustment' => 1
            ]);
        }
        elseif ($includeBalanceAdjustments) {
            $debtPayments->andWhere(['or',
                ['is_adjustment' => 0],
                ['is_adjustment' => DebtPayment::DEBT_PAYMENT_ADJUSTMENT]
            ]);
        }
        else {
            $debtPayments->andWhere([
                'is_adjustment' => 0
            ]);
        }
        return $debtPayments->all();
    }

    /**
     * The total value of transactions for a month
     * @param int $month Month.
     * @param int $year Year.
     * @param bool $includeBalanceAdjustments
     * @return float Numeric amount.
     */
    public function getMonthsDebtPaymentsAmount($month, $year, $includeBalanceAdjustments = false)
    {
        $debtPayments = $this->getMonthsDebtPayments($month, $year, $includeBalanceAdjustments);
        $amount = 0;
        foreach ($debtPayments as $t) {
            $amount += $t->amount;
        }
        return $amount;
    }

    /**
     * The debtPayments for this month
     * @return array
     */
    public function getThisMonthsDebtPayments()
    {
        if ($this->_thisMonthsDebtPayments == null) {
            $this->_thisMonthsDebtPayments = $this->getMonthsDebtPayments(date('m'), date('Y'));
        }
        return $this->_thisMonthsDebtPayments;
    }

    /**
     * The total value of transactions for this month
     * @return float Numeric amount.
     */
    public function getThisMonthsDebtPaymentsAmount()
    {
        if ($this->_thisMonthsDebtPaymentsAmount == null) {
            $debtPayments = $this->getThisMonthsDebtPayments();
            $amount = 0;
            foreach ($debtPayments as $t) {
                $amount += $t->amount;
            }
            $this->_thisMonthsDebtPaymentsAmount = $amount;
        }
        return $this->_thisMonthsDebtPaymentsAmount;
    }

    /**
     * Get list of CC transactions for a month.
     * @param int $month Month.
     * @param int $year Year.
     * @return array List of selected debt payments.
     */
    public function getMonthsCcTransactions($month, $year)
    {
        $ccTransactions = CcTransaction::find()
            ->where([
                'and',
                ['debt_id' => $this->id],
                ['MONTH(date)' => $month],
                ['YEAR(date)' => $year]
            ])
            ->all();

        return $ccTransactions;
    }

    /**
     * The ccTransactions for this month
     * @return array
     */
    public function getThisMonthsCcTransactions()
    {
        if ($this->_thisMonthsCcTransactions == null) {
            $this->_thisMonthsCcTransactions = $this->getMonthsCcTransactions(date('m'), date('Y'));
        }
        return $this->_thisMonthsCcTransactions;
    }

    /**
     * The total value of transactions for this month
     * @return float Numeric amount.
     */
    public function getThisMonthsCcTransactionsAmount()
    {
        if ($this->_thisMonthsCcTransactionsAmount == null) {
            $ccTransactions = $this->getThisMonthsCcTransactions();
            $amount = 0;
            foreach ($ccTransactions as $t) {
                $amount += $t->amount;
            }
            $this->_thisMonthsCcTransactionsAmount = $amount;
        }
        return $this->_thisMonthsCcTransactionsAmount;
    }

    /**
     * Get list of debt payments for a month.
     * @param int $month Month.
     * @param int $year Year.
     * @return array List of selected debt payments.
     */
    public function getMonthsDebtInterestPayments($month, $year)
    {
        $interestPayments = DebtPayment::find()
            ->where([
                'and',
                ['debt_id' => $this->id],
                ['=', 'debt_payment_type_id', '2'],
                ['MONTH(date)' => $month],
                ['YEAR(date)' => $year]
            ])
            ->all();

        return $interestPayments;
    }

    /**
     * The total value of transactions for a month
     * @param int $month Month.
     * @param int $year Year.
     * @return float Numeric amount.
     */
    public function getMonthsDebtInterestPaymentsAmount($month, $year)
    {
        $debtPayments = $this->getMonthsDebtInterestPayments($month, $year);
        $amount = 0;
        foreach ($debtPayments as $t) {
            $amount += $t->amount;
        }
        return $amount;
    }

    /**
     * The debtPayments for this month
     * @return array
     */
    public function getThisMonthsDebtInterestPayments()
    {
        if ($this->_thisMonthsDebtInterestPayments == null) {
            $this->_thisMonthsDebtInterestPayments = $this->getMonthsDebtInterestPayments(date('m'), date('Y'));
        }
        return $this->_thisMonthsDebtInterestPayments;
    }

    /**
     * The total value of transactions for this month
     * @return float Numeric amount.
     */
    public function getThisMonthsDebtInterestPaymentsAmount()
    {
        if ($this->_thisMonthsDebtInterestPaymentsAmount == null) {
            $debtPayments = $this->getThisMonthsDebtInterestPayments();
            $amount = 0;
            foreach ($debtPayments as $t) {
                $amount += $t->amount;
            }
            $this->_thisMonthsDebtInterestPaymentsAmount = $amount;
        }
        return $this->_thisMonthsDebtInterestPaymentsAmount;
    }

    public function getMonthsDebtAmount($month, $year)
    {
        $debtAmount = DebtAmount::find()
            ->joinWith('eoms')
            ->joinWith('debt.budget')
            ->where([
                'debt_id' => $this->id,
                'month' => date('n', strtotime("$year-$month-20")),
                'year' => date('Y', strtotime("$year-$month-20")),
                'is_mock' => 0
            ])
            ->one();
        return $debtAmount ? $debtAmount->amount : false;
    }

    /**
     * Get last month's logged debt amount.
     * @return float Numeric amount.
     */
    public function getLastMonthsDebtAmount()
    {
        if ($this->_lastMonthsDebtAmount == null) {
            $timeNow = strtotime(date('Y-m-20'));
            $this->_lastMonthsDebtAmount = $this->getMonthsDebtAmount(
                date('n', strtotime('-1 month', $timeNow)),
                date('Y', strtotime('-1 month', $timeNow))
            );
        }
        return $this->_lastMonthsDebtAmount;
    }

    /**
     * @param Eoms $eoms
     * @param float $oldAmount
     * @param float $newAmount
     */
    public function saveBalanceAdjustmentRecord(Eoms $eoms, $oldAmount, $newAmount)
    {
        $difference      = Formatter::float($oldAmount - $newAmount);
        $transactionDate = date('Y-m-t', strtotime($eoms->year . '-' . $eoms->month . '-20'));

        DebtPayment::deleteAll([
            'debt_id'              => $this->id,
            'debt_payment_type_id' => 1,
            'date'                 => $transactionDate,
            'is_adjustment'        => DebtPayment::DEBT_PAYMENT_ADJUSTMENT
        ]);

        if ($difference != 0) {
            $debtPayment = new DebtPayment();
            $debtPayment->setAttributes([
                'debt_id'              => $this->id,
                'debt_payment_type_id' => 1,
                'account_id'           => $this->account_id,
                'amount'               => $difference,
                'date'                 => $transactionDate,
                'description'          => 'Debt Amount Adjustment for ' . date('M Y', strtotime("{$eoms->year}-{$eoms->month}-20")),
                'is_adjustment'        => DebtPayment::DEBT_PAYMENT_ADJUSTMENT
            ]);
            $debtPayment->save();
        }
    }

    public function deleteCompletely()
    {
        foreach ($this->getDebtChanges() as $change) {
            DebtChangeMeta::deleteAll(['debt_change_id' => $change->id]);
        }
        DebtChange::deleteAll(['debt_id' => $this->id]);
        DebtAmount::deleteAll(['debt_id' => $this->id]);
        DebtEnd::deleteAll(['debt_id' => $this->id]);
        DebtPayment::deleteAll(['debt_id' => $this->id]);
        $this->delete();
    }


    /**
     * Determines whether it's possible to adjust the debt value
     * for the adjustment amounts provided
     * @param $oldAmount
     * @param $newAmount
     * @return bool
     */
    public static function canAdjustRecord($oldAmount, $newAmount)
    {
        // don't allow to increase debt (positive $newAmount means debt)
        if ($newAmount > $oldAmount && $newAmount > 0) {
            return false;
        }
        return true;
    }

    /**
     * @param array $formData
     * @param bool $isFts
     */
    public static function createDebtFromForm(array $formData, $isFts = false)
    {
        $debt = new Debt();
        $debt->setAttributes($formData);
        // if debt added during FTS we need to record it as an 'old debt' (only for CC debts)
        if ($debt->debt_type_id == 1 && $isFts) {
            $debt->old_amount = $debt->amount;
        }
        $debt->save();

        // check if we have the appropriate jars already in the system
        $debtJar = Jar::findOne(['budget_id' => $formData['budget_id'], 'jar_type_id' => 2]);
        if (!$debtJar) {
            $debtJar = new Jar();
            $debtJar->setAttributes([
                'budget_id' => $formData['budget_id'],
                'jar_type_id' => 2,
                'name' => 'Debts',
                'amount' => 0,
                'type' => 'debts'
            ]);
            $debtJar->save();
        }
    }

}
