<?php

namespace backend\models\db;

use Yii;

use backend\components\helpers\Calculator;

use backend\models\ChangeMetaModel;
use yii\caching\TagDependency;

/**
 * This is the model class for table "expenses".
 *
 * @property integer $id
 * @property integer $jar_id
 * @property string $name
 * @property float $amount
 * @property string $account_id
 * @property string $date
 * @property string $frequency
 * @property integer $is_unbudgeted
 * @property integer $is_adjustment
 * @property integer $tax_deductible
 * @property integer $generate_transactions
 * @property integer $archived
 *
 * @property ExpenseChange[] $thisxpenseChanges
 * @property ExpenseEvent[] $thisxpenseEvents
 * @property Jar $jar
 * @property Account $account;
 * @property Transaction[] $transactions
 */
class Expense extends ChangeMetaModel
{
    const OTHER_UNBUDGETED_EXPENSE_NAME = 'Other (unbudgeted)';
    const EXPENSE_ADJUSTMENT = 1;
    const JAR_ADJUSTMENT = 2;

    public $debt_id = null;
    public $end_date;
    public $removal_time = false;
    public $delete_time = false;
    public $start_date = false;

    private $_averageMonthlyAmount;
    private $_averageYearlyAmount;
    private $_shortfallAmount;
    private $_adjustedMonthlyAmount;
    private $_nextDue;
    private $_plannedPaymentsNumber;
    private $_amountAlreadySaved;
    private $_monthsLeftUntil;
    private $_paymentsLeftUntil;

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'expenses';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['jar_id', 'tax_deductible', 'is_adjustment', 'archived', 'account_id', 'generate_transactions'], 'integer'],
            [['date', 'is_unbudgeted'], 'safe'],
            [['name'], 'string', 'max' => 200],
            [['amount'], 'number'],
            [['frequency'], 'string', 'max' => 20]
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'jar_id' => 'Jar ID',
            'name' => 'Name',
            'amount' => 'Amount',
            'account_id' => 'Account ID',
            'date' => 'Date',
            'frequency' => 'Frequency',
            'tax_deductible' => 'Tax Deductible',
            'archived' => 'Archived',
            'generate_transactions' => 'Generate Transactions'
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getExpenseChanges()
    {
        return $this->hasMany(ExpenseChange::className(), ['expense_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getExpenseEvents()
    {
        return $this->hasMany(ExpenseEvent::className(), ['expense_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getExpenseStart()
    {
        return $this->hasOne(ExpenseStart::className(), ['expense_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getExpenseEnd()
    {
        return $this->hasOne(ExpenseEnd::className(), ['expense_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getExpenseDebt()
    {
        return $this->hasOne(ExpenseDebt::className(), ['expense_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getJar()
    {
        return $this->hasOne(Jar::className(), ['id' => 'jar_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccount()
    {
        return $this->hasOne(Account::className(), ['id' => 'account_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getTransactions()
    {
        return $this->hasMany(Transaction::className(), ['expense_id' => 'id']);
    }

    /**
     * @inheritdoc
     */
    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);
        if ($this->scenario == self::SCENARIO_CLONE) {
            return;
        }

        if ($this->jar_id) {
            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $this->jar_id);
        }

        // expense added
        if ($insert) {
            // this doesn't apply to unbudgeted or adjustment expenses, and api requests
            if (!$this->is_unbudgeted && !$this->is_adjustment && $this->scenario != 'api') {
                $today = date('Y-m-d');
                // one-time expenses have starts
                if ($this->frequency == 'one-time' && $this->generate_transactions == 1) {
                    $expenseStart = new ExpenseStart();
                    $expenseStart->date = $this->date < $today ? $this->date : $today;
                    $expenseStart->expense_id = $this->id;
                    $expenseStart->save();
                }

                // generate transaction if necessary
                $isOnetimeOneGo = ($this->frequency == 'one-time' && $this->generate_transactions == 0);
                $isRecurring = $this->isRecurring();
                if (strtotime($this->date) <= time() && ($isRecurring || $isOnetimeOneGo)) {
                    $transaction = new Transaction();
                    $transaction->account_id = $this->account_id;
                    $transaction->date = $this->date < $today ? $this->date : $today;
                    $transaction->expense_id = $this->id;
                    $transaction->amount = $this->amount;
                    $transaction->tax_deductible = $this->tax_deductible ? 1 : 0;
                    $transaction->save();
                }
            }

            // save expense change
            $expenseChange = new ExpenseChange();
            $expenseChange->setAttributes([
                'type'       => 'C',
                'expense_id' => $this->id,
                'user_id'    => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $expenseChange->save();
        }
        // expense edited
        else {
            $accountId   = array_key_exists('account_id', $changedAttributes) ?
                $changedAttributes['account_id'] : false;
            $expenseDebt = ExpenseDebt::findOne(['expense_id' => $this->id]);
            if ($expenseDebt) {
                $expenseDebt->delete();
            }

            // check if and what changed
            $changes = array();
            $expenseAttrs = $this->getAttributes();
            foreach ($changedAttributes as $key => $o) {
                if ($expenseAttrs[$key] != $o) {
                    $changes[$key] = $o;
                }
            }

            if (count($changes) > 0) {
                $expenseChange = new ExpenseChange();
                $expenseChange->setAttributes([
                    'type'       => 'E',
                    'expense_id' => $this->id,
                    'user_id'    => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                        Yii::$app->user->identity->id : null
                ]);
                $expenseChange->save();
                foreach($changes as $key => $value) {
                    $expenseChangeMeta = new ExpenseChangeMeta();
                    $expenseChangeMeta->setAttributes([
                        'expense_change_id' => $expenseChange->id,
                        'key'               => $key,
                        'value'             => (string)$value
                    ]);
                    $expenseChangeMeta->save();
                }
            }

            // if non-partial & date has changed - change transaction date || delete transaction
            if (($this->frequency == 'one-time') && ($this->generate_transactions == 0) && isset($changes['date'])) {
                if (strtotime($this->date) <= strtotime(date('Y-m-d'))) {
                    $transaction = Transaction::findOne(['expense_id' => $this->id]);
                    if (!$transaction) {
                        $transaction = new Transaction();
                    }
                    $transaction->setAttributes([
                        'expense_id' => $this->id,
                        'date' => $this->date,
                        'amount' => $this->amount,
                        'account_id' => $this->account_id,
                        'tax_deductible' => $this->tax_deductible ? 1 : 0
                    ]);
                    $transaction->save();
                }
                else {
                    Transaction::deleteAll(['expense_id' => $this->id]);
                }
            }
        }
        // expense removed
        if (isset($changedAttributes['archived']) && $this->archived == 1) {
            // log the change
            $expenseChange = new ExpenseChange();
            $expenseChange->setAttributes([
                'type'       => 'D',
                'expense_id' => $this->id,
                'user_id'    => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $expenseChange->save();
        }
    }

    /**
     * @inheritdoc
     */
    public function afterDelete()
    {
        if ($this->jar_id) {
            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $this->jar_id);
        }
        parent::afterDelete();
    }

    /**
     * Get the expense amount as it was on the day provided by $date
     * @param $month
     * @param $year
     * @return float
     */
    public function getPastAmount($month, $year)
    {
        // return the latest value for the current month
        if ("{$year}-{$month}" == date('Y-m')) {
            return $this->amount;
        }
        $edit = ExpenseChangeMeta::find()
            ->joinWith('expenseChange')
            ->where(['and',
                ['expense_id' => $this->id],
                ['type' => 'E'],
                ['key' => 'amount'],
                ['>', 'time', date('Y-m-t 23:59:59', strtotime("{$year}-{$month}-20"))]
            ])
            ->orderBy('time ASC')
            ->one();
        if ($edit) {
            return $edit->value;
        }
        return $this->amount;
    }

    /**
     * Get the expense frequency as it was on the day provided by $date
     * @param $month
     * @param $year
     * @return float
     */
    public function getPastFrequency($month, $year)
    {
        // return the latest value for the current month
        if ("{$year}-{$month}" == date('Y-m')) {
            return $this->frequency;
        }
        $edit = ExpenseChangeMeta::find()
            ->joinWith('expenseChange')
            ->where(['and',
                ['expense_id' => $this->id],
                ['type' => 'E'],
                ['key' => 'frequency'],
                ['>', 'time', date('Y-m-t 23:59:59', strtotime("{$year}-{$month}-20"))]
            ])
            ->orderBy('time ASC')
            ->one();
        if ($edit) {
            return $edit->value;
        }
        return $this->frequency;
    }

    /**
     * Get the monthly average amount for a record.
     * @param int $month
     * @param int $year
     * @return float Monthly average amount for record.
     */
    public function getAverageMonthlyAmount($month = null, $year = null)
    {
        if (!$month) {
            $month = date('m');
        }
        if (!$year) {
            $year = date('Y');
        }
        $date = date('Y-m-t', strtotime("$year-$month-20"));
        return Calculator::monthlyAverageAmount(
            $this->getPastAmount($month, $year),
            $this->getPastFrequency($month, $year),
            $this->generate_transactions ?
                $this->getPlannedPaymentsNumber(date("$year-$month") < date('Y-n') ? $date : null) :
                null
        );
    }

    /**
     * Get the yearly average amount for a record
     * @return float Yearly average amount for record
     */
    public function getAverageYearlyAmount()
    {
        if ($this->_averageYearlyAmount == null) {
            $this->_averageYearlyAmount = Calculator::yearlyAverageAmount($this->amount, $this->frequency);
        }
        return $this->_averageYearlyAmount;
    }

    /**
     * Get the shortfall amount basing on what's been saved and what's left to save
     * @param float|null $alreadySavedOverride
     * @return float
     */
    public function getShortfallAmount($alreadySavedOverride = null)
    {
        if ($this->_shortfallAmount == null) {
            $alreadySaved = $alreadySavedOverride !== null ?
                $alreadySavedOverride :
                ($this->getAmountAlreadySaved() ?
                    $this->getAmountAlreadySaved() : 0);

            $this->_shortfallAmount =
                $this->getSavedByDueDate(true) +
                $alreadySaved -   // Current Amount Saved
                $this->amount; // Total Amount
        }
        return $this->_shortfallAmount;
    }

    /**
     * Get the adjusted required monthly payment
     * @param float|null $alreadySavedOverride
     * @return float
     */
    public function getAdjustedMonthlyAmount($alreadySavedOverride = null)
    {
        if ($this->_adjustedMonthlyAmount == null) {
            $alreadySaved = $alreadySavedOverride !== null ?
                $alreadySavedOverride :
                ($this->getAmountAlreadySaved() ?
                    $this->getAmountAlreadySaved() : 0);

            // include also the last month
            $monthsLeft = $this->getMonthsLeftUntil() + 1;

            $this->_adjustedMonthlyAmount = $monthsLeft != 0 ?
                ($this->amount - // Total Amount
                $alreadySaved) / // Current Amount Saved
                $monthsLeft : // Months remaining
                0;
        }
        return $this->_adjustedMonthlyAmount;
    }

    /**
     * Get the next due date of expense.
     * @param string $startDate $startDate for the Calculator::nextDue() method.
     * @param string $dateLimit $dateLimit for the Calculator::nextDue() method.
     * @return string Date of next due date of expense.
     */
    public function getNextDue($startDate = false, $dateLimit = false, $reload = false)
    {
        if ($this->_nextDue == null || $reload) {
            if ($startDate) {
                $this->_nextDue = Calculator::nextDueDate($this->date, $this->frequency, $dateLimit, $startDate);
            }
            else {
                $this->_nextDue = Calculator::nextDueDate($this->date, $this->frequency, $dateLimit);
            }
        }
        return $this->_nextDue;
    }

    public function getNextDueOneTimeToday()
    {
        return Calculator::nextDueDate($this->date, $this->frequency, date('Y-m-d', strtotime('+1 DAY')), $this->expenseStart->date);
    }

    /**
     * Get the number of months before the expense is due.
     * @param string|bool $date Date to calculate from, format YYYY-mm-dd.
     * @return int Number of months left until next expense is due.
     */
    public function getMonthsLeftUntil($date = false)
    {
        if ($this->_monthsLeftUntil == null) {
            $date1 = ($date) ? strtotime($date) : strtotime(date('Y-m-d'));
            $date2 = strtotime($this->getNextDue());
            $months = 0;
            while (($date1 = strtotime('+1 MONTH', $date1)) <= $date2) {
                $months++;
            }
            $this->_monthsLeftUntil = $months;
        }
        return $this->_monthsLeftUntil;
    }

    /**
     * Get the number of payment before the expense is due.
     * @param string $date Date to calculate from, format YYYY-mm-dd.
     * @return int Number of months left utnil next expense is due.
     */
    public function getPaymentsLeftUntil($date = null)
    {
        if ($this->_paymentsLeftUntil == null) {
            $today = $date ? $date : date('Y-m-d');
            if ($this->frequency == 'one-time') {
                if ($this->generate_transactions) {
                    if (!$this->start_date) {
                        $expenseStart = $this->getExpenseStart()->one();
                        if ($expenseStart) {
                            $this->start_date = $expenseStart->date;
                        }
                        else {
                            $this->start_date = $today;
                        }
                    }

                    $endTime = strtotime($this->date);
                    $helpTime = $prevHelpTime = $oldHelpTime = strtotime('+1 day', strtotime($this->getNextDue($this->start_date, date('Y-m-d', strtotime($today)), true)));
                    $payments = 1;
                    while ($helpTime < $endTime) {
                        $helpTime = strtotime('+1 day', strtotime($this->getNextDue($this->start_date, date('Y-m-d', $helpTime), true)));
                        if ($helpTime == $prevHelpTime) {
                            $helpTime = strtotime(
                                '+1 day',
                                strtotime(
                                    $this->getNextDue($this->start_date, date('Y-m-d', strtotime('+2 days', $helpTime)), true)
                                )
                            );
                        }
                        $prevHelpTime = $helpTime;
                        $payments++;
                        if ($oldHelpTime == $helpTime) {
                            break;
                        }
                    }
                    $this->_paymentsLeftUntil = $payments;
                }
                else {
                    $this->_paymentsLeftUntil = strtotime($this->date) > time() ? 1 : 0;
                }
            }
            else {
                $this->_paymentsLeftUntil = false;
            }
        }
        return $this->_paymentsLeftUntil;
    }

    /**
     * Get planned payments number for one-time expense
     * basing on the start date and due date
     * @return bool|int
     */
    public function getPlannedPaymentsNumber($date = null)
    {
        if ($this->_plannedPaymentsNumber == null) {
            $today = $date ? $date : date('Y-m-d');
            if ($this->frequency == 'one-time') {
                if ($this->generate_transactions) {
                    if (!$this->start_date) {
                        if ($this->expenseStart) {
                            $this->start_date = $this->expenseStart->date;
                        }
                        else {
                            $this->start_date = $today;
                        }
                    }

                    $endTime  = strtotime($this->date);
                    $helpTime = $prevHelpTime = $oldHelpTime = strtotime('+1 day', strtotime($this->getNextDue($this->start_date, date('Y-m-d', strtotime($this->start_date)), true)));
                    $payments = 1;
                    while ($helpTime < $endTime) {
                        $helpTime = strtotime('+1 day', strtotime($this->getNextDue($this->start_date, date('Y-m-d', $helpTime), true)));
                        if ($helpTime == $prevHelpTime) {
                            $helpTime = strtotime(
                                '+1 day',
                                strtotime(
                                    $this->getNextDue($this->start_date, date('Y-m-d', strtotime('+2 days', $helpTime)), true)
                                )
                            );
                        }
                        $prevHelpTime = $helpTime;
                        $payments++;
                        if ($oldHelpTime == $helpTime) {
                            break;
                        }
                    }
                    $this->_plannedPaymentsNumber = $payments;
                }
                else {
                    return 1;
                }
            }
            else {
                $this->_plannedPaymentsNumber = false;
            }
        }
        return $this->_plannedPaymentsNumber;
    }

    /**
     * Get the one-time expense payment amount that is due in the provided month
     * @param $month
     * @param $year
     * @return float
     */
    public function getPlannedPaymentAmount($month = null, $year = null)
    {
        if (!$month) {
            $month = date('m');
        }
        if (!$year) {
            $year = date('Y');
        }

        if ($this->frequency == 'one-time') {
            $monthStart = date('Y-m-01', strtotime("$year-$month-20"));
            $monthEnd   = date('Y-m-t', strtotime("$year-$month-20"));
            if ($this->generate_transactions) {
                $expenseStart = $this->getExpenseStart()->one();
                if ($expenseStart && $expenseStart->date <= $monthEnd && $this->getPaymentsLeftUntil() > 0) {
                    $plannedPaymentAmount = $this->getAverageMonthlyAmount();
                }
                else {
                    $plannedPaymentAmount = 0;
                }
            }
            else {
                // if a really one-time payment include it only in the month of payment
                if ($this->date >= $monthStart && $this->date <= $monthEnd) {
                    $plannedPaymentAmount = $this->amount;
                }
                else {
                    $plannedPaymentAmount = 0;
                }
            }
        }
        else {
            $plannedPaymentAmount = 0;
        }
        return $plannedPaymentAmount;
    }


    public function getAmountAlreadySaved()
    {
        if ($this->_amountAlreadySaved == null) {
            if ($this->frequency == 'one-time') {
                $transactions = $this->getTransactions()->all();
                $saved = 0;
                foreach ($transactions as $t) {
                    $saved += $t->amount;
                }
                $this->_amountAlreadySaved = $saved;
            }
            else {
                $this->_amountAlreadySaved = false;
            }
        }
        return $this->_amountAlreadySaved;
    }


    /**
     * Get the amount that should have been saved for the expense by now.
     * @return float Amount supposed to get saved by now.
     */
    public function getSupposedlySavedSoFar()
    {
        if ($this->getPlannedPaymentsNumber()) {
            $expenseStart = $this->getExpenseStart()->one();
            if ($expenseStart && $expenseStart->date <= date('Y-m-t')) {
                $saved = $this->amount * ($this->getPlannedPaymentsNumber() - $this->getPaymentsLeftUntil()) / $this->getPlannedPaymentsNumber();
            }
            else {
                $saved = 0;
            }
        }
        else {
            $saved = $this->amount;
        }
        return ($saved < $this->amount) ? $saved : $this->amount;
    }

    /**
     * Get amount that should be saved by due date basing on the average
     * monthly amount and number of months left to due date
     * @return float
     */
    public function getSavedByDueDate($includeLastMonth = false)
    {
        $monthsUntil = $this->getMonthsLeftUntil();
        if ($includeLastMonth) {
            $monthsUntil++;
        }
        return $this->getAverageMonthlyAmount() * $monthsUntil;
    }

    public function getMonthsTransactionsAmount($month, $year)
    {
        $transactions = $this->getTransactions()->where([
            'between', 'date', date("Y-m-1", strtotime("$year-$month-20")), date("Y-m-t", strtotime("$year-$month-20"))
        ])->all();
        $transactionsAmount = 0;
        foreach ($transactions as $t) {
            $transactionsAmount += $t->amount;
        }
        return $transactionsAmount;
    }

    /**
     * If the expense is recurring or not
     * @return bool
     */
    public function getIsRecurring()
    {
        return $this->generate_transactions && $this->frequency != 'one-time';
    }

    public function deleteCompletely()
    {
        ExpenseEnd::deleteAll(['expense_id' => $this->id]);
        ExpenseStart::deleteAll(['expense_id' => $this->id]);
        $this->delete();
    }


    /**
     * Get expenses for "Add Transaction" dropdown
     * @param array $jarIds
     * @param bool $excludeRecurring
     * @param bool $excludeUnbudgeted
     * @return array
     */
    public static function getExpensesForTransaction(array $jarIds, $excludeRecurring = false, $excludeUnbudgeted = false)
    {
        $expenseCondition = [
            'and',
            ['IN', 'jar_id', $jarIds],
            ['archived' => 0],
            ['is_adjustment' => 0]
        ];
        if ($excludeUnbudgeted) {
            $expenseCondition[] = ['is_unbudgeted' => 0];
        }

        $regular   = Expense::find()
            ->where($expenseCondition)
            ->andWhere(['and',
                ['generate_transactions' => 0],
                ['!=', 'frequency', 'one-time']
            ])
            ->orderBy('name ASC')
            ->all();
        if ($excludeRecurring) {
            $recurring = [];
        }
        else {
            $recurring = Expense::find()
                ->where($expenseCondition)
                ->andWhere(['and',
                    ['generate_transactions' => 1],
                    ['!=', 'frequency', 'one-time']
                ])
                ->orderBy('name ASC')
                ->all();
        }
        $oneOff = Expense::find()
            ->where($expenseCondition)
            ->andWhere(['and',
                ['frequency' => 'one-time']
            ])
            ->orderBy('name ASC')
            ->all();

        return array_merge($regular, $recurring, $oneOff);
    }

    public function isRecurring()
    {
        return ($this->frequency != 'one-time') && ($this->generate_transactions == 1);
    }
}
