<?php

namespace backend\models\db;

use Yii;

use backend\models\ChangeMetaModel;

use backend\components\helpers\Calculator;
use backend\components\helpers\Formatter;
use yii\caching\TagDependency;

/**
 * This is the model class for table "incomes".
 *
 * @property integer $id
 * @property integer $budget_id
 * @property integer $account_id
 * @property integer $jar_id
 * @property string $name
 * @property string $amount
 * @property string $date
 * @property string $frequency
 * @property integer $is_adjustment
 * @property integer $archived
 *
 * @property Budget $budget
 * @property IncomeChange[] $incomeChanges
 * @property Account $account
 */
class Income extends ChangeMetaModel
{
    const OTHER_INCOME_NAME = 'Other';
    const INCOME_ADJUSTMENT = 1;
    const JAR_ADJUSTMENT = 2;

    public $start_date;

    private $_nextDue;
    private $_averageMonthlyAmount;
    private $_minimumMonthlyAmount;
    private $_realAverageMonthlyAmount;
    private $_thisMonthsIncomeTransactions;
    private $_thisMonthsIncomeTransactionsAmount;

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'incomes';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['account_id', 'archived', 'is_adjustment', 'budget_id', 'jar_id'], 'integer'],
            [['amount'], 'number'],
            [['date'], 'safe'],
            [['name'], 'string', 'max' => 200],
            [['frequency'], 'string', 'max' => 20]
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'budget_id' => 'Budget ID',
            'account_id' => 'Account ID',
            'jar_id' => 'Jar ID',
            'name' => 'Name',
            'amount' => 'Amount',
            'date' => 'Date',
            'frequency' => 'Frequency',
            'archived' => 'Archived',
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getIncomeChanges()
    {
        return $this->hasMany(IncomeChange::className(), ['income_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getIncomeTransactions()
    {
        return $this->hasMany(IncomeTransaction::className(), ['income_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccount()
    {
        return $this->hasOne(Account::className(), ['id' => 'account_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getJar()
    {
        return $this->hasOne(Jar::className(), ['id' => 'jar_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBudget()
    {
        return $this->hasOne(Budget::className(), ['id' => 'budget_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getIncomeEnd()
    {
        return $this->hasOne(IncomeEnd::className(), ['income_id' => 'id']);
    }

    /**
     * @inheritdoc
     */
    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);
        if ($this->scenario == self::SCENARIO_CLONE) {
            return;
        }

        if ($this->jar_id) {
            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $this->jar_id);
        }

        // income record created
        if ($insert) {
            // generate transaction if one-time OR today or in past (not for adjustment records and api requests)
            if (!$this->is_adjustment &&
                ($this->frequency == 'one-time' || $this->date <= date('Y-m-d')) && $this->scenario != 'api') {
                $incomeTransaction = new IncomeTransaction();
                $incomeTransaction->setAttributes([
                    'income_id' => $this->id,
                    'account_id' => $this->account_id,
                    'amount' => $this->amount,
                    'date' => $this->date
                ]);
                $incomeTransaction->save();
            }

            // save change info
            $incomeChange = new IncomeChange();
            $incomeChange->setAttributes([
                'type'      => 'C',
                'income_id' => $this->id,
                'user_id'   => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $incomeChange->save();
        }
        // income record updated
        else {
            // check if and what changed
            $changes = array();
            $incomeAttrs = $this->getAttributes();

            foreach ($changedAttributes as $key => $o) {
                if (isset($incomeAttrs[$key]) && ($incomeAttrs[$key] != $o)) {
                    $changes[$key] = $o;
                }
            }

            if (count($changes) > 0) {
                $incomeChange = new IncomeChange();
                $incomeChange->setAttributes([
                    'type'      => 'E',
                    'income_id' => $this->id,
                    'user_id'   => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                        Yii::$app->user->identity->id : null
                ]);
                $incomeChange->save();

                foreach($changes as $key => $value) {
                    $incomeChangeMeta = new IncomeChangeMeta();
                    $incomeChangeMeta->setAttributes([
                        'income_change_id' => $incomeChange->id,
                        'key'              => $key,
                        'value'            => (string)$value
                    ]);
                    $incomeChangeMeta->save();
                }
            }
        }
        // income record removed
        if (isset($changedAttributes['archived']) && $this->archived == 1) {
            $incomeChange = new IncomeChange();
            $incomeChange->setAttributes([
                'type'      => 'D',
                'income_id' => $this->id,
                'user_id'   => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $incomeChange->save();
        }
    }

    /**
     * @inheritdoc
     */
    public function afterDelete()
    {
        if ($this->jar_id) {
            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $this->jar_id);
        }
        parent::afterDelete();
    }

    /**
     * Get the next due date of income.
     * @param bool $startDate
     * @param bool $reset
     * @param bool $dateLimit
     * @param bool $startingFrom
     * @return string
     */
    public function getNextDue($startDate = false, $reset = false, $dateLimit = false)
    {
        if ($this->_nextDue == null || $reset) {
            if ($startDate) {
                $this->_nextDue = Calculator::nextDueDate($this->date, $this->frequency, $dateLimit, $startDate);
            }
            else {
                $this->_nextDue = Calculator::nextDueDate($this->date, $this->frequency, $dateLimit);
            }
        }
        return $this->_nextDue;
    }

    /**
     * Get the income amount as it was on the day provided by $date
     * @param $month
     * @param $year
     * @return float
     */
    public function getPastAmount($month, $year)
    {
        // return the latest value for the current month
        if ("{$year}-{$month}" == date('Y-m')) {
            return $this->amount;
        }
        $edit = IncomeChangeMeta::find()
            ->joinWith('incomeChange')
            ->where(['and',
                ['income_id' => $this->id],
                ['type' => 'E'],
                ['key' => 'amount'],
                ['>', 'time', date('Y-m-t 23:59:59', strtotime("{$year}-{$month}-20"))]
            ])
            ->orderBy('time ASC')
            ->one();
        if ($edit) {
            return $edit->value;
        }
        return $this->amount;
    }

    /**
     * Get the income frequency as it was on the day provided by $date
     * @param $month
     * @param $year
     * @return float
     */
    public function getPastFrequency($month, $year)
    {
        // return the latest value for the current month
        if ("{$year}-{$month}" == date('Y-m')) {
            return $this->frequency;
        }
        $edit = IncomeChangeMeta::find()
            ->joinWith('incomeChange')
            ->where(['and',
                ['income_id' => $this->id],
                ['type' => 'E'],
                ['key' => 'frequency'],
                ['>', 'time', date('Y-m-t 23:59:59', strtotime("{$year}-{$month}-20"))]
            ])
            ->orderBy('time ASC')
            ->one();
        if ($edit) {
            return $edit->value;
        }
        return $this->frequency;
    }

    public function getMinimumMonthlyAmount()
    {
        if ($this->_minimumMonthlyAmount == null) {
            $this->_minimumMonthlyAmount = Calculator::monthlyMinimumAmount($this->amount, $this->frequency);
        }
        return $this->_minimumMonthlyAmount;
    }

    /**
     * Get the monthly average amount for a record.
     * @return float Monthly average amount for record.
     */
    public function getAverageMonthlyAmount()
    {
        if ($this->_averageMonthlyAmount == null) {
            $this->_averageMonthlyAmount = Calculator::monthlyAverageAmount($this->amount, $this->frequency);
        }
        return $this->_averageMonthlyAmount;
    }

    /**
     * Get the monthly average amount for a record based on actual transactions for earlier periods.
     * @return float Monthly average amount for record.
     */
    public function getRealAverageMonthlyAmount()
    {
        if ($this->_realAverageMonthlyAmount == null) {
            $incomeChangeMeta = IncomeChangeMeta::find()
                ->joinWith('incomeChange')
                ->where([
                    'and',
                    [IncomeChange::tableName() . '.id' => $this->id],
                    [IncomeChangeMeta::tableName() . '.key' => 'date']
                ])
                ->orderBy(IncomeChangeMeta::tableName() . '.id DESC')
                ->one();

            $startDate = ($incomeChangeMeta) ? $incomeChangeMeta->value : $this->date;

            // get date for income comparison
            $day = date('d', strtotime($startDate));
            $previousMonthTime = strtotime("first day of previous month");

            if ($day < date('d')) {
                $periodEndDate = date("Y-m-$day");
            }
            else {
                if ($day > date('t', $previousMonthTime)) {
                    $day = date('t', $previousMonthTime);
                }
                $periodEndDate = date("Y-m-$day", $previousMonthTime);
            }

            $periodEndTime = strtotime($periodEndDate);
            $periodStartTime = strtotime($startDate);
            $monthsCount = date('Y', $periodEndTime) * 12 + date('m', $periodEndTime) - date('Y', $periodStartTime) * 12 - date('m', $periodStartTime);

            if ($monthsCount > 0) {
                // get transactions
                $transactions = IncomeTransaction::find()
                    ->where([
                        'and',
                        ['=', 'income_id', $this->id],
                        ['<', 'date', $periodEndDate]
                    ])
                    ->all();

                $totalAmount = 0;
                foreach ($transactions as $t) {
                    $totalAmount +=  $t->amount;
                }

                $averageAmount = $totalAmount / $monthsCount;
                $this->_realAverageMonthlyAmount = Formatter::float($averageAmount);
            }
            else {
                $this->_realAverageMonthlyAmount = false;
            }
        }
        return $this->_realAverageMonthlyAmount;
    }

    /**
     * Get list of debt payments for a month.
     * @param int $month Month.
     * @param int $year Year.
     * @return array List of selected debt payments.
     */
    public function getMonthsIncomeTransactions($month, $year)
    {
        $incomeTransactions = IncomeTransaction::find()
            ->where([
                'and',
                [ 'income_id' => $this->id],
                ['between', 'date', date("$year-$month-1"), date("$year-$month-" . date('t', strtotime("$year-$month-1")))]
            ])
            ->all();
        return $incomeTransactions;
    }

    /**
     * The debtPayments for this month
     * @return array
     */
    public function getThisMonthsIncomeTransactions()
    {
        if ($this->_thisMonthsIncomeTransactions == null) {
            $this->_thisMonthsIncomeTransactions = $this->getMonthsIncomeTransactions(date('m'), date('Y'));
        }
        return $this->_thisMonthsIncomeTransactions;
    }

    /**
     * The total value of transactions for this month
     * @return float Numeric amount.
     */
    public function getThisMonthsIncomeTransactionsAmount()
    {
        if ($this->_thisMonthsIncomeTransactionsAmount == null) {
            $incomeTransactions = $this->getThisMonthsIncomeTransactions();
            $amount = 0;
            foreach ($incomeTransactions as $i) {
                $amount += $i->amount;
            }
            $this->_thisMonthsIncomeTransactionsAmount = $amount;
        }
        return $this->_thisMonthsIncomeTransactionsAmount;
    }

    /**
     * Get the list of planned income transactions for the month
     * @param $startDate
     * @return array
     */
    public function getProjectedTransactions($startDate, $month, $year)
    {
        $transactions = [];
        $startDate = date('Y-m-d', strtotime($startDate));
        $fromDate  = date('Y-m-01', strtotime("$year-$month-20"));
        $toDate    = date('Y-m-t', strtotime("$year-$month-20"));
        // check if the date range is out of scope
        if ($startDate > $toDate) {
            return [];
        }
        elseif ($startDate > $fromDate) {
            $fromDate = $startDate;
        }
        $endDate   = null;
        $incomeEnd = $this->getIncomeEnd()->one();
        if ($incomeEnd) {
            $endDate = $incomeEnd->date;
        }

        if ($this->frequency == 'one-time' && $this->date >= $startDate &&
            $this->date >= $fromDate && $this->date <= $toDate) {
            $transaction = new IncomeTransaction();
            $transaction->setAttributes([
                'income_id' => $this->id,
                'account_id' => $this->account_id,
                'amount' => $this->amount,
                'date' => $this->date
            ]);
            $transaction->income_name = $this->name;
            $transactions[] = $transaction;
        }
        elseif ($this->frequency != 'one-time') {
            $dueDate = $this->getNextDue(false, true, $fromDate);
            while ($dueDate && $dueDate >= $fromDate && $dueDate <= $toDate &&
                   (!$endDate || $dueDate <= $endDate)) {
                if ($dueDate >= $startDate) {
                    $transaction = new IncomeTransaction();
                    $transaction->setAttributes([
                        'income_id' => $this->id,
                        'account_id' => $this->account_id,
                        'amount' => $this->amount,
                        'date' => $dueDate
                    ]);
                    $transaction->income_name = $this->name;
                    $transactions[] = $transaction;
                }
                $dueDate = $this->getNextDue(false, true, date('Y-m-d', strtotime('+1 day', strtotime($dueDate))));
            }
        }
        return $transactions;
    }

    /**
     * @return int|null
     */
    public function getMockBudgetId()
    {
        if ($this->budget->is_mock) {
            return $this->budget_id;
        }
        return null;
    }

    public function deleteCompletely()
    {
        IncomeEnd::deleteAll(['income_id' => $this->id]);
        $this->delete();
    }

    /**
     * @param int $budgetId
     * @return \yii\db\ActiveQuery
     */
    public static function getActiveIncomesQuery($budgetId)
    {
        return Income::find()
            ->where(['and',
                ['budget_id' => $budgetId],
                ['archived' => 0],
                ['is_adjustment' => 0],
                ['or',
                    ['and',
                        ['frequency' => 'one-time'],
                        ['>=', 'date', date('Y-m-1')]
                    ],
                    ['and',
                        ['!=', 'frequency', 'one-time']
                    ]
                ]
            ])
            ->orderBy('name ASC');
    }

    /**
     * Get incomes for "Add Transaction" dropdown
     * @param $budgetId
     * @param array $jarIds
     * @param bool $excludeUnbudgeted
     * @return array
     */
    public static function getIncomesForTransaction($budgetId, array $jarIds, $excludeUnbudgeted = false)
    {
        $condition = [
            'and',
            ['archived' => 0],
            ['is_adjustment' => 0],
            ['budget_id' => $budgetId]
        ];
        if ($excludeUnbudgeted) {
            $condition[] = ['is_unbudgeted' => 0];
        }
        return Income::find()
            ->where($condition)
            ->andWhere([
                'or',
                ['jar_id' => $jarIds],
                ['jar_id' => null]
            ])
            ->orderBy('name ASC')
            ->all();
    }

}
