<?php

namespace backend\models\db;

use backend\components\helpers\DripHelper;
use backend\components\helpers\Formatter;
use kartik\mpdf\Pdf;
use Yii;
use yii\helpers\Json;
use yii\helpers\Url;
use backend\components\helpers\Stripe as StripeHelper;

/**
 * This is the model class for table "invoices".
 *
 * @property integer $id
 * @property integer $user_id
 * @property string $bill_date
 * @property string $bill_name
 * @property string $bill_address_1
 * @property string $bill_address_2
 * @property string $bill_address_3
 * @property string $bill_country
 * @property string $total
 * @property string $tax
 * @property string $tax_percent
 * @property string $percent_off
 * @property string $amount_off
 * @property string $stripe_invoice_id
 * @property string $stripe_coupon_id
 * @property string $payment_data
 * @property integer $is_paid;
 *
 * @property InvoiceItem[] $invoiceItems
 * @property User $user
 */
class Invoice extends \yii\db\ActiveRecord
{
    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'invoices';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['user_id', 'bill_date', 'bill_name'], 'required'],
            [['user_id', 'is_paid'], 'integer'],
            [['bill_date'], 'safe'],
            [['total', 'tax', 'tax_percent', 'percent_off', 'amount_off'], 'number'],
            [['stripe_invoice_id', 'stripe_coupon_id', 'payment_data', 'bill_address_1', 'bill_address_2', 'bill_address_3', 'bill_country'], 'string']
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'user_id' => 'User ID',
            'bill_date' => 'Bill Date',
            'bill_name' => 'Bill Name',
            'bill_address_1' => 'Bill Address 1',
            'bill_address_2' => 'Bill Address 2',
            'bill_address_3' => 'Bill Address 3',
            'bill_country' => 'Bill Country',
            'total' => 'Total',
            'payment_data' => 'Payment Data',
        ];
    }

    /**
     * @return array Bill address in a form of an array
     */
    public function getAddressLines($includeCountry = false)
    {
        $address = [];
        if (!empty($this->bill_address_1)) {
            $address[] = $this->bill_address_1;
        }
        if (!empty($this->bill_address_2)) {
            $address[] = $this->bill_address_2;
        }
        if (!empty($this->bill_address_3)) {
            $address[] = $this->bill_address_3;
        }
        if ($includeCountry && !empty($this->bill_country)) {
            $address[] = $this->bill_country;
        }
        return $address;
    }

    /**
     * @return InvoiceItem In our case it's always the one and only invoice item
     */
    public function getInvoiceItem()
    {
        return $this->invoiceItems[0];
    }

    public function getIsPaid()
    {
        return !($this->payment_data === null);
    }

    /**
     * Create the charge on Stripe's servers -
     * this will actually charge the user's card
     * @param UserPlan|null $userPlan
     * @return bool
     */
    public function chargeCustomer(UserPlan $userPlan = null)
    {
        $paymentData = !empty($this->payment_data) ? Json::decode($this->payment_data) : null;
        if (!$paymentData) {
            return false;
        }

        $stripe = new StripeHelper();
        $customerId = $stripe->saveCustomer($this->user, $paymentData['id']);
        if (!$customerId) {
            return false;
        }

        if ($userPlan) {
            $stripe->subscribeCustomerToPlan($customerId, $userPlan);
        }

        return true;
    }

    public function payInvoiceAndChargeCustomer(array $paymentData)
    {
        if ($paymentData === null) {
            return false;
        }

        $this->payment_data = Json::encode($paymentData);
        if (!$this->save()) {
            return false;
        }
        // create plan based on the payment made ...
        $plan = UserPlan::getActivePlan()->cloneForNewPeriod();
        $invoiceItem = $this->getInvoiceItem();
        $invoiceItem->user_plan_id = $plan->id;
        if ($invoiceItem->save()) {
            return false;
        }

        if ($this->chargeCustomer()) {
            // ... and send ONLY the receipt
            $this->sendEmail(false);
        }
        else {
            // clear the reference in the invoice record and delete the userPlan
            $invoiceItem->user_plan_id = null;
            $invoiceItem->save();
            $plan->delete();
            return false;
        }

        return true;
    }

    public function updateInvoiceLines(UserPlan $userPlan)
    {
        $this->is_paid = 1;
        $this->save();
        foreach ($this->invoiceItems as $item) {
            $item->date_end = date('Y-m-d', strtotime($userPlan->valid_until));
            $item->save();
        }
    }

    public static function generateChargeInvoice(UserPlan $userPlan, array $paymentData)
    {
        $user = $userPlan->user;
        $userMeta = $user->getUserMetaSorted();

        $invoice = new Invoice();
        $invoice->attributes = [
            'user_id'        => $userPlan->user_id,
            'bill_date'      => Formatter::localDatetimeToUtcDatetime((int)$paymentData['created'], 'Y-m-d'),
            'bill_name'      => $user->getFullName(),
            'bill_address_1' => !empty($userMeta['address_line_1']) ? $userMeta['address_line_1'] : '',
            'bill_address_2' => !empty($userMeta['address_line_2']) ? $userMeta['address_line_2'] : '',
            'bill_address_3' => !empty($userMeta['address_line_3']) ? $userMeta['address_line_3'] : '',
            'bill_country'   => !empty($userMeta['country_name']) ? $userMeta['country_name'] : '',
            'total'          => round($paymentData['amount'] / 100.0, 2),
            'payment_data'   => Json::encode($paymentData),
            'is_paid'        => $paymentData['paid'] ? 1 : 0
        ];
        $invoice->save();

        // all dates/times are in UTC here because of the user plan start_time that is stored in UTC
        $invoiceItem = new InvoiceItem();
        $invoiceItem->setAttributes([
            'invoice_id'   => $invoice->id,
            'user_plan_id' => $userPlan->id,
            'description'  => $userPlan->userPlanType->description,
            'amount'       => $invoice->total,
            'date_start'   => date('Y-m-d', strtotime($userPlan->start_time)),
            'date_end'     => date('Y-m-d', strtotime($userPlan->valid_until))
        ]);
        $invoiceItem->save();

        return $invoice;
    }

    public static function generateInvoice(UserPlan $userPlan, array $paymentData)
    {
        $user = $userPlan->user;
        $userMeta = $user->getUserMetaSorted();

        $invoice = new Invoice();
        $invoice->attributes = [
            'user_id'        => $userPlan->user_id,
            'bill_date'      => Formatter::localDatetimeToUtcDatetime($paymentData['date'], 'Y-m-d'),
            'bill_name'      => $user->getFullName(),
            'bill_address_1' => !empty($userMeta['address_line_1']) ? $userMeta['address_line_1'] : '',
            'bill_address_2' => !empty($userMeta['address_line_2']) ? $userMeta['address_line_2'] : '',
            'bill_address_3' => !empty($userMeta['address_line_3']) ? $userMeta['address_line_3'] : '',
            'bill_country'   => !empty($userMeta['country_name']) ? $userMeta['country_name'] : '',
            'total'          => round($paymentData['amount_due'] / 100.0, 2),
            'stripe_invoice_id' => $paymentData['id'],
            'payment_data'   => Json::encode($paymentData),
        ];
        if (!empty($paymentData['discount'])) {
            $invoice->stripe_coupon_id = $paymentData['discount']['coupon']['id'];
            $invoice->percent_off = $paymentData['discount']['coupon']['percent_off'];
            $invoice->amount_off = Yii::$app->formatter->asDecimal($paymentData['discount']['coupon']['amount_off'] / 100, 2);
        }
        if (!empty($paymentData['tax'])) {
            $invoice->tax = Yii::$app->formatter->asDecimal($paymentData['tax'] / 100, 2);
        }
        if (!empty($paymentData['tax_percent'])) {
            $invoice->tax_percent = $paymentData['tax_percent'];
        }
        $invoice->save();

        // all dates/times are in UTC here because of the user plan start_time that is stored in UTC
        foreach ($paymentData['lines']['data'] as $line) {
            $invoiceItem = new InvoiceItem();
            $invoiceItem->setAttributes([
                'invoice_id'   => $invoice->id,
                'user_plan_id' => $userPlan->id,
                'description'  => $line['plan']['name'] . (!empty($line['description']) ? ' (' . $line['description'] . ')' : ''),
                'amount'       => round($line['amount'] / 100.0, 2),
                'date_start'   => date('Y-m-d', $line['period']['start']),
                'date_end'     => date('Y-m-d', $line['period']['end'])
            ]);
            $invoiceItem->save();
        }
        return $invoice;
    }

    public static function generateUpdateInvoice(UserPlan $userPlan, array $paymentData)
    {
        $user = $userPlan->user;
        $userMeta = $user->getUserMetaSorted();

        $invoice = new Invoice();
        $invoice->attributes = [
            'user_id'        => $userPlan->user_id,
            'bill_date'      => Formatter::localDatetimeToUtcDatetime($paymentData['date'], 'Y-m-d'),
            'bill_name'      => $user->getFullName(),
            'bill_address_1' => !empty($userMeta['address_line_1']) ? $userMeta['address_line_1'] : '',
            'bill_address_2' => !empty($userMeta['address_line_2']) ? $userMeta['address_line_2'] : '',
            'bill_address_3' => !empty($userMeta['address_line_3']) ? $userMeta['address_line_3'] : '',
            'bill_country'   => !empty($userMeta['country_name']) ? $userMeta['country_name'] : '',
            'total'          => round($paymentData['amount'] / 100.0, 2),
            'stripe_invoice_id' => $paymentData['id'],
            'payment_data'   => Json::encode($paymentData),
        ];
        $invoice->save();

        // all dates/times are in UTC here because of the user plan start_time that is stored in UTC
        $invoiceItem = new InvoiceItem();
        $invoiceItem->setAttributes([
            'invoice_id'   => $invoice->id,
            'user_plan_id' => $userPlan->id,
            'description'  => $paymentData['plan']['name'],
            'amount'       => round($paymentData['plan']['amount'] / 100.0, 2),
            'date_start'   => date('Y-m-d', $paymentData['period']['start']),
            'date_end'     => date('Y-m-d', $paymentData['period']['end'])
        ]);
        $invoiceItem->save();
        return $invoice;
    }

    /**
     * Returns the invoice record if the last issued invoice
     * is still unpaid, null otherwise
     * @param integer $userId
     * @return null|\yii\db\ActiveRecord
     */
    public static function lastUnpaidInvoice($userId = null)
    {
        if ($userId === null) {
            $userId = Yii::$app->user->identity->id;
        }
        $invoice = Invoice::find()
            ->where(['user_id' => $userId])
            ->orderBy(['bill_date' => SORT_DESC])
            ->limit(1)
            ->one();

        return $invoice->getIsPaid() ? null : $invoice;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getInvoiceItems()
    {
        return $this->hasMany(InvoiceItem::className(), ['invoice_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUser()
    {
        return $this->hasOne(User::className(), ['id' => 'user_id']);
    }

    public function renderPdfInvoice($destination = Pdf::DEST_BROWSER)
    {
        $pdf = new Pdf([
            'destination' => $destination,
            'filename' => 'invoice-'.$this->id.'.pdf',
            'content' => Yii::$app->controller->renderPartial('@backend/views/subscription/print', [
                'invoice' => $this
            ]),
            'cssInline' => '.unpaid { color: red; font-size: 2em; }',
            'options' => [
                'title' => 'Invoice #'.$this->id
            ]
        ]);
        return $pdf->render();
    }

    public function renderPdfReceipt($destination = Pdf::DEST_BROWSER)
    {
        $pdf = new Pdf([
            'destination' => $destination,
            'filename' => 'receipt-'.$this->id.'.pdf',
            'content' => Yii::$app->controller->renderPartial('@backend/views/subscription/print-receipt', [
                'invoice' => $this
            ]),
            'options' => [
                'title' => 'Receipt for invoice #'.$this->id
            ]
        ]);
        return $pdf->render();
    }

    public function sendEmail($sendInvoice = true, $sendReceipt = true)
    {
        $user = $this->user;

        DripHelper::sendEmailEvent(
            $sendReceipt ? 'receipt' : 'invoice',
            $user->email,
            $sendReceipt ? 'Payment Confirmation' : 'New Invoice from Grandma’s Jars',
            [
                'username'          => $user->getFullName(),
                'invoice_id'        => $this->id,
                'download_url'      => $sendReceipt ?
                    Url::toRoute(['/subscription/print-receipt', 'invoice' => $this->id], true) :
                    Url::toRoute(['/subscription/print-invoice', 'invoice' => $this->id], true),
                'subscriptions_url' => Url::toRoute(['/subscription/index'], true)
            ]
        );
    }
}
