<?php

namespace backend\models\db;

use backend\components\helpers\Formatter;
use backend\models\form\AdjustJarFundsRemaining;
use Yii;

use backend\models\ChangeMetaModel;
use yii\caching\TagDependency;
use yii\db\ActiveQuery;
use yii\helpers\ArrayHelper;
use yii\helpers\Url;

/**
 * This is the model class for table "jars".
 *
 * @property integer $id
 * @property integer $budget_id
 * @property integer $jar_type_id
 * @property string $name
 * @property double $amount
 * @property string $created_time
 * @property integer $archived
 * @property string $type
 * @property integer $order
 *
 * @property Debt[] $debts
 * @property Expense[] $expenses
 * @property Income[] $incomes
 * @property JarChange[] $jarChanges
 * @property Budget $budget
 */
class Jar extends ChangeMetaModel
{
    const EVERYDAY_JAR_NAME = 'Everyday';
    const EVERYDAY_JAR_TYPE = 'everyday';

    private $_thisMonthsTransactions;
    private $_thisMonthsTransactionsAmount;
    private $_thisMonthsEstimatedExpenses;
    private $_thisMonthsEstimatedExpensesAmount;
    private $_thisMonthsRecurringExpenses;
    private $_thisMonthsRecurringExpensesAmount;
    private $_thisMonthsOnetimeExpenses;
    private $_thisMonthsOnetimeExpensesAmount;
    private $_thisMonthsExpensesAmount;
    private $_thisMonthsFundsRemainingAmount;
    private $_thisMonthsMoneyInAmount;
    private $_recurringExpensesAmount;


    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'jars';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['budget_id', 'jar_type_id','archived', 'order'], 'integer'],
            [['amount'], 'number'],
            [['created_time'], 'safe'],
            [['name'], 'string', 'max' => 200],
            [['type'], 'string', 'max' => 20]
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'budget_id' => 'Budget ID',
            'jar_type_id' => 'Jar Type ID',
            'name' => 'Name',
            'amount' => 'Amount',
            'created_time' => 'Created Time',
            'archived' => 'Archived',
            'type' => 'Jar Type',
            'order' => 'Jar Order'
        ];
    }

    public function fields()
    {
        $fields = parent::fields();
        unset($fields['jar_type_id'], $fields['amount']);
        $fields['monthly_budget'] = function() {
            return $this->getMonthlyBudget();
        };
        $fields['funds_remaining'] = function() {
            return $this->getThisMonthsFundsRemainingAmount();
        };
        return $fields;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getDebts()
    {
        return $this->hasMany(Debt::className(), ['jar_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getExpenses()
    {
        return $this->hasMany(Expense::className(), ['jar_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getIncomes()
    {
        return $this->hasMany(Income::className(), ['jar_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getJarChanges()
    {
        return $this->hasMany(JarChange::className(), ['jar_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBudget()
    {
        return $this->hasOne(Budget::className(), ['id' => 'budget_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getJarType()
    {
        return $this->hasOne(JarType::className(), ['id' => 'jar_type_id']);
    }

    /**
     * @inheritdoc
     */
    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);
        if ($this->scenario == self::SCENARIO_CLONE) {
            return;
        }

        // jar created
        if ($insert) {
            $jarChange = new JarChange();
            $jarChange->setAttributes([
                'type'    => 'C',
                'jar_id'  => $this->id,
                'user_id' => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $jarChange->save();
        }
        // jar removed
        elseif (count($changedAttributes) == 1 &&
            isset($changedAttributes['archived']) &&
            $this->archived == 1) {

            $jarChange = new JarChange();
            $jarChange->setAttributes([
                'type'    => 'D',
                'jar_id'  => $this->id,
                'user_id' => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $jarChange->save();

        }
        // jar restored
        elseif (count($changedAttributes) == 1 &&
            isset($changedAttributes['archived']) &&
            $this->archived == 0) {

            $jarChange = new JarChange();
            $jarChange->setAttributes([
                'type'    => 'R',
                'jar_id'  => $this->id,
                'user_id' => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                    Yii::$app->user->identity->id : null
            ]);
            $jarChange->save();

        }
        // jar edited
        else {

            // check if and what changed
            $changes = array();
            $jarAttrs = $this->getAttributes();

            foreach ($changedAttributes as $key => $o) {
                if ($jarAttrs[$key] != $o) {
                    $changes[$key] = $o;
                }
            }

            if (count($changes) > 0) {

                $jarChange = new JarChange();
                $jarChange->setAttributes([
                    'type'    => 'E',
                    'jar_id'  => $this->id,
                    'user_id' => isset(Yii::$app->user) && !Yii::$app->user->isGuest ?
                        Yii::$app->user->identity->id : null
                ]);
                $jarChange->save();

                foreach($changes as $key => $value) {

                    $jarChangeMeta = new JarChangeMeta();
                    $jarChangeMeta->setAttributes([
                        'jar_change_id' => $jarChange->id,
                        'key'           => $key,
                        'value'         => (string)$value
                    ]);
                    $jarChangeMeta->save();

                }

            }
        }
    }

    /*public function afterSave($insert, $changedAttributes)
    {
        if($insert){
            $log = UserActivity::log(static::className(), $this->budget->user_id, 'New Jar', $this->name, [
                'amount' => $this->amount,
            ]);

        }
        parent::afterSave($insert, $changedAttributes);
    }*/


    /**
     * Return a list of currently non-archived expenses for this jar.
     * @return array The expenses.
     */
    public function getActiveExpenses()
    {
        return Expense::find()
            ->where(['jar_id' => $this->id, 'archived' => 0])
            ->orderBy('name ASC')
            ->all();
    }

    /**
     * Finds the date of the first or the last money operation for the budget
     * @param bool $first
     * @return string
     */
    public function getMoneyOperationDate($first = true)
    {
        $transaction = Transaction::find()
            ->joinWith('expense')
            ->where([
                Expense::tableName() . '.jar_id' => $this->id,
                Expense::tableName() . '.archived' => 0
            ])
            ->orderBy('date '.($first ? 'ASC' : 'DESC'))
            ->one();

        if ($this->jar_type_id != 1) {
            $debtPayment = DebtPayment::find()
                ->joinWith('debt')
                ->where([Debt::tableName() . '.budget_id' => $this->budget_id])
                ->orderBy('date ' . ($first ? 'ASC' : 'DESC'))
                ->one();
        }
        else {
            $debtPayment = null;
        }

        $income = IncomeTransaction::find()
            ->joinWith('income')
            ->where([
                Income::tableName() . '.jar_id' => $this->id,
                Income::tableName() . '.archived' => 0
            ])
            ->orderBy('date '.($first ? 'ASC' : 'DESC'))
            ->one();

        $date = date('Y-m-d');
        if ($transaction &&
            $first && $transaction->date < $date || !$first && $transaction->date > $date) {
            $date = $transaction->date;
        }
        if ($debtPayment &&
            $first && $debtPayment->date < $date || !$first && $debtPayment->date > $date) {
            $date = $debtPayment->date;
        }
        if ($income &&
            $first && $income->date < $date || !$first && $income->date > $date) {
            $date = $income->date;
        }

        return $date;
    }

    /**
     * Get the date of the first money operation for the budget
     * @return string
     */
    public function getFirstMoneyOperationDate()
    {
        return $this->getMoneyOperationDate(true);
    }

    /**
     * Get the date of the last money operation for the budget
     * @return string
     */
    public function getLastMoneyOperationDate()
    {
        return $this->getMoneyOperationDate(false);
    }

    /**
     * Get a list of transactions for a certain month and year.
     * @param int $month Month.
     * @param int $year Year.
     * @param bool $excludeBalanceAdjustments
     * @param bool $excludeJarAdjustments
     * @return array List of transactions.
     */
    public function getMonthsTransactions($month, $year, $excludeBalanceAdjustments = false, $excludeJarAdjustments = false)
    {
        $transactions = Transaction::find()
            ->joinWith('expense')
            ->joinWith('account')
            ->where(['and',
                [Account::tableName() . '.budget_id' => $this->budget_id],
                [Expense::tableName() . '.jar_id' => $this->id],
                ['between', Transaction::tableName() . '.date', date("$year-$month-1"), date("$year-$month-" . date('t', strtotime("$year-$month-1")))]
            ]);
        if ($excludeBalanceAdjustments || $excludeJarAdjustments) {
            $exclude = [];
            if ($excludeBalanceAdjustments) {
                $exclude[] = Expense::EXPENSE_ADJUSTMENT;
            }
            if ($excludeJarAdjustments) {
                $exclude[] = Expense::JAR_ADJUSTMENT;
            }
            $transactions->andWhere([
                'NOT IN', 'is_adjustment', $exclude
            ]);
        }

        return $transactions->all();
    }

    /**
     * The transactions for this month
     * @param null|string $month
     * @param null|string $year
     * @return array List of transactions
     */
    public function getThisMonthsTransactions($month = null, $year = null) {

        if ($this->_thisMonthsTransactions == null) {
            $this->_thisMonthsTransactions = $this->getMonthsTransactions(
                date('m', $month && $year ? strtotime("$year-$month-01") : time()),
                date('Y', $month && $year ? strtotime("$year-$month-01") : time())
            );
        }
        return $this->_thisMonthsTransactions;

    }


    /**
     * The total value of transactions for this month
     * @param bool $withBalanceAdjustments
     * @param bool $withJarAdjustments
     * @return float Numeric amount.
     */
    public function getThisMonthsTransactionsAmount($withBalanceAdjustments = false, $withJarAdjustments = true)
    {
        if ($this->_thisMonthsTransactionsAmount == null) {
            $this->_thisMonthsTransactionsAmount = $this->getMonthsTransactionsAmount(date('m'), date('Y'), $withBalanceAdjustments, $withJarAdjustments);
        }
        return $this->_thisMonthsTransactionsAmount;
    }

    /**
     * The total value of transactions for the given month
     * @param null|string $month
     * @param null|string $year
     * @param bool $withBalanceAdjustments
     * @param bool $withJarAdjustments
     * @return float
     */
    public function getMonthsTransactionsAmount($month, $year, $withBalanceAdjustments = false, $withJarAdjustments = true)
    {
        $amount = 0;
        switch ($this->jar_type_id) {
            case 1:
                $transactions = $this->getMonthsTransactions($month, $year, !$withBalanceAdjustments, !$withJarAdjustments);
                foreach ($transactions as $t) {
                    $amount += $t->amount;
                }
                break;
            case 2:
                $budget = Budget::findOne($this->budget_id);
                $debtPayments = array_merge(
                    $budget->getMonthsDebtPayments($month, $year, false, 1, !$withBalanceAdjustments, !$withJarAdjustments),
                    $budget->getMonthsDebtPayments($month, $year, false, 3, !$withBalanceAdjustments, !$withJarAdjustments)
                );
                foreach ($debtPayments as $t) {
                    if ($t->amount > 0) {
                        $amount += $t->amount;
                    }
                }
                break;
            case 3:
                $budget = Budget::findOne($this->budget_id);
                $debtPayments = $budget->getMonthsDebtPayments($month, $year, false, 2, !$withBalanceAdjustments, !$withJarAdjustments);
                foreach ($debtPayments as $t) {
                    if ($t->amount > 0) {
                        $amount += $t->amount;
                    }
                }
                break;
        }

        return $amount;
    }

    public function getMonthsUnbudgetedTransactionsAmount($month, $year)
    {
        if ($this->jar_type_id != 1) {
            return 0;
        }

        $total = 0;
        $transactions = Transaction::find()
            ->joinWith('expense')
            ->joinWith('account')
            ->where(['and',
                [Account::tableName() . '.budget_id' => $this->budget_id],
                [Expense::tableName() . '.jar_id' => $this->id],
                ['between', Transaction::tableName() . '.date', date("$year-$month-1"), date("$year-$month-" . date('t', strtotime("$year-$month-1")))],
                [Expense::tableName() . '.is_unbudgeted' => 1]
            ]);

        foreach ($transactions->all() as $transaction) {
            $total += $transaction->amount;
        }
        return $total;
    }

    /**
     * The Funds Remaining amount for the given month and year
     * @param null|string $month
     * @param null|string $year
     * @param bool $withBalanceAdjustments
     * @param bool $withJarAdjustments
     * @return float
     */
    public function getMonthsFundsRemainingAmount($month, $year, $withBalanceAdjustments = false, $withJarAdjustments = true)
    {
        $key = [
            'monthsFundsRemainingAmount',
            'm' => $month, 'y' => $year, 'id' => $this->id,
            'b' => $withBalanceAdjustments, 'j' => $withJarAdjustments
        ];
        $value = Yii::$app->cache->get($key);
        if ($value === false) {
            $value =
                $this->getMonthsFundsBroughtForwardAmount($month, $year, $withBalanceAdjustments) +
                $this->getMonthlyBudget($month, $year) +
                $this->getMonthsMoneyInAmount($month, $year, $withBalanceAdjustments, $withJarAdjustments) -
                $this->getMonthsTransactionsAmount($month, $year, $withBalanceAdjustments, $withJarAdjustments);
            Yii::$app->cache->set($key, $value, 0, new TagDependency(['tags' => 'jarFundsRemaining-' . $this->id]));
        }
        return $value;
    }

    public function getThisMonthsFundsRemainingAmount()
    {
        return $this->getMonthsFundsRemainingAmount(date('m'), date('Y'));
    }

    /**
     * Get the total jar balance adjustment for the given month
     * If there are no adjustments, the return value is null
     * @param $month
     * @param $year
     * @return float|null
     */
    public function getFundsRemainingAdjustment($month, $year)
    {
        switch ($this->jar_type_id) {
            case 1:
                $incomeAdjustment = Income::find()
                    ->where([
                        'jar_id'        => $this->id,
                        'date'          => date('Y-m-01', strtotime("$year-$month-20")),
                        'is_adjustment' => Income::JAR_ADJUSTMENT
                    ])
                    ->one();
                $expenseAdjustment = Expense::find()
                    ->where([
                        'jar_id'        => $this->id,
                        'date'          => date('Y-m-01', strtotime("$year-$month-20")),
                        'is_adjustment' => Expense::JAR_ADJUSTMENT
                    ])
                    ->one();

                if (!$incomeAdjustment && !$expenseAdjustment) {
                    return null;
                }
                return
                    ($incomeAdjustment ? $incomeAdjustment->amount : 0) -
                    ($expenseAdjustment ? $expenseAdjustment->amount : 0);

            case 2:
                $debtAdjustment = DebtPayment::find()
                    ->joinWith('account')
                    ->joinWith('debt')
                    ->where(['and',
                        ['or',
                            [Debt::tableName() . '.budget_id' => $this->budget_id],
                            [Account::tableName() . '.budget_id' => $this->budget_id]
                        ],
                        [DebtPayment::tableName() . '.date' => date('Y-m-01', strtotime("$year-$month-20"))],
                        ['is_adjustment' => DebtPayment::JAR_ADJUSTMENT],
                        ['!=', 'debt_payment_type_id', 2]
                    ])
                    ->one();
                if (!$debtAdjustment) {
                    return null;
                }
                return $debtAdjustment->amount;

            case 3:
                $debtAdjustment = DebtPayment::find()
                    ->joinWith('account')
                    ->joinWith('debt')
                    ->where(['and',
                        ['or',
                            [Debt::tableName() . '.budget_id' => $this->budget_id],
                            [Account::tableName() . '.budget_id' => $this->budget_id]
                        ],
                        [DebtPayment::tableName() . '.date' => date('Y-m-01', strtotime("$year-$month-20"))],
                        ['is_adjustment' => DebtPayment::JAR_ADJUSTMENT],
                        ['debt_payment_type_id' => 2]
                    ])
                    ->one();
                if (!$debtAdjustment) {
                    return null;
                }
                return $debtAdjustment->amount;
        }
    }

    /**
     * Get query for recurring expenses
     * @param $month
     * @param $year
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return ActiveQuery
     */
    public function getMonthsRecurringExpensesQuery($month, $year, $onlyBudgeted = false, $includeDeleted = true)
    {
        $expenseTableName = Expense::tableName();
        $expenseEndTableName = ExpenseEnd::tableName();
        $expenseChangeTableName = ExpenseChange::tableName();

        // active expenses or expenses removed this month
        $query = Expense::find()
            ->select("$expenseTableName.*, ee.date as removal_time, ecd.time as delete_time")
            ->leftJoin($expenseChangeTableName . ' as ecc', $expenseTableName . ".id = ecc.expense_id AND ecc.type = 'C'")
            ->leftJoin($expenseChangeTableName . ' as ecd', $expenseTableName . ".id = ecd.expense_id AND ecd.type = 'D'")
            ->leftJoin($expenseEndTableName . ' as ee', "$expenseTableName.id = ee.expense_id")
            ->where([
                'and',
                ['jar_id' => $this->id],
                ['!=' , 'frequency', 'one-time'],
                ['generate_transactions' => 1],
                [
                    'or',
                    ['ecc.time' => null],
                    ['<=', 'ecc.time', date('Y-m-t 23:59:59', strtotime("$year-$month-20"))],
                ]
            ]);
        if ($onlyBudgeted) {
            $query->andWhere([
                'is_unbudgeted' => 0
            ]);
        }
        if ($includeDeleted) {
            $query->andWhere([
                'or',
                ['archived' => 0],
                [
                    'and',
                    [$expenseTableName . '.archived' => 1],
                    ['>=', "ecd.time", "$year-$month-1 00:00:00"]
                ]
            ]);
        }
        else {
            $query->andWhere([
                'or',
                ['archived' => 0],
                ['>', 'DATE(ecd.time)', date("Y-m-t", strtotime("$year-$month-20"))]
            ]);
        }
        return $query;
    }

    /**
     * Get recurring expenses for a month.
     * @param int $month Selected month.
     * @param int $year Selected year.
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return array List of selected expenses.
     */
    public function getMonthsRecurringExpenses($month, $year, $onlyBudgeted = false, $includeDeleted = true)
    {
        return $this->getMonthsRecurringExpensesQuery($month, $year, $onlyBudgeted, $includeDeleted)
            ->all();
    }

    /**
     * Get the total amount of a month's recurring expenses.
     * @return float Total amount of a month's recurring expenses.
     */
    public function getMonthsRecurringExpensesAmount($month, $year)
    {
        $expenses = $this->getMonthsRecurringExpenses($month, $year);
        $amount = 0;
        foreach ($expenses as $e) {
            $amount += $e->getAverageMonthlyAmount();
        }
        return $amount;
    }

    /**
     * Get recurring expenses for this month.
     * @return array List of selected expenses.
     */
    public function getThisMonthsRecurringExpenses()
    {
        if ($this->_thisMonthsRecurringExpenses == null) {
            $this->_thisMonthsRecurringExpenses = $this->getMonthsRecurringExpenses(date('m'), date('Y'));
        }
        return $this->_thisMonthsRecurringExpenses;
    }

    /**
     * Get the total amount of this month's recurring expenses.
     * @return float Total amount of this month's recurring expenses.
     */
    public function getThisMonthsRecurringExpensesAmount()
    {
        if ($this->_thisMonthsRecurringExpensesAmount == null) {
            $expenses = $this->getThisMonthsRecurringExpenses();
            $amount   = 0;
            foreach ($expenses as $e) {
                if ($e->archived == 0) {
                    $amount += $e->getAverageMonthlyAmount();
                }
                else {
                    $transactions = $e->getTransactions()
                        ->where(['>=', 'date',  date('Y-m-d')])
                        ->all();
                    foreach ($transactions as $t) {
                        $amount += $t->amount;
                    }
                }
            }
            $this->_thisMonthsRecurringExpensesAmount = $amount;
        }
        return $this->_thisMonthsRecurringExpensesAmount;
    }

    /**
     * Get all active recurring expenses for this jar
     * @param string|null $month
     * @param string|null $year
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return array
     */
    public function getRecurringExpenses($month = null, $year = null, $onlyBudgeted = false, $includeDeleted = true)
    {
        if (!$month) {
            $month = date('m');
        }
        if (!$year) {
            $year = date('Y');
        }
        return $this->getMonthsRecurringExpensesQuery($month, $year, $onlyBudgeted, $includeDeleted)
            ->all();
    }

    /**
     * Get the total amount of recurring expenses
     * as displayed on the Jar's details page
     * @param string|null $month
     * @param string|null $year
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return float
     */
    public function getRecurringExpensesAmount($month = null, $year = null, $onlyBudgeted = false, $includeDeleted = true)
    {
        if (!$month) {
            $month = date('m');
        }
        if (!$year) {
            $year = date('Y');
        }
        $expenses = $this->getRecurringExpenses($month, $year, $onlyBudgeted, $includeDeleted);
        $amount = 0;
        /** @var Expense $expense */
        foreach ($expenses as $expense) {
            $amount += $expense->getAverageMonthlyAmount((int)$month, $year);
        }
        return $amount;
    }

    /**
     * Get query for estimated expenses
     * @param int $month
     * @param int $year
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return ActiveQuery
     */
    public function getMonthsEstimatedExpensesQuery($month, $year, $onlyBudgeted = false, $includeDeleted = true)
    {
        $expenseTableName = Expense::tableName();
        $expenseChangeTableName = ExpenseChange::tableName();

        // active expenses or expenses removed this month
        $query = Expense::find()
            ->select("$expenseTableName.*, ecd.time as removal_time")
            ->leftJoin($expenseChangeTableName . ' as ecc', $expenseTableName . ".id = ecc.expense_id AND ecc.type = 'C'")
            ->leftJoin($expenseChangeTableName . ' as ecd', $expenseTableName . ".id = ecd.expense_id AND ecd.type = 'D'")
            ->where([
                'and',
                ['jar_id' => $this->id],
                ['!=' , 'frequency', 'one-time'],
                ['generate_transactions' => 0],
                [
                    'or',
                    ['ecc.time' => null],
                    ['<=', 'ecc.time', date('Y-m-t 23:59:59', strtotime("$year-$month-20"))],
                ]
            ]);
        if ($onlyBudgeted) {
            $query->andWhere([
                'is_unbudgeted' => 0
            ]);
        }
        if ($includeDeleted) {
            $query->andWhere([
                'or',
                ['archived' => 0],
                [
                    'and',
                    ['archived' => 1],
                    ['>', "ecd.time", "$year-$month-1 00:00"]
                ]
            ]);
        }
        else {
            $query->andWhere([
                'or',
                ['archived' => 0],
                ['>', 'DATE(ecd.time)', date("Y-m-t", strtotime("$year-$month-20"))]
            ]);
        }
        return $query;
    }

    /**
     * Get estimated expenses for a month.
     * @param int $month Selected month.
     * @param int $year Selected year.
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return array List of selected expenses.
     */
    public function getMonthsEstimatedExpenses($month, $year, $onlyBudgeted = false, $includeDeleted = true)
    {
        return $this->getMonthsEstimatedExpensesQuery($month, $year, $onlyBudgeted, $includeDeleted)
            ->all();
    }

    /**
     * Get the total amount of a month's estimated expenses.
     * @return float Total amount of a month's estimated expenses.
     */
    public function getMonthsEstimatedExpensesAmount($month, $year)
    {
        $expenses = $this->getMonthsEstimatedExpenses($month, $year);
        $amount = 0;
        foreach ($expenses as $e) {
            $amount += $e->getAverageMonthlyAmount();
        }
        return $amount;
    }

    /**
     * Get estimated expenses for this month.
     * @return array List of selected expenses.
     */
    public function getThisMonthsEstimatedExpenses()
    {
        if ($this->_thisMonthsEstimatedExpenses == null) {
            $this->_thisMonthsEstimatedExpenses = $this->getMonthsEstimatedExpenses(date('m'), date('Y'));
        }
        return $this->_thisMonthsEstimatedExpenses;
    }

    /**
     * Get the total amount of this month's estimated expenses.
     * @return float Total amount of this month's estimated expenses.
     */
    public function getThisMonthsEstimatedExpensesAmount()
    {
        if ($this->_thisMonthsEstimatedExpensesAmount == null) {
            $expenses = $this->getThisMonthsEstimatedExpenses();
            $amount = 0;
            foreach ($expenses as $e) {
                if ($e->archived == 0) {
                    $amount += $e->getAverageMonthlyAmount();
                }
                else {
                    $transactions = $e->getTransactions()
                        ->where(['>=', 'date',  date('Y-m-d')])
                        ->all();
                    foreach ($transactions as $t) {
                        $amount += $t->amount;
                    }
                }
            }
            $this->_thisMonthsEstimatedExpensesAmount = $amount;
        }
        return $this->_thisMonthsEstimatedExpensesAmount;
    }

    /**
     * Get the total amount of all estimated (regular) expenses
     * as seen on the Jar's details page
     * @param int $month
     * @param int $year
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return float
     */
    public function getEstimatedExpensesAmount($month = null, $year = null, $onlyBudgeted = false, $includeDeleted = true)
    {
        if (!$month) {
            $month = date('m');
        }
        if (!$year) {
            $year = date('Y');
        }
        $expenses = $this->getMonthsEstimatedExpensesQuery($month, $year, $onlyBudgeted, $includeDeleted);

        $amount = 0;
        /** @var Expense $expense */
        foreach ($expenses->all() as $expense) {
            $amount += $expense->getAverageMonthlyAmount((int)$month, $year);
        }
        return $amount;
    }

    /**
     * Get query for one time expenses
     * @param $month
     * @param $year
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return ActiveQuery
     */
    public function getMonthsOnetimeExpensesQuery($month, $year, $onlyBudgeted = false, $includeDeleted = true)
    {
        $expenseTableName = Expense::tableName();
        $expenseChangeTableName = ExpenseChange::tableName();
        $expenseStartTableName = ExpenseStart::tableName();

        $expenses = Expense::find()
            ->select("$expenseTableName.*, $expenseStartTableName.date as start_date")
            ->leftJoin($expenseChangeTableName . ' as ecc', $expenseTableName . ".id = ecc.expense_id AND ecc.type = 'C'")
            ->leftJoin($expenseChangeTableName . ' as ecd', $expenseTableName . ".id = ecd.expense_id AND ecd.type = 'D'")
            ->joinWith('expenseStart')
            ->where([
                'AND',
                ['is_adjustment' => 0],
                ['jar_id' => $this->id],
                ['frequency' => 'one-time'],
                [
                    'or',
                    ['ecc.time' => null],
                    ['<=', 'ecc.time', date('Y-m-t 23:59:59', strtotime("$year-$month-20"))],
                ],
                [
                    'or',
                    [
                        'and',
                        ['not', [$expenseStartTableName . '.id' => null]],
                        ['<=', $expenseStartTableName . '.date', "$year-$month-".date('t', strtotime("$year-$month-20"))],
                    ],
                    [
                        'and',
                        [$expenseStartTableName . '.id' => null],
                        ['>=', $expenseTableName . '.date', "$year-$month-1"]
                    ]
                ]
            ]);
        if ($onlyBudgeted) {
            $expenses->andWhere([
                'is_unbudgeted' => 0
            ]);
        }
        if ($includeDeleted) {
            $expenses->andWhere([
                'or',
                ['archived' => 0],
                [
                    'and',
                    ['archived' => 1],
                    ['>', "ecd.time", "$year-$month-1 00:00:00"]
                ]
            ]);
        }
        else {
            $expenses->andWhere([
                'or',
                ['archived' => 0],
                ['>', 'DATE(ecd.time)', date("Y-m-t", strtotime("$year-$month-20"))]
            ]);
        }
        return $expenses;
    }

    /**
     * Get one-time expenses for a month.
     * @param int $month Selected month.
     * @param int $year Selected year.
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return array List of selected expenses.
     */
    public function getMonthsOnetimeExpenses($month, $year, $onlyBudgeted = false, $includeDeleted = true)
    {
        return $this->getMonthsOnetimeExpensesQuery($month, $year, $onlyBudgeted, $includeDeleted)
            ->all();
    }

    /**
     * Get the total amount of a month's one-time expenses.
     * @return float Total amount of a month's one-time expenses.
     */
    public function getMonthsOnetimeExpensesAmount($month, $year, $onlyBudgeted = false)
    {
        $expenses = $this->getMonthsOnetimeExpenses($month, $year, $onlyBudgeted);
        $amount = 0;
        foreach ($expenses as $e) {
            $amount += $e->getPlannedPaymentAmount($month, $year);
        }
        return $amount;
    }

    /**
     * Get one-time expenses for this month.
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return array List of selected expenses.
     */
    public function getThisMonthsOnetimeExpenses($onlyBudgeted = false, $includeDeleted = true)
    {
        if ($this->_thisMonthsOnetimeExpenses == null) {
            $this->_thisMonthsOnetimeExpenses = $this->getMonthsOnetimeExpenses(date('m'), date('Y'), $onlyBudgeted, $includeDeleted);
        }
        return $this->_thisMonthsOnetimeExpenses;
    }

    /**
     * Get the total amount of this month's one-time expenses.
     * @return float Total amount of this month's one-time expenses.
     */
    public function getThisMonthsOnetimeExpensesAmount()
    {
        if ($this->_thisMonthsOnetimeExpensesAmount == null) {
            $this->_thisMonthsOnetimeExpensesAmount = $this->getMonthsOnetimeExpensesAmount(date('m'), date('Y'), true);
        }
        return $this->_thisMonthsOnetimeExpensesAmount;
    }

    /**
     * Get the total of one time expenses
     * as seen on the Jar's details page
     * @param null $month
     * @param null $year
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return float
     */
    public function getOnetimeExpensesAmount($month = null, $year = null, $onlyBudgeted = false, $includeDeleted = true)
    {
        if (!$year) {
            $year = date('Y');
        }
        if (!$month) {
            $month = date('m');
        }
        $expenses = $this->getMonthsOnetimeExpensesQuery($month, $year, $onlyBudgeted, $includeDeleted);

        $amount = 0;
        /** @var Expense $expense */
        foreach ($expenses->all() as $expense) {
            $amount += $expense->getAverageMonthlyAmount((int)$month, $year);
        }
        return $amount;
    }

    /**
     * Get total amount of a month's expenses.
     * @param integer $month
     * @param integer $year
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return float Total amount.
     */
    public function getMonthsExpensesAmount($month, $year, $onlyBudgeted = false, $includeDeleted = true)
    {
        $month = date('m', strtotime($year.'-'.$month.'-20'));

        switch ($this->jar_type_id) {
            case 1:
                $amount =
                    $this->getRecurringExpensesAmount($month, $year, $onlyBudgeted, $includeDeleted) +
                    $this->getEstimatedExpensesAmount($month, $year, $onlyBudgeted, $includeDeleted) +
                    $this->getOnetimeExpensesAmount($month, $year, $onlyBudgeted, $includeDeleted);
                break;
            case 2:
                $budget = Budget::findOne($this->budget_id);
                $amount = $budget->getMonthsDebtsAmount($month, $year);
                break;
            case 3:
                // interest amounts are no longer handled separately
                $amount = 0;
                break;
            default:
                $amount = false;
                break;
        }

        return $amount;
    }

    /**
     * @param $month
     * @param $year
     * @param bool $onlyBudgeted
     * @param bool $includeDeleted
     * @return array
     */
    public function getMonthsExpenses($month, $year, $onlyBudgeted = false, $includeDeleted = true)
    {
        $expenses = [];
        if ($this->jar_type_id == '1') {
            $expenses = array_merge(
                $this->getMonthsRecurringExpenses($month, $year, $onlyBudgeted, $includeDeleted),
                $this->getMonthsEstimatedExpenses($month, $year, $onlyBudgeted, $includeDeleted),
                $this->getMonthsOnetimeExpenses($month, $year, $onlyBudgeted, $includeDeleted)
            );
        }
        elseif ($this->jar_type_id == '2') {
            $budget   = Budget::findOne($this->budget_id);
            $expenses = $budget->getMonthsDebts($month, $year, false, $includeDeleted);
        }
        elseif ($this->jar_type_id == '3') {
            $budget   = Budget::findOne($this->budget_id);
            $expenses = $budget->getMonthsDebts($month, $year, false, $includeDeleted);
        }

        return $expenses;
    }


    /**
     * Get total amount of this month's expenses.
     * @return float Total amount.
     */
    public function getThisMonthsExpensesAmount()
    {
        if ($this->_thisMonthsExpensesAmount == null) {
            $this->_thisMonthsExpensesAmount = $this->getMonthsExpensesAmount(date('m'), date('Y'));
        }
        return $this->_thisMonthsExpensesAmount;
    }

    public function getMonthsMoneyInAmount($month, $year, $withBalanceAdjustments = false, $withJarAdjustments = true)
    {
        $amount = 0;
        switch ($this->jar_type_id) {
            case 1:
                $incomes = IncomeTransaction::find()
                    ->joinWith('income')
                    ->where([
                        'jar_id' => $this->id,
                        'frequency' => 'one-time',
                        'budget_id' => $this->budget_id
                    ])
                    ->andWhere([
                        'like', IncomeTransaction::tableName().'.date', date('Y-m-', strtotime($year.'-'.$month.'-20'))
                    ]);
                if (!$withBalanceAdjustments || !$withJarAdjustments) {
                    $exclude = [];
                    if (!$withBalanceAdjustments) {
                        $exclude[] = Income::INCOME_ADJUSTMENT;
                    }
                    if (!$withJarAdjustments) {
                        $exclude[] = Income::JAR_ADJUSTMENT;
                    }
                    $incomes = $incomes
                        ->andWhere([
                            'NOT IN', 'is_adjustment', $exclude
                        ]);
                }
                $incomes = $incomes->all();
                foreach ($incomes as $i) {
                    $amount += $i->amount;
                }
                break;
            case 2:
                $budget = Budget::findOne($this->budget_id);
                $debtPayments = $budget->getMonthsDebtPayments($month, $year, false, 1, !$withBalanceAdjustments, !$withJarAdjustments);
                foreach ($debtPayments as $t) {
                    if ($t->amount < 0) {
                        $amount += -1 * $t->amount;
                    }
                }
                break;
            case 3:
                $budget = Budget::findOne($this->budget_id);
                $debtPayments = $budget->getMonthsDebtPayments($month, $year, false, 2, !$withBalanceAdjustments, !$withJarAdjustments);
                foreach ($debtPayments as $t) {
                    if ($t->amount < 0) {
                        $amount += -1 * $t->amount;
                    }
                }
                break;
        }

        return $amount;
    }

    /**
     * @param bool $withBalanceAdjustments
     * @param bool $withJarAdjustments
     * @return int|mixed
     */
    public function getThisMonthsMoneyInAmount($withBalanceAdjustments = false, $withJarAdjustments = true)
    {
        if ($this->_thisMonthsMoneyInAmount == null) {
            $this->_thisMonthsMoneyInAmount = $this->getMonthsMoneyInAmount(date('m'), date('Y'), $withBalanceAdjustments, $withJarAdjustments);
        }
        return $this->_thisMonthsMoneyInAmount;
    }

    public function getMonthlyBudget($month = null, $year = null)
    {
        if ($month == null) {
            $month = date('m');
        }
        if ($year == null) {
            $year = date('Y');
        }
        if ($this->jar_type_id == 1) {
            return $this->getMonthsExpensesAmount($month, $year, true, false);
        }
        else {
            $query = $this->getMonthsDebtsExpensesQuery($month, $year);
            $total = 0;
            /** @var Debt $debt */
            foreach ($query->all() as $debt) {
                $total += $debt->getAverageMonthlyAmount($month, $year);
            }
            return $total;
        }
    }

    public function getThisMonthsFundsBroughtForwardAmount()
    {
        return $this->getMonthsFundsBroughtForwardAmount(date('m'), date('Y'));
    }

    /**
     * Get funds brought forward for the given month
     * @param int $month
     * @param int $year
     * @param bool $withBalanceAdjustments
     * @param bool $withJarAdjustments
     * @return float
     */
    public function getMonthsFundsBroughtForwardAmount($month, $year, $withBalanceAdjustments = false, $withJarAdjustments = true)
    {
        // we're calculating this value basing on the values from the previous month
        $timeNow = strtotime($year.'-'.$month.'-20');
        $month = date('m', $timeNow);
        $prevM = date('m', strtotime('-1 month', $timeNow));
        $prevY = date('Y', strtotime('-1 month', $timeNow));

        // it's a future date
        if (date('Y-m', strtotime($year.'-'.$month.'-20')) > date('Y-m')) {
            // return previous month's remaining amount
            return $this->getMonthsFundsRemainingAmount($prevM, $prevY, $withBalanceAdjustments, $withJarAdjustments);
        }

        $prevJarAmount = JarAmount::find()
            ->joinWith('jar.budget')
            ->where([
                'month'   => (int)$prevM,
                'year'    => $prevY,
                'jar_id'  => $this->id,
                'is_mock' => 0
            ])
            ->one();
        if ($prevJarAmount) {
            $brought = $prevJarAmount->funds_brought_forward;
        }
        else {
            // get the amounts from before 2 months if necessary
            $prevTimeNow = strtotime("$prevY-$prevM-20");
            $prev2M = date('m', strtotime('-1 month', $prevTimeNow));
            $prev2Y = date('Y', strtotime('-1 month', $prevTimeNow));
            $jarAmountExists = JarAmount::find()
                ->joinWith('jar.budget')
                ->where([
                    'month'   => (int)$prev2M,
                    'year'    => $prev2Y,
                    'jar_id'  => $this->id,
                    'is_mock' => 0
                ])->exists();
            if ($jarAmountExists) {
                $brought = $this->getMonthsFundsBroughtForwardAmount($prevM, $prevY, $withBalanceAdjustments, $withJarAdjustments);
            }
            else {
                $brought = 0;
            }
        }

        return
            $brought +
            $this->getMonthlyBudget($prevM, $prevY) +
            $this->getMonthsMoneyInAmount($prevM, $prevY, $withBalanceAdjustments, $withJarAdjustments) -
            $this->getMonthsTransactionsAmount($prevM, $prevY, $withBalanceAdjustments, $withJarAdjustments);
    }

    public function getMonthsDebtsExpensesQuery($month, $year, $includeDeleted = false)
    {
        // active debts or debts removed this month (exclude CC debts)
        $query = Debt::find()
            ->select(Debt::tableName() . '.* ,' . Account::tableName() . '.name as account_name,' . Debt::tableName() . '.name as debt_name')
            ->joinWith('account')
            ->joinWith('budget')
            ->leftJoin(DebtChange::tableName() . ' as dcc', Debt::tableName() . ".id = dcc.debt_id AND dcc.type = 'C'")
            ->leftJoin(DebtChange::tableName() . ' as dcd', Debt::tableName() . ".id = dcd.debt_id AND dcd.type = 'D'")
            ->leftJoin(Account::tableName() . ' as a', Debt::tableName() . ".active_debt_account_id = a.id AND a.account_type_id != 1 AND a.archived = 0")
            ->where(['and',
                ['<=', 'DATE(dcc.time)', date('Y-m-t', strtotime("$year-$month-20"))],
                //['!=', Debt::tableName() . '.frequency', 'one-time'],
                ['or',
                    [Debt::tableName() . '.budget_id' => $this->budget_id],
                    [Account::tableName() . '.budget_id' => $this->budget_id]
                ],
            ]);
        if ($includeDeleted) {
            $query->andWhere([
                'OR',
                [Debt::tableName() . '.archived' => 0],
                ['AND',
                    ['>=', 'DATE(dcd.time)', date("$year-$month-1")],
                    ['a.id' => null]
                ]
            ]);
        }
        else {
            $query->andWhere([
                'or',
                [Debt::tableName() . '.archived' => 0],
                ['>', 'DATE(dcd.time)', date("Y-m-t", strtotime("$year-$month-20"))]
            ]);
        }
        return $query;
    }

    public function getMonthsDebtsTransactionsQuery($month, $year)
    {
        $query = DebtPayment::find()
            ->select(DebtPayment::tableName() . '.* ,' . Account::tableName() . '.name as account_name,' . Debt::tableName() . '.name as debt_name')
            ->joinWith('account')
            ->joinWith('debt')
            ->where(['and',
                ['!=', DebtPayment::tableName() . '.debt_payment_type_id', 2],
                [DebtPayment::tableName() . '.is_adjustment' => 0],
                ['BETWEEN', DebtPayment::tableName() . '.date', date('Y-m-01', strtotime("$year-$month-20")), date('Y-m-t', strtotime("$year-$month-20"))],
                ['or',
                    [Debt::tableName() . '.budget_id' => $this->budget_id],
                    [Account::tableName() . '.budget_id' => $this->budget_id]
                ],
            ]);
        return $query;
    }

    /**
     * Whether the given month and year is after the first month when the EoMS was submitted
     * (useful when getting funds brought forward and monthly budget for the given month)
     * @param $month
     * @param $year
     * @return bool
     */
    public function isAfterFirstEoms($month, $year)
    {
        $month = str_pad($month, 2, '0', STR_PAD_LEFT);
        return User::find()
            ->where(['and',
                ['id' => Yii::$app->session->get('mainUserId')],
                ['<=', 'created_at', date('Y-m-t 23:59:59', strtotime("$year-$month-20"))]
            ])
            ->exists();
        /*return JarAmount::find()
            ->joinWith('jar.budget')
            ->where(['and',
                ['<=', "CONCAT(year, '-', LPAD(month, 2, '0'))", "$year-$month"],
                ['jar_id'  => $this->id],
                ['is_mock' => 0]
            ])
            ->exists();*/
    }

    /**
     * Values have changed for the month (expenses, transactions)
     * so we need to update all relevant data
     * @param $month
     * @param $year
     */
    public function updateMonthsSummaryAmount($month, $year)
    {
        // TODO check if this is actually needed
        /*$month = date('m', strtotime($year.'-'.$month.'-20'));
        $nextM = date('m', strtotime('+1 month', strtotime("$year-$month-1")));
        $nextY = date('Y', strtotime('+1 month', strtotime("$year-$month-1")));

        $jarAmountNow = JarAmount::find()
            ->where([
                'month'  => (int)$month,
                'year'   => $year,
                'jar_id' => $this->id
            ])
            ->one();
        $jarAmountNext = JarAmount::find()
            ->where([
                'month'  => (int)$nextM,
                'year'   => $nextY,
                'jar_id' => $this->id
            ])
            ->one();

        // first update the current month
        if ($jarAmountNow && $jarAmountNow->funds_remaining_adjustment !== null) {

        }*/
    }

    public function getMonthsJarAmountRecord($month, $year)
    {
        return JarAmount::find()
            ->where([
                'month'  => (int)$month,
                'year'   => $year,
                'jar_id' => $this->id
            ])
            ->one();
    }


    /**
     * Stores funds brought forwards values for all jars for the given EomS
     * @param Eoms $eoms
     */
    public static function saveFundsBroughtForwardForEoms(Eoms $eoms)
    {
        $jars = $eoms->budget->getActiveJars();
        /** @var Jar $jar */
        foreach ($jars as $jar) {
            // delete the existing amount for the eoms and jar
            $jarAmount = JarAmount::find()
                ->joinWith('jar.budget')
                ->where([
                    'jar_id'  => $jar->id,
                    'is_mock' => 0
                ])
                ->andWhere([
                    'or',
                    ['eoms_id' => $eoms->id],
                    [
                        'and',
                        [JarAmount::tableName() . '.month' => $eoms->month],
                        [JarAmount::tableName() . '.year'  => $eoms->year]
                    ]
                ])
                ->one();
            if (!$jarAmount) {
                $jarAmount = new JarAmount();
                $jarAmount->setAttributes([
                    'jar_id' => $jar->id,
                    'eoms_id' => $eoms->id,
                    'month' => $eoms->month,
                    'year' => $eoms->year,
                ]);
                $jarAmount->save();
            }

            $brought   = $jar->getMonthsFundsBroughtForwardAmount($eoms->month, $eoms->year);
            $jarAmount->updateAttributes([
                'funds_brought_forward' => $brought
            ]);
            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jar->id);
        }
    }

    public function saveJarAdjustmentRecord(AdjustJarFundsRemaining $formData, $month, $year, $accountId)
    {
        $balance         = Formatter::float($formData->funds_remaining_adjustment - $this->getMonthsFundsRemainingAmount($month, $year, false, false));
        $transactionDate = date('Y-m-01', strtotime($year . '-' . $month . '-20'));
        $operationName   = 'Jar Funds Remaining Adjustment for ' . date('M Y', strtotime("$year-$month-20"));

        if ($this->jar_type_id == 1) {
            $this->saveAccountAdjustmentRecord($transactionDate, $operationName, $balance, $accountId);
        }
        else {
            $this->saveDebtAdjustmentRecord($transactionDate, $operationName, $balance, $accountId);
        }

        // we need to recalculate all the following funds brought forward values
        if ($balance != 0) {
            $thisYear   = (int)date('Y');
            $thisMonth  = (int)date('n');
            $startMonth = (int)date('n', strtotime('+1 month', strtotime($year.'-'.$month.'-20')));
            $startYear  = (int)date('Y', strtotime('+1 month', strtotime($year.'-'.$month.'-20')));

            // starting year
            for ($month = $startMonth; $month <= ($startYear == $thisYear ? $thisMonth : 12); $month++) {
                JarAmount::updateFundsBroughtForward($this, $month, $startYear);
            }
            // full years
            if ($thisYear - $startYear > 1) {
                for ($year = $startYear + 1; $startYear < $thisYear; $year++) {
                    for ($month = 1; $month <= 12; $month++) {
                        JarAmount::updateFundsBroughtForward($this, $month, $startYear);
                    }
                }
            }
            // end year
            if ($startYear != $thisYear) {
                for ($month = 1; $month <= $thisMonth; $month++) {
                    JarAmount::updateFundsBroughtForward($this, $month, $thisYear);
                }
            }
        }
    }

    protected function saveAccountAdjustmentRecord($transactionDate, $operationName, $balance, $accountId)
    {
        $absBalance      = Formatter::float(abs($balance));
        // delete existing adjustments
        Income::deleteAll([
            'jar_id'        => $this->id,
            'date'          => $transactionDate,
            'is_adjustment' => Income::JAR_ADJUSTMENT
        ]);
        Expense::deleteAll([
            'jar_id'        => $this->id,
            'date'          => $transactionDate,
            'is_adjustment' => Expense::JAR_ADJUSTMENT
        ]);

        // we need to create an income or expense, depending on the balance
        if ($balance > 0) {
            $income = new Income();
            $income->setAttributes([
                'budget_id'     => $this->budget_id,
                'account_id'    => $accountId,
                'jar_id'        => $this->id,
                'name'          => $operationName,
                'date'          => $transactionDate,
                'frequency'     => 'one-time',
                'is_adjustment' => Income::JAR_ADJUSTMENT
            ]);
            $income->amount = $absBalance;
            $income->save();

            $transaction = new IncomeTransaction();
            $transaction->setAttributes([
                'income_id'  => $income->id,
                'account_id' => $accountId,
                //'date'       => $income->date
            ]);
            $transaction->date   = $income->date; // for some reason mass assignment doesn't work properly
            $transaction->amount = $income->amount;
            $transaction->save();
        }
        elseif ($balance < 0) {
            $expense = new Expense();
            $expense->setAttributes([
                'jar_id'                => $this->id,
                'name'                  => $operationName,
                'account_id'            => $accountId,
                'date'                  => $transactionDate,
                'frequency'             => 'one-time',
                'is_unbudgeted'         => 1,
                'is_adjustment'         => Expense::JAR_ADJUSTMENT,
                'generate_transactions' => 0
            ]);
            $expense->amount = $absBalance;
            $expense->save();

            $transaction = new Transaction();
            $transaction->setAttributes([
                'account_id'     => $accountId,
                'expense_id'     => $expense->id,
                'date'           => $expense->date,
                'description'    => $expense->name,
                'tax_deductible' => $expense->tax_deductible ? 1 : 0
            ]);
            $transaction->amount = $expense->amount;
            $transaction->save();
        }
        TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $this->id);
    }

    protected function saveDebtAdjustmentRecord($transactionDate, $operationName, $balance, $accountId)
    {
        // delete existing adjustments
        $debtPayments = DebtPayment::find()
            ->joinWith('account')
            ->joinWith('debt')
            ->where(['and',
                ['or',
                    [Debt::tableName() . '.budget_id' => $this->budget_id],
                    [Account::tableName() . '.budget_id' => $this->budget_id]
                ],
                [DebtPayment::tableName() . '.date' => $transactionDate],
                ['is_adjustment' => DebtPayment::JAR_ADJUSTMENT]
            ])
            ->all();
        DebtPayment::deleteAll(['id' => ArrayHelper::map($debtPayments, 'id', 'id')]);

        // this is the special case that only for jar funds remaining adjustment
        // we allow a negative value to be provided (i.e. debt increased)
        if ($balance != 0) {
            // jar adjustments are saved WITHOUT any debt associated
            $debtPayment = new DebtPayment();
            $debtPayment->setAttributes([
                'debt_payment_type_id' => 1,
                'account_id'           => $accountId,
                'date'                 => $transactionDate,
                'is_adjustment'        => DebtPayment::JAR_ADJUSTMENT
            ]);
            $debtPayment->amount = -1 * Formatter::float($balance);
            $debtPayment->save();
        }
        TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $this->id);
    }

    /**
     * @return int|null
     */
    public function getMockBudgetId()
    {
        if ($this->budget->is_mock) {
            return $this->budget_id;
        }
        return null;
    }

    /**
     * Get a list of accounts for select input element.
     * @param $budgetId
     * @param bool $onlyExpenseJars
     * @return array List of jars [ID => name]
     */
    public static function getJarsForSelect($budgetId, $onlyExpenseJars = true)
    {
        $params = [
            'budget_id' => $budgetId,
            'archived' => '0'
        ];
        if ($onlyExpenseJars) {
            $params['jar_type_id'] = 1;
        }
        return ArrayHelper::map(Jar::findAll($params), 'id', 'name');
    }

}
