<?php

namespace backend\models\form;

use backend\components\helpers\Formatter;
use backend\components\validators\Currency;
use backend\components\validators\DateStart;
use backend\models\db\Account;
use backend\models\form\common\DateField;
use backend\models\form\common\InterestFreeEndDateField;
use Yii;

/**
 * New Account form
 */
class AddAccount extends \yii\base\Model
{
    use DateField, InterestFreeEndDateField;

    public $name;
    public $balance = 0;
    public $account_type_id;
    public $debt_type_id = 1;
    public $is_central_account;
    public $is_debt_active = 1;
    public $budget_id;
    public $account_id = -1;
    public $active_debt_account_id = -1;
    public $payment;
    public $interest = 0;
    public $frequency;
    public $date;
    public $has_interest_free_period;
    public $interest_free_end_date;

    protected $ccWhenBackend;
    protected $ccWhenFrontend;


    public function init()
    {
        parent::init();
        $this->ccWhenBackend = function($model) {
            return $model->account_type_id != 1;
        };
        $this->ccWhenFrontend = "function (attribute, value) {
                return $('#addaccount-account_type_id').val() != 1;
            }";
    }

    public function rules() {

        // validate input
        $validationRules = [
            ['name', 'required'],
            ['name', 'string', 'min' => 3],
            ['name', 'accountNameValid'],
            ['balance', 'required'],
            ['balance', Currency::className()],
            ['account_id', 'required'],
            ['account_type_id', 'required', 'when' => function($model) {
                return $model->account_id == -1;
            }],
            [['account_type_id', 'debt_type_id', 'account_id', 'active_debt_account_id'], 'integer'],
            ['is_central_account', 'integer'],
            ['payment', 'required', 'when' => $this->ccWhenBackend, 'whenClient' => $this->ccWhenFrontend],
            ['payment', Currency::className(), 'min' => 0],
            ['interest', 'number', 'min' => 0],
            ['frequency', 'required', 'when' => $this->ccWhenBackend, 'whenClient' => $this->ccWhenFrontend],
            ['date', 'required', 'when' => $this->ccWhenBackend, 'whenClient' => $this->ccWhenFrontend],
            ['date', DateStart::className(), 'fieldName' => 'debt', 'on' => 'create',
                'when' => $this->ccWhenBackend, 'whenClient' => $this->ccWhenFrontend],
            ['date', 'date', 'format' => Formatter::phpDateCodeToDateCode(Yii::$app->session->get('dateFormat')), 'message' => 'Incorrect date format.', 'when' => $this->ccWhenBackend, 'whenClient' => $this->ccWhenFrontend],
            ['interest_free_end_date', 'required', 'when' => function($model) {
                return $model->has_interest_free_period && !$model->is_debt_active;
            }, 'whenClient' => "function (attribute, value) {
                return $('#addaccount-has_interest_free_period').is(':checked') && !$('#addaccount-is_debt_active').is(':checked');
            }"],
            ['interest_free_end_date', 'date', 'format' => Formatter::phpDateCodeToDateCode(Yii::$app->session->get('dateFormat')), 'message' => 'Incorrect date format.'],
        ];

        // sanitize
        $filterRules = [
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'strip_tags'],
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'trim'],
            [['balance', 'payment'], 'filter', 'filter' => ['backend\components\helpers\Formatter', 'currencyToDecimal']],
            [['date', 'interest_free_end_date'], 'filter', 'filter' => ['backend\components\helpers\Formatter', 'appDateToPhpDate'], 'when' => function($model, $attribute) {
                return !empty($model->{$attribute});
            }]
        ];

        return array_merge($this->dateRules(), $this->interestFreeEndDateRules(), $validationRules, $filterRules);

    }

    /**
     * Check whether jar name is allowed.
     * @param string $attribute
     */
    public function accountNameValid($attribute)
    {
        $account = Account::find()
            ->where(['and',
                ['name' => $this->{$attribute}],
                ['budget_id' => $this->budget_id],
                ['account_type_id' => $this->account_type_id],
                ['!=', 'id', $this->active_debt_account_id]
            ])
            ->one();

        if ($account) {
            $this->addError($attribute, 'Please enter a unique account name.');
        }
    }

}
