<?php

namespace backend\models\form;

use backend\components\validators\Currency;
use backend\models\form\common\DateField;
use Yii;

use backend\models\db\Account;
use backend\models\db\Expense;

use backend\components\helpers\Formatter;
use backend\components\validators\DateStart;

/**
 * New expense form
 */
class AddExpense extends \yii\base\Model
{
    use DateField;

    public $name;
    public $amount;
    public $account_id;
    public $debt_id = false; // to make life easier in the controller
    public $date;
    public $frequency = 'one-time';
    public $tax_deductible = 0;
    public $archived = 0;
    public $jar_id;
    public $expense_id = -1;
    public $generate_transactions = 1;
    public $budget_id;

    public function rules() {
        // validate input
        $validationRules = [
            ['name', 'required'],
            ['name', 'string', 'min' => 3],
            ['name', 'expenseNameValid'],
            ['amount', 'required'],
            ['amount', Currency::className(), 'min' => '0.00'],
            ['account_id', 'required', 'message' => 'Please select an account.'],
            ['account_id', 'string'],
            ['account_id', 'accountIdValid'],
            ['date', 'required', 'when' => function($model) {
                return $model->generate_transactions == 1;
            }],
            ['date', DateStart::className(), 'fieldName' => 'expense'],
            ['date', 'date', 'format' => Formatter::phpDateCodeToDateCode(Yii::$app->session->get('dateFormat')), 'message' => 'Incorrect date format.'],
            ['frequency', 'required'],
            ['jar_id', 'required', 'message' => 'Please select a jar.'],
            ['jar_id', 'integer'],
            ['jar_id', 'jarIdValid'],
            ['tax_deductible', 'number', 'max' => 1]
        ];

        // sanitize
        $filterRules = [
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'strip_tags'],
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'trim'],
            ['amount', 'filter', 'filter' => ['backend\components\helpers\Formatter', 'currencyToDecimal']],
            ['date', 'filter', 'filter' => ['backend\components\helpers\Formatter', 'appDateToPhpDate'], 'when' => function($model, $attribute) {
                return !empty($model->{$attribute});
            }]
        ];

        return array_merge($this->dateRules(), $validationRules, $filterRules);
    }

    /**
     * Check whether expense name is allowed.
     * @param string $attribute
     */
    public function expenseNameValid($attribute)
    {
        if ($this->frequency == 'one-time') {
            $expense = Expense::find()
                ->where([
                    'and',
                    ['jar_id' => $this->jar_id],
                    ['!=', 'id', $this->expense_id],
                    ['name' => $this->{$attribute}],
                    ['!=', 'frequency', 'one-time'],
                    ['archived' => 0]
                ])
                ->one();
        }
        else {
            $expense = Expense::find()
                ->where([
                    'and',
                    ['jar_id' => $this->jar_id],
                    ['!=', 'id', $this->expense_id],
                    ['name' => $this->{$attribute}],
                    ['archived' => 0]
                ])
                ->one();
        }
        if ($expense) {
            $this->addError($attribute, 'Please enter a unique expense name.');
        }
    }

    /**
     * Check if the account ID can be accessed by this user
     */
    public function accountIdValid($attribute)
    {
        $account = Account::findOne([
            'budget_id' => $this->budget_id,
            'id' => $this->{$attribute}
        ]);
        if (!$account) {
            $this->addError($attribute, 'Invalid account selected.');
        }
    }

    /**
     *  Check if the jar ID is valid
     */
    public function jarIdValid($attribute)
    {
        if (!is_numeric($this->{$attribute}) || !Yii::$app->user->can('editJar', ['jarId' => $this->{$attribute}])) {
            $this->addError($attribute, 'Invalid jar selected.');
        }
    }

}
