<?php
namespace backend\models\form;

use backend\components\helpers\Formatter;
use backend\components\helpers\Frequencies;
use backend\models\db\TimeSlot;
use backend\models\db\TimeSlotBooking;
use backend\models\db\TimeSlotItem;
use backend\models\db\TimeZone;
use backend\models\db\User;
use backend\models\form\common\EndDateField;
use backend\models\form\common\RepeatUntilDateField;
use backend\models\form\common\StartDateField;
use Yii;


/**
 * Add Time Slot form
 */
class AddTimeSlotBooking extends \yii\base\Model
{
    use StartDateField, EndDateField;

    public $startDate;
    public $startTime;
    public $endDate;
    public $endTime;

    public function rules() {
        // validate input
        $validationRules = [
            [['startDate', 'startTime'], 'required'],
        ];
        // sanitize
        $filterRules = [
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'strip_tags'],
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'trim'],
            [['startDate', 'endDate'], 'filter', 'filter' => ['backend\components\helpers\Formatter', 'appDateToPhpDate'], 'when' => function($model, $attribute) {
                return !empty($model->{$attribute});
            }]
        ];

        return array_merge($this->startDateRules(), $this->endDateRules(), $validationRules, $filterRules);
    }

    /**
     * @param $coachId
     * @return TimeSlotItem|null
     */
    public function getMatchingTimeSlotItem($coachId)
    {
        $this->updateEndTime();
        $startTime = Formatter::localDatetimeToUtcDatetime($this->startDate . ' ' . $this->startTime . ':00', 'Y-m-d H:i:s');
        $endTime = Formatter::localDatetimeToUtcDatetime($this->endDate . ' ' . $this->endTime . ':00', 'Y-m-d H:i:s');

        return TimeSlotItem::find()
            ->joinWith('timeSlot')
            ->where(['and',
                ['coach_id' => $coachId],
                ['<=', 'start_at', $startTime],
                ['>=', 'end_at', $endTime]
            ])
            ->one();
    }

    public function checkIfOverlays($userId, $coachId)
    {
        $this->updateEndTime();
        $startTime = Formatter::localDatetimeToUtcDatetime(
            strtotime('+1 second', strtotime($this->startDate . ' ' . $this->startTime . ':00')),
            'Y-m-d H:i:s'
        );
        $endTime = Formatter::localDatetimeToUtcDatetime(
            strtotime('-1 second', strtotime($this->endDate . ' ' . $this->endTime . ':00')),
            'Y-m-d H:i:s'
        );

        return TimeSlotBooking::find()
            ->joinWith('timeSlotItem.timeSlot')
            ->where([
                'coach_id' => $coachId
            ])
            ->andWhere(['or',
                ['between', TimeSlotBooking::tableName() . '.start_at', $startTime, $endTime],
                ['between', TimeSlotBooking::tableName() . '.end_at', $startTime, $endTime]
            ])
            ->exists();
    }

    public function loadFromDb(TimeSlotBooking $slot, TimeZone $timeZone)
    {
        $this->startDate = Formatter::utcDatetimeToLocalDatetime($slot->start_at, $timeZone->code, 'Y-m-d');
        $this->startTime = Formatter::utcDatetimeToLocalDatetime($slot->start_at, $timeZone->code, 'H:i');
        $this->endDate = Formatter::utcDatetimeToLocalDatetime($slot->end_at, $timeZone->code, 'Y-m-d');
        $this->endTime = Formatter::utcDatetimeToLocalDatetime($slot->end_at, $timeZone->code, 'H:i');
    }

    public function updateEndTime()
    {
        $endTimeDate = strtotime(
            '+' . TimeSlotBooking::DEFAULT_BOOKING_MINUTES . ' minutes',
            strtotime($this->startDate . ' ' . $this->startTime . ':00')
        );
        $this->endDate = date('Y-m-d', $endTimeDate);
        $this->endTime = date('H:i', $endTimeDate);
    }

}
