<?php

namespace backend\models\form;

use backend\components\validators\Currency;
use backend\models\form\common\DateField;
use Yii;

use backend\models\db\Account;
use backend\models\db\Transfer;

use backend\components\helpers\Formatter;
use backend\components\validators\DateStart;

/**
 * New jar form
 */
class AddTransfer extends \yii\base\Model
{
    use DateField;

    public $name;
    public $amount;
    public $account_from_id;
    public $account_to_id;
    public $frequency;
    public $date;
    public $budget_id;
    public $transfer_id = -1;

    public function rules()
    {
        // validate input
        $validationRules = [
            ['name', 'required'],
            ['name', 'string', 'min' => 3],
            ['name', 'transferNameValid'],
            ['amount', 'required'],
            ['amount', Currency::className(), 'min' => '0.01'],
            ['account_from_id', 'required', 'message' => 'Please select an account.', 'when' => function($model) {
                return ($model->transfer_id == -1);
            }],
            ['account_from_id', 'integer'],
            ['account_from_id', 'accountIdValid'],
            ['account_to_id', 'required', 'message' => 'Please select an account.', 'when' => function($model) {
                return ($model->transfer_id == -1);
            }],
            ['account_to_id', 'integer'],
            ['date', 'required'],
            ['date', DateStart::className(), 'fieldName' => 'transfer'],
            ['date', 'date', 'format' => Formatter::phpDateCodeToDateCode(Yii::$app->session->get('dateFormat')), 'message' => 'Incorrect date format.'],
            ['frequency', 'required']
        ];

        // sanitize
        $filterRules = [
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'strip_tags'],
            [array_keys($this->getAttributes()), 'filter', 'filter' => 'trim'],
            ['amount', 'filter', 'filter' => ['backend\components\helpers\Formatter', 'currencyToDecimal']],
            ['date', 'filter', 'filter' => ['backend\components\helpers\Formatter', 'appDateToPhpDate'], 'when' => function($model, $attribute) {
                return !empty($model->{$attribute});
            }]
        ];

        return array_merge($this->dateRules(), $validationRules, $filterRules);
    }

    /**
     * Check whether transfer name is allowed.
     * @param string $attribute
     */
    public function transferNameValid($attribute) {

        if ($this->frequency == 'one-time') {

            $transfer = Transfer::find()
                ->leftJoin(Account::tableName(), Account::tableName() . ".id = " . Transfer::tableName() . ".account_from_id")
                ->where([
                    'and',
                    [Transfer::tableName() . '.name' => $this->{$attribute}],
                    [Account::tableName() . '.budget_id' => $this->budget_id],
                    ['!=', Transfer::tableName() . '.id', $this->transfer_id],
                    ['!=', 'frequency', 'one-time']
                ])
                ->one();

        }
        else {

            $transfer = Transfer::find()
                ->leftJoin(Account::tableName(), Account::tableName() . ".id = " . Transfer::tableName() . ".account_from_id")
                ->where([
                    'and',
                    [Transfer::tableName() . '.name' => $this->{$attribute}],
                    [Account::tableName() . '.budget_id' => $this->budget_id],
                    ['!=', Transfer::tableName() . '.id', $this->transfer_id]
                ])
                ->one();

        }

        if ($transfer) {

            $this->addError($attribute, 'Please enter a unique transfer name.');

        }

    }


    /**
     * Check if the account ID can be accessed by this user
     */
    public function accountIdValid($attribute) {

        $account = Account::findOne([
            'budget_id' => $this->budget_id,
            'id' => $this->{$attribute}
        ]);

        if (!$account) {

            $this->addError($attribute, 'Invalid account ID.');

        }

        if ($this->account_from_id == $this->account_to_id) {

            $this->addError($attribute, 'Please choose different "From Account" and "To Account".');

        }

    }

}
