<?php

namespace backend\modules\admin\controllers;

use backend\models\db\Jar;
use backend\models\db\Transaction;
use backend\models\db\UserRole;
use Yii;

use yii\db\Expression;
use yii\filters\AccessControl;

use backend\components\CustomController;

use backend\models\db\Budget;
use backend\models\db\Currency;
use backend\models\db\DebtPayment;
use backend\models\db\Eoms;
use backend\models\db\User;
use backend\models\db\UserLogin;
use backend\models\db\UserMeta;
use backend\models\db\UserParent;
use backend\models\db\UserPlan;
use backend\models\db\UserPlanType;

use yii\helpers\ArrayHelper;
use backend\components\helpers\Formatter;

class DefaultController extends CustomController
{
    protected static $cacheKeys = [
        'admin-coaches' => 'cacheCoaches',
        'admin-coachesCount' => 'cacheCoachCount',
        'admin-coachPlanBreakdown' => 'cacheCoachPlanBreakdown',
        'admin-coachUserCount12Months' => 'cacheCoachUserCount12Months',
        'admin-coachUserCount30Days' => 'cacheCoachUserCount30Days',
        'admin-allActivePlansCount' => 'cacheAllActivePlansCount',
        'admin-allActivePlansIncome' => 'cacheAllActivePlansIncome',
        'admin-planCounts' => 'cachePlanCounts',
        'admin-userCurrencies' => 'cacheUserCurrencies',
        'admin-coachEoms' => 'cacheCoachEoms',
        'admin-usersAssignedOver12Months' => 'cacheUsersAssignedOver12Months',
        'admin-inactiveUsers' => 'cacheInactiveUsers',
        'admin-inactiveUsersByCoach' => 'cacheInactiveUsersByCoach',
        'admin-overspendingUsersByCoach' => 'cacheOverspendingUsersByCoach',
        'admin-recentlyActiveUsersByCoach' => 'cacheRecentlyActiveUsersByCoach',
        'admin-recentLoginPeriods' => 'cacheRecentLoginPeriods',
        'admin-userBirthdaysByCoach' => 'cacheUserBirthdaysByCoach',
        'admin-coachBirthdays' => 'cacheCoachBirthdays',
        'admin-eomsTotalCount' => 'cacheEomsTotalCount',
        'admin-eomsCompletedCount' => 'cacheEomsCompletedCount',
        'admin-eomsPositiveCount' => 'cacheEomsPositiveCount',
        'admin-eomsPositiveAmount' => 'cacheEomsPositiveAmount',
        'admin-eomsNegativeCount' => 'cacheEomsNegativeCount',
        'admin-eomsNegativeAmount' => 'cacheEomsNegativeAmount',
        'admin-eomsAvgPositiveAmount' => 'cacheEomsAvgPositiveAmount',
        'admin-eomsAvgNegativeAmount' => 'cacheEomsAvgNegativeAmount',
        'admin-debtPayments30DaysAmounts' => 'cacheDebtPayments30DaysAmounts',
        'admin-debtPaymentsTotalAmounts' => 'cacheDebtPaymentsTotalAmounts'
    ];
    protected static $cachedValues = [];
    protected static $users = [];
    protected static $coaches = [];
    protected static $userLoginIds = [];
    protected static $budgetIds = [];

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'coach-dashboard',
                            'flush-cache',
                            'overspending',
                            'new-users',
                            'upp-chart',
                            'eoms-users'
                        ],
                        'allow' => true,
                        'roles' => ['superadmin', 'coach'],
                    ],
                    [
                        'actions' => [
                            'admin-dashboard',
                        ],
                        'allow' => true,
                        'roles' => ['superadmin'],
                    ]
                ],
            ]
        ];

    }

    public function actionIndex()
    {
        if ($this->initUserRole == 'coach') {
            return $this->redirect('coach-dashboard');
        }
        else {
            return $this->redirect('admin-dashboard');
        }
    }

    public function actionFlushCache()
    {
        Yii::$app->cache->flush();
        return $this->redirect(['/admin/default/index']);
    }

    public function actionEomsUsers()
    {
        $timeNow = strtotime(date('Y-m-20'));
        $users = User::find()
            ->select(User::tableName() . '.*, um1.value AS first_name, um2.value AS last_name')
            ->joinWith('userMeta AS um1')
            ->joinWith('userMeta AS um2')
            ->joinWith('budgets')
            ->joinWith('budgets.eoms')
            ->where([
                'and',
                [Eoms::tableName() . '.month' => (int) date('m', strtotime('-1 MONTHS', $timeNow))],
                [Eoms::tableName() . '.year' => date('Y', strtotime('-1 MONTHS', $timeNow))],
                [Eoms::tableName() . '.submitted' => 0],
                ['um1.key' => UserMeta::FIRST_NAME],
                ['um2.key' => UserMeta::LAST_NAME]
            ])
            ->orderBy([
                'last_name' => 'ASC',
                'first_name' => 'ASC'
            ]);
        if ($this->user->user_role_id == UserRole::COACH) {
            $users
                ->joinWith('userParent')
                ->andWhere([
                    'user_parent_id' => $this->user->id
                ]);
        }
        $users = $users->all();

        $this->view->title = 'Users with EoMS pending';
        return $this->renderPartial('_eoms-users', [
            'users' => $users
        ]);
    }

    public function actionOverspending($id)
    {
        if ($this->user->user_role_id == UserRole::COACH) {
            $userExists = User::find()
                ->joinWith('userParent')
                ->where([
                    'AND',
                    ['user_parent_id' => $this->user->id],
                    ['user_role_id' => 3],
                    [User::tableName() . '.id' => $id]
                ])
                ->exists();
            if (!$userExists) {
                return $this->accessError();
            }
        }
        $user = User::findOne($id);
        if (!$user) {
            return $this->accessError();
        }

        $overspendingAmounts = [];
        $budgetId = $user->getUserMeta()
            ->select('value')
            ->where('`key` = "active_budget_id"')
            ->scalar();
        $budget = Budget::findOne($budgetId);

        if ($budget) {
            $jars = [];
            $total = 0;
            /** @var Jar $j */
            foreach ($budget->getActiveJars() as $j) {
                $jarBalance = $j->getThisMonthsFundsRemainingAmount();
                if ($jarBalance < 0) {
                    $total += $jarBalance;
                    $jars[] = [
                        'name' => $j->name,
                        'balance' => $jarBalance
                    ];
                }
            }
            $overspendingAmounts = [
                'total' => $total,
                'jars' => $jars
            ];
        }

        $currencyMeta = UserMeta::find()
            ->where([
                'user_id' => $user->id,
                'key' => 'currency_id'
            ])
            ->one();
        $currency = Currency::findOne($currencyMeta ? $currencyMeta->value : Yii::$app->params['defaultCurrencyId']);

        $this->view->title = 'User\'s Overspent Jars';
        return $this->render('overspending', [
            'currency'            => $currency->code,
            'overspendingAmounts' => $overspendingAmounts
        ]);
    }

    public function actionNewUsers()
    {
        $months = (int)Yii::$app->request->getQueryParam('months', 1);


        // gets plans and user counts per plan
        $userPlans = UserPlan::find()
            ->joinWith('user.userParent')
            ->leftJoin(UserPlan::tableName() . ' as up', UserPlan::tableName() . '.user_id = up.user_id AND ' . UserPlan::tableName() . '.start_time > up.start_time')
            ->with('userPlanType')
            ->with('userPlanTypeDuration')
            ->where([
                'AND',
                ['up.start_time' => NULL],
                ['<=', 'DATE(' . UserPlan::tableName() . '.start_time)', date('Y-m-d')]
            ]);

        if ($this->user->user_role_id == UserRole::COACH) {
            $userPlans->andWhere([
                'user_parent_id' => $this->user->id
            ]);
        }

        $totalCount = 0;
        $lastYearStartTime = strtotime("-$months MONTHS");
        $planCounts = [];

        /** @var $u UserPlan */
        foreach ($userPlans->all() as $u) {
            $activationTime = strtotime($u->start_time);
            if (!$u->getIsExpired()) {
                if ($activationTime > $lastYearStartTime) {
                    $totalCount++;
                    $planCounts[$u->userPlanType->name][] = $u->user;
                }
            }
        }

        return $this->renderPartial('_new-users', [
            'newUsers' => $totalCount,
            'planCounts' => $planCounts
        ]);
    }

    public function actionUppChart()
    {
        if (Yii::$app->request->getQueryParam('data')) {
            // gets current plans and user counts per plan
            $userPlans = UserPlan::find()
                ->leftJoin(UserPlan::tableName() . ' as up', UserPlan::tableName() . '.user_id = up.user_id AND ' . UserPlan::tableName() . '.start_time > up.start_time')
                ->joinWith('user.userParent')
                ->with('userPlanTypeDuration')
                ->where([
                    'AND',
                    ['up.start_time' => NULL],
                    ['<=', 'DATE(' . UserPlan::tableName() . '.start_time)', date('Y-m-d')],
                    ['is not', UserPlan::tableName() . '.user_plan_variation_id', null]
                ]);

            if ($this->user->user_role_id == UserRole::COACH) {
                $userPlans->andWhere([
                    'user_parent_id' => $this->user->id
                ]);
            }
            $userPlans = $userPlans->all();

            if (count($userPlans)) {
                $labels = [];
                $values = [];

                foreach ($userPlans as $up) {
                    $planKey = $up->userPlanType->name;
                    $labels[$planKey] = $planKey;
                    $values[$planKey]++;
                }
                if (array_sum($values)) {
                    $response = [
                        'type' => 'Pie',
                        'labels' => array_values($labels),
                        'series' => array_values($values),
                    ];
                } else {
                    $response = false;
                }
            } else {
                $response = false;
            }
            return json_encode($response);
        }
        else {
            $this->view->title = 'Users Per Plan';
            return $this->renderPartial('_upp-chart');
        }
    }

    public function actionCoachDashboard()
    {
        $coachKey = 'coach' . $this->user->id . '-';
        $planCounts = Yii::$app->cache->get("{$coachKey}planCounts");
        if (!$planCounts || Yii::$app->request->getQueryParam('force')) {
            set_time_limit(7200);
            ini_set('memory_limit', '2G');
            // refresh the data
            self::calculateCoachDashboard($this->user->id);
            if (Yii::$app->request->getQueryParam('force')) {
                return $this->redirect(['/admin/default/coach-dashboard']);
            }
        }

        $this->view->title = 'Coach Dashboard';
        return $this->render('coach-dashboard', [
            'planCounts' => Yii::$app->cache->get("{$coachKey}planCounts"),
            'allActivePlansCount' => Yii::$app->cache->get("{$coachKey}allActivePlansCount"),
            'usersAssignedOver12Months' => Yii::$app->cache->get("{$coachKey}usersAssignedOver12Months"),
            'inactiveUsers' => Yii::$app->cache->get("{$coachKey}inactiveUsers"),
            'userCurrencies' => Yii::$app->cache->get("{$coachKey}userCurrencies"),
            'overspendingUsers' => Yii::$app->cache->get("{$coachKey}overspendingUsers"),
            'recentlyActiveUsers' => Yii::$app->cache->get("{$coachKey}recentlyActiveUsers"),
            'recentLoginPeriods' => Yii::$app->cache->get("{$coachKey}recentLoginPeriods"),
            'birthdays' => Yii::$app->cache->get("{$coachKey}birthdays"),
            'eomsTotalCount' => Yii::$app->cache->get("{$coachKey}eomsTotalCount"),
            'eomsCompletedCount' => Yii::$app->cache->get("{$coachKey}eomsCompletedCount"),
            'eomsPositiveCount' => Yii::$app->cache->get("{$coachKey}eomsPositiveCount"),
            'eomsPositiveAmount' => Yii::$app->cache->get("{$coachKey}eomsPositiveAmount"),
            'eomsPositiveUserCounts' => Yii::$app->cache->get("{$coachKey}eomsPositiveUserCounts"),
            'eomsAvgPositiveAmount' => Yii::$app->cache->get("{$coachKey}eomsAvgPositiveAmount"),
            'eomsNegativeCount' => Yii::$app->cache->get("{$coachKey}eomsNegativeCount"),
            'eomsNegativeAmount' => Yii::$app->cache->get("{$coachKey}eomsNegativeAmount"),
            'eomsNegativeUserCounts' => Yii::$app->cache->get("{$coachKey}eomsNegativeUserCounts"),
            'eomsAvgNegativeAmount' => Yii::$app->cache->get("{$coachKey}eomsAvgNegativeAmount"),
            'debtPayments30DaysAmounts' => Yii::$app->cache->get("{$coachKey}debtPayments30DaysAmounts"),
            'debtPaymentsTotalAmounts' => Yii::$app->cache->get("{$coachKey}debtPaymentsTotalAmounts")
        ]);
    }


    public function actionAdminDashboard()
    {
        //$coachesCount = Yii::$app->cache->get('admin-coachesCount');
        //if (!$coachesCount || Yii::$app->request->getQueryParam('force')) {
            //set_time_limit(7200);
            //ini_set('memory_limit', '2G');
            // refresh the data
            self::calculateAdminDashboard();
            if (Yii::$app->request->getQueryParam('force')) {
                return $this->redirect(['/admin/default/admin-dashboard']);
            }
        //}

        $this->view->title = 'Admin Dashboard';
        return $this->render('admin-dashboard', [
            'coaches' => Yii::$app->cache->get('admin-coaches'),
            'coachesCount' => Yii::$app->cache->get('admin-coachesCount'),
            'coachPlanBreakdown' => Yii::$app->cache->get('admin-coachPlanBreakdown'),
            'coachUserCount12Months' => Yii::$app->cache->get('admin-coachUserCount12Months'),
            'coachUserCount30Days' => Yii::$app->cache->get('admin-coachUserCount30Days'),
            'coachEoms' => Yii::$app->cache->get('admin-coachEoms'),
            'planCounts' => Yii::$app->cache->get('admin-planCounts'),
            'allActivePlansCount' => Yii::$app->cache->get('admin-allActivePlansCount'),
            'allActivePlansIncome' => Yii::$app->cache->get('admin-allActivePlansIncome'),
            'usersAssignedOver12Months' => Yii::$app->cache->get('admin-usersAssignedOver12Months'),
            'inactiveUsers' => Yii::$app->cache->get('admin-inactiveUsers'),
            'inactiveUsersByCoach' => Yii::$app->cache->get('admin-inactiveUsersByCoach'),
            'userCurrencies' => Yii::$app->cache->get('admin-userCurrencies'),
            'overspendingUsersByCoach' => Yii::$app->cache->get('admin-overspendingUsersByCoach'),
            'recentlyActiveUsersByCoach' => Yii::$app->cache->get('admin-recentlyActiveUsersByCoach'),
            'recentLoginPeriods' => Yii::$app->cache->get('admin-recentLoginPeriods'),
            'userBirthdaysByCoach' => Yii::$app->cache->get('admin-userBirthdaysByCoach'),
            'coachBirthdays' => Yii::$app->cache->get('admin-coachBirthdays'),
            'eomsTotalCount' => Yii::$app->cache->get('admin-eomsTotalCount'),
            'eomsCompletedCount' => Yii::$app->cache->get('admin-eomsCompletedCount'),
            'eomsPositiveCount' => Yii::$app->cache->get('admin-eomsPositiveCount'),
            'eomsPositiveAmount' => Yii::$app->cache->get('admin-eomsPositiveAmount'),
            'eomsNegativeCount' => Yii::$app->cache->get('admin-eomsNegativeCount'),
            'eomsNegativeAmount' => Yii::$app->cache->get('admin-eomsNegativeAmount'),
            'eomsAvgPositiveAmount' => Yii::$app->cache->get('admin-eomsAvgPositiveAmount'),
            'eomsAvgNegativeAmount' => Yii::$app->cache->get('admin-eomsAvgNegativeAmount'),
            'debtPayments30DaysAmounts' => Yii::$app->cache->get('admin-debtPayments30DaysAmounts'),
            'debtPaymentsTotalAmounts' => Yii::$app->cache->get('admin-debtPaymentsTotalAmounts'),
        ]);
    }

    
    protected static function cacheCoaches()
    {
        self::$cachedValues['cacheCoaches'] = [];
        foreach (self::$coaches as $c) {
            self::$cachedValues['cacheCoaches'][$c->id] = $c;
        }
        return self::$cachedValues['cacheCoaches'];
    }

    protected static function cacheCoachCount()
    {
        return count(self::$coaches);
    }

    protected static function cacheCoachPlanBreakdown()
    {
        self::$cachedValues['cacheAllActivePlansCount'] = 0;
        self::$cachedValues['cacheAllActivePlansIncome'] = 0;
        self::$cachedValues['cacheCoachPlanBreakdown'] = []; // for coaches breakdown
        self::$cachedValues['cachePlanCounts'] = [];

        $lastYearStartTime = strtotime('-1 YEARS');
        $thisMonthStartTime = strtotime(date('Y-m-01 00:00:00'));
        $thisMonthEndTime = time(); // now
        $timeNow = strtotime(date('Y-m-20'));
        $lastMonthStartTime = strtotime(date('Y-m-01 00:00:00', strtotime('-1 month', $timeNow)));
        $lastMonthEndTime = strtotime(date('Y-m-t 00:00:00', strtotime('-1 month', $timeNow)));

        // gets plans and user counts per plan
        $userPlans = UserPlan::find()
            ->joinWith('user.userParent')
            ->leftJoin(UserPlan::tableName() . ' as up', UserPlan::tableName() . '.user_id = up.user_id AND ' . UserPlan::tableName() . '.start_time > up.start_time')
            ->with('userPlanType')
            ->with('userPlanTypeDuration')
            ->where([
                'AND',
                ['up.start_time' => NULL],
                ['<=', 'DATE(' . UserPlan::tableName() . '.start_time)', date('Y-m-d')],
                ['is not', UserPlan::tableName() . '.user_plan_variation_id', null]
            ])
            ->all();
        /** @var $u UserPlan */
        foreach ($userPlans as $u) {
            $activationTime = strtotime($u->start_time);
            if (!$u || !$u->user || !$u->user->userParent) {
                continue;
            }
            $upId = $u->user->userParent->user_parent_id;
            if (!isset(self::$cachedValues['cacheCoachPlanBreakdown'][$upId]['inactive_users'])) {
                self::$cachedValues['cacheCoachPlanBreakdown'][$upId]['inactive_users'] = 0;
            }

            // general user plan statistics
            if (!$u->getIsExpired()) {
                $planIndex = $u->userPlanType->name;
                $planCost = $u->userPlanVariation->getPricePerMonth();

                self::$cachedValues['cacheAllActivePlansCount']++;
                self::$cachedValues['cacheAllActivePlansIncome'] += $planCost;
                if (!isset(self::$cachedValues['cachePlanCounts'][$planIndex])) {
                    self::$cachedValues['cachePlanCounts'][$planIndex] = [
                        'total' => 0,
                        'income' => 0,
                        'prevMonthSub' => 0,
                        'thisMonthSub' => 0
                    ];
                }
                self::$cachedValues['cachePlanCounts'][$planIndex]['total']++;
                self::$cachedValues['cachePlanCounts'][$planIndex]['income'] += $planCost;

                if ($activationTime >= $lastMonthStartTime && $activationTime <= $lastMonthEndTime) {
                    self::$cachedValues['cachePlanCounts'][$planIndex]['prevMonthSub']++;
                }
                elseif ($activationTime >= $thisMonthStartTime && $activationTime <= $thisMonthEndTime) {
                    self::$cachedValues['cachePlanCounts'][$planIndex]['thisMonthSub']++;
                }

                // coach user plan statistics
                if ($activationTime >= $lastYearStartTime) {
                    if (!isset(self::$cachedValues['cacheCoachUserCount12Months'][$upId])) {
                        self::$cachedValues['cacheCoachUserCount12Months'][$upId] = 0;
                    }
                    self::$cachedValues['cacheCoachUserCount12Months'][$upId]++;
                }
                if ($activationTime >= strtotime('-30 days')) {
                    if (!isset(self::$cachedValues['cacheCoachUserCount30Days'][$upId])) {
                        self::$cachedValues['cacheCoachUserCount30Days'][$upId] = 0;
                    }
                    self::$cachedValues['cacheCoachUserCount30Days'][$upId]++;
                }

                if (!isset(self::$cachedValues['cacheCoachPlanBreakdown'][$upId]['income'])) {
                    self::$cachedValues['cacheCoachPlanBreakdown'][$upId]['income'] = 0;
                }
                if (!isset(self::$cachedValues['cacheCoachPlanBreakdown'][$upId]['plans'][$planIndex])) {
                    self::$cachedValues['cacheCoachPlanBreakdown'][$upId]['plans'][$planIndex] = 0;
                }

                self::$cachedValues['cacheCoachPlanBreakdown'][$upId]['plans'][$planIndex]++;
                self::$cachedValues['cacheCoachPlanBreakdown'][$upId]['income'] += $planCost;
            }
            else {
                self::$cachedValues['cacheCoachPlanBreakdown'][$upId]['inactive_users']++;
            }
        }
        return self::$cachedValues['cacheCoachPlanBreakdown'];
    }

    protected static function cacheCoachUserCount12Months()
    {
        return self::$cachedValues['cacheCoachUserCount12Months'];
    }

    protected static function cacheCoachUserCount30Days()
    {
        return self::$cachedValues['cacheCoachUserCount30Days'];
    }

    protected static function cacheAllActivePlansCount()
    {
        return self::$cachedValues['cacheAllActivePlansCount'];
    }

    protected static function cacheAllActivePlansIncome()
    {
        return self::$cachedValues['cacheAllActivePlansIncome'];
    }

    protected static function cachePlanCounts()
    {
        return self::$cachedValues['cachePlanCounts'];
    }

    protected static function cacheUsersAssignedOver12Months()
    {
        // get user assignments over last 12 months
        return UserParent::find()
            ->select('COUNT(user_id)')
            ->where(['>=', 'DATE(time)', date('Y-m-d', strtotime('-1 YEARS'))])
            ->scalar();
    }

    protected static function cacheCoachEoms()
    {
        self::$cachedValues['cacheEomsTotalCount'] = 0;
        self::$cachedValues['cacheEomsCompletedCount'] = 0;
        self::$cachedValues['cacheEomsPositiveCount'] = 0;
        self::$cachedValues['cacheEomsNegativeCount'] = 0;
        self::$cachedValues['cacheEomsPositiveAmount'] = [];
        self::$cachedValues['cacheEomsNegativeAmount'] = [];
        self::$cachedValues['cacheEomsPositiveUserCounts'] = [];
        self::$cachedValues['cacheEomsNegativeUserCounts'] = [];
        self::$cachedValues['cacheEomsAvgPositiveAmount'] = [];
        self::$cachedValues['cacheEomsAvgNegativeAmount'] = [];
        self::$cachedValues['cacheCoachEoms'] = [];
        self::$cachedValues['cacheUserCurrencies'] = [];

        $timeNow = strtotime(date('Y-m-20'));
        // EoMS
        // completed EoMS
        $eoms = Eoms::find()
            ->select(Eoms::tableName() . '.*, up.user_id')
            ->joinWith('budget.user.userParent as up')
            ->where([
                'and',
                ['month' => (int) date('m', strtotime('-1 MONTHS', $timeNow))],
                ['year' => date('Y', strtotime('-1 MONTHS', $timeNow))]
            ])
            ->all();

        /** @var Eoms $e */
        foreach ($eoms as $e) {
            // coach stats
            $coachId = $e->budget->user->userParent->user_parent_id;
            if (!isset(self::$cachedValues['cacheCoachEoms'][$coachId])) {
                self::$cachedValues['cacheCoachEoms'][$coachId] = [
                    'eomsTotalCount' => 0,
                    'eomsCompletedCount' => 0,
                    'eomsPositiveCount' => 0,
                    'eomsPositiveAmount' => [],
                    'eomsPositiveUserCounts' => [],
                    'eomsAvgPositiveAmount' => [],
                    'eomsNegativeCount' => 0,
                    'eomsNegativeAmount' => [],
                    'eomsNegativeUserCounts' => [],
                    'eomsAvgNegativeAmount' => []
                ];
            }

            self::$cachedValues['cacheCoachEoms'][$coachId]['eomsTotalCount']++;
            self::$cachedValues['cacheEomsTotalCount']++;

            if ($e->submitted == 1) {
                // general stats & coach stats
                self::$cachedValues['cacheEomsCompletedCount']++;
                self::$cachedValues['cacheCoachEoms'][$coachId]['eomsCompletedCount']++;
                $eomsAmount = $e->getSummaryAmount();
                if ($eomsAmount >= 0) {
                    self::$cachedValues['cacheEomsPositiveCount']++;
                    self::$cachedValues['cacheCoachEoms'][$coachId]['eomsPositiveCount']++;
                    if (!isset(self::$cachedValues['cacheEomsPositiveAmount'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]])) {
                        self::$cachedValues['cacheEomsPositiveAmount'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]] = 0;
                    }
                    if (!isset(self::$cachedValues['cacheCoachEoms'][$coachId]['eomsPositiveAmount'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]])) {
                        self::$cachedValues['cacheCoachEoms'][$coachId]['eomsPositiveAmount'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]] = 0;
                    }
                    self::$cachedValues['cacheEomsPositiveAmount'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]] += $eomsAmount;
                    self::$cachedValues['cacheEomsPositiveUserCounts'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]][$e->user_id] = $e->user_id;
                    self::$cachedValues['cacheCoachEoms'][$coachId]['eomsPositiveAmount'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]] += $eomsAmount;
                    self::$cachedValues['cacheCoachEoms'][$coachId]['eomsPositiveUserCounts'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]][$e->user_id] = $e->user_id;
                }
                else {
                    self::$cachedValues['cacheEomsNegativeCount']++;
                    self::$cachedValues['cacheCoachEoms'][$coachId]['eomsNegativeCount']++;
                    if (!isset(self::$cachedValues['cacheEomsNegativeAmount'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]])) {
                        self::$cachedValues['cacheEomsNegativeAmount'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]] = 0;
                    }
                    if (!isset(self::$cachedValues['cacheCoachEoms'][$coachId]['eomsNegativeAmount'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]])) {
                        self::$cachedValues['cacheCoachEoms'][$coachId]['eomsNegativeAmount'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]] = 0;
                    }
                    self::$cachedValues['cacheEomsNegativeAmount'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]] += $eomsAmount;
                    self::$cachedValues['cacheEomsNegativeUserCounts'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]][$e->user_id] = $e->user_id;
                    self::$cachedValues['cacheCoachEoms'][$coachId]['eomsNegativeAmount'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]] += $eomsAmount;
                    self::$cachedValues['cacheCoachEoms'][$coachId]['eomsNegativeUserCounts'][self::$cachedValues['cacheUserCurrencies'][$e->user_id]][$e->user_id] = $e->user_id;
                }
            }

        }
        foreach (self::$cachedValues['cacheEomsPositiveAmount'] as $currency => $amount) {
            self::$cachedValues['cacheEomsAvgPositiveAmount'][$currency] = empty(self::$cachedValues['cacheEomsPositiveUserCounts'][$currency]) ?
                0 :
                $amount / count(self::$cachedValues['cacheEomsPositiveUserCounts'][$currency]);
        }
        foreach (self::$cachedValues['cacheEomsNegativeAmount'] as $currency => $amount) {
            self::$cachedValues['cacheEomsAvgNegativeAmount'][$currency] = empty(self::$cachedValues['cacheEomsNegativeUserCounts'][$currency]) ?
                0 :
                $amount / count(self::$cachedValues['cacheEomsNegativeUserCounts'][$currency]);
        }
        foreach (self::$cachedValues['cacheCoachEoms'] as $coachId => $data) {
            foreach ($data['eomsPositiveAmount'] as $currency => $amount) {
                self::$cachedValues['cacheCoachEoms'][$coachId]['eomsAvgPositiveAmount'][$currency] = empty(self::$cachedValues['cacheCoachEoms'][$coachId]['eomsPositiveUserCounts'][$currency]) ?
                    0 :
                    $amount / count(self::$cachedValues['cacheCoachEoms'][$coachId]['eomsPositiveUserCounts'][$currency]);
            }
            foreach ($data['eomsNegativeAmount'] as $currency => $amount) {
                self::$cachedValues['cacheCoachEoms'][$coachId]['eomsAvgNegativeAmount'][$currency] = empty(self::$cachedValues['cacheCoachEoms'][$coachId]['eomsNegativeUserCounts'][$currency]) ?
                    0 :
                    $amount / count(self::$cachedValues['cacheCoachEoms'][$coachId]['eomsNegativeUserCounts'][$currency]);
            }
        }

        return self::$cachedValues['cacheCoachEoms'];
    }

    protected static function cacheEomsTotalCount()
    {
        return self::$cachedValues['cacheEomsTotalCount'];
    }

    protected static function cacheEomsCompletedCount()
    {
        return self::$cachedValues['cacheEomsCompletedCount'];
    }

    protected static function cacheEomsPositiveCount()
    {
        return self::$cachedValues['cacheEomsPositiveCount'];
    }

    protected static function cacheEomsPositiveAmount()
    {
        return self::$cachedValues['cacheEomsPositiveAmount'];
    }

    protected static function cacheEomsNegativeCount()
    {
        return self::$cachedValues['cacheEomsNegativeCount'];
    }

    protected static function cacheEomsNegativeAmount()
    {
        return self::$cachedValues['cacheEomsNegativeAmount'];
    }

    protected static function cacheEomsAvgPositiveAmount()
    {
        return self::$cachedValues['cacheEomsAvgPositiveAmount'];
    }

    protected static function cacheEomsAvgNegativeAmount()
    {
        return self::$cachedValues['cacheEomsAvgNegativeAmount'];
    }

    protected static function cacheCoachBirthdays()
    {
        // get coach birthdays
        return UserMeta::find()
            ->joinWith('user')
            ->where([
                'AND',
                ['key' => 'birth_date'],
                ['!=', 'value', ''],
                ['user_role_id' => 2],
                ['BETWEEN', UserMeta::DATE_FIELD_DECODE, date('Y-m-d'), date('Y-m-d', strtotime('+1 MONTH'))]
            ])
            ->all();
    }

    protected static function cacheInactiveUsers()
    {
        self::$cachedValues['cacheInactiveUsers'] = [];
        self::$cachedValues['cacheInactiveUsersByCoach'] = [];

        $inactiveUsersList = User::find()
            ->select(User::tableName() . '.*, ul2.time as login_time')
            ->leftJoin(UserLogin::tableName() . ' as ul2', User::tableName() . '.id = ul2.user_id')
            ->where([
                'AND',
                ['ul2.id' => self::$userLoginIds],
                ['<=', 'DATE(ul2.time)', date('Y-m-d', strtotime('-7 days'))]
            ])
            ->all();
        foreach ($inactiveUsersList as $r) {
            $r->seconds_ago = ceil((time() - strtotime($r->login_time)) / 86400);
            self::$cachedValues['cacheInactiveUsers'][$r->id] = $r;
            if (!$r || !$r->userParent) {
                continue;
            }
            $coach = $r->getCoach();
            if ($coach) {
                // hack for subclients
                if ($coach instanceof UserParent) {
                    $coach = $coach->user->getCoach();
                }
                self::$cachedValues['cacheInactiveUsersByCoach'][$coach->id][] = $r;
            }
        }
    }

    protected static function cacheInactiveUsersByCoach()
    {
        // sort by last activity time
        foreach (self::$cachedValues['cacheInactiveUsersByCoach'] as $coachId => $users) {
            usort($users, function($a, $b) {
                return $b->seconds_ago <=> $a->seconds_ago;
            });
            self::$cachedValues['cacheInactiveUsersByCoach'][$coachId] = $users;
        }
        return self::$cachedValues['cacheInactiveUsersByCoach'];
    }

    protected static function cacheUserBirthdaysByCoach()
    {
        // get user birthdays
        $userBirthdays = UserMeta::find()
            ->joinWith('user')
            ->where([
                'AND',
                ['key' => 'birth_date'],
                ['!=', 'value', ''],
                ['user_role_id' => [3, 4]],
                ['BETWEEN', UserMeta::DATE_FIELD_DECODE, date('Y-m-d'), date('Y-m-d', strtotime('+1 MONTH'))]
            ])
            ->all();
        $userBirthdaysByCoach = [];
        foreach ($userBirthdays as $r) {
            $coach = $r->user->getCoach();
            if ($coach) {
                // hack for subclients
                if ($coach instanceof UserParent) {
                    $coach = $coach->user->getCoach();
                }
                $userBirthdaysByCoach[$coach->id][] = $r;
            }
            else {
                // very unlikely that this will happen
                $userBirthdaysByCoach[0][] = $r;
            }
        }
        return $userBirthdaysByCoach;
    }

    protected static function cacheUserCurrencies()
    {
        $userIds = ArrayHelper::map(self::$users, 'id', 'id');

        // we'll also need user currencies
        $currencyMeta = UserMeta::find()
            ->where([
                'and',
                ['user_id' => $userIds],
                ['key' => 'currency_id']
            ])
            ->all();
        $currencyMetaSorted = ArrayHelper::map($currencyMeta, 'user_id', 'value');

        $currencies = Currency::find()->all();
        $currenciesSorted = ArrayHelper::map($currencies, 'id', 'code');

        foreach ($userIds as $u) {
            $currencyId = isset($currencyMetaSorted[$u]) ? $currencyMetaSorted[$u] : Yii::$app->params['defaultCurrencyId'];
            self::$cachedValues['cacheUserCurrencies'][$u] = $currenciesSorted[$currencyId];
        }
        return self::$cachedValues['cacheUserCurrencies'];
    }

    protected static function cacheOverspendingUsersByCoach()
    {
        self::$cachedValues['cacheOverspendingUsersByCoach'] = [];
        foreach (self::$users as $u) {
            $budgetId = $u->getUserMeta()
                ->select('value')
                ->where('`key` = "active_budget_id"')
                ->scalar();
            $budget = Budget::findOne($budgetId);

            if ($budget) {
                $balance = 0;
                /** @var Jar $j */
                foreach ($budget->getActiveJars() as $j) {
                    $jarBalance = $j->getThisMonthsFundsRemainingAmount();
                    if ($jarBalance < 0) {
                        $balance += $jarBalance;
                    }
                }

                if ($balance < 0) {
                    $coach = $u->getCoach();
                    if ($coach) {
                        // hack for subclients
                        if ($coach instanceof UserParent) {
                            $coach = $coach->user->getCoach();
                        }
                        self::$cachedValues['cacheOverspendingUsersByCoach'][$coach->id][] = [
                            'user' => $u,
                            'total' => $balance
                        ];
                    } else {
                        // very unlikely that this will happen
                        self::$cachedValues['cacheOverspendingUsersByCoach'][0][] = [
                            'user' => $u,
                            'total' => $balance
                        ];
                    }
                }
            }
        }
        // sort by overspent amount
        foreach (self::$cachedValues['cacheOverspendingUsersByCoach'] as $coachId => $users) {
            usort($users, function($a, $b) {
                return $a['total'] <=> $b['total'];
            });
            self::$cachedValues['cacheOverspendingUsersByCoach'][$coachId] = $users;
        }
        return self::$cachedValues['cacheOverspendingUsersByCoach'];
    }

    protected static function cacheDebtPayments30DaysAmounts()
    {
        self::$cachedValues['cacheDebtPayments30DaysAmounts'] = [];
        // debt amount paid back by all
        $relativeDate = date('Y-m-d', strtotime('-30 days'));
        $debtPayments30Days = DebtPayment::find()
            ->select(DebtPayment::tableName() . '.*')
            ->joinWith('debt d')
            ->where(['and',
                ['budget_id' => self::$budgetIds],
                ['is_adjustment' => 1],
                ['>=', DebtPayment::tableName() . '.date', $relativeDate]
            ])
            ->all();

        $budgetToUser = array_flip(self::$budgetIds);
        foreach ($debtPayments30Days as $d) {
            $currencyKey = self::$cachedValues['cacheUserCurrencies'][$budgetToUser[$d->debt->budget_id]];
            $debtTypeKey = $d->debt->debtType->name;

            if (!isset(self::$cachedValues['cacheDebtPayments30DaysAmounts'][$currencyKey]['total'])) {
                self::$cachedValues['cacheDebtPayments30DaysAmounts'][$currencyKey]['total'] = 0;
            }
            if (!isset(self::$cachedValues['cacheDebtPayments30DaysAmounts'][$currencyKey][$debtTypeKey])) {
                self::$cachedValues['cacheDebtPayments30DaysAmounts'][$currencyKey][$debtTypeKey] = 0;
            }
            self::$cachedValues['cacheDebtPayments30DaysAmounts'][$currencyKey][$debtTypeKey] += $d->amount;
            self::$cachedValues['cacheDebtPayments30DaysAmounts'][$currencyKey]['total'] += $d->amount;
        }
        return self::$cachedValues['cacheDebtPayments30DaysAmounts'];
    }

    protected static function cacheDebtPaymentsTotalAmounts()
    {
        self::$cachedValues['cacheDebtPaymentsTotalAmounts'] = [];
        // debt amount paid back by all
        $debtPaymentsTotal = DebtPayment::find()
            ->select(DebtPayment::tableName() . '.*')
            ->joinWith('debt d')
            ->where([
                'budget_id' => self::$budgetIds,
                'is_adjustment' => 1
            ])
            ->all();

        $budgetToUser = array_flip(self::$budgetIds);
        foreach ($debtPaymentsTotal as $d) {
            $currencyKey = self::$cachedValues['cacheUserCurrencies'][$budgetToUser[$d->debt->budget_id]];
            $debtTypeKey = $d->debt->debtType->name;

            if (!isset(self::$cachedValues['cacheDebtPaymentsTotalAmounts'][$currencyKey]['total'])) {
                self::$cachedValues['cacheDebtPaymentsTotalAmounts'][$currencyKey]['total'] = 0;
            }
            if (!isset(self::$cachedValues['cacheDebtPaymentsTotalAmounts'][$currencyKey][$debtTypeKey])) {
                self::$cachedValues['cacheDebtPaymentsTotalAmounts'][$currencyKey][$debtTypeKey] = 0;
            }
            self::$cachedValues['cacheDebtPaymentsTotalAmounts'][$currencyKey][$debtTypeKey] += $d->amount;
            self::$cachedValues['cacheDebtPaymentsTotalAmounts'][$currencyKey]['total'] += $d->amount;
        }
        return self::$cachedValues['cacheDebtPaymentsTotalAmounts'];
    }

    protected static function cacheRecentlyActiveUsersByCoach()
    {
        self::$cachedValues['cacheRecentlyActiveUsersByCoach'] = [];
        self::$cachedValues['cacheRecentLoginPeriods'] = [];

        $recentlyActiveUsers = User::find()
            ->select(User::tableName() . '.*, ul2.time as login_time, duration as login_duration')
            ->leftJoin(UserLogin::tableName() . ' as ul2', User::tableName() . '.id = ul2.user_id')
            ->where([
                'AND',
                ['ul2.id' => $userLoginIds],
                ['>=', new Expression('`ul2`.`time` + INTERVAL `ul2`.`duration` SECOND'), date_format(date_create('-6 HOURS', timezone_open('UTC')), 'Y-m-d H:i')],
                //['!=', 'ul2.user_id', Yii::$app->user->id]
            ])
            ->all();

        foreach ($recentlyActiveUsers as $r) {
            $r->seconds_ago = time() - strtotime(Formatter::datetime($r->login_time, 'UTC', 'Y-m-d', 'php:H:i:s')) - $r->login_duration;
            self::$cachedValues['cacheRecentLoginPeriods'][$r->id] = Formatter::period($r->seconds_ago);
            if (!$r || !$r->userParent) {
                continue;
            }
            $coach = $r->getCoach();
            if ($coach) {
                // hack for subclients
                if ($coach instanceof UserParent) {
                    $coach = $coach->user->getCoach();
                }
                self::$cachedValues['cacheRecentlyActiveUsersByCoach'][$coach->id][] = $r;
            }
        }
        // sort by last activity time
        foreach (self::$cachedValues['cacheRecentlyActiveUsersByCoach'] as $coachId => $users) {
            usort($users, function($a, $b) {
                return $a->seconds_ago <=> $b->seconds_ago;
            });
            self::$cachedValues['cacheRecentlyActiveUsersByCoach'][$coachId] = $users;
        }
        return self::$cachedValues['cacheRecentlyActiveUsersByCoach'];
    }

    protected static function cacheRecentLoginPeriods()
    {
        return self::$cachedValues['cacheRecentLoginPeriods'];
    }
    
    public static function calculateAdminDashboard()
    {
        self::$users = User::find()
            ->joinWith('userParent')
            ->where(['user_role_id' => 3])
            ->all();
        foreach (self::$users as $u) {
            $budgetId = $u->getUserMeta()
                ->select('value')
                ->where('`key` = "active_budget_id"')
                ->scalar();
            $budget = Budget::findOne($budgetId);
            if ($budget) {
                self::$budgetIds[$u->id] = $budgetId;
            }
        }
        self::$coaches = User::findAll(['user_role_id' => 2]);
        // get inactive & recently active users
        $userLoginCommand = Yii::$app->getDb()->createCommand("
            SELECT 
                id
            FROM 
                user_logins
            WHERE
                id IN (
                    SELECT MAX(id)
                    FROM user_logins
                    GROUP BY user_id
                )
            ;
        ");
        foreach ($userLoginCommand->queryAll() as $ul) {
            self::$userLoginIds[$ul['id']] = $ul['id'];
        }

        foreach (self::$cacheKeys as $keyName => $function) {
            if (!Yii::$app->cache->exists($keyName)) {
                Yii::$app->cache->set($keyName, call_user_func("self::{$function}"));
            } else {
                self::$cachedValues[$function] = Yii::$app->cache->get($keyName);
            }
        }
    }

    public static function calculateCoachDashboard($coachId)
    {
        $coachKey = 'coach' . $coachId . '-';

        // gets current plans and user counts per plan
        $userPlans = UserPlan::find()
            ->leftJoin(UserPlan::tableName() . ' as up', UserPlan::tableName() . '.user_id = up.user_id AND ' . UserPlan::tableName() . '.start_time > up.start_time')
            ->joinWith('user.userParent')
            ->with('userPlanTypeDuration')
            ->where([
                'AND',
                ['up.start_time' => NULL],
                ['<=', 'DATE(' . UserPlan::tableName() . '.start_time)', date('Y-m-d')],
                ['is not', UserPlan::tableName() . '.user_plan_variation_id', null],
                ['user_parent_id' => $coachId]
            ])
            ->all();

        $planCounts = [];
        $allActivePlansCount = 0;

        $thisMonthStartTime = strtotime(date('Y-m-01 00:00:00'));
        $thisMonthEndTime = time(); // now
        $timeNow = strtotime(date('Y-m-20'));
        $lastMonthStartTime = strtotime(date('Y-m-01 00:00:00', strtotime('-1 month', $timeNow)));
        $lastMonthEndTime = strtotime(date('Y-m-t 00:00:00', strtotime('-1 month', $timeNow)));

        /** @var $u UserPlan */
        foreach ($userPlans as $u) {
            $activationTime = strtotime($u->start_time);

            if (!$u->getIsExpired()) {
                $planIndex = $u->userPlanType->name;

                if (!isset($planCounts[$planIndex])) {
                    $planCounts[$planIndex] = [
                        'total' => 0,
                        'prevMonthSub' => 0,
                        'thisMonthSub' => 0
                    ];
                }
                $planCounts[$planIndex]['total']++;
                $allActivePlansCount++;
                if ($activationTime >= $lastMonthStartTime && $activationTime <= $lastMonthEndTime) {
                    $planCounts[$planIndex]['prevMonthSub']++;
                }
                elseif ($activationTime >= $thisMonthStartTime && $activationTime <= $thisMonthEndTime) {
                    $planCounts[$planIndex]['thisMonthSub']++;
                }
            }
        }

        // get user assignments over last 12 months
        $usersAssignedOver12Months = UserParent::find()
            ->select('COUNT(DISTINCT user_id)')
            ->where([
                'and',
                ['user_parent_id' => $coachId],
                ['>=', 'DATE(time)', date('Y-m-d', strtotime('-1 YEAR'))]
            ])
            ->scalar();

        // get inactive & recently active users
        $userLoginQuery = UserLogin::find()
            ->select(UserLogin::tableName() . '.id')
            ->leftJoin(UserLogin::tableName() . ' as ul', UserLogin::tableName() . '.user_id = ul.user_id AND '
                . UserLogin::tableName() . '.time < ul.time')
            ->where(['ul.time' => NULL]);

        $inactiveUsers = User::find()
            ->select(User::tableName() . '.*, ul2.time as login_time')
            ->leftJoin(UserLogin::tableName() . ' as ul2', User::tableName() . '.id = ul2.user_id')
            ->joinWith('userParent')
            ->where([
                'AND',
                ['ul2.id' => $userLoginQuery],
                ['<=', 'DATE(ul2.time)', date('Y-m-d', strtotime('-7 days'))],
                ['user_parent_id' => $coachId]
            ])
            ->all();
        foreach ($inactiveUsers as $i => $iu) {
            $inactiveUsers[$i]->seconds_ago = ceil((time() - strtotime($iu->login_time)) / 86400);
        }
        // sort by last activity time
        usort($inactiveUsers, function($a, $b) {
            return $b->seconds_ago <=> $a->seconds_ago;
        });

        $recentlyActiveUsers = User::find()
            ->select(User::tableName() . '.*, ul2.time as login_time, duration as login_duration')
            ->leftJoin(UserLogin::tableName() . ' as ul2', User::tableName() . '.id = ul2.user_id')
            ->joinWith('userParent')
            ->where([
                'AND',
                ['ul2.id' => $userLoginQuery],
                ['>=', new Expression('`ul2`.`time` + INTERVAL `ul2`.`duration` SECOND'), date_format(date_create('-6 HOURS', timezone_open('UTC')), 'Y-m-d H:i')],
                ['user_parent_id' => $coachId],
                ['!=', 'ul2.user_id', $coachId]
            ])
            ->all();

        $recentLoginPeriods = [];
        foreach ($recentlyActiveUsers as $r) {
            $r->seconds_ago = time() - strtotime(Formatter::datetime($r->login_time, 'UTC', 'Y-m-d', 'php:H:i:s')) - $r->login_duration;
            $recentLoginPeriods[$r->id] = Formatter::period($r->seconds_ago);
        }
        // sort by last activity time
        usort($recentlyActiveUsers, function($a, $b) {
            return $a->seconds_ago <=> $b->seconds_ago;
        });

        // get overspending users
        $users = User::find()
            ->joinWith('userParent')
            ->where([
                'AND',
                ['user_parent_id' => $coachId],
                ['user_role_id' => 3]
            ])
            ->all();
        $userIds = ArrayHelper::map($users, 'id', 'id');

        // we'll also need user currencies
        $userCurrencies = [];
        $currencyMeta = UserMeta::find()
            ->where([
                'and',
                ['user_id' => $userIds],
                ['key' => 'currency_id']
            ])
            ->all();
        $currencyMetaSorted = ArrayHelper::map($currencyMeta, 'user_id', 'value');

        $currencies = Currency::find()->all();
        $currenciesSorted = ArrayHelper::map($currencies, 'id', 'code');

        foreach ($userIds as $u) {
            $currencyId = isset($currencyMetaSorted[$u]) ? $currencyMetaSorted[$u] : Yii::$app->params['defaultCurrencyId'];
            $userCurrencies[$u] = $currenciesSorted[$currencyId];
        }

        $overspendingUsers = [];
        $budgetIds = [];
        foreach ($users as $u) {
            $budgetId = $u->getUserMeta()
                ->select('value')
                ->where('`key` = "active_budget_id"')
                ->scalar();
            $budget = Budget::findOne($budgetId);

            if ($budget) {
                $budgetIds[$u->id] = $budgetId;
                $balance = 0;
                /** @var Jar $j */
                foreach ($budget->getActiveJars() as $j) {
                    $jarBalance = $j->getThisMonthsFundsRemainingAmount();
                    if ($jarBalance < 0) {
                        $balance += $jarBalance;
                    }
                }
                if ($balance < 0) {
                    $overspendingUsers[] = [
                        'user' => $u,
                        'total' => $balance
                    ];
                }
            }
        }
        // sort by overspent amount
        usort($overspendingUsers, function($a, $b) {
            return $a['total'] <=> $b['total'];
        });

        // get user birthdays
        $birthdays = UserMeta::getUsersBirthdays(array_merge([$coachId], $userIds));

        // EoMS
        // completed EoMS
        $eoms = Eoms::find()
            ->select(Eoms::tableName() . '.*, up.user_id')
            ->joinWith('budget.user.userParent as up')
            ->where([
                'and',
                ['month' => (int) date('m', strtotime('-1 MONTHS', $timeNow))],
                ['year' => date('Y', strtotime('-1 MONTHS', $timeNow))],
                ['user_parent_id' => $coachId]
            ])
            ->all();

        $eomsTotalCount = 0;
        $eomsCompletedCount = 0;
        $eomsPositiveCount = 0;
        $eomsPositiveAmount = [];
        $eomsPositiveUserCounts = [];
        $eomsAvgPositiveAmount = [];
        $eomsNegativeCount = 0;
        $eomsNegativeAmount = [];
        $eomsNegativeUserCounts = [];
        $eomsAvgNegativeAmount = [];
        foreach ($eoms as $e) {
            $eomsTotalCount++;
            if ($e->submitted == 1) {
                $eomsCompletedCount++;
                $eomsAmount = $e->getSummaryAmount();
                if ($eomsAmount >= 0) {
                    $eomsPositiveCount++;
                    if (!isset($eomsPositiveAmount[$userCurrencies[$e->user_id]])) {
                        $eomsPositiveAmount[$userCurrencies[$e->user_id]] = 0;
                    }
                    $eomsPositiveAmount[$userCurrencies[$e->user_id]] += $eomsAmount;
                    $eomsPositiveUserCounts[$userCurrencies[$e->user_id]][$e->user_id] = $e->user_id;
                }
                else {
                    $eomsNegativeCount++;
                    if (!isset($eomsNegativeAmount[$userCurrencies[$e->user_id]])) {
                        $eomsNegativeAmount[$userCurrencies[$e->user_id]] = 0;
                    }
                    $eomsNegativeAmount[$userCurrencies[$e->user_id]] += $eomsAmount;
                    $eomsNegativeUserCounts[$userCurrencies[$e->user_id]][$e->user_id] = $e->user_id;
                }
            }
        }
        foreach ($eomsPositiveAmount as $currency => $amount) {
            $eomsAvgPositiveAmount[$currency] = empty($eomsPositiveUserCounts[$currency]) ?
                0 :
                $amount / count($eomsPositiveUserCounts[$currency]);
        }
        foreach ($eomsNegativeAmount as $currency => $amount) {
            $eomsAvgNegativeAmount[$currency] = empty($eomsNegativeUserCounts[$currency]) ?
                0 :
                $amount / count($eomsNegativeUserCounts[$currency]);
        }

        // debt amount paid back by all
        $relativeDate = date('Y-m-d', strtotime('-30 days'));
        $debtPayments30Days = DebtPayment::find()
            ->select(DebtPayment::tableName() . '.*')
            ->joinWith('debt d')
            ->where([
                'and',
                ['budget_id' => $budgetIds],
                ['is_adjustment' => 1],
                ['>=', DebtPayment::tableName() . '.date', $relativeDate]
            ])
            ->all();

        $budgetToUser = array_flip($budgetIds);
        $debtPayments30DaysAmounts = [];
        foreach ($debtPayments30Days as $d) {

            $currencyKey = $userCurrencies[$budgetToUser[$d->debt->budget_id]];
            $debtTypeKey = $d->debt->debtType->name;

            if (!isset($debtPayments30DaysAmounts[$currencyKey]['total'])) {
                $debtPayments30DaysAmounts[$currencyKey]['total'] = 0;
            }
            if (!isset($debtPayments30DaysAmounts[$currencyKey][$debtTypeKey])) {
                $debtPayments30DaysAmounts[$currencyKey][$debtTypeKey] = 0;
            }
            $debtPayments30DaysAmounts[$currencyKey][$debtTypeKey] += $d->amount;
            $debtPayments30DaysAmounts[$currencyKey]['total'] += $d->amount;

        }

        // debt amount paid back by all
        $debtPaymentsTotal = DebtPayment::find()
            ->select(DebtPayment::tableName() . '.*')
            ->joinWith('debt d')
            ->where([
                'budget_id' => $budgetIds,
                'is_adjustment' => 1
            ])
            ->all();

        $debtPaymentsTotalAmounts = [];
        foreach ($debtPaymentsTotal as $d) {
            $currencyKey = $userCurrencies[$budgetToUser[$d->debt->budget_id]];
            $debtTypeKey = $d->debt->debtType->name;

            if (!isset($debtPaymentsTotalAmounts[$currencyKey]['total'])) {
                $debtPaymentsTotalAmounts[$currencyKey]['total'] = 0;
            }
            if (!isset($debtPaymentsTotalAmounts[$currencyKey][$debtTypeKey])) {
                $debtPaymentsTotalAmounts[$currencyKey][$debtTypeKey] = 0;
            }
            $debtPaymentsTotalAmounts[$currencyKey][$debtTypeKey] += $d->amount;
            $debtPaymentsTotalAmounts[$currencyKey]['total'] += $d->amount;
        }

        Yii::$app->cache->set("{$coachKey}allActivePlansCount", $allActivePlansCount);
        Yii::$app->cache->set("{$coachKey}usersAssignedOver12Months", $usersAssignedOver12Months);
        Yii::$app->cache->set("{$coachKey}inactiveUsers", $inactiveUsers);
        Yii::$app->cache->set("{$coachKey}userCurrencies", $userCurrencies);
        Yii::$app->cache->set("{$coachKey}overspendingUsers", $overspendingUsers);
        Yii::$app->cache->set("{$coachKey}recentlyActiveUsers", $recentlyActiveUsers);
        Yii::$app->cache->set("{$coachKey}recentLoginPeriods", $recentLoginPeriods);
        Yii::$app->cache->set("{$coachKey}birthdays", $birthdays);
        Yii::$app->cache->set("{$coachKey}eomsTotalCount", $eomsTotalCount);
        Yii::$app->cache->set("{$coachKey}eomsCompletedCount", $eomsCompletedCount);
        Yii::$app->cache->set("{$coachKey}eomsPositiveCount", $eomsPositiveCount);
        Yii::$app->cache->set("{$coachKey}eomsPositiveAmount", $eomsPositiveAmount);
        Yii::$app->cache->set("{$coachKey}eomsPositiveUserCounts", $eomsPositiveUserCounts);
        Yii::$app->cache->set("{$coachKey}eomsAvgPositiveAmount", $eomsAvgPositiveAmount);
        Yii::$app->cache->set("{$coachKey}eomsNegativeCount", $eomsNegativeCount);
        Yii::$app->cache->set("{$coachKey}eomsNegativeAmount", $eomsNegativeAmount);
        Yii::$app->cache->set("{$coachKey}eomsNegativeUserCounts", $eomsNegativeUserCounts);
        Yii::$app->cache->set("{$coachKey}eomsAvgNegativeAmount", $eomsAvgNegativeAmount);
        Yii::$app->cache->set("{$coachKey}debtPayments30DaysAmounts", $debtPayments30DaysAmounts);
        Yii::$app->cache->set("{$coachKey}debtPaymentsTotalAmounts", $debtPaymentsTotalAmounts);
    }
}
