<?php

namespace backend\modules\admin\controllers;

use backend\components\helpers\Stripe;
use backend\models\db\UserMeta;
use backend\models\db\UserPlanType;
use kartik\mpdf\Pdf;
use Yii;

use backend\components\helpers\JsonTools;
use backend\modules\admin\models\search\UserSearch;
use backend\components\CustomController;
use yii\web\HttpException;
use yii\helpers\Url;
use yii\filters\AccessControl;
use dmstr\bootstrap\Tabs;

use backend\models\db\Budget;
use backend\models\db\User;
use backend\models\db\UserLogin;
use backend\models\db\UserParent;
use backend\models\db\UserPlan;
use backend\models\db\UserRegistration;
use backend\models\db\UserRole;

use backend\modules\admin\models\form\AddUser;

use yii\helpers\ArrayHelper;
use backend\components\helpers\Password;

class UserController extends CustomController
{

    public $enableCsrfValidation = false;
    public $fileActions = [
        'print-consolidated-report'
    ];

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'coaches',
                            'view',
                            'create',
                            'update',
                            'activate',
                            'delete',
                            'reset',
                            'switch',
                            'sync',
                            'cancel'
                        ],
                        'allow' => true,
                        'roles' => ['superadmin'],
                    ],
                    [
                        'actions' => [
                            'index',
                            'view',
                            'update',
                            'consolidated-report',
                            'print-consolidated-report',
                            'switch',
                            'switch-back',
                        ],
                        'allow' => true,
                        'roles' => ['coach'],
                    ]
                ],
            ]
        ];

    }


    /**
     * Lists client & subClient User models.
     * @return mixed
     */
    public function actionIndex() {

        $userRoleIds = [3, 4];

        $searchModel = new UserSearch;
        $dataProvider = $searchModel->search($_GET, $userRoleIds, ($this->userRole == 'coach') ? $this->user->id : false);

        Tabs::clearLocalStorage();

        Url::remember();
        Yii::$app->session['__crudReturnUrl'] = null;

        $this->view->title = 'Users';

        return $this->render('index', [
            'dataProvider' => $dataProvider,
            'searchModel' => $searchModel,
            'userRoles' => UserRole::find()->where(['IN', 'id', $userRoleIds])->all(),
            'planTypes' => UserPlanType::find()->all(),
            'userRoleIds' => $userRoleIds,
            'menuHtmlId' => 'admin-users',
        ]);
    }


    /**
     * Lists coach User models.
     * @return mixed
     */
    public function actionCoaches() {

        $userRoleIds = [UserRole::COACH];

        $searchModel = new UserSearch;
        $dataProvider = $searchModel->search($_GET, $userRoleIds);

        Tabs::clearLocalStorage();

        Url::remember();
        Yii::$app->session['__crudReturnUrl'] = null;

        $this->view->title = 'Coaches';

        return $this->render('index', [
            'dataProvider' => $dataProvider,
            'searchModel' => $searchModel,
            'userRoles' => UserRole::find()->where(['IN', 'id', $userRoleIds])->all(),
            'planTypes' => [],
            'userRoleIds' => $userRoleIds,
            'menuHtmlId' => 'admin-coaches'
        ]);

    }

    /**
     * Displays a single User model.
     * @param integer $id
     *
     * @return mixed
     */
    public function actionView($id) {

        Yii::$app->session['__crudReturnUrl'] = Url::previous();
        Url::remember();
        Tabs::rememberActiveState();

        $user = $this->findModel($id);
        if (in_array($user->user_role_id, [UserRole::SUPERADMIN, UserRole::COACH]) && (Yii::$app->user->identity->userRole->name != 'superadmin')){

            return $this->accessError();

        }

        $this->view->title = 'User Detail';

        return $this->render('view', [
            'model' => $user,
        ]);

    }

    public function actionCancel($id)
    {
        try {
            $userPlan = UserPlan::findOne($id);
            $stripe = new Stripe();
            $stripe->cancelSubscription($userPlan);
            $userPlan->is_cancelled = 1;
            $userPlan->save();
            Yii::$app->session->setFlash('success', 'Subscription successfully cancelled.');
        }
        catch (\Exception $e) {
            Yii::$app->session->setFlash('error', 'Could not cancel subscription. ' . $e->getMessage());
        }

        return $this->redirect(['user/view', 'id' => $userPlan->user_id]);
    }

    /**
     * Creates a new User model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate() {

        $model = new AddUser;

        if ($model->load($_POST)) {

            if (\Yii::$app->request->isAjax) {

                if ($model->validate()) {

                    // save user
                    $user = new User();
                    $user->username = $model->username;
                    $user->email = $model->email;
                    $user->user_role_id = $model->user_role_id;
                    if (strlen($model->password)) {

                        $password = new Password($model->password);
                        $user->password_salt = $password->getSalt();
                        $user->password = $password->getHash();

                    }
                    $user->save();

                    // save user registration
                    $registration = new UserRegistration();
                    $registration->setAttributes([
                        'user_id' => $user->id,
                        'time' => date('Y-m-d H:i:s'),
                        'auth_token' => Yii::$app->security->generateRandomString(),
                        'auth_time' => date('Y-m-d H:i:s')
                    ]);
                    $registration->save();

                    // save user parent
                    if ((is_numeric($model->user_parent_id) && ($user->user_role_id == UserRole::SUBCLIENT)) || (is_numeric($model->coach_id) && ($user->user_role_id == UserRole::CLIENT))) {

                        $parent = new UserParent();
                        $parent->user_id = $user->id;
                        $parent->user_parent_id = ($user->user_role_id == UserRole::SUBCLIENT) ? $model->user_parent_id : $model->coach_id;
                        $parent->save();

                    }

                    return JsonTools::successMessage('User successfully created.');

                }
                else {

                    return JsonTools::formErrorMessage($model);

                }
            }

        }
        else {

            $this->view->title = 'Add User';

            return $this->render('create', [
                'formModel' => $model,
                'userRoles' => ArrayHelper::map(UserRole::find()->all(), 'id', 'name'),
                'parents' => ArrayHelper::map(User::find()->where(['user_role_id' => 3])->orderBy('username asc')->all(), 'id', 'username'),
                'coaches' => ArrayHelper::map(User::find()->where(['user_role_id' => 2])->orderBy('username asc')->all(), 'id', 'username')
            ]);

        }

    }

    /**
     * Updates an existing User model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param integer $id
     * @return mixed
     */
    public function actionUpdate($id) {

        $user = User::findOne($id);

        // if coach, check if can edit
        if ($this->initUserRole == 'coach') {

            $parent = $user->getUserParent()->one();
            if (!$parent || ($parent->user_parent_id != Yii::$app->user->identity->id)) {

                return $this->redirect(['index']);

            }

        }

        $model = new AddUser();

        if ($model->load($_POST)) {

            if (Yii::$app->request->isAjax) {

                $model->username = $user->username;
                $model->user_role_id = $user->user_role_id;

                if ($model->validate()) {

                    $user->email = $model->email;

                    if (strlen($model->password)) {

                        $password = new Password($model->password);
                        $user->password_salt = $password->getSalt();
                        $user->password = $password->getHash();

                    }

                    if ($model->user_role_id == UserRole::CLIENT) {
                        $parent = $user->getUserParent()->one();
                        // coach has changed
                        if ($model->coach_id != $parent->user_parent_id) {
                            $parent->user_parent_id = $model->coach_id;
                            $parent->time = date('Y-m-d H:i:s');
                            $parent->save();

                            // send notification about coach change to both coach and client
                            $user->addNotification('coach_changed_client', 'Coach', ['coach_name' => $parent->userParent->getFullName()]);
                            $parent->userParent->addNotification('coach_changed_coach', 'Coach', ['client_name' => $user->getFullName()]);
                        }
                    }

                    $user->save();

                    return JsonTools::successMessage('User successfully edited.');

                }
                else {

                    return JsonTools::formErrorMessage($model);

                }
            }

        }
        else {

            $model->setAttributes($user->getAttributes());
            $model->password = '';
            if ($model->user_role_id == UserRole::CLIENT) {
                $parent = $user->getUserParent()->one();
                $model->coach_id = $parent->user_parent_id;
            }

            $this->view->title = 'Update User';

            return $this->render('update', [
                'formModel' => $model,
                'userId' => $user->id,
                'userRoleId' => $user->user_role_id,
                'parents' => ArrayHelper::map(User::find()->where(['user_role_id' => 3])->all(), 'id', 'username'),
                'coaches' => ArrayHelper::map(User::find()->where(['user_role_id' => 2])->all(), 'id', 'username')
            ]);
        }
    }

    /**
     * Re-activates an existing User model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionActivate($id) {

        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if ($confirm) {

            $user = $this->findModel($id);
            $user->archived = 0;
            $user->save();

            echo JsonTools::successMessage('User successfully activated.');
            die();

        }
        else {

            $this->view->title = 'Activate User';

            return $this->render('activate', [
                'userId' => $id
            ]);

        }

    }

    /**
     * Deletes an existing User model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionDelete($id)
    {
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);
        if ($confirm) {
            $user = $this->findModel($id);
            $user->deleteMembershipUser();
            $user->archived = 1;
            $user->save();

            echo JsonTools::successMessage('User successfully archived.');
            die();
        }
        else {
            $this->view->title = 'Archive User';
            return $this->render('remove-user', [
                'userId' => $id
            ]);
        }
    }

    /**
     * Resets existing user's account.
     * If reset is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionReset($id)
    {
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);
        if ($confirm) {
            $user = $this->findModel($id);
            $user->resetAccount();

            echo JsonTools::successMessage('User\'s account successfully reset.');
            die();
        }
        else {
            $this->view->title = 'Reset User\'s Account';
            return $this->render('reset-user', [
                'userId' => $id
            ]);
        }
    }

    public function actionSync($id)
    {
        $user = User::findOne($id);
        if ($user->createAndSaveMembershipUser('no-password', $user->getMetaValueByKey(UserMeta::FIRST_NAME), $user->getMetaValueByKey(UserMeta::LAST_NAME))) {
            Yii::$app->session->setFlash('success', 'User synchronized successfully. Please ask the user to reset the password on the bootcamp site to get access.');
        }
        return $this->redirect(['/admin/user/index']);
    }

    /**
     * Switch to another user as SA or coach
     */
    public function actionSwitch($id)
    {
        // if coach, check if can edit
        $user = User::findOne($id);
        if ($this->initUserRole == 'coach') {
            $parent = $user->getUserParent()->one();
            if (!$parent || ($parent->user_parent_id != Yii::$app->user->identity->id)) {
                return $this->redirect(['/admin']);
            }
        }

        // remove session info that needs to be reloaded for the freshly switched user
        Yii::$app->session->remove('adminSelectedUserIdentity');
        Yii::$app->session->remove('userMeta');
        Yii::$app->session->remove('mainUserId');
        Yii::$app->session->remove('coach');
        Yii::$app->session->remove('userRole');
        Yii::$app->session->remove('forceEoms');
        Yii::$app->session->remove('showEomsWarning');
        Yii::$app->session->remove('lastCheckDate');
        Yii::$app->session->remove('currency');
        Yii::$app->session->remove('timeZone');
        Yii::$app->session->remove('timeFormat');
        Yii::$app->session->remove('dateFormat');
        Yii::$app->session->remove('thousandSeparator');
        Yii::$app->session->remove('decimalSeparator');
        Yii::$app->session->remove('defaultAccountId');
        Yii::$app->session->remove('user');

        Yii::$app->session->set('adminSelectedUserId', $id);

        if ($user->user_role_id == 2) {
            $this->redirect(['/admin/default/coach-dashboard']);
        }
        else {
            $this->redirect(['/budget/dashboard']);
        }
    }

    /**
     * Switch back to initial coach/SA account
     */
    public function actionSwitchBack()
    {
        Yii::$app->session->remove('adminSelectedUserId');
        Yii::$app->session->remove('adminSelectedUserIdentity');
        Yii::$app->session->remove('userMeta');
        Yii::$app->session->remove('mainUserId');
        Yii::$app->session->remove('coach');
        Yii::$app->session->remove('userRole');
        Yii::$app->session->remove('forceEoms');
        Yii::$app->session->remove('showEomsWarning');
        Yii::$app->session->remove('eomsOverdue');
        Yii::$app->session->remove('lastCheckDate');
        Yii::$app->session->remove('currency');
        Yii::$app->session->remove('timeZone');
        Yii::$app->session->remove('timeFormat');
        Yii::$app->session->remove('dateFormat');
        Yii::$app->session->remove('thousandSeparator');
        Yii::$app->session->remove('decimalSeparator');
        Yii::$app->session->remove('defaultAccountId');
        Yii::$app->session->remove('user');

        if (Yii::$app->user->identity->user_role_id == 1) {
            $this->redirect(['/admin/default/admin-dashboard']);
        }
        elseif (Yii::$app->user->identity->user_role_id == 2) {
            $this->redirect(['/admin/default/coach-dashboard']);
        }
    }

    public function actionConsolidatedReport()
    {
        return $this->_handleConsolidatedReport();
    }

    public function actionPrintConsolidatedReport()
    {
        $pdf = new Pdf([
            'content' => $this->_handleConsolidatedReport(true),
            'cssInline' => '.row { margin-bottom: 1.5em } .label-group { margin-bottom: 0.5em }',
            'options' => [
                'title' => 'Consolidated Report'
            ]
        ]);
        return $pdf->render();
    }

    private function _handleConsolidatedReport($print = false)
    {
        $userId = Yii::$app->getRequest()->getQueryParam('id');
        $user   = User::findOne($userId);

        if ($user && (Yii::$app->user->isSuperadmin || ($user->userParent->user_parent_id == Yii::$app->user->id))) {

            $meta = $user->getUserMetaSorted();
            $registration = $user->getUserRegistrations()->one();
            $userPlan = UserPlan::getActivePlan($userId);

            // login data
            $userLogins = UserLogin::find()
                ->where([
                    'and',
                    ['user_id' => $user->id],
                    ['MONTH(time)' => date('m')],
                    ['YEAR(time)' => date('Y')]
                ])
                ->all();

            $loginDurationRaw = 0;
            $loginCount = 0;
            foreach ($userLogins as $u) {

                $loginCount++;
                $loginDurationRaw += $u->duration;

            }

            $hours = floor($loginDurationRaw / 3600);
            $hours = ($hours > 9) ? $hours : '0' . $hours;
            $minutes = floor(($loginDurationRaw - $hours * 3600) / 60);
            $minutes = ($minutes > 9) ? $minutes : '0' . $minutes;
            $seconds = $loginDurationRaw % 60;
            $seconds = ($seconds > 9) ? $seconds : '0' . $seconds;
            $loginDuration = "$hours:$minutes:$seconds";

            // expenses changes
            $budget = Budget::findOne($meta['active_budget_id']);
            $expenseChanges = $budget ?
                $budget->getExpenseChangesForTable(date('Y'), date('m'), $user->id) :
                [];

            // overspending jars
            $overspendingJars = [];
            $jars = $budget ? $budget->getActiveJars() : [];
            foreach ($jars as $j) {
                $balance = $j->getThisMonthsFundsRemainingAmount();
                if ($balance < 0) {
                    $overspendingJars[] = [
                        'name' => $j->name,
                        'balance' => $balance
                    ];
                }
            }

            // notes
            $notes = $user->getClientNotes()
                ->where(['user_id' => Yii::$app->user->id])
                ->orderBy('time_edited desc')
                ->all();

            $this->view->title = 'Consolidated report for ' . $user->getFullName(true);

            return $this->render($print ? 'consolidated-report-print' : 'consolidated-report', [
                'user' => $user,
                'meta' => $meta,
                'registration' => $registration,
                'userPlan' => $userPlan,
                'loginCount' => $loginCount,
                'loginDuration' => $loginDuration,
                'expenseChanges' => $expenseChanges,
                'overspendingJars' => $overspendingJars,
                'notes' => $notes
            ]);

        }

        return '';
    }

    /**
     * Finds the User model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return User the loaded model
     * @throws HttpException if the model cannot be found
     */
    protected function findModel($id) {

        if (($model = User::findOne($id)) !== null) {

            return $model;

        }
        else {

            throw new HttpException(404, 'The requested page does not exist.');

        }

    }
}
