<?php
 
namespace console\controllers;

use backend\components\helpers\DripHelper;
use backend\components\helpers\Stripe;
use backend\models\db\DebtChange;
use backend\models\db\DebtEnd;
use backend\models\db\DebtPayment;
use backend\models\db\TimeSlotBooking;
use backend\models\db\UserPlan;
use backend\models\db\UserRole;
use backend\modules\admin\controllers\DefaultController;
use Yii;
use yii\console\Controller;
use backend\models\db\Appointment;
use backend\models\db\Budget;
use backend\models\db\CcTransaction;
use backend\models\db\Debt;
use backend\models\db\Eoms;
use backend\models\db\Expense;
use backend\models\db\ExpenseDebt;
use backend\models\db\ExpenseEnd;
use backend\models\db\ExpenseStart;
use backend\models\db\Income;
use backend\models\db\IncomeChange;
use backend\models\db\IncomeTransaction;
use backend\models\db\Jar;
use backend\models\db\TimeFormat;
use backend\models\db\TimeZone;
use backend\models\db\Transaction;
use backend\models\db\Transfer;
use backend\models\db\TransferChange;
use backend\models\db\TransferEvent;
use backend\models\db\User;
use backend\models\db\UserMeta;
use backend\models\db\Update;

use yii\db\Query;
use yii\helpers\ArrayHelper;
use yii\helpers\Url;

use backend\components\helpers\Formatter;

class DataController extends Controller {
    
    private $_debug = false; // change this to turn debug mode ON/OFF
    private $_detailedOutput = true;
    private $_debugBudgetIds = []; // set ID of testing budget
    private $_forcedDateMode = false;
    private $_forcedDate = '2018-05-17'; // Y-m-d format


    public function actionMonitor()
    {
        $failedCrons = Update::getFailedCrons('hourly');
        $missingCrons = Update::getTimesToProcess('hourly', true, true);
        if (count($failedCrons) > 0 || count($missingCrons) > 0) {
            $failedCronsArray = [];
            foreach ($failedCrons as $cron) {
                $failedCronsArray[] = [
                    'time' => Formatter::localDatetimeToUtcDatetime($cron->start_time),
                    'step' => $cron->step
                ];
            }
            /*DripHelper::sendEmailEvent(
                'failed-crons',
                'bartek@procreative.eu',
                'Failed Crons Report',
                [
                    'failed_crons_count' => count($failedCronsArray),
                    'failed_crons' => $failedCronsArray,
                    'missing_crons_count' => count($missingCrons),
                    'missing_crons' => $missingCrons
                ]
            );*/
            Yii::$app->mail->sendEmail(
                'bartek@procreative.eu',
                [
                    'failed_crons_count' => count($failedCronsArray),
                    'failed_crons' => $failedCronsArray,
                    'missing_crons_count' => count($missingCrons),
                    'missing_crons' => $missingCrons
                ],
                'failed-crons',
                'Failed Crons Report'
            );
        }
        echo "Failed Cron Report resulted in ".count($failedCrons).
            " failed crons and ".count($missingCrons)." missing crons...\n";
    }

    public function actionDashboard()
    {
        $this->_refreshDashboardCalculations();
    }

    /*public function actionHubspot()
    {
        $count = 0;
        $users = [];
        $userList = User::find()->where(['archived' => 0])->orderBy('user_role_id ASC')->all();
        foreach ($userList as $user) {
            if ($user->user_role_id == 3) {
                $users[strtolower($user->email)] = $user;
            }
            $userPlan = UserPlan::getActivePlan($user->id);
            $verified = false;
            foreach ($user->getUserRegistrations()->all() as $registration) {
                if ($registration->auth_time) {
                    $verified = true;
                }
            }
            // skip sub-clients with the same email as the parent client
            if ($user->user_role_id == 4 && isset($users[strtolower($user->email)])) {
                continue;
            }
            if ($user->hubspotVerified($userPlan, $verified, true)) {
                $count++;
            }
        }
        echo "Updated {$count} users...\n";
    }*/

    public function actionMembership()
    {
        if (!isset(Yii::$app->getComponents()['siteConnector'])) {
            echo "Site Connector not found...\n";
            return false;
        }
        $membershipUsers = Yii::$app->siteConnector->getAllUsers();
        $count = 0;
        $updated = 0;
        $already = 0;
        /** @var User $user */
        foreach (User::find()->all() as $user) {
            if (!$user->isMembershipUser($membershipUsers)) {
                if ($user->createAndSaveMembershipUser('no password', $user->getMetaValueByKey(UserMeta::FIRST_NAME), $user->getMetaValueByKey(UserMeta::LAST_NAME))) {
                    $count++;
                }
            }
            else {
                $userPlan = UserPlan::getActivePlan($user->id);
                if ($userPlan) {
                    if ($user->updateMembershipLevel($userPlan)) {
                        $updated++;
                    }
                }
                $already++;
            }
        }
        echo "Added {$count} users, updated {$updated} users, {$already} users was already on the membership site...\n";
    }

    public function actionFix()
    {
        /** @var User $user */
        foreach (User::find()->all() as $user) {
            // fix only for clients and subclients
            if ($user->user_role_id < 3) {
                continue;
            }
            $query = new Query();
            $query
                ->from('auth_assignment')
                ->where(['user_id' => (string) $user->id]);
            if (!$query->exists(Yii::$app->db)) {
                Yii::$app->db->createCommand()->insert('auth_assignment', [
                    'item_name' => $user->user_role_id == 3 ? 'client' : 'sub-client',
                    'user_id' => $user->id,
                    'created_at' => time()
                ])->execute();
            }
        }
    }

    public function actionJars($force = 0)
    {
        $runningKey = 'calcRunning';
        if (!$force && Yii::$app->cache->exists($runningKey)) {
            return;
        }
        Yii::$app->cache->set($runningKey, true);
        $keyName = 'monthsFundsRemainingAmountCalcId';
        $startFrom = (int) Yii::$app->cache->get($keyName);
        if (!$startFrom) {
            $startFrom = 1;
        }
        $today = date('Y-m');
        $jars = Jar::find()
            ->joinWith('budget')
            ->innerJoin(User::tableName(), User::tableName() . '.id = ' . Budget::tableName() . '.user_id')
            ->where([
                Jar::tableName() . '.archived' => 0,
                Budget::tableName() . '.archived' => 0,
                User::tableName() . '.archived' => 0
            ])
            ->andWhere([
                '>=', Jar::tableName() . '.id', $startFrom
            ])
            ->orderBy('id ASC')
            ->limit(1000)
            ->all();
        /** @var Jar $jar */
        foreach ($jars as $jar) {
            echo "\nJAR #{$jar->id}...\n";
            $year = substr($jar->created_time, 0, 4);
            $month = substr($jar->created_time, 5, 2);
            $current = date('Y-m', strtotime("{$year}-{$month}-20"));
            while ($current <= $today) {
                //echo " - {$year}-{$month}\n";
                $jar->getMonthsFundsRemainingAmount($month, $year, false, true);
                $jar->getMonthsFundsRemainingAmount($month, $year, false, false);
                $jar->getMonthsFundsRemainingAmount($month, $year, true, true);
                $jar->getMonthsFundsRemainingAmount($month, $year, true, false);
                $current = date('Y-m', strtotime('+1 month', strtotime($current.'-20')));
                $year = substr($current, 0, 4);
                $month = substr($current, 5, 2);
            }
            $startFrom = $jar->id;
            Yii::$app->cache->set($keyName, $startFrom);
        }
        Yii::$app->cache->delete($runningKey);
    }

    public function actionTimezones()
    {
        $timezones = TimeZone::find()->all();
        $utcTime = Formatter::localDatetimeToUtcDatetime(time());
        echo "UTC: ".date('Y-m-d H:i:s', strtotime($utcTime))."\n";
        foreach ($timezones as $t) {
            $local = Formatter::utcDatetimeToLocalDatetime($utcTime, $t->code, 'Y-m-d H:i:s');
            $hour = (int) Formatter::utcDatetimeToLocalDatetime($utcTime, $t->code, 'G');
            echo $t->code . ": " . $local . " ($hour) - timezone ID: $t->id\n";
        }
    }

    public function actionFixindex()
    {
        set_time_limit(3600);
        ini_set('memory_limit', '2G');

        $toProcess = Update::getTimesToProcess('hourly', true, false, 1000);

        foreach ($toProcess as $time) {
            if (!$this->_debug) {
                ob_start();
            }
            $utcTime = Formatter::localDatetimeToUtcDatetime($time, 'Y-m-d H:i:s');
            echo "Starting daily update [" . $utcTime . "].\n";

            // get timezones where it's currently midnight
            $timezones = TimeZone::find()->all();
            $midnightTimezones = [];
            if ($this->_debug) {
                echo "System UTC time: $utcTime\n";
            }
            $currentDate = null;
            foreach ($timezones as $t) {
                $hour = (int)Formatter::utcDatetimeToLocalDatetime($utcTime, $t->code, 'G');
                if ($hour == 0) {
                    $midnightTimezones[] = $t->id;
                    if (!$currentDate) {
                        $currentDate = Formatter::utcDatetimeToLocalDatetime($utcTime, $t->code, 'Y-m-d');
                    }
                }
            }
            if ($this->_detailedOutput) {
                echo 'Midnight timezones: ' . print_r($midnightTimezones, true) . "\n\n";
            }

            // get main users which have these set as their chosen timezone
            $updatingUsers = User::find()
                ->select(User::tableName() . '.id')
                ->joinWith('userMeta')
                ->where([
                    'key' => 'time_zone_id',
                    'value' => $midnightTimezones
                ])
                ->all();
            $updatingUserIds = ArrayHelper::map($updatingUsers, 'id', 'id');

            // get active budgets
            if (!$this->_debug && empty($this->_debugBudgetIds)) {
                $activeBudgetMeta = UserMeta::findAll(['key' => 'active_budget_id', 'user_id' => $updatingUserIds]);
                $activeBudgetIds = [];
                foreach ($activeBudgetMeta as $m) {
                    $budget = Budget::findOne((int)$m->value);
                    // skip archived budgets
                    if ($budget && !$budget->archived) {
                        $activeBudgetIds[] = $budget->id;
                    }
                }
                if (!$this->_debug) {
                    // track start
                    $update = new Update();
                    $update->setAttributes([
                        'step' => 'start',
                        'type' => 'hourly',
                        'start_time' => Formatter::localDatetimeToUtcDatetime($time)
                    ]);
                    $update->save();
                }
            } else {
                $activeBudgetIds = $this->_debugBudgetIds;
            }
            $date = $this->_forcedDateMode ? $this->_forcedDate : $currentDate;

            if ($this->_detailedOutput) {
                echo 'Active Budgets: ' . print_r($activeBudgetIds, true) . "\n\n";
                echo 'Date: ' . $date . "\n\n";
            }

            // add missing EoMS records
            if (!$this->_debug) {
                $update->step = 'missing eoms';
                $update->save();
            }
            $this->_addMissingEoms($activeBudgetIds, $date);

            // generate the daily expenses
            if (!$this->_debug) {
                $update->step = 'transactions';
                $update->save();
            }
            $this->_generateTransactions($activeBudgetIds, $date);

            // generate daily debt payments
            if (!$this->_debug) {
                $update->step = 'debt payments';
                $update->save();
            }
            $this->_generateDebtPayments($activeBudgetIds, $date);

            // generate daily income transactions
            if (!$this->_debug) {
                $update->step = 'income transactions';
                $update->save();
            }
            $this->_generateIncomeTransactions($activeBudgetIds, $date);

            // generate daily transfer events
            if (!$this->_debug) {
                $update->step = 'transfer events';
                $update->save();
            }
            $this->_generateTransferEvents($activeBudgetIds, $date);

            if (!$this->_debug) {
                $update->step = 'done';
                $update->end_time = Formatter::localDatetimeToUtcDatetime(time());
                $update->save();
            }

            echo "Finished daily update [" . $utcTime . "].\n";

            if (!$this->_debug) {
                $debug = ob_get_clean();
                $update->debug = $debug;
                $update->save();
                echo $debug;
            }
        }
    }
    
    // the hourly update script
    public function actionIndex()
    {
        set_time_limit(3600);
        ini_set('memory_limit', '2G');

        $toProcess = Update::getTimesToProcess('hourly');

        foreach ($toProcess as $time) {
            if (!$this->_debug) {
                ob_start();
            }
            $utcTime = Formatter::localDatetimeToUtcDatetime($time, 'Y-m-d H:i:s');
            echo "Starting daily update [" . $utcTime . "].\n";

            // get timezones where it's currently midnight
            $timezones = TimeZone::find()->all();
            $midnightTimezones = [];
            if ($this->_debug) {
                echo "System UTC time: $utcTime\n";
            }
            $currentDate = null;
            foreach ($timezones as $t) {
                $hour = (int)Formatter::utcDatetimeToLocalDatetime($utcTime, $t->code, 'G');
                if ($hour == 0) {
                    $midnightTimezones[] = $t->id;
                    if (!$currentDate) {
                        $currentDate = Formatter::utcDatetimeToLocalDatetime($utcTime, $t->code, 'Y-m-d');
                    }
                }
            }
            if ($this->_detailedOutput) {
                echo 'Midnight timezones: ' . print_r($midnightTimezones, true) . "\n\n";
            }

            // get main users which have these set as their chosen timezone
            $updatingUsers = User::find()
                ->select(User::tableName() . '.id')
                ->joinWith('userMeta')
                ->where([
                    'key' => 'time_zone_id',
                    'value' => $midnightTimezones
                ])
                ->all();
            $updatingUserIds = ArrayHelper::map($updatingUsers, 'id', 'id');

            // get active budgets
            if (!$this->_debug && empty($this->_debugBudgetIds)) {
                $activeBudgetMeta = UserMeta::findAll(['key' => 'active_budget_id', 'user_id' => $updatingUserIds]);
                $activeBudgetIds = [];
                foreach ($activeBudgetMeta as $m) {
                    $budget = Budget::findOne((int)$m->value);
                    // skip archived budgets
                    if ($budget && !$budget->archived) {
                        $activeBudgetIds[] = $budget->id;
                    }
                }
                if (!$this->_debug) {
                    // track start
                    $update = new Update();
                    $update->setAttributes([
                        'step' => 'start',
                        'type' => 'hourly',
                        'start_time' => Formatter::localDatetimeToUtcDatetime($time)
                    ]);
                    $update->save();
                }
            } else {
                $activeBudgetIds = $this->_debugBudgetIds;
            }
            $date = $this->_forcedDateMode ? $this->_forcedDate : $currentDate;

            if ($this->_detailedOutput) {
                echo 'Active Budgets: ' . print_r($activeBudgetIds, true) . "\n\n";
                echo 'Date: ' . $date . "\n\n";
            }

            // add missing EoMS records
            if (!$this->_debug) {
                $update->step = 'missing eoms';
                $update->save();
            }
            $this->_addMissingEoms($activeBudgetIds, $date);

            // generate the daily expenses
            if (!$this->_debug) {
                $update->step = 'transactions';
                $update->save();
            }
            $this->_generateTransactions($activeBudgetIds, $date);

            // generate daily debt payments
            if (!$this->_debug) {
                $update->step = 'debt payments';
                $update->save();
            }
            $this->_generateDebtPayments($activeBudgetIds, $date);

            // generate daily income transactions
            if (!$this->_debug) {
                $update->step = 'income transactions';
                $update->save();
            }
            $this->_generateIncomeTransactions($activeBudgetIds, $date);

            // generate daily transfer events
            if (!$this->_debug) {
                $update->step = 'transfer events';
                $update->save();
            }
            $this->_generateTransferEvents($activeBudgetIds, $date);

            // generate daily invoice notifications
            /*if (!$this->_debug) {
                $update->step = 'invoice notifications';
                $update->save();
                $this->_generateInvoiceNotifications($updatingUserIds);
            }*/

            // generate daily birthday notifications
            if (!$this->_debug) {
                $update->step = 'birthday notifications';
                $update->save();
                $this->_generateBirthdaysNotifications($updatingUserIds);
            }

            // generate daily overdue eoms notifications
            if (!$this->_debug) {
                $update->step = 'eoms notifications';
                $update->save();
                $this->_generateOverdueEomsNotifications($activeBudgetIds, $time);
            }

            // generate upcoming appointment notifications
            if (!$this->_debug) {
                $update->step = 'appointment notifications';
                $update->save();
                $this->_generateUpcomingAppointmentNotifications($updatingUserIds, $time);
            }

            // generate interest-free period end notifications
            if (!$this->_debug) {
                $update->step = 'appointment notifications';
                $update->save();
                $this->_generateExpiringInterestFreePeriodNotifications($activeBudgetIds);
            }

            if (!$this->_debug) {
                $update->step = 'done';
                $update->end_time = Formatter::localDatetimeToUtcDatetime(time());
                $update->save();
            }

            echo "Finished daily update [" . $utcTime . "].\n";

            if (!$this->_debug) {
                $debug = ob_get_clean();
                $update->debug = $debug;
                $update->save();
                echo $debug;
            }
        }

        $this->_refreshDashboardCalculations();
        $this->_syncStripePlans();
    }

    public function actionStripe()
    {
        $this->_syncStripePlans();
    }

    /**
     * Synchronize Stripe Plans with the database
     */
    public function _syncStripePlans()
    {
        try {
            $stripe = new Stripe();
            $stripe->syncStripePlans();
        }
        catch (\Exception $e) {
            echo 'Sync Stipe Plans ERROR: ' . $e->getMessage();
        }
        echo "Stripe Plans Synced\n";
    }

    public function _refreshDashboardCalculations()
    {
        DefaultController::calculateAdminDashboard();
        echo "Refreshed Admin Dashboard\n";
        $coaches = User::find()
            ->where([
                'user_role_id' => UserRole::COACH,
                'archived' => 0
            ])
            ->all();
        foreach ($coaches as $coach) {
            DefaultController::calculateCoachDashboard($coach->id);
        }
        echo "Refreshed Coaches Dashboard\n";
    }

    public function _addMissingEoms($activeBudgetIds, $todayDate = false)
    {
        $eomsAdded = [];
        $todayDate = (!$todayDate) ? date('Y-m-d') : $todayDate;
        foreach ($activeBudgetIds as $activeBudgetId) {
            $eomsAdded = array_merge(
                $eomsAdded,
                Eoms::fixMissingEoms($activeBudgetId, $todayDate)
            );
        }
        if ($this->_detailedOutput) {
            echo 'Missing EoMS added: ' . print_r($eomsAdded, true) . "\n\n";
        }
    }

    // update monthly numbers
    public function actionMonthlyUpdate()
    {
        set_time_limit(3600);

        if (!$this->_debug) {
            ob_start();
        }
        // get timezones where it's currently midnight
        $timezones = TimeZone::find()->all();
        $midnightTimezones = [];
        $utcTime = Formatter::localDatetimeToUtcDatetime(time());
        foreach ($timezones as $t) {
            $hour = (int) Formatter::utcDatetimeToLocalDatetime($utcTime, $t->code, 'G');
            $day = (int) Formatter::utcDatetimeToLocalDatetime($utcTime, $t->code, 'j');
            if (($hour == 0) && ($day == 1)){
                $midnightTimezones[] = $t->id;
            }
        }
        
        // get main users which have these set as their chosen timezone
        $updatingUsers = User::find()
            ->select(User::tableName() . '.id')
            ->joinWith('userMeta')
            ->where([
                'key' => 'time_zone_id',
                'value' => $midnightTimezones
            ])
            ->all();
        $updatingUserIds = ArrayHelper::map($updatingUsers, 'id', 'id');
                
        // get active budgets
        if (!$this->_debug) {
            $activeBudgetMeta = UserMeta::findAll(['key' => 'active_budget_id', 'user_id' => $updatingUserIds]);
            $activeBudgetIds = [];
            foreach ($activeBudgetMeta as $m) {
                $budget = Budget::findOne((int)$m->value);
                // skip mock and archived budgets
                if ($budget && !$budget->is_mock && !$budget->archived) {
                    $activeBudgetIds[] = $budget->id;
                }
            }
        }
        else {
            $activeBudgetIds = $this->_debugBudgetIds;
        }

        // track start
        $update = new Update();
        $update->setAttributes([
            'step' => 'start',
            'type' => 'monthly',
            'start_time' => Formatter::localDatetimeToUtcDatetime(time())
        ]);
        $update->save();
        
        // is it still the previous month or next month?
        if (date('t') == date('d')) {
            $month = date('m');
            $year = date('Y');
        }
        else {
            $month = date('m', strtotime('-1 MONTH', strtotime(date('Y-m-20'))));
            $year = date('Y', strtotime('-1 MONTH', strtotime(date('Y-m-20'))));
        }

        $eoms = Eoms::findAll([
            'month' => $month, 
            'year' => $year,
            'budget_id' => $activeBudgetIds
        ]);

        foreach ($eoms as $e) {
            $e->runMonthlyBudgetUpdate();
        }

        $update->end_time = Formatter::localDatetimeToUtcDatetime(time());
        $update->step = 'done';
        echo "Finished monthly update [" . date('Y-m-d H:i:s') . "].\n";
        if (!$this->_debug) {
            $debug = ob_get_clean();
            $update->debug = $debug;
            $update->save();
            echo $debug;
        }
    }
    
    
    /**
     * Generate the daily transactions which the users have set up.
     * @param array $activeBudgetIds Array of active budget IDs from the user meta table.
     * @param string|bool $todayDate Today's date, in case today's date needs to be overwritten.
     * @param integer $skip Skip a number of records.
     * @return boolean Whether or not all due records were successfully saved.
     */
    private function _generateTransactions(array $activeBudgetIds, $todayDate = false, $skip = 0)
    {
        if ($this->_detailedOutput) {
            echo "========= TRANSACTIONS ========\n\n";
        }
        
        // get expenses
        $todayDate = (!$todayDate) ? date('Y-m-d') : $todayDate;
        $expensesQuery = Expense::find()
            ->select(Expense::tableName() . '.*, ' . ExpenseStart::tableName() . '.date as start_date, ' . ExpenseEnd::tableName() . '.date as end_date, ' . ExpenseDebt::tableName() . '.debt_id as debt_id')
            ->joinWith('jar')
            ->joinWith('expenseStart')
            ->joinWith('expenseEnd')
            ->joinWith('expenseDebt')
            ->leftJoin(Budget::tableName(), Jar::tableName() . '.budget_id = ' . Budget::tableName() . '.id')    
            ->where([
                'and',
                [Jar::tableName() . '.archived' => '0'],
                [Expense::tableName() . '.archived' => '0'],
                [Expense::tableName() . '.is_adjustment' => '0'],
                [ExpenseEnd::tableName() . '.date' => null],
                ['in', Budget::tableName() . '.id', $activeBudgetIds]
            ])
            ->andWhere([
                'or',
                [Expense::tableName() . '.generate_transactions' => '1'],
                [
                    'and',
                    [Expense::tableName() . '.generate_transactions' => '0'],
                    [Expense::tableName() . '.frequency' => 'one-time']
                ]
            ])
            ->andWhere([
                'or',
                ['!=', Expense::tableName() . '.frequency', 'one-time'],
                ['!=', Expense::tableName() . '.date', $todayDate],
                ['!=', ExpenseStart::tableName() . '.date', $todayDate]
            ]);
        
        if ($this->_debug) {
            echo $expensesQuery->createCommand()->rawSql . "\n---------------\n\n";
        }
        
        $expenses = $expensesQuery->all();
        $skipCounter = 0;
        $success = true;
        /** @var Expense $e */
        foreach ($expenses as $e) {
            if ($skipCounter < $skip) {
                $skipCounter++;
            }
            else {
                if ($e->frequency == 'one-time') {
                    $nextDue = $e->date;
                }
                else {
                    $nextDue = $e->getNextDue(false, $todayDate);
                }

                if ($nextDue == $todayDate) {
                    $amount = $e->amount;
                    if ($amount == 0) {
                        continue;
                    }

                    $description = 'Generated Expense from ' . $e->name;

                    if ($this->_detailedOutput) {
                        echo '[' . $e->id . '] ' . $e->name . ' ' . $nextDue . ' INSERT [' . $amount . '] ' . $description;
                    }
                    if (!$this->_debug) {
                        if ($e->account_id != null) {
                            $transaction = new Transaction();
                            $transaction->setAttributes([
                                'expense_id' => $e->id,
                                'account_id' => $e->account_id,
                                'date' => $todayDate,
                                'amount' => $amount,
                                'description' => $description,
                                'tax_deductible' => $e->tax_deductible ? 1 : 0
                            ]);
                            $saved = $transaction->save();
                            if (!$saved) {
                                if ($this->_detailedOutput) {
                                    echo ' (ERROR: ' . print_r($transaction->errors, true) . ')';
                                }
                                $success = false;
                            }
                            elseif ($this->_detailedOutput) {
                                echo ' (OK)';
                            }
                        } else {
                            $transaction = new CcTransaction();
                            $transaction->setAttributes([
                                'expense_id' => $e->id,
                                'debt_id' => $e->debt_id,
                                'date' => $todayDate,
                                'amount' => $amount
                            ]);
                            $transaction->save();

                            // adjust the debt
                            $debt = Debt::findOne($e->debt_id);
                            $debt->amount += $transaction->amount;
                            $saved = $debt->save();
                            if (!$saved) {
                                if ($this->_detailedOutput) {
                                    echo ' (ERROR: ' . print_r($debt->errors, true) . ')';
                                }
                                $success = false;
                            }
                            elseif ($this->_detailedOutput) {
                                echo ' (OK)';
                            }
                        }
                    }
                    if ($this->_detailedOutput) {
                        echo "\n";
                    }
                }
            }
        }
        return $success;
    }

    /**
     * Generate the daily debt payments which the users have set up.
     * @param array $activeBudgetIds Array of active budget IDs from the user meta table.
     * @param string|bool $todayDate Today's date, in case today's date needs to be overwritten.
     * @param integer $skip Skip a number of records.
     * @return boolean Whether or not all due records were successfully saved.
     */
    private function _generateDebtPayments(array $activeBudgetIds, $todayDate = false, $skip = 0)
    {
        if ($this->_detailedOutput) {
            echo "\n========= DEBT PAYMENTS ========\n\n";
        }
        $todayDate = (!$todayDate) ? date('Y-m-d') : $todayDate;

        // get debts
        $debtsQuery = Debt::find()
            ->select(Debt::tableName() . '.*, ' . DebtEnd::tableName() . '.date as end_date')
            ->joinWith('budget')
            ->joinWith('debtEnd')
            ->leftJoin(DebtChange::tableName(), DebtChange::tableName() . '.debt_id = ' . Debt::tableName() . '.id AND type = "C"')
            ->where([
                'and',
                [Debt::tableName() . '.archived' => '0'],
                ['in', Budget::tableName() . '.id', $activeBudgetIds]
            ])
            ->andWhere([
                'or',
                ['!=', Debt::tableName() . '.frequency', 'one-time'],
                ['!=', Debt::tableName() . '.date', $todayDate],
                ['!=', 'date_format(' . DebtChange::tableName() . '.time, \'%Y-%m-%d\')', $todayDate]
            ]);

        if ($this->_debug) {
            echo $debtsQuery->createCommand()->rawSql . "\n---------------\n\n";
        }

        $debts = $debtsQuery->all();
        $skipCount = 0;
        $success = true;
        /** @var Debt $d */
        foreach ($debts as $d) {
            if ($skipCount < $skip) {
                $skipCount++;
            }
            else {
                $hasEnded = isset($d->end_date) && (strtotime($d->end_date) <= strtotime(date('Y-m-d')));
                if (!$d->archived && !$hasEnded) {
                    if ($d->frequency == 'one-time') {
                        $nextDueDebt = $d->date;
                    }
                    else {
                        $nextDueDebt = $d->getNextDue(false, false, $todayDate);
                    }
                    // debt payment
                    if (strtotime($nextDueDebt) == strtotime($todayDate) && $d->payment) {
                        if ($this->_detailedOutput) {
                            echo '[' . $d->id . '] ' . $d->name . ' ' . $nextDueDebt . ' INSERT DEBT PAYMENT [' . $d->payment . '] ';
                        }
                        if (!$this->_debug) {
                            $description = 'Generated Debt Payment from ' . $d->name;

                            $debtPayment = new DebtPayment();
                            $debtPayment->setAttributes([
                                'debt_id' => $d->id,
                                'account_id' =>  $d->account_id,
                                'date' => $todayDate,
                                'amount' => $d->payment,
                                'debt_payment_type_id' => 1,
                                'description' => $description
                            ]);
                            $saved = $debtPayment->save();
                            if (!$saved) {
                                if ($this->_detailedOutput) {
                                    echo ' (ERROR: ' . print_r($debtPayment->errors, true) . ')';
                                }
                                $success = false;
                            }
                            elseif ($this->_detailedOutput) {
                                echo ' (OK)';
                            }
                        }
                        if ($this->_detailedOutput) {
                            echo "\n";
                        }
                    }
                }
            }
        }
        return $success;
    }

    /**
     * Generate the daily income transactions which the users have set up.
     * @param array $activeBudgetIds Array of active budget IDs from the user meta table.
     * @param string|bool $todayDate Today's date, in case today's date needs to be overwritten.
     * @param integer $skip Skip a number of records.
     * @return boolean Whether or not all due records were successfully saved.
     */
    private function _generateIncomeTransactions(array $activeBudgetIds, $todayDate = false, $skip = 0)
    {
        if ($this->_detailedOutput) {
            echo "\n========= INCOME TRANSACTIONS ========\n\n";
        }
        $todayDate = (!$todayDate) ? date('Y-m-d') : $todayDate;

        // get incomes
        $incomesQuery = Income::find()
            ->select(Income::tableName() . '.*')
            ->join('LEFT JOIN', Budget::tableName(), Income::tableName() . '.budget_id = ' . Budget::tableName() . '.id')    
            ->leftJoin(IncomeChange::tableName(), IncomeChange::tableName() . '.income_id = ' . Income::tableName() . '.id AND type = "C"')
            ->joinWith('incomeEnd')
            ->where(['and',
                [Income::tableName() . '.archived' => '0'],
                [Income::tableName() . '.is_adjustment' => '0'],
                ['!=', Income::tableName() . '.frequency', 'one-time'],
            ])
            ->andWhere([
                'or',
                ['!=', Income::tableName() . '.date', $todayDate],
                ['!=', 'date_format(' . IncomeChange::tableName() . '.time, \'%Y-%m-%d\')', $todayDate]
            ])
            ->andWhere(['in', Budget::tableName() . '.id', $activeBudgetIds]);
        
        if ($this->_debug) {
            echo $incomesQuery->createCommand()->rawSql . "\n---------------\n\n";
        }
        
        $incomes = $incomesQuery->all();
        $skipCount = 0;
        $success = true;
        /** @var Income $i */
        foreach ($incomes as $i) {
            if ($skipCount < $skip) {
                $skipCount++;
            }
            else {
                $hasEnded = isset($i->incomeEnd->date) && (strtotime($i->incomeEnd->date) <= strtotime(date('Y-m-d')));

                if (!$i->archived && !$hasEnded) {
                    $nextDueIncome = $i->getNextDue(false, false, $todayDate);
                    // income payment
                    if (strtotime($nextDueIncome) == strtotime($todayDate) && $todayDate != $i->start_date) {
                        $amount = $i->amount;
                        $description = 'Generated Income from ' . $i->name;

                        if ($this->_detailedOutput) {
                            echo '[' . $i->id . '] ' . $i->name . ' ' . $nextDueIncome . ' (start date: ' . $i->start_date . ') INSERT [' . $amount . '] ' . $description;
                        }
                        if (!$this->_debug) {
                            $incomeTransaction = new IncomeTransaction();
                            $incomeTransaction->setAttributes([
                                'income_id' => $i->id,
                                'account_id' =>  $i->account_id,
                                'date' => $todayDate,
                                'amount' => $amount,
                                'description' => $description
                            ]);
                            $saved = $incomeTransaction->save();
                            if (!$saved) {
                                if ($this->_detailedOutput) {
                                    echo ' (ERROR: ' . print_r($incomeTransaction->errors, true) . ')';
                                }
                                $success = false;
                            }
                            elseif ($this->_detailedOutput) {
                                echo ' (OK)';
                            }
                        }
                        if ($this->_detailedOutput) {
                            echo "\n";
                        }
                    }
                }
            }
        }
        return $success;
    }
    
    
    /**
     * Generate the daily income transactions which the users have set up.
     * @param array $activeBudgetIds Array of active budget IDs from the user meta table.
     * @param string|bool $todayDate Today's date, in case today's date needs to be overwritten.
     * @param integer $skip Skip a number of records.
     * @return boolean Whether or not all due records were successfully saved.
     */
    private function _generateTransferEvents(array $activeBudgetIds, $todayDate = false, $skip = 0)
    {
        if ($this->_detailedOutput) {
            echo "\n========= TRANSFER EVENTS ========\n\n";
        }
        $todayDate = (!$todayDate) ? date('Y-m-d') : $todayDate;
        
        // get transfers
        $transfersQuery = Transfer::find()
            ->join('LEFT JOIN', Budget::tableName(), Transfer::tableName() . '.budget_id = ' . Budget::tableName() . '.id')
            ->leftJoin(TransferChange::tableName(), TransferChange::tableName() . '.transfer_id = ' . Transfer::tableName() . '.id AND type = "C"')
            ->joinWith('transferEnd')
            ->where(['and',
                [Transfer::tableName() . '.archived' => '0'],
                [Transfer::tableName() . '.is_adjustment' => '0'],
                ['!=', Transfer::tableName() . '.frequency', 'one-time']
            ])
            ->andWhere([
                'or',
                ['!=', Transfer::tableName() . '.date', $todayDate],
                ['!=', 'date_format(' . TransferChange::tableName() . '.time, \'%Y-%m-%d\')', $todayDate]
            ])
            ->andWhere(['in', Budget::tableName() . '.id', $activeBudgetIds]);
                
        if ($this->_debug) {
            echo $transfersQuery->createCommand()->rawSql . "\n---------------\n\n";
        }
        
        $transfers = $transfersQuery->all();
        $skipCount = 0;
        $success = true;
        /** @var Transfer $t */
        foreach ($transfers as $t) {
            if ($skipCount < $skip) {
                $skipCount++;
            }
            else {
                $hasEnded = isset($t->transferEnd->date) && (strtotime($t->transferEnd->date) <= strtotime(date('Y-m-d')));

                if (!$t->archived && !$hasEnded) {
                    $nextDueTransfer = $t->getNextDue(false, false, $todayDate);
                    // transfer payment
                    if (strtotime($nextDueTransfer) == strtotime($todayDate)) {
                        if ($this->_detailedOutput) {
                            echo '[' . $t->id . '] ' . $t->name . ' ' . $nextDueTransfer . ' INSERT [' . $t->amount . ']';
                        }
                        if (!$this->_debug) {
                            $transferEvent = new TransferEvent();
                            $transferEvent->setAttributes([
                                'transfer_id' => $t->id,
                                'date' => $todayDate,
                                'amount' => $t->amount,
                            ]);
                            $saved = $transferEvent->save();
                            if (!$saved) {
                                if ($this->_detailedOutput) {
                                    echo ' (ERROR: ' . print_r($transferEvent->errors, true) . ')';
                                }
                                $success = false;
                            }
                            elseif ($this->_detailedOutput) {
                                echo ' (OK)';
                            }
                        }
                        if ($this->_detailedOutput) {
                            echo "\n";
                        }
                    }
                }
            }
        }
        return $success;
    }

    
    /**
     * Generate invoice notifications
     * @param array $limitToUsers List of users the notifications can actually be sent to
     */
    /*private function _generateInvoiceNotifications(array $limitToUsers)
    {
        if ($this->_detailedOutput) {
            echo "\n========= INVOICE/PLAN NOTIFICATIONS ========\n\n";
        }
        // get user plans
        $plans = UserPlan::find()->where(['user_id' => $limitToUsers])->all();

        foreach ($plans as $plan) {
            $nextAction = $plan->getNextAction(UserPlan::EXPIRY_NOTIFICATION_DAYS);
            if ($nextAction == UserPlan::INVOICE) {
                echo "Plan invoice: " . Json::encode($plan->toArray()) . "\n";
                $plan->user->addNotification('new_invoice', [
                    'plan_name' => $plan->userPlanType->name
                ]);
                Invoice::generateInvoice($plan);
            }
            elseif ($nextAction == UserPlan::CHANGE_PLAN) {
                echo "Plan change: " . Json::encode($plan->toArray()) . "\n";
                $plan->user->addNotification('change_plan', [
                    'plan_name'   => $plan->userPlanType->name,
                    'expire_days' => UserPlan::EXPIRY_NOTIFICATION_DAYS
                ]);
            }
        }
        if ($this->_detailedOutput) {
            echo "\n";
        }
    }*/

    private function _generateBirthdaysNotifications(array $limitToUsers)
    {
        $coaches = User::find()->where([
            'user_role_id' => UserRole::COACH,
            'id' => $limitToUsers
        ])->all();
        /** @var User $coach */
        foreach ($coaches as $coach) {
            $birthdays = UserMeta::getUsersBirthdays([$coach->id], true);
            foreach ($birthdays as $birthday) {
                $user = User::findOne($birthday->user_id);
                $coach->addNotification('birthday', [
                    'name' => $user->getFullName(true),
                    'birthday' => Formatter::date($birthday->date, false, 'php:'.$coach->getDateFormat())
                ]);
            }
        }
    }
    
    /**
     * Generate overdue EoMS notifications.
     */
    private function _generateOverdueEomsNotifications(array $activeBudgetIds, $time)
    {
        // make sure there are Eoms records generated, even if the users didn't open the app today
        foreach ($activeBudgetIds as $bId) {
            $budget = Budget::findOne($bId);
            if ($budget) {
                Eoms::checkForceEoms($bId, $budget->user_id);
            }
        }
        if ($this->_detailedOutput) {
            echo "\n========= OVERDUE EOMS NOTIFICATIONS ========\n\n";
        }
        // get active eoms
        $eoms = Eoms::find()
            ->select(Eoms::tableName() . '.*, user_id')
            ->leftJoin(UserMeta::tableName(), UserMeta::tableName() . ".key = 'active_budget_id' AND value = " . Eoms::tableName() . '.budget_id')
            ->where([
                'AND',
                ['IS NOT', 'value', null],
                ['submitted' => 0],
                ['budget_id' => $activeBudgetIds]
            ])
            ->all();
        $lastMonth = date('Y-n', strtotime('-1 month', strtotime(date('Y-m-20'))));

        foreach ($eoms as $e) {
            // add notification
            $user = User::findOne($e->user_id);
            $userMeta = UserMeta::findAllSorted(['user_id' => $user->id]);
            $userTimezoneCode = TimeZone::findOne((isset($userMeta['time_zone_id'])) ? $userMeta['time_zone_id'] : Yii::$app->params['defaultTimeZoneId'])->code;

            $is10DaysAfter =
                Formatter::utcDatetimeToLocalDatetime($time, $userTimezoneCode, 'd') == 10 &&
                $lastMonth == "{$e->year}-{$e->month}" ? true : false;
            $is16DaysAfter =
                Formatter::utcDatetimeToLocalDatetime($time, $userTimezoneCode, 'd') == 16 &&
                $lastMonth == "{$e->year}-{$e->month}" ? true : false;
            $is21DaysAfter =
                Formatter::utcDatetimeToLocalDatetime($time, $userTimezoneCode, 'd') == 21 &&
                $lastMonth == "{$e->year}-{$e->month}" ? true : false;
            $is28DaysAfter =
                Formatter::utcDatetimeToLocalDatetime($time, $userTimezoneCode, 'd') == 28 &&
                $lastMonth == "{$e->year}-{$e->month}" ? true : false;
            $isLastDay  =
                date('t', strtotime(Formatter::utcDatetimeToLocalDatetime($time, $userTimezoneCode, 'Y-m-d'))) -
                Formatter::utcDatetimeToLocalDatetime($time, $userTimezoneCode, 'd') <= 0 ? true : false;

            $name    = $user->getFullName();
            $url     = Url::toRoute(['/budget/end-of-month-summary']);
            $eomsUrl = Url::toRoute(['/budget/end-of-month-summary']) . '?video';

            if ($is10DaysAfter) {
                if ($this->_detailedOutput) {
                    echo "EoMS [{$e->month}/{$e->year}, {$e->budget_id}]" . ' eoms_overdue_10_days';
                }
                $user->addNotification(
                    'eoms_overdue_10_days',
                    [
                        'name' => $name,
                        'url' => $url,
                        'eoms_url' => $eomsUrl
                    ]
                );
            }
            elseif ($is16DaysAfter) {
                if ($this->_detailedOutput) {
                    echo "EoMS [{$e->month}/{$e->year}, {$e->budget_id}]" . ' eoms_overdue_16_days';
                }
                $user->addNotification(
                    'eoms_overdue_16_days',
                    [
                        'name' => $name,
                        'url' => $url,
                        'eoms_url' => $eomsUrl
                    ]
                );
            }
            elseif ($is21DaysAfter) {
                if ($this->_detailedOutput) {
                    echo "EoMS [{$e->month}/{$e->year}, {$e->budget_id}]" . ' eoms_overdue_21_days';
                }
                $user->addNotification(
                    'eoms_overdue_21_days',
                    [
                        'name' => $name,
                        'url' => $url,
                        'eoms_url' => $eomsUrl
                    ]
                );
                // this is also the moment the EoMS gets locked
                $user->addNotification(
                    'eoms_overdue_locked',
                    [
                        'name' => $name,
                        'url' => $url,
                        'eoms_url' => $eomsUrl
                    ]
                );
            }
            elseif ($is28DaysAfter) {
                if ($this->_detailedOutput) {
                    echo "EoMS [{$e->month}/{$e->year}, {$e->budget_id}]" . ' eoms_overdue_28_days';
                }
                $user->addNotification(
                    'eoms_overdue_28_days',
                    [
                        'name' => $name,
                        'url' => $url,
                        'eoms_url' => $eomsUrl
                    ]
                );
            }
            elseif ($isLastDay) {
                if ($this->_detailedOutput) {
                    echo "EoMS [{$e->month}/{$e->year}, {$e->budget_id}]" . ' eoms_due';
                }
                $user->addNotification(
                    'eoms_overdue',
                    [
                        'name' => $name,
                        'url' => $url,
                        'eoms_url' => $eomsUrl
                    ]
                );
            }
            if ($this->_detailedOutput) {
                echo "\n";
            }
        }
    }
    
    
    /**
     * Generate upcoming appointment notifications (appointments today & tomorrow)
     */
    private function _generateUpcomingAppointmentNotifications(array $limitToUsers, $time)
    {
        if ($this->_detailedOutput) {
            echo "\n========= UPCOMING APPOINTMENTS NOTIFICATIONS ========\n\n";
        }
        $endDate = date('Y-m-d', strtotime('+2 days'));
        $startDate = date('Y-m-d', strtotime('-1 days'));
        $upcomingAppointments = TimeSlotBooking::find()
            ->where(['BETWEEN', "DATE(start_at)", $startDate, $endDate])
            ->andWhere(['user_id' => $limitToUsers])
            ->all();

        /** @var TimeSlotBooking $u */
        foreach ($upcomingAppointments as $u) {
            $clientTimezone = $u->user->getTimeZone();
            $coach = $u->user->getCoach();
            $nowDatetime = Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($time, $clientTimezone->code, 'Y-m-d'));
            $appDatetime = Formatter::datetime(Formatter::utcDatetimeToLocalDatetime($u->start_at, $clientTimezone->code, 'Y-m-d'));
            $hoursToGo = (strtotime($appDatetime) - strtotime($nowDatetime)) / 3600;
            $today = $hoursToGo == 0;

            if ($hoursToGo <= 24) {
                $u->sendUpcomingNotification($coach, $coach->getTimeZone(), $today);
                if ($this->_detailedOutput) {
                    echo ($today ? 'TODAY' : 'TOMORROW') . " at {$appDatetime} with {$u->user->fullName}\n";
                }
            }
        }
        if ($this->_detailedOutput) {
            echo "\n";
        }
    }
    
    
    /**
     * expiring debt's interest-free period notification, 1 week ahead
     */
    private function _generateExpiringInterestFreePeriodNotifications(array $activeBudgetIds)
    {
        $date = date('Y-m-d', strtotime('+7 days'));
        $debts = Debt::find()
            ->joinWith('budget.user')
            ->leftJoin(UserMeta::tableName(), UserMeta::tableName() . ".key = 'active_budget_id' AND value = " . Debt::tableName() . '.budget_id')
            ->where([
                'AND',
                ['interest_free_end_date' => $date],
                ['IS NOT', 'value', null],
                ['budget_id' => $activeBudgetIds]
            ])
            ->all();
        
        foreach ($debts as $d) {
            $user = User::findOne($d->budget->user->id);
            $user->addNotification('interest_free_period_end', [
                'name' => $d->name,
                'url' => Url::toRoute(['/debts/debt-detail', 'id' => $d->id]),
                'date' => Formatter::date($date, false, $user->getDateFormat())
            ]);
        }
    }

}