<?php

use backend\components\helpers\Formatter;
use backend\models\db\Budget;
use backend\models\db\Debt;
use backend\models\db\DebtChange;
use backend\models\db\DebtEnd;
use backend\models\db\DebtPayment;
use backend\models\db\Expense;
use backend\models\db\ExpenseDebt;
use backend\models\db\ExpenseEnd;
use backend\models\db\ExpenseStart;
use backend\models\db\Income;
use backend\models\db\IncomeChange;
use backend\models\db\IncomeTransaction;
use backend\models\db\Jar;
use backend\models\db\Transaction;
use yii\db\Migration;

class m170130_105657_fix_recurring_transactions extends Migration
{
    public function up()
    {
        $dates = ['2017-01-25', '2017-01-26', '2017-01-27', '2017-01-28'];
        foreach ($dates as $date) {
            $this->generateTransactions($date);
            $this->generateIncomeTransactions($date);
            $this->generateDebtPayments($date);
        }
    }

    protected function generateTransactions($todayDate)
    {
        // expense transactions
        $expensesQuery = Expense::find()
            ->select(Expense::tableName() . '.*, ' . ExpenseStart::tableName() . '.date as start_date, ' . ExpenseEnd::tableName() . '.date as end_date, ' . ExpenseDebt::tableName() . '.debt_id as debt_id')
            ->joinWith('jar')
            ->joinWith('expenseStart')
            ->joinWith('expenseEnd')
            ->joinWith('expenseDebt')
            ->leftJoin(Budget::tableName(), Jar::tableName() . '.budget_id = ' . Budget::tableName() . '.id')
            ->where([
                'and',
                [Jar::tableName() . '.archived' => '0'],
                [Expense::tableName() . '.archived' => '0'],
                [Expense::tableName() . '.is_adjustment' => '0'],
                [ExpenseEnd::tableName() . '.date' => null]
            ])
            ->andWhere([
                'or',
                [Expense::tableName() . '.generate_transactions' => '1'],
                [
                    'and',
                    [Expense::tableName() . '.generate_transactions' => '0'],
                    [Expense::tableName() . '.frequency' => 'one-time']
                ]
            ])
            ->andWhere([
                'or',
                ['!=', Expense::tableName() . '.frequency', 'one-time'],
                ['!=', Expense::tableName() . '.date', $todayDate],
                ['!=', ExpenseStart::tableName() . '.date', $todayDate]
            ]);

        $expenses = $expensesQuery->all();
        /** @var Expense $e */
        foreach ($expenses as $e) {
            if ($e->frequency == 'one-time') {
                $nextDue = $e->date;
            }
            else {
                $nextDue = $e->getNextDue(false, $todayDate);
            }

            if (strtotime($nextDue) == strtotime($todayDate)) {
                $amount = $e->amount;
                if ($amount == 0) {
                    continue;
                }
                if (Transaction::find()->where(['and',
                    ['expense_id' => $e->id],
                    ['account_id' => $e->account_id],
                    ['date' => $todayDate],
                    ['amount' => $amount],
                    ['like', 'description', 'Generated Expense from']
                ])->exists()) {
                    continue;
                }

                if ($e->jar_id) {
                    $dateFormat = $e->jar->budget->user->getDateFormat();
                }
                else {
                    $dateFormat = $e->account->budget->user->getDateFormat();
                }
                $description = 'Generated Expense from ' . Formatter::date($todayDate, false, 'php:'.$dateFormat);

                if ($e->account_id != null) {
                    $transaction = new Transaction();
                    $transaction->setAttributes([
                        'expense_id' => $e->id,
                        'account_id' => $e->account_id,
                        'date' => $todayDate,
                        'amount' => $amount,
                        'description' => $description
                    ]);
                    $saved = $transaction->save();
                    if (!$saved) {
                        $success = false;
                    }
                }
            }
        }
    }

    protected function generateIncomeTransactions($todayDate)
    {
        $incomesQuery = Income::find()
            ->select(Income::tableName() . '.*')
            ->join('LEFT JOIN', Budget::tableName(), Income::tableName() . '.budget_id = ' . Budget::tableName() . '.id')
            ->leftJoin(IncomeChange::tableName(), IncomeChange::tableName() . '.income_id = ' . Income::tableName() . '.id AND type = "C"')
            ->joinWith('incomeEnd')
            ->where(['and',
                [Income::tableName() . '.archived' => '0'],
                [Income::tableName() . '.is_adjustment' => '0'],
                ['!=', Income::tableName() . '.frequency', 'one-time'],
            ])
            ->andWhere([
                'or',
                ['!=', Income::tableName() . '.date', $todayDate],
                ['!=', 'date_format(' . IncomeChange::tableName() . '.time, \'%Y-%m-%d\')', $todayDate]
            ]);

        $incomes = $incomesQuery->all();
        /** @var Income $i */
        foreach ($incomes as $i) {
            $hasEnded = isset($i->incomeEnd->date) && (strtotime($i->incomeEnd->date) <= strtotime(date('Y-m-d')));

            if (!$i->archived && !$hasEnded) {
                $nextDueIncome = $i->getNextDue(false, false, $todayDate);
                // income payment
                if (strtotime($nextDueIncome) == strtotime($todayDate) && $todayDate != $i->start_date) {
                    $dateFormat = $i->budget->user->getDateFormat();
                    $amount = $i->amount;
                    $description = 'Generated Income from ' . Formatter::date($todayDate, false, 'php:'.$dateFormat);

                    if (IncomeTransaction::find()->where(['and',
                        ['income_id' => $i->id],
                        ['account_id' => $i->account_id],
                        ['date' => $todayDate],
                        ['amount' => $amount],
                        ['like', 'description', 'Generated Income from']
                    ])->exists()) {
                        continue;
                    }

                    $incomeTransaction = new IncomeTransaction();
                    $incomeTransaction->setAttributes([
                        'income_id' => $i->id,
                        'account_id' =>  $i->account_id,
                        'date' => $todayDate,
                        'amount' => $amount,
                        'description' => $description
                    ]);
                    $saved = $incomeTransaction->save();
                    if (!$saved) {
                        $success = false;
                    }
                }
            }
        }
    }

    protected function generateDebtPayments($todayDate)
    {
        // get debts
        $debtsQuery = Debt::find()
            ->select(Debt::tableName() . '.*, ' . DebtEnd::tableName() . '.date as end_date')
            ->joinWith('budget')
            ->joinWith('debtEnd')
            ->leftJoin(DebtChange::tableName(), DebtChange::tableName() . '.debt_id = ' . Debt::tableName() . '.id AND type = "C"')
            ->where([
                'and',
                [Debt::tableName() . '.archived' => '0']
            ])
            ->andWhere([
                'or',
                ['!=', Debt::tableName() . '.frequency', 'one-time'],
                ['!=', Debt::tableName() . '.date', $todayDate],
                ['!=', 'date_format(' . DebtChange::tableName() . '.time, \'%Y-%m-%d\')', $todayDate]
            ]);

        $debts = $debtsQuery->all();
        /** @var Debt $d */
        foreach ($debts as $d) {
            $hasEnded = isset($d->end_date) && (strtotime($d->end_date) <= strtotime(date('Y-m-d')));
            if (!$d->archived && !$hasEnded) {
                if ($d->frequency == 'one-time') {
                    $nextDueDebt = $d->date;
                }
                else {
                    $nextDueDebt = $d->getNextDue(false, false, $todayDate);
                }
                // debt payment
                if (strtotime($nextDueDebt) == strtotime($todayDate) && $d->payment) {
                    $dateFormat = $d->budget->user->getDateFormat();
                    $description = 'Generated Debt Payment from ' . Formatter::date($todayDate, false, 'php:'.$dateFormat);

                    if (DebtPayment::find()->where(['and',
                        ['debt_id' => $d->id],
                        ['account_id' => $d->account_id],
                        ['date' => $todayDate],
                        ['amount' => $d->payment],
                        ['debt_payment_type_id' => 1],
                        ['like', 'description', 'Generated Debt Payment from']
                    ])->exists()) {
                        continue;
                    }

                    $debtPayment = new DebtPayment();
                    $debtPayment->setAttributes([
                        'debt_id' => $d->id,
                        'account_id' =>  $d->account_id,
                        'date' => $todayDate,
                        'amount' => $d->payment,
                        'debt_payment_type_id' => 1,
                        'description' => $description
                    ]);
                    $saved = $debtPayment->save();
                    if (!$saved) {
                        $success = false;
                    }
                }
            }
        }
    }

    public function down()
    {
        echo "m170130_105657_fix_recurring_transactions cannot be reverted.\n";

        return false;
    }

    /*
    // Use safeUp/safeDown to run migration code within a transaction
    public function safeUp()
    {
    }

    public function safeDown()
    {
    }
    */
}
