<?php

namespace backend\components;

use Yii;
use yii\base\Exception;
use yii\base\InvalidParamException;
use yii\caching\Cache;
use yii\caching\FileDependency;
use yii\helpers\BaseFileHelper;
use yii\helpers\FileHelper;
use yii\helpers\StringHelper;

/**
 * Class CustomAssetManager
 * Customized asset manager that handles paths to Google Cloud Storage correctly
 * @package backend\components
 */
class CustomAssetManager extends \yii\web\AssetManager
{
    public $cacheComponent;
    protected $_published = [];

    /**
     * @return Cache
     * @throws Exception
     */
    private function getCache()
    {
        if (!$this->cacheComponent || !isset(Yii::$app->getComponents()[$this->cacheComponent])) {
            throw new Exception('You need to configure a cache storage or set the variable cacheComponent');
        }
        return Yii::$app->{$this->cacheComponent};
    }

    private function getCacheKey($path)
    {
        return $this->hash(Yii::$app->request->serverName).'.'.$path;
    }

    public function init()
    {
        $originalBasePath = $this->basePath;
        parent::init();
        $this->basePath = trim(\Yii::getAlias($originalBasePath), '/');
    }

    public function publish($path, $options = [])
    {
        $path = Yii::getAlias($path);
        if (isset($this->_published[$path])) {
            return $this->_published[$path];
        }
        if (!is_string($path) || ($src = realpath($path)) === false) {
            throw new InvalidParamException("The file or directory to be published does not exist: $path");
        }

        $cacheVal = $this->getCache()->get($this->getCacheKey($path));
        if ($cacheVal === false) {
            if (is_file($src)) {
                $this->_published[$path] = $this->publishFile($src);
            } else {
                $this->_published[$path] = $this->publishDirectory($src, $options);
            }
            $this->getCache()->set($this->getCacheKey($path), $this->_published[$path], 0, new FileDependency(['fileName' => $src]));
        }
        else {
            $this->_published[$path] = $cacheVal;
        }

        return $this->_published[$path];
    }

    public function publishDirectory($src, $options)
    {
        $dir = $this->hash($src);
        $dstDir = $this->basePath . DIRECTORY_SEPARATOR . $dir;
        if ($this->linkAssets) {
            if (!is_dir($dstDir)) {
                FileHelper::createDirectory(dirname($dstDir), $this->dirMode, true);
                symlink($src, $dstDir);
            }
        } elseif (!empty($options['forceCopy']) || ($this->forceCopy && !isset($options['forceCopy'])) || !is_dir($dstDir)) {
            $opts = array_merge(
                $options,
                [
                    'dirMode' => $this->dirMode,
                    'fileMode' => $this->fileMode,
                ]
            );
            if (!isset($opts['beforeCopy'])) {
                if ($this->beforeCopy !== null) {
                    $opts['beforeCopy'] = $this->beforeCopy;
                } else {
                    $opts['beforeCopy'] = function ($from, $to) {
                        return strncmp(basename($from), '.', 1) !== 0;
                    };
                }
            }
            if (!isset($opts['afterCopy']) && $this->afterCopy !== null) {
                $opts['afterCopy'] = $this->afterCopy;
            }
            static::copyDirectory($src, $dstDir, $opts);
        }
        return [$dstDir, $this->baseUrl . '/' . $dir];
    }

    protected static function copyDirectory($src, $dst, $options = [])
    {
        if ($src === $dst || strpos($dst, $src . DIRECTORY_SEPARATOR) === 0) {
            throw new InvalidParamException('Trying to copy a directory to itself or a subdirectory.');
        }
        if (!is_dir($dst)) {
            BaseFileHelper::createDirectory($dst, isset($options['dirMode']) ? $options['dirMode'] : 0775, true);
        }
        $handle = opendir($src);
        if ($handle === false) {
            throw new InvalidParamException("Unable to open directory: $src");
        }
        if (!isset($options['basePath'])) {
            // this should be done only once
            $options['basePath'] = realpath($src);
            $options = self::normalizeOptions($options);
        }
        while (($file = readdir($handle)) !== false) {
            if ($file === '.' || $file === '..') {
                continue;
            }
            $from = $src . DIRECTORY_SEPARATOR . $file;
            $to = $dst . DIRECTORY_SEPARATOR . $file;
            if (BaseFileHelper::filterPath($from, $options)) {
                if (isset($options['beforeCopy']) && !call_user_func($options['beforeCopy'], $from, $to)) {
                    continue;
                }
                if (is_file($from)) {
                    copy($from, $to);
                    if (isset($options['fileMode'])) {
                        @chmod($to, $options['fileMode']);
                    }
                } else {
                    // recursive copy, defaults to true
                    if (!isset($options['recursive']) || $options['recursive']) {
                        static::copyDirectory($from, $to, $options);
                    }
                }
                if (isset($options['afterCopy'])) {
                    call_user_func($options['afterCopy'], $from, $to);
                }
            }
        }
        closedir($handle);
    }

    private static function normalizeOptions(array $options)
    {
        if (!array_key_exists('caseSensitive', $options)) {
            $options['caseSensitive'] = true;
        }
        if (isset($options['except'])) {
            foreach ($options['except'] as $key => $value) {
                if (is_string($value)) {
                    $options['except'][$key] = self::parseExcludePattern($value, $options['caseSensitive']);
                }
            }
        }
        if (isset($options['only'])) {
            foreach ($options['only'] as $key => $value) {
                if (is_string($value)) {
                    $options['only'][$key] = self::parseExcludePattern($value, $options['caseSensitive']);
                }
            }
        }
        return $options;
    }

    private static function parseExcludePattern($pattern, $caseSensitive)
    {
        if (!is_string($pattern)) {
            throw new InvalidParamException('Exclude/include pattern must be a string.');
        }
        $result = [
            'pattern' => $pattern,
            'flags' => 0,
            'firstWildcard' => false,
        ];
        if (!$caseSensitive) {
            $result['flags'] |= BaseFileHelper::PATTERN_CASE_INSENSITIVE;
        }
        if (!isset($pattern[0])) {
            return $result;
        }
        if ($pattern[0] === '!') {
            $result['flags'] |= BaseFileHelper::PATTERN_NEGATIVE;
            $pattern = StringHelper::byteSubstr($pattern, 1, StringHelper::byteLength($pattern));
        }
        if (StringHelper::byteLength($pattern) && StringHelper::byteSubstr($pattern, -1, 1) === '/') {
            $pattern = StringHelper::byteSubstr($pattern, 0, -1);
            $result['flags'] |= BaseFileHelper::PATTERN_MUSTBEDIR;
        }
        if (strpos($pattern, '/') === false) {
            $result['flags'] |= BaseFileHelper::PATTERN_NODIR;
        }
        $result['firstWildcard'] = self::firstWildcardInPattern($pattern);
        if ($pattern[0] === '*' && self::firstWildcardInPattern(StringHelper::byteSubstr($pattern, 1, StringHelper::byteLength($pattern))) === false) {
            $result['flags'] |= BaseFileHelper::PATTERN_ENDSWITH;
        }
        $result['pattern'] = $pattern;
        return $result;
    }

    private static function firstWildcardInPattern($pattern)
    {
        $wildcards = ['*', '?', '[', '\\'];
        $wildcardSearch = function ($r, $c) use ($pattern) {
            $p = strpos($pattern, $c);
            return $r === false ? $p : ($p === false ? $r : min($r, $p));
        };
        return array_reduce($wildcards, $wildcardSearch, false);
    }
}