<?php

namespace backend\components\helpers;

use backend\models\db\Transaction;
use Yii;
use yii\base\Component;
use yii\helpers\Url;


class Calculator extends Component {

    /**
     * Get monthly average amount based on operation amount and frequency.
     * @param float $amount The amount of a single occurence of this operation.
     * @param string $frequency The frequency.
     * @param int $multiplePayments Whether a one-time expense should be saved for regularly
     * @return float Amount paid on average over the course of 1 month.
     */
    public static function monthlyAverageAmount($amount, $frequency, $multiplePayments = null)
    {
        switch ($frequency) {
            case 'daily':
                $monthlyAmount = $amount * 365 / 12;
                break;
            case 'weekly':
                $monthlyAmount = $amount / 7 * 365 / 12;
                break;
            case 'fortnightly':
                $monthlyAmount = $amount / 14 * 365 / 12;
                break;
            case 'monthly':
                $monthlyAmount = $amount;
                break;
            case 'quarterly':
                $monthlyAmount = $amount / 3;
                break;
            case 'semi-annually':
                $monthlyAmount = $amount / 6;
                break;
            case 'annually':
                $monthlyAmount = $amount / 12;
                break;
            case 'one-time':
                $monthlyAmount =
                    $multiplePayments ? $amount / $multiplePayments :
                    $amount;
                break;
            default:
                $monthlyAmount = $amount;
                break;
        }

        $monthlyAmountRound = ceil($monthlyAmount * 100) / 100;
        return number_format($monthlyAmountRound, 2, '.', '');
    }

    public static function yearlyAverageAmount($amount, $frequency)
    {
        switch ($frequency) {
            case 'daily':
                $yearlyAmount = $amount * 365;
                break;
            case 'weekly':
                $yearlyAmount = $amount / 7 * 365;
                break;
            case 'fortnightly':
                $yearlyAmount = $amount / 14 * 365;
                break;
            case 'monthly':
                $yearlyAmount = $amount * 12;
                break;
            case 'quarterly':
                $yearlyAmount = $amount * 4;
                break;
            case 'semi-annually':
                $yearlyAmount = $amount * 2;
                break;
            case 'annually':
                $yearlyAmount = $amount;
                break;
            case 'one-time':
            default:
                $yearlyAmount = $amount;
                break;
        }

        $yearlyAmountRound = ceil($yearlyAmount * 100) / 100;
        return number_format($yearlyAmountRound, 2, '.', '');
    }

    public static function monthlyMinimumAmount($amount, $frequency, $multiplePayments = null)
    {
        switch ($frequency) {
            case 'daily':
                $monthlyAmount = $amount * 28;
                break;
            case 'weekly':
                $monthlyAmount = $amount * 4;
                break;
            case 'fortnightly':
                $monthlyAmount = $amount * 2;
                break;
            case 'monthly':
                $monthlyAmount = $amount;
                break;
            case 'quarterly':
                $monthlyAmount = $amount / 3;
                break;
            case 'semi-annually':
                $monthlyAmount = $amount / 6;
                break;
            case 'annually':
                $monthlyAmount = $amount / 12;
                break;
            case 'one-time':
                $monthlyAmount =
                    $multiplePayments ? $amount / $multiplePayments :
                        $amount;
                break;
            default:
                $monthlyAmount = $amount;
                break;
        }

        $monthlyAmountRound = ceil($monthlyAmount * 100) / 100;
        return number_format($monthlyAmountRound, 2, '.', '');
    }


    /**
     * Get the next due date based on date and a frequency.
     * @param string $date
     * @param string $frequency
     * @param string|bool $today Overrides today's date.
     * @param string|bool $startDate Used for one-time expenses, in that case $date is last day of expense, $startDate is beginning date
     * @return string
     */
    public static function nextDueDate($date, $frequency, $today = false, $startDate = false)
    {
        $endDate = $today ? $today : date('Y-m-d');

        $key = [
            'nextDueDate',
            'date' => $date,
            'frequency' => $frequency,
            'endDate' => $endDate,
            'startDate' => $startDate
        ];
        $value = Yii::$app->cache->get($key);
        if ($value === false) {
            if ($frequency == 'one-time' || strtotime($date) >= strtotime($endDate)) {
                if ($startDate) {
                    $day = date('j', strtotime($date));
                    if ($day > 28) {
                        $date = strtr($date, array($day => 28));
                        $endDate = date('Y-m-28', strtotime($endDate));
                    }

                    $startTime = strtotime($startDate);
                    $nextTime = $newNextTime = strtotime($date);
                    $endTime = strtotime($endDate);
                    while (($newNextTime = strtotime('-1 MONTHS', $newNextTime)) >= $endTime && $newNextTime != $startTime) {
                        $nextTime = $newNextTime;
                    };

                    $nextDate = date('Y-m-d', $nextTime);
                    $daysInNextDateMonth = cal_days_in_month(CAL_GREGORIAN, date('m', $nextTime), date('Y', $nextTime));
                    $replacementDay = ($day > $daysInNextDateMonth) ? $daysInNextDateMonth : $day;
                    $nextDate = strtr($nextDate, array('28' => $replacementDay));
                    $value = $nextDate;
                } else {
                    $value = $date;
                }
            } else {
                switch ($frequency) {
                    case 'daily':
                        $timeInt = '+1 DAY';
                        break;
                    case 'weekly':
                        $timeInt = '+1 WEEK';
                        break;
                    case 'fortnightly':
                        $timeInt = '+2 WEEKS';
                        break;
                    case 'monthly':
                        $timeInt = '+1 MONTH';
                        break;
                    case 'quarterly':
                        $timeInt = '+3 MONTHS';
                        break;
                    case 'semi-annually':
                        $timeInt = '+6 MONTHS';
                        break;
                    case 'annually':
                        $timeInt = '+1 YEAR';
                        break;
                    default:
                        $timeInt = false;
                }
                if ($timeInt) {
                    // set dates
                    // adjust for different number of last days
                    $day = date('j', strtotime($date));
                    $endDay = date('j', strtotime($endDate));
                    if (strpos($timeInt, 'MONTH')) {
                        if ($day > 28) {
                            $date = strtr($date, array($day => 28));
                            $endDate = date('Y-m-28', strtotime($endDate));
                        }
                    }

                    $nextTime = strtotime($date);
                    $endTime = strtotime($endDate);
                    while ($nextTime < $endTime) {
                        $nextTime = strtotime($timeInt, $nextTime);
                    }
                    // fix for $endDate = 2017-03-31 and $date = 2017-01-30
                    if (strpos($timeInt, 'MONTH') && $day > 28 && $endDay > $day) {
                        $nextTime = strtotime($timeInt, $nextTime);
                    }
                    $nextDate = date('Y-m-d', $nextTime);

                    // month days correction, pt. 2
                    if (strpos($timeInt, 'MONTH') && ($day > 28)) {
                        $daysInNextDateMonth = cal_days_in_month(CAL_GREGORIAN, date('m', $nextTime), date('Y', $nextTime));
                        $replacementDay = ($day > $daysInNextDateMonth) ? $daysInNextDateMonth : $day;
                        $nextDate = strtr($nextDate, array('28' => $replacementDay));
                    }
                    $value = $nextDate;
                }
                else {
                    $value = $date;
                }
            }
            Yii::$app->cache->set($key, $value);
        }
        return $value;
    }

    /**
     * Get the previous due date based on date and a frequency. @todo Check-out what's going on here. Clean-up needed.
     * @param string $date
     * @param string $frequency
     * @param string|bool $dateLimit Overrides today's date.
     * @param string|bool $startDate Used for one-time expenses
     * @return string
     */
    public static function prevDueDate($date, $frequency, $dateLimit = false, $startDate = false)
    {
        $endTime = strtotime($dateLimit ? $dateLimit : date('Y-m-d'));

        $key = [
            'prevDueDate',
            'date' => $date,
            'frequency' => $frequency,
            'endTime' => $endTime,
            'startDate' => $startDate
        ];
        $value = Yii::$app->cache->get($key);
        if ($value === false) {
            if ($frequency == 'one-time' || strtotime($date) <= $endTime) {
                if ($startDate) {
                    $oneMonthDate = self::prevDueDate($startDate, 'monthly');
                    if (strtotime($oneMonthDate) > strtotime($date)) {
                        $value = $oneMonthDate;
                    } else {
                        $value = $date;
                    }
                } else {
                    $value = $date;
                }
            } else {
                switch ($frequency) {
                    case 'daily':
                        $timeInt = '-1 DAY';
                        break;
                    case 'weekly':
                        $timeInt = '-1 WEEK';
                        break;
                    case 'fortnightly':
                        $timeInt = '-2 WEEKS';
                        break;
                    case 'monthly':
                        $timeInt = '-1 MONTH';
                        break;
                    case 'quarterly':
                        $timeInt = '-3 MONTHS';
                        break;
                    case 'semi-annually':
                        $timeInt = '-6 MONTHS';
                        break;
                    case 'annually':
                        $timeInt = '-1 YEAR';
                        break;
                    default:
                        $timeInt = false;
                }
                if ($timeInt) {
                    // set dates
                    // adjust for different number of last days
                    $day = date('j', strtotime($date));
                    if (strpos($timeInt, 'MONTH')) {
                        if ($day > 28) {
                            $date = strtr($date, array($day => 28));
                        }
                    }
                    $prevTime = strtotime($date);
                    while ($prevTime > $endTime) {
                        $prevTime = strtotime($timeInt, $prevTime);
                    }
                    $prevDate = date('Y-m-d', $prevTime);

                    // month days correction, pt. 2
                    if (strpos($timeInt, 'MONTH') && ($day > 28)) {
                        $daysInNextDateMonth = cal_days_in_month(CAL_GREGORIAN, date('m', $prevTime), date('Y', $prevTime));
                        $replacementDay = ($day > $daysInNextDateMonth) ? $daysInNextDateMonth : $day;
                        $prevDate = strtr($prevDate, array('28' => $replacementDay));
                    }
                    $value = $prevDate;
                }
                else {
                    $value = $date;
                }
            }
            Yii::$app->cache->set($key, $value);
        }
        return $value;
    }

    /**
     * Get available years list for transactions
     * @param $linkRoute
     * @param null $currentYear
     * @param null $jarIds
     * @param null $expenseIds
     * @param null $expenseId
     * @param null $order
     * @return array
     */
    public static function getYearsList($linkRoute, $currentYear = null, $jarIds = null, $expenseIds = null, $expenseId = null, $order = null)
    {
        $result = self::_getFirstLastTransactionYears($jarIds, $expenseIds, $order);
        $firstYear = $result['firstYear'];
        if ($currentYear && $currentYear < $firstYear) {
            $firstYear = $currentYear;
        }
        $lastYear = $result['lastYear'];
        if ($currentYear && $currentYear > $lastYear) {
            $lastYear = $currentYear;
        }

        return self::getYearsListFromTo($linkRoute, $firstYear, $lastYear, $currentYear, $jarIds, $expenseId);
    }

    public static function getYearsListFromTo($linkRoute, $firstYear, $lastYear, $currentYear = null, $jarIds = null, $expenseId = null)
    {
        // get years
        $yearsList = [];
        for ($i = $firstYear; $i <= $lastYear; $i++) {
            $route = is_array($linkRoute) ?
                array_merge(
                    $linkRoute,
                    ['year' => $i]
                ) :
                [
                    $linkRoute,
                    'year' => $i
                ];
            if ($expenseId) {
                $route['expense_id'] = $expenseId;
            }
            elseif (!is_array($jarIds)) {
                $route['jar_id'] = $jarIds;
            }
            $active = $currentYear && $i == $currentYear ? ' active' : '';
            $yearsList[] = [
                'id' => $i,
                'label' => strval($i),
                'url' => Url::toRoute($route),
                'options' => ['class' => $active],
                'linkOptions' => ['class' => 'system-link']
            ];
        }
        return $yearsList;
    }

    /**
     * Get months list for the given year for transactions
     * @param $linkRoute
     * @param null $currentYear
     * @param null $currentMonth
     * @param null $jarIds
     * @param null $expenseIds
     * @param null $expenseId
     * @param null $order
     * @return array
     */
    public static function getMonthsList($linkRoute, $currentYear = null, $currentMonth = null,
                                         $jarIds = null, $expenseIds = null, $expenseId = null, $order = null)
    {
        $result = self::_getFirstLastTransactionYears($jarIds, $expenseIds, $order);
        $firstTransaction = $result['firstTransaction'];
        $firstYear        = $result['firstYear'];
        $lastYear         = $result['lastYear'];
        if ($currentYear == $firstYear) {
            if ($firstTransaction) {
                $firstMonth = date('m', strtotime($firstTransaction->date));
            }
            else {
                $firstMonth = date('m');
            }
            if ($currentMonth && $currentMonth < $firstMonth) {
                $firstMonth = $currentMonth;
            }
        }
        else {
            $firstMonth = 1;
        }
        $lastMonth = 12;

        return self::getMonthsListFromTo($linkRoute, $firstMonth, $firstYear, $lastMonth, $lastYear, $currentMonth, $currentYear, $expenseId);
    }

    public static function getMonthsListFromTo($linkRoute, $firstMonth, $firstYear, $lastMonth, $lastYear, $currentMonth = null, $currentYear = null, $expenseId = null)
    {
        // get months
        $monthsList = [];
        if ($firstYear < $currentYear) {
            $firstMonth = 1;
        }
        if ($lastYear > $currentYear) {
            $lastMonth = 12;
        }

        if ($currentMonth === null) {
            $currentMonth = ($currentYear == $lastYear) ? $lastMonth : $firstMonth;
        }

        for ($i = $firstMonth; $i <= $lastMonth; $i++) {
            $route = is_array($linkRoute) ? array_merge(
                $linkRoute, [
                    'year' => $currentYear,
                    'month' => $i
                ]) :
                [
                    $linkRoute,
                    'year' => $currentYear,
                    'month' => $i
                ];
            if ($expenseId) {
                $route['expense_id'] = $expenseId;
            }
            $active = ($i == $currentMonth) ? ' active' : '';
            $monthsList[] = [
                'id' => (int)$i,
                'label' => date('F', strtotime('2010-' . $i . '-01')),
                'url' => Url::toRoute($route),
                'options' => ['class' => $active],
                'linkOptions' => ['class' => 'system-link']
            ];
        }
        return $monthsList;
    }

    private static function _getFirstLastTransactionYears($jarIds = null, $expenseIds = null, $order = null)
    {
        $firstTransactionQuery = Transaction::find()
            ->joinWith('expense')
            ->orderBy(($order ? $order : 'id') . ' asc');
        $lastTransactionQuery = Transaction::find()
            ->joinWith('expense')
            ->orderBy(($order ? $order : 'id') . ' desc');

        if ($expenseIds && $jarIds) {
            $firstTransaction = $firstTransactionQuery
                ->where([
                    'or',
                    'expense_sid' => $expenseIds,
                    'jar_id' => $jarIds
                ])
                ->one();
            $lastTransaction = $lastTransactionQuery
                ->where([
                    'or',
                    'expense_id' => $expenseIds,
                    'jar_id' => $jarIds
                ])
                ->one();
        }
        elseif ($expenseIds) {
            $firstTransaction = $firstTransactionQuery
                ->where(['expense_id' => $expenseIds])
                ->one();
            $lastTransaction = $lastTransactionQuery
                ->where(['expense_id' => $expenseIds])
                ->one();
        }
        else {
            $firstTransaction = $firstTransactionQuery
                ->where(['jar_id' => $jarIds])
                ->one();
            $lastTransaction = $lastTransactionQuery
                ->where(['jar_id' => $jarIds])
                ->one();
        }
        if ($firstTransaction) {
            $firstYear = date('Y', strtotime($firstTransaction->date));
            $lastYear = date('Y', strtotime($lastTransaction->date));
        }
        else {
            $firstYear = $lastYear = date('Y');
        }

        return [
            'firstTransaction' => $firstTransaction,
            'lastTransaction'  => $lastTransaction,
            'firstYear'        => $firstYear,
            'lastYear'         => $lastYear
        ];
    }

    public static function getRelativeTime($date, $wrapUnit = false) {
        $diff = strtotime($date) - time();
        $time = '';
        $unit = '';
        if (0 > $diff) {
            $time = 'today';
        } else {
            $days = ceil($diff / (24 * 3600));
            if ($days == 1) {
                $time = 'tomorrow';
            } elseif (30 > $days) {
                $time = $days;
                $unit = 'days';
            } else {
                $months = floor($days / 30);
                $time = $months;
                $unit = 'months';
            }
        }
        if ($wrapUnit) {
            return "$time <span class='time-unit'>$unit</span>";
        }
        return "$time $unit";
    }

    /**
     * Whether the given $date is within the given month
     * @param string|null $date
     * @param string|null $month
     * @param string|null $year
     * @return bool
     */
    public static function happenedThisMonth($date = null, $month = null, $year = null)
    {
        if (!$date) {
            return false;
        }
        $month = $month ? (int)$month : date('n');
        $year = $year ? (int)$year : date('Y');
        $date = date('Y-n-d H:i:s', strtotime($date));
        if ($date >= "$year-$month-01 00:00:00" && $date <= date("Y-n-t 23:59:59", strtotime("$year-$month-20"))) {
            return true;
        }
        return false;
    }

    /**
     * Returns a list of hours in the day
     * @param int $everyMinutes
     * @return array
     */
    public static function getTimeOfDayHours($everyMinutes = 15)
    {
        $times = [];
        for ($h = 0; $h < 24; $h++) {
            for ($m = 0; $m < 60; $m += $everyMinutes) {
                $time =
                    str_pad($h, 2, '0', STR_PAD_LEFT) .
                    ':' .
                    str_pad($m, 2, '0', STR_PAD_LEFT);
                $times[$time] = $time;
            }
        }
        return $times;
    }
}
