<?php

namespace backend\components\helpers;

use Yii;


class DripHelper
{
    const GET  = 1;
    const POST = 2;
    const DELETE = 3;
    const PUT = 4;
    const API_ENDPOINT = 'https://api.getdrip.com/v2/';

    public static $error_message;
    public static $error_code;


    public static function addSubscriber($email, $userId, $tags = array())
    {
        if (!isset(Yii::$app->params['drip']) || !empty(Yii::$app->params['drip']['is_dev'])) {
            return true;
        }
        $status = false;
        $account_id = Yii::$app->params['drip']['account_id'];

        $api_action = "$account_id/subscribers";
        $url = self::API_ENDPOINT . $api_action;

        // The API wants the params to be JSON encoded
        $req_params = array('subscribers' => array(
            array(
                'email' => $email,
                'user_id' => $userId,
                'tags' => $tags
            )
        ));

        try {
            $res = self::make_request(Yii::$app->params['drip']['api_token'], $url, $req_params, self::POST);
            if ($res['http_code'] == 201) {
                $status = true;
            }
            else {
                $status = print_r($res, true);
            }
        }
        catch (\Exception $e) {
            return $e->getMessage();
        }

        return $status;
    }

    public static function applyTagToSubscriber($subscriberEmail, $tag)
    {
        if (!isset(Yii::$app->params['drip']) || !empty(Yii::$app->params['drip']['is_dev'])) {
            return true;
        }
        $status = false;
        $account_id = Yii::$app->params['drip']['account_id'];

        $api_action = "$account_id/tags";
        $url = self::API_ENDPOINT . $api_action;

        // The API wants the params to be JSON encoded
        $req_params = array('tags' => array(
            array(
                'email' => $subscriberEmail,
                'tag' => $tag
            )
        ));

        try {
            $res = self::make_request(Yii::$app->params['drip']['api_token'], $url, $req_params, self::POST);
            if ($res['http_code'] == 201) {
                $status = true;
            }
            else {
                $status = print_r($res, true);
            }
        }
        catch (\Exception $e) {
            return $e->getMessage();
        }

        return $status;
    }

    public static function sendEmailEvent($type, $toEmail, $subject, $params = array())
    {
        if (!isset(Yii::$app->params['drip']) || !empty(Yii::$app->params['drip']['is_dev'])) {
            Yii::$app->mail->sendEmail($toEmail, $params, $type, $subject);
            return true;
        }
        try {
            self::record_event([
                'account_id' => Yii::$app->params['drip']['account_id'],
                'action' => 'email',
                'email' => $toEmail,
                'properties' => array_merge($params, [
                    'type' => $type,
                    'to' => $toEmail,
                    'subject' => $subject
                ])
            ]);
            return true;
        } catch (\Exception $e) {
            return $e->getMessage();
        }
    }

    public static function sendCustomEvent($action, $email, $params = array())
    {
        if (!isset(Yii::$app->params['drip']) || !empty(Yii::$app->params['drip']['is_dev'])) {
            return true;
        }
        try {
            self::record_event([
                'account_id' => Yii::$app->params['drip']['account_id'],
                'action' => $action,
                'email' => $email,
                'properties' => $params
            ]);
            return true;
        } catch (\Exception $e) {
            return $e->getMessage();
        }
    }

    /**
     *
     * Posts an event specified by the user.
     *
     * @param array $params
     * @param bool
     * @return bool
     * @throws \Exception
     */
    protected static function record_event($params)
    {
        $status = false;
        if (empty($params['account_id'])) {
            throw new \Exception("Account ID not specified");
        }
        if (empty($params['action'])) {
            throw new \Exception("Action was not specified");
        }

        $account_id = $params['account_id'];
        unset($params['account_id']); // clear it from the params

        $api_action = "$account_id/events";
        $url = self::API_ENDPOINT . $api_action;

        // The API wants the params to be JSON encoded
        $req_params = array('events' => array($params));

        $res = self::make_request(Yii::$app->params['drip']['api_token'], $url, $req_params, self::POST);

        if ($res['http_code'] == 204) {
            $status = true;
        }

        return $status;
    }

    /**
     *
     * @param string $api_token
     * @param string $url
     * @param array $params
     * @param int $req_method
     * @return array
     * @throws \Exception
     */
    protected static function make_request($api_token, $url, $params = array(), $req_method = self::GET) {
        if (!function_exists('curl_init')) {
            throw new \Exception("Cannot find cURL php extension or it's not loaded.");
        }

        $ch = curl_init();

        /*if ($this->debug) {
            //curl_setopt($ch, CURLOPT_HEADER, true);
            // TRUE to output verbose information. Writes output to STDERR, or the file specified using CURLOPT_STDERR.
            curl_setopt($ch, CURLOPT_VERBOSE, true);
        }*/

        curl_setopt($ch, CURLOPT_FRESH_CONNECT, true);
        curl_setopt($ch, CURLOPT_FORBID_REUSE, true);

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30);
        curl_setopt($ch, CURLOPT_USERAGENT, empty($params['user_agent']) ? 'GrandmasJars (app.grandmasjars.com)' : $params['user_agent']);

        if ($req_method == self::POST) { // We want post but no params to supply. Probably we have a nice link structure which includes all the info.
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
        } elseif ($req_method == self::DELETE) {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "DELETE");
        } elseif ($req_method == self::PUT) {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "PUT");
        }

        if (!empty($params)) {
            if ((isset($params['__req']) && strtolower($params['__req']) == 'get')
                || $req_method == self::GET) {
                unset($params['__req']);
                $url .= '?' . http_build_query($params);
            } elseif ($req_method == self::POST || $req_method == self::DELETE) {
                $params_str = is_array($params) ? json_encode($params) : $params;
                curl_setopt($ch, CURLOPT_POSTFIELDS, $params_str);
            }
        }

        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
            'Authorization: Basic ' . base64_encode($api_token),
            'Accept:application/json, text/javascript, */*; q=0.01',
            'Content-Type: application/vnd.api+json',
        ));

        $buffer = curl_exec($ch);
        $status = !empty($buffer);

        $data = array(
            'url'       => $url,
            'params'    => $params,
            'status'    => $status,
            'error'     => empty($buffer) ? curl_error($ch) : '',
            'error_no'  => empty($buffer) ? curl_errno($ch) : '',
            'http_code' => curl_getinfo($ch, CURLINFO_HTTP_CODE),
            //'debug'     => $this->debug ? curl_getinfo($ch) : '',
        );

        curl_close($ch);

        // remove some weird headers HTTP/1.1 100 Continue or HTTP/1.1 200 OK
        $buffer = preg_replace('#HTTP/[\d.]+\s+\d+\s+\w+[\r\n]+#si', '', $buffer);
        $buffer = trim($buffer);
        $data['buffer'] = $buffer;

        self::_parse_error($data);
        //$this->recent_req_info = $data;

        return $data;
    }

    /**
     * Some keys are removed from the params so they don't get send with the other data to Drip.
     *
     * @param array $res
     */
    protected static function _parse_error($res) {
        if (empty($res['http_code']) || $res['http_code'] >= 200 && $res['http_code'] <= 299) {
            self::$error_message = null;
            self::$error_code = null;
            return;
        }

        if (empty($res['buffer'])) {
            self::$error_message = "Response from the server.";
            self::$error_code = $res['http_code'];
        } elseif (!empty($res['buffer'])) {
            $json_arr = json_decode($res['buffer'], true);

            // The JSON error response looks like this.
            /*
             {
                "errors": [{
                  "code": "authorization_error",
                  "message": "You are not authorized to access this resource"
                }]
              }
             */
            if (!empty($json_arr['errors'])) { // JSON
                $messages = $error_codes = array();

                foreach ($json_arr['errors'] as $rec) {
                    $messages[] = $rec['message'];
                    $error_codes[] = $rec['code'];
                }

                self::$error_code = join(", ", $error_codes);
                self::$error_message = join("\n", $messages);
            } else { // There's no JSON in the reply so we'll extract the message from the HTML page by removing the HTML.
                $msg = $res['buffer'];

                $msg = preg_replace('#.*?<body[^>]*>#si', '', $msg);
                $msg = preg_replace('#</body[^>]*>.*#si', '', $msg);
                $msg = strip_tags($msg);
                $msg = preg_replace('#[\r\n]#si', '', $msg);
                $msg = preg_replace('#\s+#si', ' ', $msg);
                $msg = trim($msg);
                $msg = substr($msg, 0, 256);

                self::$error_code = $res['http_code'];
                self::$error_message = $msg;
            }
        } elseif ($res['http_code'] >= 400 || $res['http_code'] <= 499) {
            self::$error_message = "Not authorized.";
            self::$error_code = $res['http_code'];
        } elseif ($res['http_code'] >= 500 || $res['http_code'] <= 599) {
            self::$error_message = "Internal Server Error.";
            self::$error_code = $res['http_code'];
        }
    }
}