<?php

namespace backend\components\helpers;

use backend\models\db\UserMeta;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ServerException;
use yii\base\InvalidParamException;
use yii\helpers\Json;


class SiteConnector extends \yii\base\Component
{
    const LEVELS = [
        '1' => 'Bootcamp',
        '2' => 'Membership',
        '3' => 'Coaching',
        '5' => 'Bootcamp Access'
    ];

    public $apiUrl;
    public $login;
    public $password;

    /** @var Client */
    protected $httpClient;

    public function init()
    {
        if (empty($this->apiUrl)) {
            throw new InvalidParamException('Missing API URL.');
        }
        if (empty($this->login)) {
            throw new InvalidParamException('Missing API login.');
        }
        if (empty($this->password)) {
            throw new InvalidParamException('Missing API password.');
        }

        $this->httpClient = new Client([
            'base_uri' => $this->apiUrl,
            'headers' => [
                'Accept' => 'application/json',
                'Authorization' => 'Basic ' . base64_encode($this->login . ':' . $this->password),
            ]
        ]);

        parent::init();
    }

    public function createMembershipUser(array $data, $level = '1')
    {
        try {
            $result = $this->httpClient->request('POST', 'users', [
                'json' => array_merge($data, [
                    'roles' => [
                        'subscriber'
                    ],
                    'ihc_user_levels' => (string)$level
                ]),
                'timeout' => 10
            ]);
        }
        catch (ClientException $e) {
            return false;
        }
        catch (ServerException $e) {
            return false;
        }
        return $result->getStatusCode() == 201 ?
            Json::decode($result->getBody()->getContents()) :
            false;
    }

    public function updateMembershipUserDetails($membershipUserId, array $data, $level = null)
    {
        try {
            if ($level) {
                $data = array_merge($data, [
                    'ihc_user_levels' => $level
                ]);
            }
            $result = $this->httpClient->request('POST', 'users/' . (int)$membershipUserId, [
                'json' => $data,
                'timeout' => 10
            ]);
        }
        catch (ClientException $e) {
            return false;
        }
        catch (ServerException $e) {
            return false;
        }
        return $result->getStatusCode() == 200 ?
            Json::decode($result->getBody()->getContents()) :
            false;
    }

    public function deleteMembershipUserDetails($membershipUserId)
    {
        try {
            $result = $this->httpClient->request('DELETE', 'users/' . (int)$membershipUserId, [
                'json' => [
                    'force' => true,
                    'reassign' => 8 // api manager
                ],
                'timeout' => 10
            ]);
        }
        catch (ClientException $e) {
            return false;
        }
        catch (ServerException $e) {
            return false;
        }
        return $result->getStatusCode() == 200 ?
            Json::decode($result->getBody()->getContents()) :
            false;
    }

    public function getAllUsers()
    {
        $allUsers = [];
        $page = 1;
        try {
            do {
                $response = $this->httpClient->request('GET', 'users?per_page=100&page=' . $page, [
                    'timeout' => 10
                ]);
                if ($response->getStatusCode() == 200) {
                    $results = Json::decode($response->getBody()->getContents());
                    if (!empty($results)) {
                        $allUsers = array_merge($allUsers, $results);
                        $page++;
                    }
                } else {
                    $results = null;
                }
            } while (!empty($results));
        } catch (\Exception $e) {
            return $allUsers;
        }
        return $allUsers;
    }

    public function getUserByUsername($username)
    {
        $username = strtr(strtolower($username), [
            '.' => '-',
            '&' => '-'
        ]);
        try {
            $response = $this->httpClient->request('GET', 'users?slug=' . urlencode($username) . '&per_page=1&page=1', [
                'timeout' => 10
            ]);
            if ($response->getStatusCode() == 200) {
                $results = Json::decode($response->getBody()->getContents());
                if (!empty($results) && is_array($results)) {
                    return $results[0];
                }
            }
        } catch (\Exception $e) {
            return null;
        }
        return null;
    }

    public static function getMembershipLevelDescription(array $levels)
    {
        $result = [];
        foreach ($levels as $level) {
            $level = (string)$level;
            if (isset(self::LEVELS[$level])) {
                $result[] = self::LEVELS[$level];
            }
            else {
                $result[] = 'N/A';
            }
        }
        return implode(' / ', $result);
    }
}