<?php

namespace backend\components\helpers;

use backend\models\db\Invoice;
use backend\models\db\User;
use backend\models\db\UserMeta;
use backend\models\db\UserPlan;
use backend\models\db\UserPlanVariation;
use backend\models\db\UserPlanType;
use Stripe\Coupon;
use Stripe\Customer;
use Stripe\Error\InvalidRequest;
use Stripe\Plan;
use Stripe\Subscription as StripeSubscription;
use Stripe\Charge;
use Stripe\Stripe as StripeApi;
use Yii;
use yii\base\Component;


class Stripe extends Component
{
    const GRACE_PERIOD_DAYS = 7;

    protected $api;


    public function __construct()
    {
        StripeApi::setApiKey(Yii::$app->stripe->privateKey);
    }

    public function getCouponDiscount($code)
    {
        try {
            $coupon = Coupon::retrieve($code);
        }
        catch (InvalidRequest $e) {
            return null;
        }
        if (!$coupon->valid) {
            return null;
        }
        if (!empty($coupon->amount_off)) {
            return ['amount' => $coupon->amount_off];
        }
        elseif (!empty($coupon->percent_off)) {
            return ['percent' => Yii::$app->formatter->asDecimal($coupon->percent_off / 100, 2)];
        }
        return null;
    }

    public function getCustomer($customerId)
    {
        return Customer::retrieve($customerId);
    }

    public function getDefaultCard(Customer $customer)
    {
        if (!empty($customer->cards->data)) {
            foreach ($customer->cards->data as $card) {
                if ($card->id == $customer->default_card) {
                    return $card;
                }
            }
        }
        return null;
    }

    public function saveCustomer(User $user, $token = null)
    {
        $metaKey = UserMeta::findAllSorted([
            'user_id' => $user->id,
            'key' => 'stripe_customer_id'
        ]);
        if (empty($metaKey)) {
            $params = ['email' => $user->email];
            if ($token) {
                $params['source'] = $token;
            }
            $customer = Customer::create($params);
            if ($customer) {
                $metaKey = new UserMeta();
                $metaKey->setAttributes([
                    'user_id' => $user->id,
                    'key' => 'stripe_customer_id',
                    'value' => $customer->id,
                    'visible' => 0
                ]);
                $metaKey->save();
                return $customer->id;
            }
            else {
                return null;
            }
        }
        else {
            return $metaKey['stripe_customer_id'];
        }
    }

    public function chargeCustomer($customerId, UserPlan $userPlan, $customAmount = null)
    {
        $amount = round(($customAmount ? $customAmount : $userPlan->userPlanVariation->price) * 100);
        $chargeParams = [
            'customer' => $customerId,
            'amount' => $amount,
            'currency' => 'aud'
            //'tax_percent' => Yii::$app->params['gstTaxPercent'] // FIXME: change depending on user's location
        ];
        if (!empty($userPlan->stripe_coupon_id)) {
            $subscriptionParams['coupon'] = $userPlan->stripe_coupon_id;
        }
        try {
            $charge = Charge::create($chargeParams);
        }
        catch (\Exception $e) {
            return $e->getMessage();
        }
        if ($charge) {
            $userPlan->stripe_charge_id = $charge->id;
            $userPlan->save();
            return $charge->id;
        }
        return null;
    }

    public function cancelSubscription(UserPlan $userPlan)
    {
        $subscription = \Stripe\Subscription::retrieve($userPlan->stripe_subscription_id);
        if ($subscription) {
            $subscription->cancel(['at_period_end' => true]);
        }
    }

    public function subscribeCustomerToPlan($customerId, UserPlan $userPlan)
    {
        $subscriptionParams = [
            'customer' => $customerId,
            'plan' => $userPlan->userPlanVariation->stripe_plan_id,
            'tax_percent' => Yii::$app->params['gstTaxPercent'] // FIXME: change depending on user's location
        ];
        if (!empty($userPlan->stripe_coupon_id)) {
            $subscriptionParams['coupon'] = $userPlan->stripe_coupon_id;
        }
        try {
            $subscription = StripeSubscription::create($subscriptionParams);
        }
        catch (\Exception $e) {
            return $e->getMessage();
        }
        if ($subscription) {
            $userPlan->stripe_subscription_id = $subscription->id;
            $userPlan->save();
            return $subscription->id;
        }
        return null;
    }

    public function changeCustomerPlan(UserPlan $userPlan, $stripePlanId, $token = null)
    {
        if ($userPlan->stripe_subscription_id) {
            $subscription = StripeSubscription::retrieve($userPlan->stripe_subscription_id);
            $subscription->plan = $stripePlanId;
            return $subscription->save();
        }
        elseif ($token) {
            $customerId = $this->saveCustomer($userPlan->user, $token);
            if (!$customerId) {
                return false;
            }
            return $this->subscribeCustomerToPlan($customerId, $userPlan);
        }
        return false;
    }

    /**
     * Creates an invoice and sends it basing on the Stripe invoice.created event
     * @param array $eventData
     * @return bool
     */
    public function createInvoice(array $eventData)
    {
        // invoice.created event
        if (isset($eventData['lines']['data'][0]['id'])) {
            $i = 0;
            do {
                $userPlan = UserPlan::getByStripeSubscriptionID($eventData['lines']['data'][$i]['id']);
                $i++;
            }
            while (!$userPlan && isset($eventData['lines']['data'][$i]));
            if (!$userPlan) {
                // try to find user plan by customer_id (or email)
                $userPlan = UserPlan::getByStripeCustomerID($eventData['customer'], $eventData['lines']['data']);
                if (!$userPlan) {
                    return false;
                }
            }
            $invoice = Invoice::generateInvoice($userPlan, $eventData);
            // send the invoice
            //$invoice->sendEmail(true, false);
            return $invoice;
        }
        // invoiceitem.created event
        elseif (isset($eventData['plan'])) {
            $userPlan = UserPlan::getByStripeSubscriptionID($eventData['subscription']);
            if (!$userPlan) {
                return false;
            }
            $invoice = Invoice::generateUpdateInvoice($userPlan, $eventData);
            // send the invoice
            //$invoice->sendEmail(true, false);
            return $invoice;
        }
        return false;
    }

    /**
     * Marks invoice as paid and updates valid_until field basing on the invoice.payment_succeeded Stripe event
     * @param array $eventData
     * @param Invoice|null $invoice
     * @return bool
     */
    public function invoicePaid(array $eventData, $invoice)
    {
        if (!$invoice) {
            return false;
        }
        if (!isset($eventData['lines']['data'][0]['id'])) {
            return false;
        }
        $i = 0;
        do {
            $subscription = $eventData['lines']['data'][$i];
            $userPlan = UserPlan::getByStripeSubscriptionID($subscription['id']);
            $i++;
        }
        while (!$userPlan && isset($eventData['lines']['data'][$i]));
        if (!$userPlan) {
            return false;
        }
        // extend the plan validity for the next period
        // (just get the end date provided by Stripe and add some grace period)
        $updateValidUntil = true;
        $validUntil = $subscription['period']['end'];
        if (!empty($userPlan->userPlanVariation->access_length)) {
            // this WORKS DIFFERENTLY if the plan has access_length field set
            $maxValidUntil = date(
                'Y-m-d',
                strtotime(UserPlanVariation::getDaysMonths($userPlan->userPlanVariation->access_length), strtotime($userPlan->start_time))
            );
            if (date('Y-m-d', $validUntil) > $maxValidUntil) {
                $updateValidUntil = false;
            }
        }
        if ($updateValidUntil) {
            $userPlan->valid_until = date('Y-m-d 23:59:59', strtotime('+' . self::GRACE_PERIOD_DAYS . ' days', $validUntil));
        }
        // need to cancel the subscription if CANCEL_AFTER is set
        if (!empty($userPlan->userPlanVariation->cancel_after)) {
            //$daysMonths = $userPlan->userPlanVariation->cancel_after . ($userPlan->userPlanVariation->cancel_after > 12 ? ' days' : ' months');
            // cancel after will always be in months for now
            $daysMonths = $userPlan->userPlanVariation->cancel_after . ' months';
            $shouldBeCanceledOn = date('Y-m-d', strtotime('+' . $daysMonths, strtotime($userPlan->start_time)));
            if (date('Y-m-d', $validUntil) >= $shouldBeCanceledOn) {
                $this->cancelSubscription($userPlan);
            }
        }
        if ($userPlan->save()) {
            $invoice->updateInvoiceLines($userPlan);
            // send the receipt
            $invoice->sendEmail(false);
        }
        return true;
    }

    public function syncStripePlans()
    {
        $plans = $this->getPlans();
        foreach ($plans as $plan) {
            $planLength = null;
            $isWeekly = false;
            switch ($plan->interval) {
                case 'year':
                    $planLength = $plan->interval_count * 12;
                    break;
                case 'month':
                    $planLength = $plan->interval_count;
                    break;
                case 'week':
                    $planLength = $plan->interval_count;
                    $isWeekly = true;
                    break;
            }
            // ignore plans with duration less than a month
            if ($planLength == null) {
                continue;
            }

            $variation = UserPlanVariation::findOne(['stripe_plan_id' => $plan->id]);
            if ($variation) {
                // just update fields if necessary
                $variation->setAttributes([
                    'name' => $plan->name,
                    'plan_length' => $planLength,
                    'plan_length_in_weeks' => $isWeekly ? 1 : 0,
                    'price' => round($plan->amount / 100.0, 2)
                ]);
            }
            else {
                // prepare plan name
                $planName = trim(strtr($plan->name, [
                    'Yearly' => '',
                    'Monthly' => '',
                    'Weekly' => '',
                    'yearly' => '',
                    'monthly' => '',
                    'weekly' => ''
                ]));
                $userPlanType = UserPlanType::find()->where([
                    'like', 'name', $planName
                ])->one();
                if (!$userPlanType) {
                    $userPlanType = new UserPlanType();
                    $userPlanType->name = $planName;
                    $userPlanType->save();
                }

                $variation = new UserPlanVariation();
                $variation->setAttributes([
                    'name' => $plan->name,
                    'stripe_plan_id' => $plan->id,
                    'user_plan_type_id' => $userPlanType->id,
                    'plan_length' => $planLength,
                    'plan_length_in_weeks' => $isWeekly ? 1 : 0,
                    'price' => round($plan->amount / 100.0, 2)
                ]);
            }
            $variation->save();
        }
    }

    public function getPlans()
    {
        return Plan::all(['limit' => 100])->data;
    }
}