<?php

namespace backend\controllers;

use backend\components\helpers\DataTables;
use backend\components\helpers\DripHelper;
use backend\components\helpers\Frequencies;
use backend\models\db\AccountAdjustment;
use backend\models\db\AccountBalance;
use backend\models\db\Eoms;
use backend\models\form\AddAccount;
use backend\widgets\Tooltip;
use Yii;

use yii\filters\AccessControl;

use yii\helpers\ArrayHelper;
use backend\components\helpers\Formatter;

use backend\components\CustomController;

use backend\models\db\Account;
use backend\models\db\Budget;
use backend\models\db\Debt;
use backend\models\db\Expense;
use backend\models\db\Income;
use backend\models\db\Jar;
use backend\models\db\Transfer;

use backend\models\form\RemoveAccount;

use backend\components\helpers\JsonTools;
use yii\helpers\Url;


/**
 * Accounts controller
 */
class AccountsController extends CustomController
{
    public $enableCsrfValidation = false;

    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'account-detail',
                            'account-detail-info',
                            'account-transactions-table-data',
                            'balance-sheet-table',
                            'balance-sheet-adjustments-table',
                            'add-account',
                            'edit-account',
                            'remove-account',
                            'adjust-balance'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];
    }


    public function beforeAction($action)
    {
        // for mock budgets navigation
        if (in_array($action->id, ['account-detail', 'account-detail-info'])) {
            $account = Account::findOne(['id' => Yii::$app->request->getQueryParam('id')]);
            if ($account && $account->budget->is_mock) {
                $this->mockBudgetId = $account->budget_id;
                $this->view->params['mockBudgetId'] = $account->budget_id;
            }
        }
        elseif ($budgetId = Yii::$app->request->getQueryParam('budget_id')) {
            $budget = Budget::findOne($budgetId);
            if ($budget->is_mock) {
                $this->mockBudgetId = $budget->id;
                $this->view->params['mockBudgetId'] = $budget->id;
            }
        }
        return parent::beforeAction($action);
    }

    /**
     * Show accounts
     */
    public function actionIndex()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        $accounts = $this->_getAccountsMenuList($budgetId);

        $this->view->title = 'My Accounts';
        // pass mock budget specific data
        $mockBudgetData = [];
        if ($budgetId == $this->mockBudgetId) {
            $this->mockBudgetId = $budgetId;
            $budget = Budget::findOne($budgetId);
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['budgetId'] = $budget->id;
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['isMockBudget'] = true;
        }

        return $this->render('index', $mockBudgetData + [
            'accounts' => $accounts,
            'defaultAccountId' => $this->defaultAccountId,
            'allowEdit' => true
        ]);
    }


    /**
     * Add a new account
     */
    public function actionAddAccount()
    {
        $accountType = Yii::$app->request->get('type') == 'cc' ? 2 : 1;
        $budgetId    = $this->getBudgetIdFromUrl();
        $this->mockBudgetId = $budgetId;

        $accountForm = new AddAccount();
        $accountForm->account_type_id = $accountType;
        if (Yii::$app->request->isPost && $accountForm->load(Yii::$app->request->post())) {
            $accountForm->budget_id = $budgetId;

            // set validation attributes
            if ($accountForm->validate()) {
                $formData = $accountForm->getAttributes();
                // check whether to create a debt (for cc)
                if ($accountForm->account_type_id == 2) {
                    $budget = Budget::findOne([$this->activeBudgetId]);

                    $account = new Account();
                    $account->setAttributes($formData);
                    $account->budget_id = $budgetId;
                    $account->archived = $accountForm->is_debt_active ? 0 : 1;
                    $account->save();

                    // need to create associated debt anyway
                    // if CC account value is positive, there's no debt on the card (debt amount = 0)
                    $formData['amount'] = $formData['balance'] < 0 ? -1 * $formData['balance'] : 0;
                    $formData['account_id'] = $budget->default_account_id;
                    $formData['active_debt_account_id'] = $account->id;
                    $formData['archived'] = 0;
                    Debt::createDebtFromForm($formData, $this->userMeta['fts_step'] != 'completed');
                    if (Yii::$app->request->isAjax) {
                        echo JsonTools::successMessage(
                            'Account and associated debt successfully created.'
                        );
                        die();
                    }
                }
                // or just a regular account
                else {
                    $account = new Account();
                    $account->setAttributes($formData);
                    $account->budget_id = $budgetId;
                    $account->save();

                    // see if this is the default account
                    if ($accountForm->is_central_account) {
                        $budget = Budget::findOne([$this->activeBudgetId]);
                        $budget->default_account_id = $account->id;
                        $budget->save();
                        Yii::$app->session->set('defaultAccountId', $account->id);
                    }

                    DripHelper::sendCustomEvent('Account created', $this->user->email, [
                        'name' => $account->name
                    ]);

                    if (Yii::$app->request->isAjax) {
                        echo JsonTools::successMessage('Account successfully created.');
                        die();
                    }
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($accountForm);
                die();
            }
        }
        else {
            $frequencies = Frequencies::getAll();
            $this->view->title = $accountType == 1 ? 'Add Account' : 'Add Credit Card Account';
            $this->jsCallbackToView();

            return $this->render($accountType == 1 ? 'add-account' : 'add-cc-account', [
                'frequencies' => $frequencies,
                'budgetId' => $budgetId,
                'formModel' => $accountForm,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId)
            ]);
        }
    }


    /**
     * Edit an account
     */
    public function actionEditAccount()
    {
        $accountId      = Yii::$app->getRequest()->getQueryParam('id', false);
        $firstTimeSetup = Yii::$app->getRequest()->getQueryParam('first_time_setup', false);

        if (!$firstTimeSetup) {
            if (!is_numeric($accountId)) {
                return $this->accessError();
            }
            $account = Account::findOne($accountId);
        }
        else {
            $account = Account::findOne($this->defaultAccountId);
        }
        if (!$account || !Yii::$app->user->can('accessBudget', ['budgetId' => $account->budget_id])) {
            return $this->accessError();
        }
        $this->mockBudgetId = $account->getMockBudgetId();

        $accountDebt = null;
        $accountForm = new AddAccount();
        if ($account->account_type_id != 1) {
            // pre-init the record with associated debt, if possible
            if ($account->getAccountDebts()->exists()) {
                $accountDebt = $account->getAccountDebts()->one();
                $accountForm->setAttributes($accountDebt->getAttributes());
                // make sure we don't overwrite the core account details
                $accountForm->name = $account->name;
                $accountForm->balance = $account->balance;
            }
            // we always need to have an associated debt for debt-/cc-type account
            else {
                $accountDebt = new Debt();
                $accountDebt->active_debt_account_id = $account->id;
            }
        }

        if (Yii::$app->request->isPost && $accountForm->load(Yii::$app->request->post())) {
            $accountForm->budget_id       = $this->activeBudgetId;
            $accountForm->account_type_id = $account->account_type_id;
            $accountForm->active_debt_account_id = $account->id;

            // set validation attributes
            if ($accountForm->validate()) {
                $formData = $accountForm->getAttributes();
                // check whether to move to a debt (for inactive debt)
                if ($account->account_type_id != 1 && !$accountForm->is_debt_active) {
                    $account->setAttributes($formData);
                    $account->archived = 1;
                    $account->save();

                    if ($accountForm->account_type_id == 2) {
                        $formData['amount'] = $formData['balance'] < 0 ? -1 * $formData['balance'] : 0;
                    }
                    else {
                        $formData['amount'] = $formData['balance'];
                    }
                    // update the debt
                    if ($accountDebt) {
                        $accountDebt->setAttributes($formData);
                        $accountDebt->archived = 0;
                        $accountDebt->save();
                    }
                    else {
                        // create a new debt record
                        Debt::createDebtFromForm($formData, $this->userMeta['fts_step'] != 'completed');
                    }

                    if (Yii::$app->request->isAjax) {
                        echo JsonTools::successMessage('Account successfully moved to debts'.($account->account_type_id == 2 ? ' as an inactive credit card' : '').'.');
                        die();
                    }
                }
                else {
                    $account->setAttributes($formData);
                    $account->save();
                    // see if this is the default account
                    if ($accountForm->is_central_account) {
                        $budget = Budget::findOne([$this->activeBudgetId]);
                        $budget->default_account_id = $account->id;
                        $budget->save();
                        Yii::$app->session->set('defaultAccountId', $account->id);
                    }
                    if (Yii::$app->request->isAjax) {
                        echo JsonTools::successMessage('Account successfully edited.');
                        die();
                    }
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($accountForm);
                die();
            }
        }
        else {
            $frequencies = Frequencies::getAll();
            $noClose = Yii::$app->getRequest()->getQueryParam('no_close', false);
            $accountForm->setAttributes($account->getAttributes());
            $this->view->title = 'Edit '.($account->account_type_id == 2 ? 'Credit Card' : ($account->account_type_id == 3 ? 'Debt' : '')).' Account';
            $this->jsCallbackToView();

            return $this->render($account->account_type_id == 1 ? 'edit-account' : 'edit-cc-account', [
                'frequencies' => $frequencies,
                'formModel' => $accountForm,
                'accountId' => $account->id,
                'isDefaultAccount' => $this->defaultAccountId == $account->id,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'noClose' => $noClose,
                'adjustable' => Eoms::getLastEoms($this->activeBudgetId, false) == null
            ]);
        }
    }

    /**
     * Show account detail
     */
    public function actionAccountDetail()
    {
        $accountId = Yii::$app->getRequest()->getQueryParam('id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content
        if (!is_numeric($accountId)) {
            return $this->accessError();
        }
        $account = Account::findOne($accountId);
        if ($account->archived) {
            return $this->redirect(['/accounts/index']);
        }
        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $account->budget_id])) {
            return $this->accessError();
        }

        $this->view->title = $account->name . ' account';
        // pass mock budget specific data
        $mockBudgetData = [];
        if ($account->budget->is_mock) {
            $budget = $account->getBudget()->one();
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['budgetId'] = $budget->id;
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['isMockBudget'] = true;
        }

        if ($this->pageLoadRequest || $fullLoad) {
            $accounts = $this->_getAccountsMenuList($account->budget_id);
            return $this->render('account-detail', $mockBudgetData + [
                'account' => $account,
                'accounts' => $accounts,
                'defaultAccountId' => $this->defaultAccountId,
                'allowEdit' => true
            ]);
        }
        else {
            return $this->render('account-detail-content', $mockBudgetData + [
                'account' => $account,
                'allowEdit' => true
            ]);
        }
    }

    public function actionAccountDetailInfo()
    {
        $accountId = Yii::$app->getRequest()->getQueryParam('id', false);
        $account = Account::findOne($accountId);
        if (!$account || !Yii::$app->user->can('accessBudget', ['budgetId' => $account->budget_id])) {
            return $this->accessError();
        }
        echo json_encode([
            'account-name' => $account->name,
            'account-current-balance' => Formatter::currency($account->getCurrentBalance())
        ]);
        die();
    }

    /**
     * Delete an existing account
     */
    public function actionRemoveAccount()
    {
        // check number of accounts left for this budget
        $accounts = $this->_getAccountsMenuList();
        if (count($accounts) < 2) {
            echo JsonTools::errorMessage('You cannot remove all accounts from your budget.');
            die();
        }
        $accountId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);
        $hardDelete = Yii::$app->getRequest()->getQueryParam('hard', false);
        if (!is_numeric($accountId)) {
            return $this->accessError();
        }
        $account = Account::findOne($accountId);
        if (!$account || !Yii::$app->user->can('accessBudget', ['budgetId' => $account->budget_id])) {
            return $this->accessError();
        }
        $this->mockBudgetId = $account->getMockBudgetId();
        $associatedDebt = Debt::findOne(['active_debt_account_id' => $account->id]);

        $hardDeletable = Eoms::isEntryHardDeletable($this->activeBudgetId, $account);
        if ($associatedDebt) {
            // check also if the related debt is hard deletable
            $hardDeletable = $hardDeletable && Eoms::isEntryHardDeletable($this->activeBudgetId, $associatedDebt);
        }

        $accountForm = new RemoveAccount();
        // find accounts to choose from
        $otherAccounts = ArrayHelper::map(Account::findAll(['budget_id' => $this->activeBudgetId, 'archived' => 0]), 'id', 'name');
        unset($otherAccounts[$account->id]);
        $accountForm->is_default_account = ($this->defaultAccountId == $account->id);

        $expenses = Expense::find()
            ->where([
                'account_id' => $account->id,
                'is_adjustment' => 0,
                'archived' => 0
            ])
            ->orderBy('name asc')
            ->all();

        // find debts to assign to other accounts
        $debts = Debt::find()
            ->where([
                'account_id' => $account->id,
                'archived' => 0
            ])
            ->orderBy('name asc')
            ->all();

        // find incomes to assign to other accounts
        $incomes = Income::find()
            ->where([
                'account_id' => $account->id,
                'is_adjustment' => 0,
                'archived' => 0
            ])
            ->orderBy('name asc')
            ->all();

        // find transfers to assign to other accounts
        $transfers = Transfer::find()
            ->where(['or',
                ['account_from_id' => $account->id],
                ['account_to_id' => $account->id]
            ])
            ->andWhere([
                'is_adjustment' => 0,
                'archived' => 0
            ])
            ->orderBy('name asc')
            ->all();

        if ($confirm) {
            if (Yii::$app->request->isPost && $accountForm->load(Yii::$app->request->post())) {
                $accountForm->budget_id = $this->activeBudgetId;

                // set validation attributes
                if ($accountForm->validate()) {
                    // transfer default account attr
                    if ($accountForm->is_default_account) {
                        $budget = Budget::findOne($this->activeBudgetId);
                        $budget->default_account_id = $accountForm->default_account_id;
                        $budget->save();
                        Yii::$app->session->set('defaultAccountId', $accountForm->default_account_id);
                    }

                    // add balance to other account
                    if (!$hardDelete && $accountForm->new_account_id > 0) {
                        $transfer = new Transfer();
                        $transfer->setAttributes([
                            'budget_id' => $this->activeBudgetId,
                            'account_from_id' => $account->id,
                            'account_to_id' => $accountForm->new_account_id,
                            'name' => "Funds from deleted '{$account->name}' account",
                            'frequency' => 'one-time',
                            'date' => date('Y-m-d'),
                            'amount' => $account->balance
                        ]);
                        $transfer->save();
                    }

                    // update debt account IDs
                    if (!empty($accountForm->debt_account_id) && !empty($debts)) {
                        Debt::updateAll([
                            'account_id' => $accountForm->debt_account_id
                        ], [
                            'id' => ArrayHelper::map($debts, 'id', 'id')
                        ]);
                    }

                    // update expense account IDs
                    if (!empty($accountForm->expense_account_id) && !empty($expenses)) {
                        Expense::updateAll([
                            'account_id' => $accountForm->expense_account_id
                        ], [
                            'id' => ArrayHelper::map($expenses, 'id', 'id')
                        ]);
                    }

                    // update income account IDs
                    if (!empty($accountForm->income_account_id) && !empty($incomes)) {
                        Income::updateAll([
                            'account_id' => $accountForm->income_account_id
                        ], [
                            'id' => ArrayHelper::map($incomes, 'id', 'id')
                        ]);
                    }

                    // update transfer account IDs
                    if (!empty($accountForm->transfer_account_id) && !empty($transfers)) {
                        Transfer::updateAll([
                            'account_from_id' => $accountForm->transfer_account_id
                        ], [
                            'id' => ArrayHelper::map($transfers, 'id', 'id'),
                            'account_from_id' => $account->id
                        ]);
                        Transfer::updateAll([
                            'account_to_id' => $accountForm->transfer_account_id
                        ], [
                            'id' => ArrayHelper::map($transfers, 'id', 'id'),
                            'account_to_id' => $account->id
                        ]);
                    }

                    if (Yii::$app->user->hasFinishedFts()) {
                        if ($hardDelete && $hardDeletable) {
                            if ($associatedDebt) {
                                $associatedDebt->deleteCompletely();
                            }
                            $account->deleteCompletely();
                        }
                        else {
                            $account->archived = 1;
                            $account->save();
                            if ($associatedDebt) {
                                $associatedDebt->archived = 1;
                                $associatedDebt->save();
                            }
                        }
                    }
                    else {
                        if ($associatedDebt) {
                            $associatedDebt->deleteCompletely();
                        }
                        $account->deleteCompletely();
                    }
                    echo JsonTools::successMessage('Account '.($associatedDebt ? 'with the associated debt ' : '').'successfully removed.');
                    die();
                }
                elseif (Yii::$app->request->isAjax) {
                    echo JsonTools::formErrorMessage($accountForm);
                    die();
                }
            }
        }

        $this->view->title = 'Remove Account';
        $this->jsCallbackToView();
        return $this->render('remove-account', [
            'formModel' => $accountForm,
            'accountId' => $accountId,
            'otherAccountsWithNone' => [0 => 'None'] + $otherAccounts,
            'otherAccounts' => $otherAccounts,
            'debts' => $debts,
            'expenses' => $expenses,
            'incomes' => $incomes,
            'transfers' => $transfers,
            'isDefaultAccount' => ($this->defaultAccountId == $account->id),
            'hardDeletable' => $hardDeletable
        ]);
    }

    public function actionAdjustBalance() {

        $operationType = Yii::$app->getRequest()->getQueryParam('type', false);
        $operationId   = Yii::$app->getRequest()->getQueryParam('id', false);
        $jsCallback    = Yii::$app->getRequest()->getQueryParam('js_callback', false);
        $accountId     = Yii::$app->getRequest()->getQueryParam('account_id', false);
        if (!is_numeric($operationId) || !$operationType || !is_numeric($accountId)) {
            return $this->accessError();
        }
        $account = Account::findOne($accountId);
        // check if user can edit jar
        if (!$account || !Yii::$app->user->can('accessBudget', ['budgetId' => $account->budget_id])) {
            return $this->accessError();
        }

        $amount       = Yii::$app->request->post('amount', false);
        $amountBefore = Yii::$app->request->post('amount_before', 0);

        if ($amount !== false && Formatter::float($amount) != Formatter::float($amountBefore)) {

            $accountAdj = new AccountAdjustment();
            $accountAdj->setAttributes([
                'account_id'     => $account->id,
                'row_key'        => $operationType . $operationId,
                'balance'        => $amount,
                'balance_before' => $amountBefore
            ]);
            $accountAdj->save();

        }

        echo json_encode([
            'value' => Formatter::currency($amount),
            'jsCallback' => $jsCallback
        ]);
        die();

    }


    public function actionBalanceSheetAdjustmentsTable() {

        $accountId = Yii::$app->getRequest()->getQueryParam('account_id', false);
        if (!is_numeric($accountId)) {
            return $this->accessError();
        }
        $account = Account::findOne($accountId);

        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $account->budget_id])) {
            return $this->accessError();
        }

        $columnNames = [
            'date',
            'name',
            'balance_before',
            'balance'
        ];

        $adjustments = AccountAdjustment::find()->where(['account_id' => $accountId])->all();
        $responseData = DataTables::getFilteredData($columnNames, $adjustments, true);

        foreach ($responseData['items'] as $i) {

            $responseData['response']['data'][] = [
                Formatter::date($i->date),
                $i->name,
                Formatter::currency($i->balance_before),
                Formatter::currency($i->balance),
                ''
            ];

        }

        echo json_encode($responseData['response']);
        die();

    }

    public function actionBalanceSheetTable() {

        $accountId = Yii::$app->getRequest()->getQueryParam('account_id', false);
        if (!is_numeric($accountId)) {
            return $this->accessError();
        }
        $account = Account::findOne($accountId);

        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $account->budget_id])) {
            return $this->accessError();
        }

        $bsStart = Yii::$app->getRequest()->getQueryParam('bs_start', false);
        $bsStart = $bsStart ? date('Y-m-d', strtotime($bsStart)) : date('Y-m-d', strtotime('-30 days'));
        $bsEnd   = Yii::$app->getRequest()->getQueryParam('bs_end', false);
        $bsEnd   = $bsEnd ? date('Y-m-d', strtotime($bsEnd)) : date('Y-m-d');

        $columnNames = [
            'checked', // non-sortable
            'date',
            'name',
            'amount'
        ];

        $adjustments = [];
        $displayOperations = [];
        $balances = [];
        $thisMonthStart = date('Y-m-01');

        // in order to calculate the balances correctly, we always need to get the operations
        // from the 1st of the month
        $operations = $account->getTimePeriodMoneyOperations(date('Y-m-01', strtotime($bsStart)), $bsEnd);

        // sort chronologically to calculate balances first
        $responseData = DataTables::getFilteredData($columnNames, $operations, [
            'start' => '0',
            'length' => '999999',
            'orderColumn' => ['date', 'id'],
            'orderDirection' => 'asc'
        ]);
        // calculate balances for each balance sheet row
        // for the entries from this month
        $timeNow = strtotime(date('Y-m-20'));
        $lastMonthMonth = date('m', strtotime('-1 month', $timeNow));
        $lastMonthYear  = date('Y', strtotime('-1 month', $timeNow));
        $currentBalance = $account->getCurrentBalance(
            $lastMonthMonth,
            $lastMonthYear
        );
        foreach ($responseData['items'] as $i) {
            $theDate = date('Y-m-d', strtotime($i->date));
            if ($theDate >= $thisMonthStart) {
                $operationKey = Account::getOperationKey($i);
                $adjustment   = AccountAdjustment::find()->where([
                    'account_id' => $accountId,
                    'row_key'    => $operationKey
                ])->one();
                $currentBalance = $adjustment ?
                    $adjustment->balance :
                    $account->getBalanceAfterOperation($currentBalance, $i);
                $balances[$operationKey] = $currentBalance;
                $adjustments[$operationKey] = $adjustment ? true : false;
            }
            if ($theDate >= $bsStart && $theDate <= $bsEnd) {
                $displayOperations[] = $i;
            }
        }

        // calculate balances for each balance sheet row
        // for the entries from previous months
        $responseData = DataTables::getFilteredData($columnNames, $operations, [
            'start' => '0',
            'length' => '999999',
            'orderColumn' => ['date', 'id'],
            'orderDirection' => 'desc'
        ]);
        $currentBalance = $account->getCurrentBalance(
            date('m', strtotime('-1 month', $timeNow)),
            date('Y', strtotime('-1 month', $timeNow))
        );
        foreach ($responseData['items'] as $ind => $i) {
            $theDate = date('Y-m-d', strtotime($i->date));
            if ($theDate < $thisMonthStart) {
                $operationKey = Account::getOperationKey($i);
                $bankStatType = Account::getOperationType($i);
                $adjustment   = AccountAdjustment::find()->where([
                    'account_id' => $accountId,
                    'row_key'    => $bankStatType . $i->id
                ])->one();
                $balances[$operationKey] = $adjustment ?
                    $adjustment->balance : $currentBalance;
                $currentBalance = $account->getBalanceAfterOperation($currentBalance, $i, true);
                $adjustments[$operationKey] = $adjustment ? true : false;
            }
        }

        $responseData = DataTables::getFilteredData($columnNames, $displayOperations, true);
        $responseData['balances'] = $balances;

        foreach ($responseData['items'] as $ind => $i) {

            $bankStatType     = Account::getOperationType($i);
            $bankStatName     = Account::getOperationKey($i);
            $checked          = $i->checked ? ' checked' : '';
            $amount           = $account->getAmountWithSign($i);
            $balance          = $responseData['balances'][$bankStatName];
            $fmtAmount        = Formatter::currency($amount);
            $fmtBalance       = Formatter::currency($balance);
            $negAmountCls     = 0 > $amount ? 'gj-is-negative' : '';
            $negBalanceCls    = 0 > $balance ? 'gj-is-negative' : '';

            $amountEditUrl = Url::toRoute(['/accounts/adjust-balance', 'account_id' => $accountId, 'id' => $i->id, 'type' => $bankStatType, 'js_callback' => 'reloadAccountsAfterBalanceEditAmount']);
            $responseData['response']['data'][] = [
                '<div class="checkbox bank-statement-check" data-operation-type="' . $bankStatType . '" data-operation-id="' . $i->id . '">
                    <input type="hidden" name="Check[' . $bankStatName . ']" value="0">
                    <input type="checkbox" id="check-balance-' . $bankStatName . '" name="Check[' . $bankStatName . ']" value="1"' . $checked . '>
                    <label for="check-balance-' . $bankStatName . '"></label>
                </div><span class="center-content"></span>' . ($i->is_adjustment ? '<span class="row-bg-blue"></span>' : ''),
                Formatter::date($i->date),
                isset($i->name) ? $i->name : (isset($i->expense_name) ? $i->expense_name : (isset($i->account_from_name) ? $i->account_from_name . ' -> ' . $i->account_to_name : '-')),
                !empty($i->jar_name) ? $i->jar_name : '-',
                "<span class='$negAmountCls' data-value>$fmtAmount</span>",
                "<span class='$negBalanceCls' data-value>$fmtBalance</span>",
                $adjustments[$bankStatName] ? Tooltip::widget(['text' => 'This row has a balance adjustment.']) :
                    ($i->is_adjustment ? Tooltip::widget(['text' => 'This row is an adjustment transaction.']) : '')
            ];

        }

        echo json_encode($responseData['response']);
        die();
    }

    private function _getAccountsMenuList($budgetId = false)
    {
        if (!$budgetId) {
            $budgetId = $this->activeBudgetId;
        }

        $accounts = Account::find()
            ->where(['budget_id' => $budgetId, 'archived' => 0])
            ->orderBy('name ASC')
            ->all();
        return $accounts;
    }

}
