<?php

namespace backend\controllers;

use backend\models\db\SearchLog;
use backend\models\db\UserRole;
use Yii;

use yii\filters\AccessControl;

use yii\helpers\ArrayHelper;
use yii\helpers\Url;

use backend\components\CustomController;

use backend\models\db\Budget;
use backend\models\db\DebtChange;
use backend\models\db\ExpenseChange;
use backend\models\db\IncomeChange;
use backend\models\db\JarChange;
use backend\models\db\TransferChange;
use backend\models\db\User;
use backend\models\db\UserParent;

use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;


/**
 * Activity log controller
 */
class ActivityLogController extends CustomController
{
    public $fileActions = [
        'search-log'
    ];

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'activity-log-detail',
                            'activity-log-table'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                    [
                        'actions' => [
                            'activity-log-by-coach-table'
                        ],
                        'allow' => true,
                        'roles' => ['coach', 'superadmin'],
                    ],
                    [
                        'actions' => [
                            'search-log'
                        ],
                        'allow' => true,
                        'roles' => ['superadmin']
                    ]
                ],
            ]
        ];

    }


    /**
     * Index page with the activity log
     */
    public function actionIndex()
    {
        $this->view->title = 'Activity Log';
        return $this->render('index');
    }

    public function actionSearchLog()
    {
        $logs = SearchLog::exportSearches();
        $content = '"Query","User Name","User Email","Date and Time"'."\n";
        foreach ($logs as $log) {
            $columns = [];
            foreach ($log as $col) {
                $columns[] = strtr($col, ['"' => '""']);
            }
            $content .= '"'.implode('","', $columns).'"'."\n";
        }
        return Yii::$app->response->sendContentAsFile($content, 'search_logs.csv', [
            'mimeType' => 'text/csv'
        ]);
    }

    /**
     * The detail of an activity log, used primarily for edit activities
     */
    public function actionActivityLogDetail() {

        $id = Yii::$app->getRequest()->getQueryParam('id', false);
        $type = Yii::$app->getRequest()->getQueryParam('type', false);

        $canAccess = true;

        if ($type == 'Debt') {

            $change = DebtChange::findOne($id);
            $parent = $change->debt;

            if (!$parent) {

                $canAccess = false;

            }
            else {

                $budget = $parent->budget;

                if (!$budget || !Yii::$app->user->can('accessBudget', ['budgetId' => $budget->id])) {

                    $canAccess = false;

                }
                else {

                    $labels = $parent->attributeLabels();
                    $meta = $this->_getMetaForActivityLogDetail([
                        'parent' => $parent,
                        'parentId' => 'debt_id',
                        'change' => $change,
                        'metaChangesMethodName' => 'debtChangeMetas',
                        'changeModelName' => 'backend\models\db\DebtChange'
                    ]);

                }
            }

        }
        elseif ($type == 'Expense') {

            $change = ExpenseChange::findOne($id);
            $parent = $change->expense;

            if (!$parent) {

                $canAccess = false;

            }
            else {

                $jar = $parent->jar;
                $budget = $jar->budget;

                if (!$budget || !Yii::$app->user->can('accessBudget', ['budgetId' => $budget->id])) {

                    $canAccess = false;

                }
                else {

                    $labels = $parent->attributeLabels();
                    $meta = $this->_getMetaForActivityLogDetail([
                        'parent' => $parent,
                        'parentId' => 'expense_id',
                        'change' => $change,
                        'metaChangesMethodName' => 'expenseChangeMetas',
                        'changeModelName' => 'backend\models\db\ExpenseChange'
                    ]);

                }
            }

        }
        elseif ($type == 'Income') {

            $change = IncomeChange::findOne($id);
            $parent = $change->income;

            if (!$parent) {

                $canAccess = false;

            }
            else {

                $budget = $parent->budget;

                if (!$budget || !Yii::$app->user->can('accessBudget', ['budgetId' => $budget->id])) {

                    $canAccess = false;

                }
                else {

                    $labels = $parent->attributeLabels();
                    $meta = $this->_getMetaForActivityLogDetail([
                        'parent' => $parent,
                        'parentId' => 'income_id',
                        'change' => $change,
                        'metaChangesMethodName' => 'incomeChangeMetas',
                        'changeModelName' => 'backend\models\db\IncomeChange'
                    ]);

                }
            }

        }
        elseif ($type == 'Jar') {

            $change = JarChange::findOne($id);
            $parent = $change->jar;

            if (!$parent) {

                $canAccess = false;

            }
            else {

                $budget = $parent->budget;

                if (!$budget || !Yii::$app->user->can('accessBudget', ['budgetId' => $budget->id])) {

                    $canAccess = false;

                }
                else {

                    $labels = $parent->attributeLabels();
                    $meta = $this->_getMetaForActivityLogDetail([
                        'parent' => $parent,
                        'parentId' => 'jar_id',
                        'change' => $change,
                        'metaChangesMethodName' => 'jarChangeMetas',
                        'changeModelName' => 'backend\models\db\JarChange'
                    ]);

                }
            }

        }
        elseif ($type == 'Transfer') {

            $change = TransferChange::findOne($id);
            $parent = $change->transfer;

            if (!$parent) {

                $canAccess = false;

            }
            else {

                $budget = $parent->budget;

                if (!$budget || !Yii::$app->user->can('accessBudget', ['budgetId' => $budget->id])) {

                    $canAccess = false;

                }
                else {

                    $labels = $parent->attributeLabels();
                    $meta = $this->_getMetaForActivityLogDetail([
                        'parent' => $parent,
                        'parentId' => 'transfer_id',
                        'change' => $change,
                        'metaChangesMethodName' => 'transferChangeMetas',
                        'changeModelName' => 'backend\models\db\TransferChange'
                    ]);

                }
            }

        }

        if ($canAccess) {

            $this->view->title = 'Activity Log Detail';

            return $this->render('activity-log-detail', [
                'type' => $type,
                'parent' => $parent,
                'labels' => $labels,
                'oldMetaSorted' => $meta['oldMetaSorted'],
                'newMetaSorted' => $meta['newMetaSorted']
            ]);

        }

    }

    /**
     * Activity log table data
     */
    public function actionActivityLogTable() {

        /**
         * Notes: After comparing view performance vs. running the query, the query was significantly faster, which is
         * why it is used.
         */
        $activityQuery = "
            SELECT * FROM (
                SELECT time, 'Debt' as type, debts.name as name, debt_changes.type as operation, user_id, debt_changes.id
                FROM debt_changes
                LEFT JOIN debts ON debts.id = debt_changes.debt_id
                UNION
                SELECT time, 'Expense' as type, expenses.name as name, expense_changes.type as operation, user_id, expense_changes.id
                FROM expense_changes
                LEFT JOIN expenses ON expenses.id = expense_changes.expense_id
                UNION
                SELECT time, 'Income' as type, incomes.name as name, income_changes.type as operation, user_id, income_changes.id
                FROM income_changes
                LEFT JOIN incomes ON incomes.id = income_changes.income_id
                UNION
                SELECT time, 'Jar' as type, jars.name as name, jar_changes.type as operation, user_id, jar_changes.id
                FROM jar_changes
                LEFT JOIN jars ON jars.id = jar_changes.jar_id
                UNION
                SELECT time, 'Transfer' as type, transfers.name as name, transfer_changes.type as operation, user_id, transfer_changes.id
                FROM transfer_changes
                LEFT JOIN transfers ON transfers.id = transfer_changes.transfer_id
            ) activity
            WHERE user_id = :user_id
            ORDER BY time ASC
        ";

        $activity = Yii::$app->db
            ->createCommand($activityQuery, [':user_id' => Yii::$app->user->id])
            ->queryAll();

        // filtering and getting record
        $columnNames = [
            'time',
            'type',
            'name',
            'operation',
        ];

        $responseData = DataTables::getFilteredData($columnNames, $activity);

        foreach ($responseData['items'] as $i) {

            $operationName = strtr($i['operation'], [
                'C' => 'Create',
                'E' => 'Edit',
                'P' => 'Pause',
                'K' => 'Continue',
                'D' => 'Delete',
                'R' => 'Remove',
                'U' => 'Edit'
            ]);

            $action = (($i['operation'] == 'E') || ($i['operation'] == 'U'))? '<a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['/activity-log/activity-log-detail', 'type' => $i['type'], 'id' => $i['id']]) . '"><i class="glyphicon glyphicon-list-alt gi-2x"></i></a>' : '-';

            $responseData['response']['data'][] = [
                Formatter::datetime(strtotime($i['time'])),
                $i['type'],
                $i['name'],
                $operationName,
                $action
            ];

        }

        echo json_encode($responseData['response']);
        die();

    }


    /**
     * Activity log data for coach's dashboard or admin's dashboard
     */
    public function actionActivityLogByCoachTable() {

        if (Yii::$app->user->isCoach) {

            $coachIds = [Yii::$app->user->id];

        }
        else {

            $coaches = User::findAll(['user_role_id' => UserRole::COACH]);
            $coachIds = ArrayHelper::map($coaches, 'id', 'id');

        }

        // get budget IDs here so we don't have to add more joins to the query that follows
        $budgetIds = Budget::find()
            ->select(Budget::tableName() . '.id')
            ->joinWith('user.userParent')
            ->where([UserParent::tableName() . '.user_parent_id' => $coachIds])
            ->column();

        $userIds = User::find()
            ->select(User::tableName() . '.id')
            ->joinWith('userParent')
            ->where([UserParent::tableName() . '.user_parent_id' => $coachIds])
            ->column();
        // add subclients
        $userIds = array_merge($userIds, User::find()
            ->select(User::tableName() . '.id')
            ->joinWith('userParent')
            ->where([UserParent::tableName() . '.user_parent_id' => $userIds])
            ->column());

        $activityQuery = "
            SELECT * FROM (
                SELECT time, 'Debt' as type, debts.name as name, debt_changes.type as operation, user_id, debt_changes.id, budget_id
                FROM debt_changes
                LEFT JOIN debts ON debts.id = debt_changes.debt_id
                UNION
                SELECT time, 'Expense' as type, expenses.name as name, expense_changes.type as operation, b.user_id, expense_changes.id, budget_id
                FROM expense_changes
                LEFT JOIN expenses ON expenses.id = expense_changes.expense_id
                LEFT JOIN jars as j ON j.id = expenses.jar_id
                LEFT JOIN budgets as b ON b.id = j.budget_id
                UNION
                SELECT time, 'Income' as type, incomes.name as name, income_changes.type as operation, user_id, income_changes.id, budget_id
                FROM income_changes
                LEFT JOIN incomes ON incomes.id = income_changes.income_id
                UNION
                SELECT time, 'Jar' as type, jars.name as name, jar_changes.type as operation, user_id, jar_changes.id, budget_id
                FROM jar_changes
                LEFT JOIN jars ON jars.id = jar_changes.jar_id
                UNION
                SELECT time, 'Transfer' as type, transfers.name as name, transfer_changes.type as operation, user_id, transfer_changes.id, budget_id
                FROM transfer_changes
                LEFT JOIN transfers ON transfers.id = transfer_changes.transfer_id
            ) activity
            WHERE 
                " . (!empty($budgetIds) ? "budget_id IN (" . implode(',', $budgetIds) . ") AND" : 'FALSE AND') . "
                " . (!empty($userIds) ? "user_id IN (" . implode(',', $userIds) . ") AND" : 'FALSE AND') . "
                DATE(time) > DATE_SUB(NOW(), INTERVAL 1 MONTH)
        ";

        $activity = Yii::$app->db
            ->createCommand($activityQuery)
            ->queryAll();

        // let's fetch and sort all the needed users now, since we'll need them all in a moment
        $users = User::find()
            ->where(['id' => $userIds])
            ->all();
        $sortedUsers = [];
        foreach ($users as $u) {
            $sortedUsers[$u->id] = $u;
        }

        // assign the names
        foreach($activity as $i => $a) {

            if (strlen($a['user_id']) && isset($a['user_id']) && isset($sortedUsers[$a['user_id']])) {

                $activity[$i]['username'] = $sortedUsers[$a['user_id']]->getFullName(true);
                if ($sortedUsers[$a['user_id']]->user_role_id == UserRole::COACH) {

                    $parent = $sortedUsers[$a['user_id']]->userParent->userParent;
                    $activity[$i]['parent_username'] = $parent ? $parent->getFullName(true) : '-';
                    $activity[$i]['parent_user_id'] = $parent ? $parent->id : '-';
                    $activity[$i]['coach_username'] = false;

                }
                elseif ($sortedUsers[$a['user_id']]->userParent->userParent->user_role_id == UserRole::COACH) {

                    $parent = $sortedUsers[$a['user_id']]->userParent->userParent;
                    $activity[$i]['coach_username'] = $parent ? $parent->getFullName(true) : '-';
                    $activity[$i]['coach_user_id'] = $parent ? $parent->id : '-';
                    $activity[$i]['parent_username'] = false;

                }
                else {

                    $activity[$i]['parent_username'] = false;
                    $activity[$i]['coach_username'] = false;

                }
            }
            else {

                $activity[$i]['username'] = false;
                $activity[$i]['parent_username'] = false;
                $activity[$i]['coach_username'] = false;

            }
        }

        // filtering and getting record
        $columnNames = [
            'time',
            'username',
            'type',
            'name',
            'operation',
            'parent_username',
            'coach_username'
        ];

        $responseData = DataTables::getFilteredData($columnNames, $activity);

        foreach ($responseData['items'] as $i) {

            $operationName = strtr($i['operation'], [
                'C' => 'Create',
                'E' => 'Edit',
                'P' => 'Pause',
                'K' => 'Continue',
                'D' => 'Delete',
                'R' => 'Remove',
                'U' => 'Edit'
            ]);

            $action = (($i['operation'] == 'E') || ($i['operation'] == 'U'))? '<a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['/activity-log/activity-log-detail', 'type' => $i['type'], 'id' => $i['id']]) . '"><i class="glyphicon glyphicon-list-alt gi-2x"></i></a>' : '-';
            if ($i['username']) {

                if ($i['parent_username']) {

                    $username       = $i['username'];
                    $parentUsername = '<a href="' . Url::toRoute(['/admin/user/switch', 'id' => $i['parent_user_id']]) . '">' . $i['parent_username'] . '</a>';
                    $coachUsername  = '-';

                }
                elseif ($i['coach_username']) {

                    $username       = $i['username'];
                    $parentUsername = '-';
                    $coachUsername  = $i['coach_username'];

                }
                else {

                    $username       = '<a href="' . Url::toRoute(['/admin/user/switch', 'id' => $i['user_id']]) . '">' . $i['username'] . '</a>';
                    $parentUsername = '-';
                    $coachUsername  = '-';

                }

            }
            else {

                $username       = '-';
                $parentUsername = '-';
                $coachUsername  = '-';

            }

            $responseData['response']['data'][] = [
                Formatter::datetime(strtotime($i['time'])),
                $username,
                $i['type'],
                $i['name'],
                $operationName,
                $parentUsername,
                $coachUsername,
                $action
            ];

        }

        echo json_encode($responseData['response']);
        die();

    }


    /**
     * Get the meta data for an activity log detail
     * @param array $params [
     *     object $parent,
     *     object $change,
     *     string $metaChangesMethodName,
     *     string $changeModelName
     *     integer $parentId
     * ]
     * @return array New and old meta lists.
     */
    private function _getMetaForActivityLogDetail($params) {

        $oldMeta = $params['change']->{$params['metaChangesMethodName']};
        $oldMetaSorted = ArrayHelper::map($oldMeta, 'key', 'value');
        $oldMetaCount = count($oldMetaSorted);
        $newMetaCount = 0;
        $newMetaSorted = [];
        $previousChangeId = $params['change']->id;

        // get the new values
        while ($newMetaCount < $oldMetaCount) {

            $newChange = call_user_func([$params['changeModelName'], 'find'])
                ->where([
                    'and',
                    [$params['parentId'] => $params['change']->{$params['parentId']}],
                    ['>', 'id', $previousChangeId],
                    ['type' => $params['change']->type]
                ])
                ->orderBy('id asc')
                ->one();

            if ($newChange) {

                $previousChangeId = $newChange->id;
                $newChangeMeta = $newChange->{$params['metaChangesMethodName']};

                foreach ($newChangeMeta as $meta) {

                    if (isset($oldMetaSorted[$meta->key]) && !isset($newMetaSorted[$meta->key])) {

                        $newMetaSorted[$meta->key] = $meta->value;
                        $newMetaCount++;

                    }

                }

            }
            else {

                foreach ($oldMetaSorted as $key => $o) {

                    if (!isset($newMetaSorted[$key])) {

                        $newMetaSorted[$key] = $params['parent']->{$key};

                    }

                }

                break;

            }

        }

        return [
            'oldMetaSorted' => $oldMetaSorted,
            'newMetaSorted' => $newMetaSorted
        ];

    }

}
