<?php

namespace backend\controllers;

use Yii;

use backend\models\form\AddAppointment;

use backend\models\db\Appointment;
use backend\models\db\AppointmentUser;

use yii\filters\AccessControl;
use yii\helpers\Url;
use backend\components\CustomController;
use backend\components\helpers\Formatter;
use backend\components\helpers\JsonTools;


/**
 * Coach controller
 */
class AppointmentsController extends CustomController
{

    public $enableCsrfValidation = false;

    public function behaviors()
    {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'cancel-appointment',
                            'get-day-appointment-data',
                            'book-appointment'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                    [
                        'actions' => [
                            'add-appointment',

                            'remove-appointment',
                        ],
                        'allow' => true,
                        'roles' => ['coach', 'superadmin'],
                    ]
                ],
            ]
        ];

    }


    /**
     * Add appointment time slot as coach or SA.
     */
    public function actionAddAppointment() {

        $appointmentForm = new AddAppointment();
        $amPm = Yii::$app->session->get('timeFormat') == 'g:ia';
        $appointmentForm->amPm = $amPm;
        if (Yii::$app->request->isPost && $appointmentForm->load(Yii::$app->request->post())) {

            $appointmentForm->coachId = $this->userIdentity->id;

            if ($appointmentForm->validate()) {

                foreach ($appointmentForm->day as $key => $d) {

                    $appointment = new Appointment();
                    $hour = ($amPm) ? date('H', strtotime($appointmentForm->hour[$key] . ' ' . ($appointmentForm->pm ? 'PM' : 'AM'))) : $appointmentForm->hour[$key];
                    $datetime = $appointmentForm->year . '-' . $appointmentForm->month . '-' . $appointmentForm->day[$key] . ' ' . $hour . ':' . $appointmentForm->minute[$key];
                    $appointment->setAttributes([
                        'datetime' => Formatter::localDatetimeToUtcDatetime($datetime),
                        'user_id' => $appointmentForm->coachId
                    ]);
                    $appointment->save();

                }

                if (Yii::$app->request->isAjax) {

                    echo JsonTools::successMessage('Appointment Time-slot(s) successfully created.');
                    die();

                }

            }
            elseif (Yii::$app->request->isAjax) {

                echo JsonTools::formErrorMessage($appointmentForm);
                die();

            }

        }
        else {

            $years = [date('Y') => date('Y'), date('Y') + 1 => date('Y') + 1];

            $months = [];
            for($i = 1; $i <= 12; $i++) {

                $months[$i] = date('F', strtotime(date("Y-$i-1")));

            }

            $appointmentForm->minute[0] = '00';
            $appointmentForm->month = (int) date('m');
            $appointmentForm->year = (int) date('Y');

            $this->view->title = 'Add Appointment Time-slots';

            return $this->render('add-appointment', [
                'formModel' => $appointmentForm,
                'years' => $years,
                'months' => $months,
                'amPm' => $amPm
            ]);

        }


    }


    /**
     * Get appointment data for a specific day
     */
    public function actionGetDayAppointmentData() {

        $dateRaw = Yii::$app->getRequest()->getQueryParam('date', date('Y-m-d'));
        $date = date('Y-m-d', strtotime($dateRaw));

        if (in_array($this->userIdentity->user_role_id, [3, 4]) ) {

            $appointmentData = $this->_getClientDayAppointmentData($date);

        }
        elseif ($this->userIdentity->user_role_id == 2) {

            $appointmentData = $this->_getCoachDayAppointmentData($date);

        }

        echo json_encode([
            'data' => $appointmentData,
            'role' => $this->userIdentity->userRole->name,
            'date' => Formatter::date(strtotime($date)),
            'status' => 'success'
        ]);
        die();

    }


    /**
     * Book an appointment as client/sub-client
     */
    public function actionBookAppointment() {

        $appointmentId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if (!$appointmentId || !$appointmentId) {

            echo JsonTools::errorMessage('Invalid appointment ID.');
            die();

        }

        $appointment = Appointment::findOne($appointmentId);
        $appointmentUser = $appointment->appointmentUser;

        if (!$appointment) {

            echo JsonTools::errorMessage('Appointment doesn\'t exist.');
            die();

        }

        if (($appointment->user_id != $this->coach->id) || $appointmentUser) {

            echo JsonTools::errorMessage('Invalid appointment or already booked.');
            die();

        }

        if ($confirm) {

            $saved = $appointment->book($this->userIdentity->id);

            if ($saved) {

                $this->coach->addNotification('booked_appointment', [
                    'user_name' => $this->user->getFullName(true),
                    'datetime' => Formatter::datetime(strtotime($appointment->datetime))
                ]);

                echo JsonTools::successMessage('Appointment successfully booked.');
                die();

            }
            else {

                echo JsonTools::errorMessage('Appointment booking failed.');
                die();

            }

        }
        else {

            $this->view->title = 'Book Appointment';

            return $this->render('book-appointment', [
                'appointmentId' => $appointment->id
            ]);

        }

    }


    /**
     * Cancel an appointment
     */
    public function actionCancelAppointment() {

        $appointmentId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if (!$appointmentId || !$appointmentId) {

            echo JsonTools::errorMessage('Invalid appointment ID.');
            die();

        }

        $appointment = Appointment::findOne($appointmentId);
        $appointmentUser = $appointment->appointmentUser;

        if (!$appointment) {

            echo JsonTools::errorMessage('Appointment doesn\'t exist.');
            die();

        }

        if (!$appointmentUser || (($appointmentUser->user_id != $this->userIdentity->id) && ($appointment->user_id != $this->userIdentity->id))) {

            echo JsonTools::errorMessage('Invalid appointment.');
            die();

        }

        if ($confirm) {

            if ($this->userIdentity->user_role_id == 2) {

                $userToNotify = $appointmentUser->user;

            }
            else {
                $userToNotify = $appointment->user;
            }

            $removed = $appointment->cancelBooking();

            if ($removed) {

                $userToNotify->addNotification('cancelled_appointment', [
                    'datetime' => Formatter::datetime(strtotime($appointment->datetime)),
                    'user_name' => $this->user->getFullName(true)
                ]);

                echo JsonTools::successMessage('Appointment successfully cancelled.');
                die();

            }
            else {

                echo JsonTools::errorMessage('Appointment cancellation failed.');
                die();

            }

        }
        else {

            $this->view->title = 'Cancel Appointment';

            return $this->render('cancel-appointment', [
                'appointmentId' => $appointment->id
            ]);

        }

    }


    /**
     * Remove an appointment as coach or SA.
     */
    public function actionRemoveAppointment() {

        $appointmentId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if (!$appointmentId || !$appointmentId) {

            echo JsonTools::errorMessage('Invalid appointment ID.');
            die();

        }

        $appointment = Appointment::findOne($appointmentId);

        if (!$appointment) {

            echo JsonTools::errorMessage('Appointment doesn\'t exist.');
            die();

        }

        if ($appointment->user_id != $this->userIdentity->id) {

            echo JsonTools::errorMessage('Invalid appointment.');
            die();

        }

        if ($confirm) {

            $appointmentUsers = $appointment->appointmentUsers;

            if (count($appointmentUsers)) {

                foreach ($appointmentUsers as $a) {

                    $user = $a->user;
                    $user->addNotification('cancelled_appointment', [
                        'datetime' => Formatter::datetime(strtotime($appointment->datetime)),
                        'user_name' => $this->user->getFullName(true)
                    ]);
                    $a->delete();

                }

            }

            $deleted = $appointment->delete();

            if ($deleted) {

                echo JsonTools::successMessage('Appointment successfully deleted.');
                die();

            }
            else {

                echo JsonTools::errorMessage('Appointment deletion failed.');
                die();

            }

        }
        else {

            $this->view->title = 'Remove Appointment';

            return $this->render('remove-appointment', [
                'appointmentId' => $appointment->id
            ]);

        }

    }


    /**
     * Get data for a calendar day for a coach
     */
    private function _getCoachDayAppointmentData($date) {

        $appointments = Appointment::find()
            ->with('appointmentUser')
            ->where([
                'and',
                ['DATE(datetime)' => $date],
                ['user_id' => $this->userIdentity->id]
            ])
            ->orderBy('datetime asc')
            ->all();

        $appointmentsData = [];
        foreach($appointments as $a) {

            $data = $a->getAttributes();
            $data['timeFrom'] = Formatter::time($a->datetime, 'UTC');
            $data['timeTo'] = Formatter::time(strtotime($a->datetime) + 1800, 'UTC');
            $data['removeUrl'] = Url::toRoute(['/appointments/remove-appointment', 'id' => $a->id]);
            if ($a->appointmentUser) {

                $data['cancelUrl'] = Url::toRoute(['/appointments/cancel-appointment', 'id' => $a->id]);
                $data['client'] = [
                    'id' => $a->appointmentUser->user_id,
                    'name' => $a->appointmentUser->user->getFullName(true)
                ];

            }
            $appointmentsData[] = $data;

        }

        return $appointmentsData;

    }


    /**
     * Get data for a calendar day for a client
     */
    private function _getClientDayAppointmentData($date) {

        $appointments = Appointment::find()
            ->joinWith('appointmentUser')
            ->with('appointmentUser')
            ->where([
                'and',
                ['DATE(datetime)' => $date],
                [Appointment::tableName() . '.user_id' => $this->coach->id],
                [
                    'or',
                    [AppointmentUser::tableName() . '.user_id' => $this->userIdentity->id],
                    [AppointmentUser::tableName() . '.user_id' => null]
                ]
            ])
            ->orderBy('datetime asc')
            ->all();

        $appointmentsData = [];
        foreach($appointments as $a) {

            $data = $a->getAttributes();
            $data['timeFrom'] = Formatter::time($a->datetime, 'UTC');
            $data['timeTo'] = Formatter::time(strtotime($a->datetime) + 1800, 'UTC');
            if ($a->appointmentUser) {

                $data['cancelUrl'] = Url::toRoute(['/appointments/cancel-appointment', 'id' => $a->id]);
                $data['booked'] = true;

            }
            else {

                $data['bookUrl'] = Url::toRoute(['/appointments/book-appointment', 'id' => $a->id]);

            }
            $appointmentsData[] = $data;

        }

        return $appointmentsData;

    }




}
