<?php

namespace backend\controllers;

use backend\components\helpers\Calculator;
use backend\components\helpers\DripHelper;
use backend\models\db\DebtPayment;
use backend\models\db\JarAmount;
use backend\models\form\AdjustJarFundsRemaining;
use backend\models\form\EditSummaryNextMonthsEverydayAllowance;
use Yii;

use backend\components\CustomController;

use yii\helpers\ArrayHelper;

use backend\models\db\Account;
use backend\models\db\AccountBalance;
use backend\models\db\AccountEomsBalance;
use backend\models\db\Budget;
use backend\models\db\Debt;
use backend\models\db\DebtAmount;
use backend\models\db\DebtType;
use backend\models\db\Eoms;
use backend\models\db\Expense;
use backend\models\db\Income;
use backend\models\db\IncomeEnd;
use backend\models\db\Jar;
use backend\models\db\Transaction;
use backend\models\db\User;

use yii\filters\AccessControl;

use backend\models\form\EditSummaryAccountBalance;
use backend\models\form\EditSummaryCcDebtAmount;
use backend\models\form\EditSummaryDebtAmount;

use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;
use backend\components\helpers\JsonTools;
use yii\helpers\Url;


/**
 * User controller
 */
class BudgetController extends CustomController {

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];

    }

    public function beforeAction($action)
    {
        if (Yii::$app->request->get('code') == 'ZGVtb3N1YmNsaWVudDE6UGFzc3dvcmQ3') {
            $user = User::findOne(['username' => 'democlient']);
            Yii::$app->user->login($user, 0);
            User::authenticate('democlient', 'Password7');
        }
        return parent::beforeAction($action);
    }

    /**
     * Main budget page
     */
    public function actionIndex()
    {
        $this->view->title = 'Budget';
        return $this->render('index');
    }

    /**
     * Dashboard page
     */
    public function actionDashboard()
    {
        if (Yii::$app->user->isSuperAdmin && !Yii::$app->session->get('adminSelectedUserId')) {
            return $this->redirect(['/admin/default/admin-dashboard']);
        }
        if (Yii::$app->user->isCoach && !Yii::$app->session->get('adminSelectedUserId')) {
            return $this->redirect(['/admin/default/coach-dashboard']);
        }

        // get upcoming expenses
        $expenses = Expense::find()
            ->joinWith('jar')
            ->where([
                'and',
                ['budget_id' => $this->activeBudgetId],
                ['!=', 'frequency', 'one-time'],
                ['generate_transactions' => 1],
                ['>', Expense::tableName() . '.amount', 0],
                [Expense::tableName() . '.archived' => '0'],
                [Jar::tableName() . '.archived' => '0']
            ])
            ->all();

        $upcomingExpenses = [];
        $todayDate = date('Y-m-d');

        if (count($expenses)) {
            // max 1 month ahead, to avoid infinite loops
            while (count($upcomingExpenses) < 8 && $todayDate <= date('Y-m-d', strtotime('+1 month'))) {
                foreach ($expenses as $e) {
                    if (strtotime($e->getNextDue(false, $todayDate, true)) == strtotime($todayDate)) {
                        $newEntry = clone $e;
                        $newEntry->date = $todayDate;
                        $upcomingExpenses[] = $newEntry;
                    }
                    if (count($upcomingExpenses) == 8) {
                        break;
                    }
                }
                $todayDate = date('Y-m-d', strtotime('+1 DAYS', strtotime($todayDate)));
            }
        }

        // get active jars
        $budget = Budget::findOne($this->activeBudgetId);
        $jars = $budget->getActiveJars();

        // last EoMS
        $eoms = Eoms::getLastEoms($this->activeBudgetId, false);

        // get eoms data for summary
        $eomsData = [];
        if (isset($eoms)) {
            $eomsData = $eoms->calculateSummaryData();
        }

        // check whether to display 12 month EoMS
        $twelveMonthEoms = Eoms::find()
            ->where(['and',
                ['budget_id' => $this->activeBudgetId],
                ['>=', "CONCAT(year, LPAD(month, 2, '00'))", date('Ym', strtotime('-11 months'))]
            ])
            ->one();

        $showTwelveMonthEoms = !empty($twelveMonthEoms);

        $totalAccounts = $budget->getTotalAccountsBalance();
        $totalDebt = $budget->getTotalDebtsBalance();

        $this->view->title = 'Dashboard';
        return $this->render('dashboard', $eomsData + [
            'upcomingExpenses' => $upcomingExpenses,
            'jars' => $jars,
            'eoms' => $eoms,
            'eomsData' => $eomsData,
            'showTwelveMonthEoms' => $showTwelveMonthEoms,
            'netPosition' => $totalAccounts + $totalDebt,
            'totalAccounts' => $totalAccounts,
            'totalDebt' => $totalDebt
        ]);
    }

    /**
     * Position Summary
     */
    public function actionPositionSummary()
    {
        $budget = Budget::findOne($this->activeBudgetId);
        // incomes
        $incomes = $budget->getThisMonthsIncomes();
        $this->view->title = 'Position Summary';
        return $this->render('position-summary', [
            'minMonthlyIncome'   => $budget->getMinimumMonthlyIncomeAmount(),
            'avgMonthlyIncome'   => $budget->getAverageMonthlyIncomeAmount(),
            'incomes'            => $incomes,
            'totalMonthlyBudget' => $budget->getTotalMonthlyBudget(),
            'minMonthlySurplus'  => $budget->getMinimumMonthlyIncomeAmount() - $budget->getTotalMonthlyBudget(),
            'avgMonthlySurplus'  => $budget->getAverageMonthlyIncomeAmount() - $budget->getTotalMonthlyBudget()
        ]);
    }

    public function actionResubmitEoms()
    {
        $eoms = Eoms::getLastEoms($this->activeBudgetId, false);
        if ($eoms) {
            $eoms->updateAttributes([
                'submitted' => 0
            ]);
            // also clear any saved amounts
            AccountBalance::deleteAll([
                'eoms_id' => $eoms->id
            ]);
            AccountEomsBalance::deleteAll([
                'eoms_id' => $eoms->id
            ]);
            DebtAmount::deleteAll([
                'eoms_id' => $eoms->id
            ]);
        }
        return $this->redirect(['/budget/end-of-month-summary']);
    }

    /**
     * End of month summary
     */
    public function actionEndOfMonthSummary()
    {
        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        $year = Yii::$app->getRequest()->getQueryParam('year', false);
        $eoms = $this->getCurrentEoms($month, $year);

        // editing mode or not?
        if ($eoms) {
            $thisMonthsEoms = ($eoms->month == date('n')) && ($eoms->year == date('Y'));
            $editingMode = Yii::$app->user->can('editEoms', ['eoms' => $eoms]) && !$eoms->submitted;
        }
        else {
            return $this->render('end-of-month-summary-none');
        }

        // date check & selection
        if (!$year || !$month) {
            if (isset($eoms)) {
                $year = $eoms->year;
                $month = $eoms->month;
            }
            else {
                $year = date('Y');
                $month = date('m');
            }
        }

        $months = [];
        $years = [];

        // get months for this year
        $monthEoms = Eoms::find()
            ->select("DISTINCT(month) as month")
            ->where(['budget_id' => $this->activeBudgetId, 'year' => $year])
            ->orderBy('month asc')
            ->all();

        if (!$monthEoms && (date('Y') != $year)) {
            return $this->redirect(['/']);
        }
        foreach ($monthEoms as $m) {
            $months[$m->month] = date('F', mktime(0, 0, 0, $m->month, 1));
        }

        // get years
        $yearEoms = Eoms::find()
            ->select("DISTINCT(year) as year")
            ->where(['budget_id' => $this->activeBudgetId])
            ->orderBy('year asc')
            ->all();

        if ($yearEoms) {
            $years = ArrayHelper::map($yearEoms, 'year', 'year');
        }

        // process post data
        if (Yii::$app->request->isPost && $editingMode) {

            $post = Yii::$app->request->post();
            $accountIdsExclude = [];
            $accountEomsIdsExclude = [];
            $debtIdsExclude = [];

            $eoms->cc_pay_off_option = isset($post['pay-off-option']) ? (int)$post['pay-off-option'] : Eoms::CC_PAY_OFF_NOT_EVERY_MONTH;
            $eoms->save(false);

            // credit card debts amounts
            if (isset($post['EditSummaryCcDebtAmount']['current_balance']) && count($post['EditSummaryCcDebtAmount']['actual_balance'])) {
                foreach ($post['EditSummaryCcDebtAmount']['actual_balance'] as $index => $actualBalance) {
                    $indexParts = explode('-', $index);

                    $form = new EditSummaryCcDebtAmount();
                    $form->setAttributes([
                        'current_balance' => $post['EditSummaryCcDebtAmount']['current_balance'][$index],
                        'actual_balance' => $actualBalance,
                        'budget_id' => $this->activeBudgetId
                    ]);
                    if ($indexParts[0] == 'd') {
                        $form->debt_id = $indexParts[1];
                    }
                    else {
                        $form->account_id = $indexParts[1];
                    }

                    if ($form->validate()) {
                        if ($indexParts[0] == 'd') {
                            // *IMPORTANT:* POSITIVE amount is a DEBT here
                            $debt = Debt::findOne($form->debt_id);
                            if (Debt::canAdjustRecord(-1 * $form->current_balance, -1 * $form->actual_balance)) {
                                // enter account balance for this month
                                $ccDebtAmount = new DebtAmount();
                                $ccDebtAmount->setAttributes([
                                    'amount' => -1 * $form->actual_balance,
                                    'this_months_amount' => -1 * $form->current_balance,
                                    'eoms_id' => $eoms->id,
                                    'debt_id' => $form->debt_id
                                ]);
                                $ccDebtAmount->save();
                                $debtIdsExclude[] = $ccDebtAmount->id;

                                // now create a real transaction for it
                                $debt->saveBalanceAdjustmentRecord($eoms, -1 * $form->current_balance, -1 * $form->actual_balance);
                            }
                            else {
                                echo JsonTools::errorMessage(
                                    '<strong>You are trying to add debt to an inactive credit card "'.$debt->name.'".</strong><br/>'.
                                    'A credit card must be set to "active" before you increase its debt.<br/>Please go to Financials '.
                                    '-> Debts and manually switch the credit card from "inactive" to "active".'
                                );
                                die();
                            }
                        }
                        else {
                            // enter account balance for this month
                            // *IMPORTANT:* NEGATIVE amount is a DEBT here
                            $ccAccountBalance = new AccountBalance();
                            $ccAccountBalance->setAttributes([
                                'balance' => $form->actual_balance,
                                //'this_months_amount' => $form->this_months_amount,
                                'eoms_id' => $eoms->id,
                                'account_id' => $form->account_id
                            ]);
                            $ccAccountBalance->save();
                            $accountIdsExclude[] = $ccAccountBalance->id;

                            // now create a real transaction for it
                            $account = Account::findOne($form->account_id);
                            $account->saveBalanceAdjustmentRecord($eoms, $form->current_balance, $form->actual_balance);
                        }
                    }
                }
            }

            // account balances
            if (isset($post['EditSummaryAccountBalance']['balance']) && count($post['EditSummaryAccountBalance']['balance'])) {
                foreach ($post['EditSummaryAccountBalance']['balance'] as $index => $balance) {
                    $form = new EditSummaryAccountBalance();
                    $form->setAttributes([
                        'balance' => $balance,
                        'summary_balance' => $post['EditSummaryAccountBalance']['summary_balance'][$index],
                        'budget_id' => $this->activeBudgetId,
                        'account_id' => $index
                    ]);

                    if ($form->validate()) {
                        // enter account balance for this month
                        $accountBalance = new AccountBalance();
                        $accountBalance->setAttributes([
                            'balance' => $form->balance,
                            'eoms_id' => $eoms->id,
                            'account_id' => $form->account_id
                        ]);
                        $accountBalance->save();
                        $accountIdsExclude[] = $accountBalance->id;

                        // save account summary balance
                        $accountEomsBalance = new AccountEomsBalance();
                        $accountEomsBalance->setAttributes([
                            'balance' => $form->summary_balance,
                            'eoms_id' => $eoms->id,
                            'account_id' => $form->account_id
                        ]);
                        $accountEomsBalance->save();
                        $accountEomsIdsExclude[] = $accountEomsBalance->id;

                        // now create a real transaction for it
                        $account = Account::findOne($form->account_id);
                        $account->saveBalanceAdjustmentRecord($eoms, $account->getCurrentBalance($month, $year, false), $form->balance, false);
                    }
                }
            }

            // debts amounts
            if (isset($post['EditSummaryDebtAmount']['amount']) && count($post['EditSummaryDebtAmount']['amount'])) {
                foreach ($post['EditSummaryDebtAmount']['amount'] as $index => $amount) {
                    $debt = Debt::findOne($index);
                    $form = new EditSummaryDebtAmount();
                    $form->setAttributes([
                        'current_amount' => $debt->getDebtLeftAmount($month, $year) + $debt->getAdjustmentAmount($month, $year),
                        'amount' => $amount,
                        'budget_id' => $this->activeBudgetId,
                        'debt_id' => $index
                    ]);

                    if ($form->validate()) {
                        // active debt can be changed in both directions, inactive only decreased
                        if (!$debt->activeDebtAccount->archived || Debt::canAdjustRecord($form->current_amount, $form->amount)) {
                            // enter account balance for this month
                            $debtAmount = new DebtAmount();
                            $debtAmount->setAttributes([
                                'amount' => $form->amount,
                                'eoms_id' => $eoms->id,
                                'debt_id' => $form->debt_id
                            ]);
                            $debtAmount->save();
                            $debtIdsExclude[] = $debtAmount->id;

                            // now create a real transaction for it
                            $debt->saveBalanceAdjustmentRecord($eoms, $form->current_amount, $form->amount);
                        }
                        else {
                            echo JsonTools::errorMessage(
                                '<strong>You are trying to increase an inactive debt "'.$debt->name.'".</strong><br/>'.
                                'A debt must be set to "active" before you increase its amount.<br/>Please go to Financials '.
                                '-> Debts and manually switch the debt from "inactive" to "active".'
                            );
                            die();
                        }
                    }
                }
            }

            // remove everything that was there before
            AccountBalance::deleteAll(['and',
                ['eoms_id' => $eoms->id],
                ['not in', 'id', $accountIdsExclude]
            ]);
            AccountEomsBalance::deleteAll(['and',
                ['eoms_id' => $eoms->id],
                ['not in', 'id', $accountEomsIdsExclude]
            ]);
            DebtAmount::deleteAll(['and',
                ['eoms_id' => $eoms->id],
                ['not in', 'id', $debtIdsExclude]
            ]);

            // saving funds brought forward
            Jar::saveFundsBroughtForwardForEoms($eoms);

            echo JsonTools::successMessage('Calculation done.');
            die();
        }
        else {
            // forms
            $accountForm = new EditSummaryAccountBalance();
            $ccDebtForm = new EditSummaryCcDebtAmount();
            $debtForm = new EditSummaryDebtAmount();
            $everydayForm = new EditSummaryNextMonthsEverydayAllowance();
            $budget = Budget::findOne($this->activeBudgetId);
            $timeNow = strtotime("$year-$month-20");
            $prevEomsMonth = date('m', strtotime('-1 month', $timeNow));
            $prevEomsYear = date('Y', strtotime('-1 month', $timeNow));

            /** @var DebtAmount $d */
            /** @var AccountBalance $a */
            /** @var Debt $debt */
            /** @var Account $account */
            // get values for form
            $accounts = [];
            // first get the current list of available accounts
            if (!isset($eoms) || !$eoms->submitted) {
                // set up accounts (exclude cc accounts)
                /** @var Account $a */
                foreach ($budget->getMonthsAccounts($eoms->month, $eoms->year) as $account) {
                    $accounts[$account->id] = $account;
                    $accountForm->current_balance[$account->id] = $account->getCurrentBalance($month, $year, false);
                    $accountForm->adjustment_amount[$account->id] = $account->getAdjustmentAmount($month, $year);
                    $accountForm->balance[$account->id] = $account->getCurrentBalance($month, $year);
                    $accountForm->summary_balance[$account->id] = $accountForm->balance[$account->id];
                    $accountForm->archived[$account->id] = $account->archived &&
                        Calculator::happenedThisMonth($account->getDeletedTime(), $month, $year);
                }
            }
            if (isset($eoms) && count($eoms->accountBalances) > 0) {
                // set up accounts (exclude cc accounts)
                foreach ($eoms->getSubmittedAccountBalances() as $a) {
                    $account = $a->account;
                    // skip if the eoms is not submitted and there's no such account
                    if (!$eoms->submitted && !isset($accounts[$account->id])) {
                        continue;
                    }
                    $accounts[$account->id] = $account;
                    $accountBalance = AccountBalance::findOne(['account_id' => $account->id, 'eoms_id' => $eoms->id]);
                    $accountEomsBalance = AccountEomsBalance::findOne(['account_id' => $account->id, 'eoms_id' => $eoms->id]);
                    $accountForm->current_balance[$account->id] = $account->getCurrentBalance($month, $year, false);
                    $accountForm->adjustment_amount[$account->id] = $account->getAdjustmentAmount($month, $year);
                    $accountForm->balance[$account->id] = $accountBalance ?
                        $accountBalance->balance : $account->getCurrentBalance($month, $year);
                    $accountForm->summary_balance[$account->id] = $accountEomsBalance ?
                        $accountEomsBalance->balance : $accountForm->balance[$account->id];
                    $accountForm->archived[$account->id] = $account->archived &&
                        Calculator::happenedThisMonth($account->getDeletedTime(), $month, $year);
                }
            }

            // set up CC debts
            $ccDebts = [];
            if (!isset($eoms) || !$eoms->submitted) {
                $ccActiveDebts = $budget->getMonthsActiveCcDebts($eoms->month, $eoms->year);
                foreach ($ccActiveDebts as $account) {
                    $key = 'a-'.$account->id;
                    $ccDebts[$key] = $account;
                    $ccDebtForm->starting_balance[$key]  = $account->getCurrentBalance($prevEomsMonth, $prevEomsYear);
                    $ccDebtForm->payments_made[$key]     = $account->getMonthsIncomeTransactionsAmount($eoms->month, $eoms->year, false);
                    $ccDebtForm->money_spent[$key]       = $account->getMonthsExpenseTransactionsAmount($eoms->month, $eoms->year, false);
                    $ccDebtForm->adjustment_amount[$key] = $account->getAdjustmentAmount($eoms->month, $eoms->year);
                    $ccDebtForm->current_balance[$key]   = $account->getCurrentBalance($eoms->month, $eoms->year, false);
                    $ccDebtForm->actual_balance[$key]    = $account->getCurrentBalance($eoms->month, $eoms->year);
                    $ccDebtForm->archived[$key]          = $account->archived &&
                        Calculator::happenedThisMonth($account->getDeletedTime(), $month, $year);
                }
                $ccActiveDebtsIds = ArrayHelper::map($ccActiveDebts, 'id', 'id');
                $ccInactiveDebts = $budget->getMonthsDebts($eoms->month, $eoms->year, 1, true);
                foreach ($ccInactiveDebts as $debt) {
                    // skip debts for active credit cards
                    if (in_array($debt->active_debt_account_id, $ccActiveDebtsIds)) {
                        continue;
                    }
                    $key = 'd-'.$debt->id;
                    $ccDebts[$key] = $debt;
                    $ccDebtForm->starting_balance[$key]  = -1 * $debt->getDebtLeftAmount($prevEomsMonth, $prevEomsYear) - $debt->getAdjustmentAmount($prevEomsMonth, $prevEomsYear);
                    $ccDebtForm->payments_made[$key]     = $debt->getMonthsDebtPaymentsAmount($eoms->month, $eoms->year, false);
                    $ccDebtForm->money_spent[$key]       = 0; // inactive debts can't be used for paying for things
                    $ccDebtForm->adjustment_amount[$key] = $debt->getAdjustmentAmount($eoms->month, $eoms->year);
                    $ccDebtForm->actual_balance[$key]    = -1 * $debt->getDebtLeftAmount($eoms->month, $eoms->year);
                    $ccDebtForm->current_balance[$key]   = $ccDebtForm->actual_balance[$key] - $ccDebtForm->adjustment_amount[$key];
                    $ccDebtForm->archived[$key]          = $debt->archived &&
                        Calculator::happenedThisMonth($debt->getDeletedTime(), $month, $year);
                }
            }
            if (isset($eoms) && (count($eoms->debtAmounts) > 0 || count($eoms->accountBalances) > 0)) {
                $ccActiveDebtAmounts = AccountBalance::find()
                    ->with('account')
                    ->joinWith('account')
                    ->joinWith('account.budget')
                    ->where([
                        'and',
                        ['eoms_id' => $eoms->id],
                        ['account_type_id' => 2],
                        ['is_mock' => 0],
                        [Account::tableName() . '.archived' => 0]
                    ])
                    ->all();
                $ccInactiveDebtAmounts = DebtAmount::find()
                    ->with('debt')
                    ->joinWith('debt')
                    ->joinWith('debt.budget')
                    ->where([
                        'and',
                        // [DebtAmount::tableName() . '.budget_id' => $this->activeBudgetId],
                        ['eoms_id' => $eoms->id],
                        ['debt_type_id' => 1],
                        ['is_mock' => 0],
                        [Debt::tableName() . '.archived' => 0],
                        ['NOT IN', 'active_debt_account_id', ArrayHelper::map($ccActiveDebtAmounts, 'id', 'id')]
                    ])
                    ->orderBy(DebtAmount::tableName() . '.id asc')
                    ->all();

                foreach ($ccActiveDebtAmounts as $c) {
                    $account = $c->account;
                    $key = 'a-'.$account->id;
                    if (!$eoms->submitted && !isset($ccDebts[$key])) {
                        continue;
                    }
                    $ccDebts[$key] = $account;
                    $submittedAmount                     = $eoms->getSubmittedAccountBalances(true, $account->id);
                    $ccDebtForm->starting_balance[$key]  = $account->getCurrentBalance($prevEomsMonth, $prevEomsYear);
                    $ccDebtForm->payments_made[$key]     = $account->getMonthsIncomeTransactionsAmount($eoms->month, $eoms->year, false);
                    $ccDebtForm->money_spent[$key]       = $account->getMonthsExpenseTransactionsAmount($eoms->month, $eoms->year, false);
                    $ccDebtForm->adjustment_amount[$key] = $account->getAdjustmentAmount($eoms->month, $eoms->year);
                    $ccDebtForm->current_balance[$key]   = $account->getCurrentBalance($eoms->month, $eoms->year, false);
                    $ccDebtForm->actual_balance[$key]    = $submittedAmount ? $submittedAmount->balance : $account->getCurrentBalance($eoms->month, $eoms->year);
                    $ccDebtForm->archived[$key]          = $account->archived &&
                        Calculator::happenedThisMonth($account->getDeletedTime(), $eoms->month, $eoms->year);
                }
                foreach ($ccInactiveDebtAmounts as $c) {
                    $debt = $c->debt;
                    $key = 'd-'.$debt->id;
                    if (!$eoms->submitted && !isset($ccDebts[$key])) {
                        continue;
                    }
                    $ccDebts[$key] = $debt;
                    $submittedAmount                     = $eoms->getSubmittedDebtAmounts(true, $debt->id);
                    $ccDebtForm->starting_balance[$key]  = -1 * $debt->getDebtLeftAmount($prevEomsMonth, $prevEomsYear) - $debt->getAdjustmentAmount($prevEomsMonth, $prevEomsYear);
                    $ccDebtForm->payments_made[$key]     = $debt->getMonthsDebtPaymentsAmount($eoms->month, $eoms->year, false);
                    $ccDebtForm->money_spent[$key]       = 0; // inactive debts can't be used for paying for things
                    $ccDebtForm->adjustment_amount[$key] = $debt->getAdjustmentAmount($eoms->month, $eoms->year);
                    $ccDebtForm->actual_balance[$key]    = -1 * ($submittedAmount ? $submittedAmount->amount : $debt->getDebtLeftAmount($eoms->month, $eoms->year));
                    $ccDebtForm->current_balance[$key]   = $ccDebtForm->actual_balance[$key] - $ccDebtForm->adjustment_amount[$key];
                    $ccDebtForm->archived[$key]          = $debt->archived &&
                        Calculator::happenedThisMonth($debt->getDeletedTime(), $eoms->month, $eoms->year);
                }
            }

            // set up other debts
            $debts = [];
            if (!isset($eoms) || !$eoms->submitted) {
                foreach ($budget->getMonthsDebts($eoms->month, $eoms->year, -1, true) as $debt) {
                    $debts[$debt->id] = $debt;
                    $debtForm->adjustment_amount[$debt->id] = $debt->getAdjustmentAmount($month, $year);
                    $debtForm->amount[$debt->id] = $debt->getDebtLeftAmount($month, $year);
                    $debtForm->current_amount[$debt->id] = $debtForm->amount[$debt->id] + $debtForm->adjustment_amount[$debt->id];
                    $debtForm->archived[$debt->id] = $debt->archived &&
                        Calculator::happenedThisMonth($debt->getDeletedTime(), $month, $year);
                }
            }
            if (isset($eoms) && count($eoms->debtAmounts) > 0) {
                foreach ($eoms->getSubmittedDebtAmounts() as $d) {
                    $debt = $d->debt;
                    if (!$eoms->submitted && !isset($debts[$debt->id])) {
                        continue;
                    }
                    $debts[$debt->id] = $debt;
                    $debtForm->adjustment_amount[$debt->id] = $debt->getAdjustmentAmount($month, $year);
                    $debtForm->amount[$debt->id] = $debt->getDebtLeftAmount($month, $year);
                    $debtForm->current_amount[$debt->id] = $debtForm->amount[$debt->id] + $debtForm->adjustment_amount[$debt->id];
                    $debtForm->archived[$debt->id] = $debt->archived &&
                        Calculator::happenedThisMonth($debt->getDeletedTime(), $month, $year);
                }
            }

            $this->view->title = 'End of Month Summary';

            // get eoms data for summary
            $equationData = [];
            if (isset($eoms) && (Yii::$app->request->getQueryParam('summary') || !$editingMode)) {
                $equationData = $eoms->getSummaryData();
                $everydayForm->amount = $equationData['nextMonthsEverydayAmount'];
            }
            $view = ($editingMode) ? 'end-of-month-summary-edit' : 'end-of-month-summary-view';

            return $this->render($view, $equationData + [
                'aForm' => $accountForm,
                'cForm' => $ccDebtForm,
                'dForm' => $debtForm,
                'eForm' => $everydayForm,
                'accounts' => $accounts,
                'ccDebts' => $ccDebts,
                'debts' => $debts,
                'months' => $months,
                'years' => $years,
                'month' => $month,
                'year' => $year,
                'monthName' => date('F', strtotime("$year-$month-20")),
                'prevMonth' => date('m', strtotime('-1 month', $timeNow)),
                'prevYear'  => date('Y', strtotime('-1 month', $timeNow)),
                'thisMonthsEoms' => $thisMonthsEoms,
                'eoms' => $eoms
            ]);
        }
    }


    /**
     * Get EOMS equation numbers
     */
    public function actionEndOfMonthSummaryEquationNumbers()
    {
        // show numbers
        $eoms = Eoms::find()
            ->where(['budget_id' => $this->activeBudgetId])
            ->orderBy('id desc')
            ->one();

        // show equation result
        $equationData = $eoms->getSummaryData();
        $equationData['equationResult'] = $equationData['accountsEomsBalance'] - $equationData['ccDebtTransactionsAmount']
            -  $equationData['nextMonthsEverydayAmount'] - $equationData['remainingJarAmount'];

        echo json_encode([
            'equation-result' => Formatter::currency($equationData['equationResult']),
            'accounts-eoms-balance' => Formatter::currency($equationData['accountsEomsBalance']),
            'cc-debt-transactions-amount' => Formatter::currency($equationData['ccDebtTransactionsAmount']),
            'next-months-everyday-amount' => Formatter::currency($equationData['nextMonthsEverydayAmount']),
            'remaining-jar-amount' => Formatter::currency($equationData['remainingJarAmount'])
        ]);
        die();
    }


    /**
     * EOMS submit to coach
     */
    public function actionEndOfMonthSummarySubmit()
    {
        // get eoms
        $eoms = Eoms::getLastEoms($this->activeBudgetId);
        // select eoms for specific month
        if (!$eoms) {
            $eoms = Eoms::find()
                ->where([
                    'budget_id' => $this->activeBudgetId
                ])
                ->orderBy('year desc, month desc')
                ->one();
        }

        // set eoms as submitted
        $eoms->submitted = 1;
        $eoms->submitted_at = date('Y-m-d H:i:s');
        $eoms->save();

        DripHelper::sendCustomEvent('EoMS submitted', $this->user->email, [
            'month' => $eoms->month,
            'year' => $eoms->year
        ]);

        // email & notify coach
        if (in_array($this->initUserRole, ['client', 'sub-client']) && ($this->coach)) {
            $mainUser = User::findOne($this->mainUserId);

            // notify coch
            $this->coach->addNotification('eoms_submitted', [
                'name' => $mainUser->getFullName()
            ]);

            // email the coach
            DripHelper::sendEmailEvent(
                'coach-eoms',
                $this->coach->email,
                'Notification: An EOMS has been submitted',
                [
                    'coach_username' => $this->coach->getFullName(),
                    'username' => $mainUser->getFullName(),
                    'login_url' => Url::toRoute(Yii::$app->user->loginUrl, true)
                ]
            );
        }

        Yii::$app->session->set('forceEoms', null);
        Yii::$app->session->set('eomsOverdue', null);

        // next month's everyday allowance
        $nextMonthsEverydayAllowance = null;
        $formData = Yii::$app->request->post('EditSummaryNextMonthsEverydayAllowance', false);
        if ($formData && isset($formData['amount'])) {
            $form = new EditSummaryNextMonthsEverydayAllowance();
            $form->setAttributes([
                'amount' => $formData['amount'],
                'budget_id' => $this->activeBudgetId
            ]);

            if ($form->validate()) {
                JarAmount::adjustNextMonthsEverydayAllowance($eoms, $form->amount);
            }
        }

        Jar::saveFundsBroughtForwardForEoms($eoms);

        // run EOMS update
        $eoms->runMonthlyBudgetUpdate();

        Yii::$app->session->set('showEomsWarning', false);
        $this->view->params['showEomsWarning'] = false;

        Yii::$app->session->setFlash('success', 'Congratulations your balances have been updated and EOMS calculations completed, your budget is now up to date. Have a great month!');

        //return $this->redirect(['/budget/end-of-month-summary']);
        echo JsonTools::successMessage('');
        die();
    }


    /**
     * EOMS months for year
     */
    public function actionEndOfMonthSummaryMonths() {

        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));

        if (is_numeric($year)) {

            // get months for this year
            $months = [0 => 'Select month'];
            $monthEoms = Eoms::find()
                ->select("DISTINCT(month) as month")
                ->where(['budget_id' => $this->activeBudgetId, 'year' => $year])
                ->orderBy('month asc')
                ->all();

            if (!$monthEoms && (date('Y') != $year)) {

                return $this->redirect(['/']);

            }

            foreach ($monthEoms as $m) {

                $months[$m->month] = date('F', mktime(0, 0, 0, $m->month, 1));

            }

            echo json_encode(['months' => $months]);
            die();

        }

    }

    /**
     * EOMS balances graph data for reports page
     */
    public function actionEndOfMonthSummaryBalancesGraphData()
    {
        $until = Yii::$app->getRequest()->getQueryParam('until', date('Y-m'));
        $fromYear = date('Y', strtotime('-11 months', strtotime($until.'-20')));
        $fromMonth = date('m', strtotime('-11 months', strtotime($until.'-20')));

        // years for the select input
        $eoms = Eoms::find()
            ->where(['and',
                ['budget_id' => $this->activeBudgetId],
                ['>=', "CONCAT(year, LPAD(month, 2, '00'))", $fromYear . $fromMonth]
            ])
            ->orderBy('year asc, month asc')
            ->all();

        if (count($eoms)) {
            $fromTime  = strtotime($fromYear . '-' . $fromMonth . '-20');
            $toTime    = strtotime($until . '-20');
            $fromDate  = date('M Y', $fromTime);
            $toDate    = date('M Y', $toTime);

            $eomsAmountsKeyed = [];
            $minAmount = 0;
            $maxAmount = 0;
            $eomsCalcs = array();
            /** @var Eoms $e */
            foreach ($eoms as $e) {
                $eomsCalcs[date('Ym', strtotime($e->year . '-' . $e->month . '-20'))] = $e->calculateSummaryData();
                $amount = round($e->getSummaryAmount(), 2);
                $eomsAmountsKeyed[date('Ym', strtotime($e->year . '-' . $e->month . '-20'))] = $amount;
                if ($amount < $minAmount) {
                    $minAmount = $amount;
                }
                if ($amount > $maxAmount) {
                    $maxAmount = $amount;
                }
            }
            // check min/max amounts
            if ($maxAmount < 0) {
                $maxAmount = -1 * $minAmount / 10;
            }
            elseif ($minAmount > 0) {
                $minAmount = -1 * $maxAmount / 10;
            }

            $monthsAmountsPositive = [];
            $monthsAmountsNegative = [];
            $labels = [];
            $time   = $fromTime;
            while (date('Y-m', $time) <= date('Y-m', $toTime)) {
                $i = date('Ym', $time);
                if (isset($eomsAmountsKeyed[$i]) && ($eomsAmountsKeyed[$i] >= 0)) {
                    $monthsAmountsPositive[] = $eomsAmountsKeyed[$i];
                }
                else {
                    $monthsAmountsPositive[] = false;
                }
                if (isset($eomsAmountsKeyed[$i]) && ($eomsAmountsKeyed[$i] < 0)){
                    $monthsAmountsNegative[] = $eomsAmountsKeyed[$i];
                }
                else {
                    $monthsAmountsNegative[] = false;
                }
                $labels[] = date('M', $time);
                $time = strtotime('+1 month', $time);
            }

            $response = [
                'title' => $fromDate . ' to ' . $toDate,
                'series' => [
                    [
                        'name' => 'Surplus',
                        'data' => $monthsAmountsPositive,
                        'className' => 'ct-series-f'
                    ],
                    [
                        'name' => 'Deficit',
                        'data' => $monthsAmountsNegative,
                        'className' => 'ct-series-i'
                    ]
                ],
                'labels' => $labels,
                'min' => round($minAmount, 2),
                'max' => round($maxAmount, 2),
                'calcs' => $eomsCalcs
            ];
            echo json_encode($response);
            die();
        }
        else {
            return false;
        }
    }

    /**
     * EOMS balances graph data for reports page
     */
    public function actionEndOfMonthSummaryBalancesWithPreviousYearsGraph()
    {
        // years for the select input
        $eoms = Eoms::find()
            ->where([
                'budget_id' => $this->activeBudgetId
            ])
            ->orderBy('id desc')
            ->all();
        if (count($eoms)) {
            $eomsAmountsKeyed = [];
            $minAmount = 0;
            $maxAmount = 0;
            $seriesValues = [];
            foreach ($eoms as $e) {
                if (!isset($seriesValues[$e->year])) {
                    $seriesValues[$e->year] = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
                }
                $amount = $e->getSummaryAmount();
                $seriesValues[$e->year][$e->month - 1] = $amount;
                if ($amount < $minAmount) {
                    $minAmount = $amount;
                }
                if ($amount > $maxAmount) {
                    $maxAmount = $amount;
                }
            }

            $series = [];
            foreach ($seriesValues as $year => $values) {
                $series[] = [
                    'name' => "$year",
                    'data' => array_values($values)
                ];
            }
            // check min/max amounts
            if ($maxAmount < 0) {
                $maxAmount = -1*$minAmount/10;
            }
            elseif ($minAmount > 0) {
                $minAmount = -1*$maxAmount/10;
            }
            $response = [
                'title' => 'Year\'s comparison',
                'series' => $series,
                'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
                'min' => $minAmount,
                'max' => $maxAmount
            ];
            echo json_encode($response);
            die();
        }
        else {
            return false;
        }
    }

    public function actionEndOfMonthBankAccountBalancesTableData() {

        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        $year = Yii::$app->getRequest()->getQueryParam('year', false);

        if (is_numeric($month) && ($month > 0) && is_numeric($year)) {

            $accounts = AccountBalance::find()
                ->select(AccountBalance::tableName() . '.balance as amount, ' . Account::tableName() . '.name, ' . Eoms::tableName() . '.year, ' . Eoms::tableName() . '.month')
                ->joinWith('eoms')
                ->joinWith('account')
                ->joinWith('account.budget')
                ->where([
                    'AND',
                    [Eoms::tableName() . '.budget_id' => $this->activeBudgetId],
                    ['year' => $year],
                    ['month' => $month],
                    ['is_mock' => 0]
                ])
                ->all();

            $loans = DebtAmount::find()
                ->select(DebtAmount::tableName() . '.amount, debt_id, ' . Debt::tableName() . '.name, ' . Eoms::tableName() . '.year, ' . Eoms::tableName() . '.month')
                ->joinWith('eoms')
                ->joinWith('debt')
                ->joinWith('debt.budget')
                ->leftJoin(DebtType::tableName(), Debt::tableName() . '.debt_type_id = ' . DebtType::tableName() . '.id')
                ->where([
                    'AND',
                    [Eoms::tableName() . '.budget_id' => $this->activeBudgetId],
                    ['year' => $year],
                    ['month' => $month],
                    ['is_mock' => 0],
                    ['like', DebtType::tableName() . '.name', 'loan']
                ])
                ->all();

        }
        else {

            $accounts = AccountBalance::find()
                ->select(AccountBalance::tableName() . '.balance as amount, ' . Account::tableName() . '.name, ' . Eoms::tableName() . '.year, ' . Eoms::tableName() . '.month')
                ->joinWith('eoms')
                ->joinWith('account')
                ->joinWith('account.budget')
                ->where([
                    'AND',
                    [Eoms::tableName() . '.budget_id' => $this->activeBudgetId],
                    ['is_mock' => 0]
                ])
                ->all();

            $loans = DebtAmount::find()
                ->select(DebtAmount::tableName() . '.amount, debt_id, ' . Debt::tableName() . '.name, ' . Eoms::tableName() . '.year, ' . Eoms::tableName() . '.month')
                ->joinWith('eoms')
                ->joinWith('debt')
                ->joinWith('debt.budget')
                ->leftJoin(DebtType::tableName(), Debt::tableName() . '.debt_type_id = ' . DebtType::tableName() . '.id')
                ->where([
                    'AND',
                    [Eoms::tableName() . '.budget_id' => $this->activeBudgetId],
                    ['like', DebtType::tableName() . '.name', 'loan'],
                    ['is_mock' => 0]
                ])
                ->all();

        }

        foreach ($accounts as $i => $a) {
            $accounts[$i]->type = 'Account';
        }

        foreach ($loans as $i => $l) {
            $loans[$i]->type = 'Loan';
            $loans[$i]->amount -= $l->debt->getDebtPaymentsAmountUpUntil(date('Y-m-t', strtotime("{$l->year}-{$l->month}-1")));
        }

        $entries = array_merge($accounts, $loans);

        $columnNames = [
            'name',
            'amount',
            'type',
            'year',
            'month'
        ];

        $responseData = DataTables::getFilteredData($columnNames, $entries);

        foreach ($responseData['items'] as $i) {
            $responseData['response']['data'][] = [
                $i->name,
                Formatter::currency($i->amount),
                $i->type,
                $i->year,
                date("F", strtotime("2010-{$i->month}-1"))
            ];
        }

        echo json_encode($responseData['response']);
        die();
    }


    /**
     * Retrieve monthly incode graph data via json
     */
    public function actionMonthlyIncomeGraphData()
    {
        $budgetId = Yii::$app->getRequest()->getQueryParam('id', -1);
        if ($budgetId == -1) {
            $budgetId = $this->activeBudgetId;
        }
        elseif (is_numeric($budgetId)) {
            if (!Yii::$app->user->can('accessBudget', ['budgetId' => $budgetId])) {
                return $this->accessError();
            }
        }
        else {
            return $this->accessError();
        }

        // incomes
        $incomes = Income::find()
            ->select([
                Income::tableName() . '.*',
            ])
            ->leftJoin(IncomeEnd::tableName(), IncomeEnd::tableName() . '.income_id = ' . Income::tableName() . '.id')
            ->where([
                'and',
                ['budget_id' => $budgetId, 'archived' => '0'],
                ['!=', 'frequency', 'one-time'],
                [IncomeEnd::tableName() . '.date' => null]
            ])
            ->all();

        if (count($incomes)) {
            $labels = [];
            $values = [];

            foreach ($incomes as $i) {
                $labels[] = $i->name;
                $values[] = $i->getAverageMonthlyAmount();
            }
            if (array_sum($values)) {
                $response = [
                    'type'      => 'Pie',
                    'chartData'      => [
                        'labels'    => $labels,
                        'series'    => $values,
                    ]
                ];
            }
            else {
                $response = false;
            }
        }
        else {
            $response = false;
        }
        return json_encode($response);
        die();
    }

    /**
     * Retrieve jar allocations graph data via json
     */
    public function actionJarAllocationsGraphData()
    {
        $budgetId = Yii::$app->getRequest()->getQueryParam('id', -1);

        if ($budgetId == -1) {
            $budgetId = $this->activeBudgetId;
        }
        elseif (is_numeric($budgetId)) {
            if (!Yii::$app->user->can('accessBudget', ['budgetId' => $budgetId])) {
                return $this->accessError();
            }
        }
        else {
            return $this->accessError();
        }

        // jars
        $jars = Jar::find()
            ->where(['budget_id' => $budgetId, 'archived' => 0])
            ->orderBy('name ASC')
            ->all();

        $labels = [];
        $values = [];

        foreach ($jars as $j) {
            $labels[] = $j->name;
            $values[] = $j->getMonthlyBudget();
        }

        if (array_sum($values)) {
            $response = [
                'type'          => 'Pie',
                'chartData'     => [
                    'labels'    => $labels,
                    'series'    => $values,
                ]
            ];
        }
        else {
            return false;
        }

        return json_encode($response);
        die();
    }

    /**
     * Retrieve compare graph data via json
     */
    public function actionCompareGraphData()
    {
        $budgetId = Yii::$app->getRequest()->getQueryParam('id', -1);

        if ($budgetId == -1) {
            $budgetId = $this->activeBudgetId;
        }
        elseif (is_numeric($budgetId)) {
            if (!Yii::$app->user->can('accessBudget', ['budgetId' => $budgetId])) {
                return $this->accessError();
            }
        }
        else {
            return $this->accessError();
        }

        // @todo - this is going to be optimized in the future, so we won't need to request the same data multiple times
        // incomes
        $incomes = Income::find()
            ->select([
                Income::tableName() . '.*',
            ])
            ->leftJoin(IncomeEnd::tableName(), IncomeEnd::tableName() . '.income_id = ' . Income::tableName() . '.id')
            ->where([
                'and',
                ['budget_id' => $budgetId, 'archived' => '0'],
                [
                    'or',
                    ['!=', 'frequency', 'one-time'],
                    ['>=', Income::tableName() . '.date', date('Y-m-1')]
                ],
                [IncomeEnd::tableName() . '.date' => null]
            ])
            ->all();

        $minMonthlyIncome = 0;
        $avgMonthlyIncome = 0;
        foreach ($incomes as $i) {
            $minMonthlyIncome += $i->getAverageMonthlyAmount();
        }

        // jars
        $jars = Jar::find()
            ->where(['budget_id' => $budgetId, 'archived' => 0])
            ->orderBy('name ASC')
            ->all();

        $totalMonthlyBudget = 0;
        foreach ($jars as $j) {
            $totalMonthlyBudget += $j->getMonthlyBudget();
        }

        $response = [
            'type'          => 'Bar',
            'chartData'     => [
                'labels'    => ['Monthly Income', 'Monthly Jar Allocations'],
                'series'    => [
                    [
                        'name' => 'Incoming',
                        'data' => [$minMonthlyIncome,0]
                    ],
                    [
                        'name' => 'Outgoing',
                        'data' => [0,$totalMonthlyBudget]
                    ]
                ]
            ],
            'chartOptions'   => [
                'axisX'     => ['showGrid' => false],
            ],
            'chartEvents' => [
                ['eventName' => 'strokeWidth', 'data' => '50px']
            ],
        ];

        return json_encode($response);
        die();
    }

    public function actionJarSummary()
    {
        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $monthInt = Yii::$app->getRequest()->getQueryParam('month', date('n'));
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', null);

        if ($jarId) {
            $jar = Jar::findOne($jarId);
            // check access
            if (!$jar || !Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
                echo JsonTools::errorMessage('Invalid jar.');
                die();
            }
        }

        // get jars
        $budget = Budget::findOne($this->activeBudgetId);
        $jars = $budget->getActiveJars();

        $allJarsList = [
            [
                'id' => 0,
                'label' => '[All jars]',
                'url' => Url::toRoute(['/budget/jar-summary', 'year' => $year, 'month' => $monthInt]),
                'linkOptions' => ['class' => 'system-link']
            ]
        ];
        foreach ($jars as $j) {
            $active = ($j->id ==  $jarId) ? ' active' : '';
            $allJarsList[] = [
                'id' => $j->id,
                'label' => strval($j->name),
                'url' => Url::toRoute(['/budget/jar-summary', 'year' => $year, 'month' => $monthInt, 'jar_id' => $j->id]),
                'options' => ['class' => $active],
                'linkOptions' => ['class' => 'system-link']
            ];
        }

        $firstMoneyOperationDate = $budget->getFirstMoneyOperationDate();
        $lastMoneyOperationDate = $budget->getLastMoneyOperationDate();
        $yearsList = Calculator::getYearsListFromTo(
            '/budget/jar-summary',
            date('Y', strtotime($firstMoneyOperationDate)),
            date('Y', strtotime($lastMoneyOperationDate)),
            $year
        );
        $monthsList = Calculator::getMonthsListFromTo(
            '/budget/jar-summary',
            date('n', strtotime($firstMoneyOperationDate)),
            date('Y', strtotime($firstMoneyOperationDate)),
            date('n', strtotime($lastMoneyOperationDate)),
            date('Y', strtotime($lastMoneyOperationDate)),
            $monthInt,
            $year
        );

        $jarSummary = [];
        if ($jar) {
            for ($m = 1; $m <= 12; $m++) {
                $brought = $jar->getMonthsFundsBroughtForwardAmount($m, $year);
                $mBudget = $jar->getMonthlyBudget($m, $year);
                $spent   = $jar->getMonthsTransactionsAmount($m, $year, false, false);
                $moneyIn = $jar->getMonthsMoneyInAmount($m, $year, false, false);
                $remain  = $jar->getMonthsFundsRemainingAmount($m, $year);
                $jarAmnt = $jar->getMonthsJarAmountRecord($m, $year);
                $jarSummary[] = [
                    'month'           => date('M', strtotime("$year-$m-1")),
                    'month_int'       => date('n', strtotime("$year-$m-1")),
                    'year'            => $year,
                    'brought_forward' => $brought,
                    'monthly_budget'  => $mBudget,
                    'amount_spent'    => $spent,
                    'money_in'        => $moneyIn,
                    'funds_remaining' => $remain,
                    'funds_remaining_adjustment' => $jar->getFundsRemainingAdjustment($m, $year),
                    'jarAmnt'         => $jarAmnt,
                    'jar'             => $jar
                ];
            }
        }

        // month summary
        $month    = date('F', strtotime("$year-$monthInt-20"));
        $this->view->title = 'Jar Summary';

        return $this->render('jar-summary', [
            'years' => $yearsList,
            'months' => $monthsList,
            'allJars' => $allJarsList,
            'jars' => $jars,
            'jarName' => $jar ? $jar->name : '[All jars]',
            'jarId' => $jarId,
            'jarSummary' => $jarSummary,
            'year' => $year,
            'month' => $month,
            'monthInt' => $monthInt,
            'currentMonth' => $monthInt == date('n') && $year == date('Y') ? true : false
        ]);
    }

    public function actionAdjustJarFundsRemaining()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        $year  = Yii::$app->getRequest()->getQueryParam('year', false);

        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId]) ||
            !$month || !$year) {
            return $this->accessError();
        }
        $jar   = Jar::findOne($jarId);
        $month = (int)$month;
        $year  = (int)$year;
        // don't allow to adjust funds remaining for the dates before first EoMS
        if (!$jar->isAfterFirstEoms($month, $year)) {
            echo JsonTools::errorMessage('These values can not be changed until you complete your first End of Month summary report.');
            die();
        }

        $adjustFundsForm = new AdjustJarFundsRemaining();
        $adjustFundsForm->setAttributes([
            'jar_id' => $jarId,
            'month'  => $month,
            'year'   => $year,
            'funds_remaining_current' => $jar->getMonthsFundsRemainingAmount($month, $year)
        ]);

        if (Yii::$app->request->isPost && $adjustFundsForm->load(Yii::$app->request->post())) {
            if ($adjustFundsForm->validate() && !$jar->archived) {
                $eoms = Eoms::find()
                    ->where([
                        'budget_id' => $this->activeBudgetId,
                        'month' => $month,
                        'year' => $year
                    ])
                    ->one();
                if ($eoms) {
                    Jar::saveFundsBroughtForwardForEoms($eoms);
                }
                $jar->saveJarAdjustmentRecord($adjustFundsForm, $month, $year, $this->defaultAccountId);
                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Remaining jar funds successfully adjusted.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($adjustFundsForm);
                die();
            }
        }
        else {
            $this->view->title = 'Adjust Funds Remaining';
            $this->jsCallbackToView();
            $jars = $jar->budget->getActiveJars();
            $yearsList  = Calculator::getYearsList('/budget/adjust-jar-funds-remaining', $year, $jarId);
            $monthsList = Calculator::getMonthsList('/budget/adjust-jar-funds-remaining', $year, $month, $jarId);

            return $this->render('adjust-jar-funds-remaining', [
                'formModel' => $adjustFundsForm,
                'jar' => $jar,
                'jarsList' => ArrayHelper::map($jars, 'id', 'name'),
                'months' => ArrayHelper::map($monthsList, 'id', 'label'),
                'years' => ArrayHelper::map($yearsList, 'id', 'label'),
                'year' => $year,
                'month' => $month
            ]);
        }
    }

    public function actionJarSummaryTooltip()
    {
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        $year  = Yii::$app->getRequest()->getQueryParam('year', false);

        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId]) ||
            !$month || !$year) {
            return $this->accessError();
        }
        $jar   = Jar::findOne($jarId);
        $month = (int)$month;
        $year  = (int)$year;

        $jarAmount  = $jar->getMonthsJarAmountRecord($month, $year);
        $expenses   = [];
        $budgeted   = 0;
        $unbudgeted = 0;
        $spent      = 0;
        $date       = date('Y-m', strtotime($year.'-'.$month.'-20'));

        // we can get detailed info
        if ($date <= date('Y-m')) {
            $expenses   = $jar->getMonthsExpenses($month, $year, true, false);
            $budgeted   = $jar->getMonthlyBudget($month, $year);
            $spent      = $jar->getMonthsTransactionsAmount($month, $year, false, false);
            $unbudgeted = $jar->getMonthsUnbudgetedTransactionsAmount($month, $year);
        }

        $this->view->title = 'Summary Details';
        return $this->render('jar-summary-tooltip', [
            'month'      => $month,
            'year'       => $year,
            'expenses'   => $expenses,
            'budgeted'   => $budgeted,
            'unbudgeted' => $unbudgeted,
            'spent'      => $spent,
            'jarAmount'  => $jarAmount,
            'jarFundsRemainingAdjustment' => $jar->getFundsRemainingAdjustment($month, $year)
        ]);
    }

    public function actionCalculator()
    {
        $this->view->title = 'Budget Shortfall Calculator';
        return $this->render('calculator', [
            'budgetId' => $this->activeBudgetId
        ]);
    }

    public function actionFinancialPositionGraph()
    {
        // month + year = last month of the chart (10/2016)
        $monthEnd = (int)Yii::$app->getRequest()->getQueryParam('month', date('m'));
        $yearEnd = Yii::$app->getRequest()->getQueryParam('year', date('Y')); // last year
        // how many months to look back (10/2016 - 6 = 4/2016)
        $length = Yii::$app->getRequest()->getQueryParam('length', 6);

        $timeEnd = strtotime($yearEnd.'-'.$monthEnd.'-20');
        $budget = Budget::findOne($this->activeBudgetId);
        $amounts = [];
        $min = 0;
        $max = 0;

        for ($i = $length; $i >= 0; $i--) {
            $time = strtotime("-$i months", $timeEnd);
            $month = date('n', $time);
            $year = date('Y', $time);

            //var_dump($month);
            //var_dump($year);

            $totalAccounts = $budget->getTotalAccountsBalance($month, $year);
            $totalDebts = $budget->getTotalDebtsBalance($month, $year);
            $net = $totalAccounts + $totalDebts;
            $amounts['net'][] = $net;
            $amounts['accounts'][] = $totalAccounts;
            $amounts['debts'][] = $totalDebts;
            if ($net < $min || $totalAccounts < $min || $totalDebts < $min) {
                $min = min($net, $totalAccounts, $totalDebts);
            }
            if ($net > $max || $totalAccounts > $max || $totalDebts > $max) {
                $max = max($net, $totalAccounts, $totalDebts);
            }
        }

        $response = [
            'title' => '',
            'labels' => ['','','','','',''],
            'series' => [
                [
                    'name' => 'Net Position',
                    'data' => $amounts['net']
                ], [
                    'name' => 'Total Accounts',
                    'data' => $amounts['accounts']
                ], [
                    'name' => 'Total Debt',
                    'data' => $amounts['debts']
                ]
            ],
            'min' => $min,
            'max' => $max
        ];
        return json_encode($response);
    }

    public function actionStartOver()
    {
        $eoms = $this->getCurrentEoms();
        if ($eoms && Yii::$app->user->can('editEoms', ['eoms' => $eoms]) && !$eoms->submitted) {
            AccountBalance::deleteAll(['eoms_id' => $eoms->id]);
            AccountEomsBalance::deleteAll(['eoms_id' => $eoms->id]);
            DebtAmount::deleteAll(['eoms_id' => $eoms->id]);

            $date = date('M Y', strtotime("{$eoms->year}-{$eoms->month}-20"));
            foreach ($eoms->budget->getMonthsAccounts($eoms->month, $eoms->year) as $account) {
                $params = [
                    'is_adjustment' => 1,
                    'account_id' => $account->id,
                    'frequency' => 'one-time'
                ];
                Income::deleteAll(array_merge($params, [
                    'budget_id' => $eoms->budget_id,
                    'name' => 'Account Amount Adjustment for ' . $date
                ]));
                Income::deleteAll(array_merge($params, [
                    'budget_id' => $eoms->budget_id,
                    'name' => 'Credit Card Amount Adjustment for ' . $date
                ]));
                Expense::deleteAll(array_merge($params, [
                    'name' => 'Account Amount Adjustment for ' . $date
                ]));
                Expense::deleteAll(array_merge($params, [
                    'name' => 'Credit Card Amount Adjustment for ' . $date
                ]));
            }
            foreach ($eoms->budget->getMonthsActiveCcDebts($eoms->month, $eoms->year) as $debt) {
                DebtPayment::deleteAll([
                    'debt_id' => $debt->id,
                    'description' => 'Debt Amount Adjustment for ' . $date
                ]);
            }
            foreach ($eoms->budget->getMonthsDebts($eoms->month, $eoms->year, 1, true) as $debt) {
                DebtPayment::deleteAll([
                    'debt_id' => $debt->id,
                    'description' => 'Debt Amount Adjustment for ' . $date
                ]);
            }
        }
        return $this->redirect(['/budget/end-of-month-summary']);
    }

    protected function getCurrentEoms($month = null, $year = null)
    {
        // select the oldest not submitted eoms
        $eoms = Eoms::getLastEoms($this->activeBudgetId);
        // select eoms for specific month
        if ($month && $year || !$eoms) {
            $eomsQuery = Eoms::find()
                ->where([
                    'budget_id' => $this->activeBudgetId
                ])
                ->orderBy('year desc, month desc');
            if ($month && $year) {
                $eomsQuery->andWhere([
                    'month'     => (int)$month,
                    'year'      => (int)$year
                ]);
            }
            $eoms = $eomsQuery->one();
        }
        return $eoms;
    }
}
