<?php

namespace backend\controllers;

use backend\models\db\Jar;
use backend\models\db\Transfer;
use backend\models\db\TransferEvent;
use Yii;

use yii\filters\AccessControl;

use yii\helpers\ArrayHelper;
use yii\helpers\Url;

use backend\components\CustomController;

use backend\models\db\Account;
use backend\models\db\Budget;
use backend\models\db\Debt;
use backend\models\db\DebtPayment;

use backend\models\form\AddDebtPayment;

use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;
use backend\components\helpers\JsonTools;

/**
 * Debt payments controller
 */
class DebtPaymentsController extends CustomController {

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'add-debt-payment',
                            'debt-payments-table',
                            'debt-payments-table-info',
                            'debt-payments-table-full',
                            'debt-transactions-table-full',
                            'debt-jar-debt-payments-table-info',
                            'interest-jar-interest-payments-table-info'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];

    }


    /**
     * Show debt payments
     */
    public function actionIndex() {

        $budget = Budget::findOne($this->activeBudgetId);

        $debtPayments = $budget->getThisMonthsDebtPayments();

        $debts = Debt::findAll(['budget_id' => $this->activeBudgetId, 'archived' => 0]);

        return $this->render('_debt-payments-table-full.php', [
            'payments' => $debtPayments,
            'debtNames' => ArrayHelper::map($debts, 'id', 'name'),
        ]);

    }

    /**
     * Add a new debt payment
     */
    public function actionAddDebtPayment()
    {
        $debtId = Yii::$app->getRequest()->getQueryParam('debt_id', false);
        if (!is_numeric($debtId)) {
            return $this->accessError();
        }
        $debt = Debt::findOne($debtId);
        if ($debt->budget_id != $this->activeBudgetId) {
            return $this->accessError();
        }

        $debtPaymentForm = new AddDebtPayment();
        if (Yii::$app->request->isPost && $debtPaymentForm->load(Yii::$app->request->post())) {
            $debtPaymentForm->debt_id = $debtId;
            $debtPaymentForm->budget_id = $this->activeBudgetId;

            // set validation attributes
            if ($debtPaymentForm->validate()) {
                $formData = $debtPaymentForm->getAttributes();
                $formData['account_id'] = $formData['account_from_id'];

                // whether we pay off the old debt or the new debt on the CC
                $formData['debt_payment_type_id'] = $debtPaymentForm->old_debt_payment ? 3 : 1;

                // now if the old debt payment amount is greater than the old debt amount that's actually left
                // we need to split this payment:
                // 1) the first part will cover the remaining old debt amount
                // 2) the second part will just apply to the new debt amount
                if ($formData['debt_payment_type_id'] == 3 && $formData['amount'] > $debt->getOldDebtLeftAmount()) {
                    // first part
                    $fullAmount = $formData['amount'];
                    $oldAmount = $debt->getOldDebtLeftAmount();
                    $formData['amount'] = $oldAmount;
                    $payment = new DebtPayment();
                    $payment->setAttributes($formData);
                    $payment->save();
                    // second part
                    $formData['debt_payment_type_id'] = 1;
                    $formData['amount'] = $fullAmount - $oldAmount;
                }
                $payment = new DebtPayment();
                $payment->setAttributes($formData);
                $payment->debt_payment_type_id = 1;
                $payment->save();

                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Debt payment successfully added.');
                    die();
                }

            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($debtPaymentForm);
                die();
            }
        }
        else {
            $accounts = Account::findAll(['budget_id' => $this->activeBudgetId, 'archived' => 0]);
            $this->view->title = 'Add Debt Payment';
            $this->jsCallbackToView();

            return $this->render('add-debt-payment', [
                'formModel' => $debtPaymentForm,
                'accounts' => ArrayHelper::map($accounts, 'id', 'name'),
                'debtId' => $debtId,
                'debt' => $debt
            ]);
        }
    }

    public function actionDebtPaymentsTable()
    {
        $debtId = Yii::$app->getRequest()->getQueryParam('debt_id', false);

        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'date',
                'select' => DebtPayment::tableName() . '.date'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => DebtPayment::tableName() . '.amount'
            ],
            [
                'type' => 'normal',
                'as' => 'description',
                'select' => DebtPayment::tableName() . '.description'
            ],
            [
                'type' => 'normal',
                'as' => 'account_name',
                'select' => Account::tableName() . '.name'
            ],
            [
                'type' => 'function',
                'as' => 'jarName',
                'name' => function($i) {
                    return $i->debtPaymentType->name;
                }
            ]
        ];

        $debt = Debt::findOne($debtId);

        if (!$debt || ($debt->budget_id != $this->activeBudgetId)) {
            // just want to have empty results in this case
            $countFiltered  = DebtPayment::find()->where(['id' => 0]);
            $selectFiltered = DebtPayment::find()->where(['id' => 0]);
        }
        else {
            $countFiltered = DebtPayment::find()
                ->where([
                    'and',
                    [DebtPayment::tableName() . '.debt_id' => $debtId],
                    ['Month(' . DebtPayment::tableName() . '.date)' => date('m')],
                    ['Year(' . DebtPayment::tableName() . '.date)' => date('Y')]
                ])
                ->andWhere(['or',
                    ['is_adjustment' => 0],
                    ['is_adjustment' => DebtPayment::DEBT_PAYMENT_ADJUSTMENT]
                ]);

            $selectFiltered = DebtPayment::find()
                ->select(DebtPayment::tableName() . '.* ,' . Account::tableName() . '.name as account_name')
                ->joinWith('account')
                ->where([
                    'and',
                    [DebtPayment::tableName() . '.debt_id' => $debtId],
                    ['Month(' . DebtPayment::tableName() . '.date)' => date('m')],
                    ['Year(' . DebtPayment::tableName() . '.date)' => date('Y')]
                ])
                ->andWhere(['or',
                    ['is_adjustment' => 0],
                    ['is_adjustment' => DebtPayment::DEBT_PAYMENT_ADJUSTMENT]
                ]);
        }

        $responseData = DataTables::getResponseData($tableColumns, $countFiltered, $selectFiltered);

        foreach ($responseData['items'] as $i) {
            $actionClass = get_class($i);
            $isEditable = $i->date >= date('Y-m-01') && $i->date <= date('Y-m-t');
            $row = [
                'isEditable' => $isEditable
            ];
            if ($isEditable) {
                $row['id'] = $i->id;
                $row['type'] = $actionClass;
                $row['editUrl'] = '/money-operations/edit-transaction';
                $row['deleteUrl'] = Url::toRoute(['/money-operations/delete-transaction', 'transaction_id' => $i->id, 'transaction_type' => $actionClass, 'js_callback' => 'reloadDebtsAfterDebtPaymentAmountEdit']);
                $row['callback'] = 'reloadDebtsAfterDebtPaymentAmountEdit';
            }
            $columns = [
                [
                    'key' => 'date',
                    'type' => 'date',
                    'value' => $i->date,
                    'isEditable' => true
                ], [
                    'key' => 'amount',
                    'type' => 'currency',
                    'value' => -($i->amount),
                    'isEditable' => true
                ],  [
                    'key' => 'description',
                    'type' => 'text',
                    'value' => $i->description,
                    'isEditable' => true
                ], [
                    'key' => 'account_name',
                    'type' => 'text',
                    'value' => $i->account_name,
                ], [ # this has 'Jar' table header but I doubt treating it as a jar_id editor would work, let's not make it editable for now
                    'key' => 'debt_type',
                    'type' => 'text',
                    'value' => $i->debtPaymentType->name
                ]
            ];
            $responseData['response']['data'][] = DataTables::getRow($row, $columns);
        }
        echo json_encode($responseData['response']);
        die();
    }


    /**
     * Info for debt payments table footer
     */
    public function actionDebtPaymentsTableInfo()
    {
        $debtId = Yii::$app->getRequest()->getQueryParam('debt_id', false);
        if (!is_numeric($debtId)) {
            return $this->accessError();
        }
        $debt = Debt::findOne($debtId);
        if ($debt->budget_id != $this->activeBudgetId) {
            return $this->accessError();
        }

        echo json_encode([
            'this-months-debt-payments-amount' => Formatter::currency($debt->getThisMonthsDebtPaymentsAmount() + $debt->getThisMonthsDebtInterestPaymentsAmount())
        ]);
        die();
    }

    /**
     * Debt payments table displayed on jars page
     */
    public function actionDebtPaymentsTableFull()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        $debtId = Yii::$app->getRequest()->getQueryParam('debt_id', false);

        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'debt_name',
                'select' => Debt::tableName() . '.name'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => Debt::tableName() . '.payment'
            ],
            [
                'type' => 'normal',
                'as' => 'frequency',
                'select' => Debt::tableName() . '.frequency'
            ],
            [
                'type' => 'method',
                'as' => 'monthlyAverage',
                'select' => 'getAverageMonthlyAmount'
            ],
            [
                'type' => 'normal',
                'as' => 'account_name',
                'select' => Account::tableName() . '.name'
            ],
            [
                'type' => 'method',
                'as' => 'nextDueDate',
                'select' => 'getNextDue'
            ],
        ];

        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));
        $budget = Budget::findOne($budgetId);
        $jar = $budget->getDebtsJar();
        if (!$jar) {
            echo JsonTools::errorMessage('You have no debts yet.');
            die();
        }

        $countFiltered = $jar->getMonthsDebtsExpensesQuery($month, $year);
        $selectFiltered = $jar->getMonthsDebtsExpensesQuery($month, $year);

        $responseData = DataTables::getResponseData($tableColumns, $countFiltered, $selectFiltered);

        /** @var Debt $i */
        foreach ($responseData['items'] as $i) {
            $editJsCallback = 'reloadDebtJarAfterDebtPaymentAmountEdit';

            $responseData['response']['data'][] = [
                $i->name,
                Formatter::currency($i->payment),
                $i->frequency,
                Formatter::currency($i->getAverageMonthlyAmount()),
                $i->account->name,
                Formatter::date($i->getNextDue()),
                '&nbsp;'
            ];

        }
        echo json_encode($responseData['response']);
        die();
    }

    /**
     * Debt payments table displayed on jars page
     */
    public function actionDebtTransactionsTableFull()
    {
        $budgetId = $this->getBudgetIdFromUrl();

        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'date',
                'select' => DebtPayment::tableName() . '.date'
            ],
            [
                'type' => 'normal',
                'as' => 'debt_name',
                'select' => Debt::tableName() . '.name'
            ],
            [
                'type' => 'normal',
                'as' => 'description',
                'select' => DebtPayment::tableName() . '.description'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => DebtPayment::tableName() . '.amount'
            ],
            [
                'type' => 'normal',
                'as' => 'account_name',
                'select' => Account::tableName() . '.name'
            ]
        ];

        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));
        $budget = Budget::findOne($budgetId);
        $jar = $budget->getDebtsJar();

        $countFiltered = $jar->getMonthsDebtsTransactionsQuery($month, $year);
        $selectFiltered = $jar->getMonthsDebtsTransactionsQuery($month, $year);

        $responseData = DataTables::getResponseData($tableColumns, $countFiltered, $selectFiltered);

        /** @var DebtPayment $i */
        foreach ($responseData['items'] as $i) {
            $editJsCallback = 'reloadDebtJarAfterDebtPaymentAmountEdit';

            $responseData['response']['data'][] = [
                Formatter::date($i->date),
                $i->debt->name,
                $i->description,
                Formatter::currency($i->amount),
                $i->account->name,
                '&nbsp;'
            ];

        }
        echo json_encode($responseData['response']);
        die();
    }


    /**
     * Info for debt jar debt payments table footer
     */
    public function actionDebtJarDebtPaymentsTableInfo() {

        $budgetId = $this->getBudgetIdFromUrl();

        // get the budget
        $budget = Budget::findOne($budgetId);

        echo json_encode([
            'this-months-jar-debt-payments-amount' => Formatter::currency($budget->getThisMonthsDebtPaymentsAmount())
        ]);
        die();

    }


    /**
     * Info for interest jar debt payments table footer
     */
    public function actionInterestsInterestPaymentsTableInfo() {

        $budgetId = $this->getBudgetIdFromUrl();

        // get the budget
        $budget = Budget::findOne($budgetId);

        echo json_encode([
            'this-months-jar-interest-payments-amount' => Formatter::currency($budget->getThisMonthsDebtInterestPaymentsAmount())
        ]);
        die();

    }

}
