<?php

namespace backend\controllers;


use backend\components\helpers\DripHelper;
use backend\models\db\Budget;
use Yii;
use yii\filters\AccessControl;

use yii\helpers\ArrayHelper;

use backend\components\CustomController;
use backend\components\helpers\Calculator;
use backend\components\helpers\JsonTools;

use backend\models\db\Account;
use backend\models\db\Currency;
use backend\models\db\DateFormat;
use backend\models\db\Debt;
use backend\models\db\Expense;
use backend\models\db\Income;
use backend\models\db\IncomeTransaction;
use backend\models\db\Jar;
use backend\models\db\TimeFormat;
use backend\models\db\TimeZone;
use backend\models\db\Transaction;
use backend\models\db\User;
use backend\models\db\UserMeta;

use backend\models\form\EditUserBasicInfo;
use backend\models\form\IncomeTransactionEntry;
use backend\models\form\TransactionEntry;

/**
 * First Time Setup controller
 */
class FirstTimeSetupController extends CustomController {

    public $enableCsrfValidation = false;
    public $steps = [
        'Account Settings',
        'Bank Accounts',
        'Debts',
        'Income',
        'Jars & Expenses',
        //'Summary',
        //'This Month\'s Update'
    ];

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'welcome',
                            'step-1',
                            'step-2',
                            'step-3',
                            'step-4',
                            'step-5',
                            'step-5-jar-detail',
                            'step-6',
                            'step-7',
                            'complete',
                        ],
                        'allow'   => true,
                        'roles'   => ['@'],
                    ],
                ],
            ]
        ];

    }

    public function init()
    {
        parent::init();
        $this->layout = 'minimal';
    }

    public function beforeAction($action)
    {
        parent::beforeAction($action);

        if ($this->mainUserId != Yii::$app->user->identity->id && !in_array(Yii::$app->user->identity->userRole->name, ['superadmin', 'coach']) ||
            $this->userMeta['fts_step'] == 'complete') {
            $this->redirect(['/']);
            return false;
        }
        else {
            return true;
        }
    }

    public function actionWelcome()
    {
        $this->view->title = 'Welcome!';
        return $this->render('welcome');
    }


    /**
     * Step 1 of FTS
     */
    public function actionStep1() {

        if ((string) $this->userMeta['fts_step'] == '0') {

            $userMeta = UserMeta::findOne(['key' => 'fts_step', 'user_id' => $this->mainUserId]);
            $userMeta->value = '1';
            $userMeta->save();

            $this->userMeta['fts_step'] = 1;
            Yii::$app->session->set('userMeta', $this->userMeta);
            $stepId = 1;

        }

        $stepId = strtr($this->userMeta['fts_step'], ['d' => '']);
        $users = User::getAllUsersForUserId($this->mainUserId);

        $basicInfoForm = new EditUserBasicInfo();
        $basicInfoForm->setAttributes([
            'currency_id' => $this->getUserMeta('currency_id'),
            'time_zone_id' => $this->getUserMeta('time_zone_id'),
            'time_format_id' => $this->getUserMeta('time_format_id'),
            'date_format_id' => $this->getUserMeta('date_format_id')
        ]);

        $this->view->title = 'Step 1 - First Time Setup';

        return $this->render('step-layout', [
            'steps' => $this->steps,
            'currentStepId' => 1,
            'stepProgress' => $stepId,
            'stepProgressFull' => $this->userMeta['fts_step'],
            'stepValues' => [
                'currencies' => Currency::getList(),
                'timeZones' => ArrayHelper::map(TimeZone::find()->all(), 'id', 'name'),
                'timeFormats' => ArrayHelper::map(TimeFormat::find()->all(), 'id', 'name'),
                'dateFormats' => ArrayHelper::map(DateFormat::find()->all(), 'id', 'name'),
                'currencyFormats' => Currency::CURRENCY_FORMATS,
                'users' => $users,
                'mainUserId' => $this->mainUserId
            ],
            'formModel' => $basicInfoForm,
            'showNext' => false
        ]);

    }


    /**
     * Step 2 of FTS
     */
    public function actionStep2() {

        $stepId = strtr($this->userMeta['fts_step'], ['d' => '']);
        if ($stepId == '1') {

            $userMeta = UserMeta::findOne(['key' => 'fts_step', 'user_id' => $this->mainUserId]);
            $userMeta->value = '2';
            $userMeta->save();

            $this->userMeta['fts_step'] = 2;
            Yii::$app->session->set('userMeta', $this->userMeta);
            $stepId = 2;

        }
        elseif ($stepId < 2) {

            if ($stepId == 0) {

                $this->redirect(['/first-time-setup/welcome']);
                return false;

            }
            else {

                $this->redirect(['/first-time-setup/step-' . $stepId]);
                return false;

            }

        }

        $accounts = $this->_getAccountsList();

        $this->view->title = 'Step 2 - First Time Setup';

        return $this->render('step-layout', [
            'steps' => $this->steps,
            'currentStepId' => 2,
            'stepProgress' => $stepId,
            'stepProgressFull' => $this->userMeta['fts_step'],
            'stepValues' => [
                'accounts' => $accounts,
                'defaultAccountId' => $this->defaultAccountId
            ],
            'showNext' => true
        ]);

    }


    /**
     * Step 3 of FTS
     */
    public function actionStep3() {

        $stepId = strtr($this->userMeta['fts_step'], ['d' => '']);

        if ((string) $this->userMeta['fts_step'] == '2') {

            $userMeta = UserMeta::findOne(['key' => 'fts_step', 'user_id' => $this->mainUserId]);
            $userMeta->value = '3';
            $userMeta->save();

            $this->userMeta['fts_step'] = 3;
            Yii::$app->session->set('userMeta', $this->userMeta);
            $stepId = 3;

        }
        elseif ($stepId < 3) {

            if ($stepId == 0) {

                $this->redirect(['/first-time-setup/welcome']);
                return false;

            }
            else {

                $this->redirect(['/first-time-setup/step-' . $stepId]);
                return false;

            }

        }

        $debts = $this->_getDebtsList();

        // get total debt
        $totalDebtsAmount = 0;
        foreach ($debts as $d) {
            $totalDebtsAmount -= $d->amount;
        }

        // get total account balance
        $accounts = Account::findAll(['budget_id' => $this->activeBudgetId, 'archived' => 0]);
        $totalAccountsBalance = 0;
        foreach ($accounts as $a) {
            $totalAccountsBalance += $a->balance;
        }

        $totalMoney = $totalAccountsBalance + $totalDebtsAmount;
        if ($totalMoney != 0) {
            $fullBar = abs($totalAccountsBalance) + abs($totalDebtsAmount);
            $accountsPctg = abs(round(abs($totalAccountsBalance) / $fullBar * 100));
            $debtsPctg = 100 - $accountsPctg;
        }
        else {
            $accountsPctg = 50;
            $debtsPctg = 50;
        }

        $this->view->title = 'Step 3 - First Time Setup';

        return $this->render('step-layout', [
            'steps' => $this->steps,
            'currentStepId' => 3,
            'stepProgress' => $stepId,
            'stepProgressFull' => $this->userMeta['fts_step'],
            'stepValues' => [
                'debts' => $debts,
                'defaultAccountId' => $this->defaultAccountId,
                'totalAccountsBalance' => $totalAccountsBalance,
                'isAccountNegative' => ($totalAccountsBalance < 0),
                'totalDebtsAmount' => $totalDebtsAmount,
                'totalMoney' => $totalMoney,
                'isTotalNegative' => ($totalMoney < 0),
                'accountsPctg' => $accountsPctg,
                'debtsPctg' => $debtsPctg
            ],
            'showNext' => true
        ]);

    }


    /**
     * Step 4 of FTS
     */
    public function actionStep4() {

        $stepId = strtr($this->userMeta['fts_step'], ['d' => '']);
        if ((string) $this->userMeta['fts_step'] == '3') {

            $userMeta = UserMeta::findOne(['key' => 'fts_step', 'user_id' => $this->mainUserId]);
            $userMeta->value = '4';
            $userMeta->save();

            $this->userMeta['fts_step'] = 4;
            Yii::$app->session->set('userMeta', $this->userMeta);
            $stepId = 4;

        }
        elseif ($stepId < 4) {

            if ($stepId == 0) {

                $this->redirect(['/first-time-setup/welcome']);
                return false;

            }
            else {

                $this->redirect(['/first-time-setup/step-' . $stepId]);
                return false;

            }

        }

        $incomes = $this->_getIncomesList();

        $this->view->title = 'Step 4 - First Time Setup';

        return $this->render('step-layout', [
            'steps' => $this->steps,
            'currentStepId' => 4,
            'stepProgress' => $stepId,
            'stepProgressFull' => $this->userMeta['fts_step'],
            'stepValues' => [
                'incomes' => $incomes,
                'defaultAccountId' => $this->defaultAccountId
            ],
            'showNext' => true
        ]);

    }


    /**
     * Step 5 of FTS
     */
    public function actionStep5() {

        $stepId = strtr($this->userMeta['fts_step'], ['d' => '']);

        if ((string) $this->userMeta['fts_step'] == '4') {

            $userMeta = UserMeta::findOne(['key' => 'fts_step', 'user_id' => $this->mainUserId]);
            $userMeta->value = '5';
            $userMeta->save();

            $this->userMeta['fts_step'] = 5;
            Yii::$app->session->set('userMeta', $this->userMeta);
            $stepId = 5;

        }
        elseif ($stepId < 5) {

            if ($stepId == 0) {

                $this->redirect(['/first-time-setup/welcome']);
                return false;

            }
            else {

                $this->redirect(['/first-time-setup/step-' . $stepId]);
                return false;

            }

        }

        $jars = $this->_getJarsMenuItems();
        $expVsInc = $this->_getExpensesVsIncomesBarData($jars);

        $this->view->title = 'Step 5 - First Time Setup';

        return $this->render('step-layout', [
            'steps' => $this->steps,
            'currentStepId' => 5,
            'stepProgress' => $stepId,
            'stepProgressFull' => $this->userMeta['fts_step'],
            'stepValues' => [
                'jars' => $jars,
                'totalIncomesAmount' => $expVsInc['totalIncomesAmount'],
                'totalExpensesAmount' => $expVsInc['totalExpensesAmount'],
                'incomesPctg' => $expVsInc['incomesPctg'],
                'expensesPctg' => $expVsInc['expensesPctg']
            ],
            'showNext' => true
        ]);

    }


    /**
     * Jar detail for step 5 of FTS
     */
    public function actionStep5JarDetail()
    {
        $stepId = strtr($this->userMeta['fts_step'], ['d' => '']);

        if ($stepId < 5) {
            if ($stepId == 0) {
                $this->redirect(['/first-time-setup/welcome']);
                return false;
            }
            else {
                $this->redirect(['/first-time-setup/step-' . $stepId]);
                return false;
            }
        }

        $jarId = Yii::$app->getRequest()->getQueryParam('id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        if (!is_numeric($jarId) || !Yii::$app->user->can('editJar', ['jarId' => $jarId])) {
            return $this->accessError();
        }

        // get the jar
        $jar = Jar::findOne(['id' => $jarId]);
        $this->view->title = $jar->name . ' jar - Step 5 - First Time Setup';

        if ($this->pageLoadRequest || $fullLoad) {
            $jars = $this->_getJarsMenuItems();
            $expVsInc = $this->_getExpensesVsIncomesBarData($jars);

            return $this->render('step-5-jar-detail', [
                'steps' => $this->steps,
                'currentStepId' => 5,
                'stepProgress' => $stepId,
                'stepProgressFull' => $this->userMeta['fts_step'],
                'jars' => $jars,
                'jar' => $jar,
                'totalIncomesAmount' => $expVsInc['totalIncomesAmount'],
                'totalExpensesAmount' => $expVsInc['totalExpensesAmount'],
                'incomesPctg' => $expVsInc['incomesPctg'],
                'expensesPctg' => $expVsInc['expensesPctg'],
                'showNext' => true,
                'budget' => $jar->getBudget()->one()
            ]);
        }
        else {
            return $this->render('step-5-jar-detail-content', [
                'steps' => $this->steps,
                'currentStepId' => 5,
                'stepProgress' => $stepId,
                'stepProgressFull' => $this->userMeta['fts_step'],
                'jar' => $jar,
                'showNext' => true,
                'budget' => $jar->getBudget()->one()
            ]);
        }
    }


    /**
     * Step 6 of FTS
     */
    public function actionStep6()
    {
        // Steps 6 and 7 are no longer used
        return $this->redirect(['/first-time-setup/complete']);

        /*$stepId = strtr($this->userMeta['fts_step'], ['d' => '']);

        if ($this->userMeta['fts_step'] == '5') {

            $userMeta = UserMeta::findOne(['key' => 'fts_step', 'user_id' => $this->mainUserId]);
            $userMeta->value = '6';
            $userMeta->save();

            $this->userMeta['fts_step'] = 6;
            Yii::$app->session->set('userMeta', $this->userMeta);
            $stepId = 6;

        }
        elseif ($stepId < 6) {

            if ($stepId == 0) {

                $this->redirect(['/first-time-setup/welcome']);
                return false;

            }
            else {

                $this->redirect(['/first-time-setup/step-' . $stepId]);
                return false;

            }

        }

        $users    = User::getAllUsersForUserId($this->mainUserId);
        $accounts = $this->_getAccountsList();
        $debts    = $this->_getDebtsList();
        $incomes  = $this->_getIncomesList();
        $jars     = $this->_getJarsMenuItems();
        $budget   = Budget::findOne($this->activeBudgetId);

        $this->view->title = 'Step 6 - First Time Setup';

        return $this->render('step-layout', [
            'steps'             => $this->steps,
            'currentStepId'     => 6,
            'stepProgress'      => $stepId,
            'stepProgressFull'  => $this->userMeta['fts_step'],
            'stepValues'        => [
                'users'               => $users,
                'accounts'            => $accounts,
                'debts'               => $debts,
                'incomes'             => $incomes,
                'jars'                => $jars,
                'minMonthlyIncome'    => $budget->getMinimumMonthlyIncomeAmount(),
                'avgMonthlyIncome'    => $budget->getAverageMonthlyIncomeAmount(),
                'totalMonthlyBudget'  => $budget->getTotalMonthlyBudget(),
                'minMonthlySurplus'   => $budget->getMinimumMonthlyIncomeAmount() - $budget->getTotalMonthlyBudget(),
                'avgMonthlySurplus'   => $budget->getAverageMonthlyIncomeAmount() - $budget->getTotalMonthlyBudget(),
                'budgetId'            => $budget->id
            ],
            'showNext' => true
        ]);*/
    }


    /**
     * Step 7 of FTS
     */
    public function actionStep7()
    {
        // Steps 6 and 7 are no longer used
        return $this->redirect(['/first-time-setup/complete']);

        /*$stepId = strtr($this->userMeta['fts_step'], ['d' => '']);

        if ($this->userMeta['fts_step'] == '6') {
            $userMeta = UserMeta::findOne(['key' => 'fts_step', 'user_id' => $this->mainUserId]);
            $userMeta->value = '7';
            $userMeta->save();

            $this->userMeta['fts_step'] = 7;
            Yii::$app->session->set('userMeta', $this->userMeta);
            $stepId = 7;
        }
        elseif ($stepId < 7) {
            if ($stepId == 0) {
                $this->redirect(['/first-time-setup/welcome']);
                return false;
            }
            else {
                $this->redirect(['/first-time-setup/step-' . $stepId]);
                return false;
            }
        }

        // incomes
        $incomesUnindexed = $this->_getIncomesList();
        $incomes = [];
        foreach ($incomesUnindexed as $i) {
            $incomes[$i->id] = $i;
        }

        // expenses
        $expensesUnindexed = $this->_getExpensesList();
        $expenses = [];
        $jarIds = [];
        foreach ($expensesUnindexed as $e) {
            $expenses[$e->id] = $e;
            if (!in_array($e->jar_id, $jarIds)) {
                $jarIds[] = $e->jar_id;
            }
        }

        // add, edit or delete entries
        if (Yii::$app->request->isPost) {
            $post = Yii::$app->request->post();
            // save income transactions
            if (isset($post['IncomeTransactionEntry']['generate']) && count($post['IncomeTransactionEntry']['generate'])) {
                foreach ($post['IncomeTransactionEntry']['generate'] as $index => $generate) {
                    $entry = new IncomeTransactionEntry();
                    $entry->setAttributes([
                        'generate' => $generate,
                        'income_id' => $post['IncomeTransactionEntry']['income_id'][$index],
                        'amount' => $post['IncomeTransactionEntry']['amount'][$index],
                        'date' => $post['IncomeTransactionEntry']['date'][$index],
                        'budget_id' => $this->activeBudgetId
                    ]);

                    if ($entry->validate()) {
                        // add or edit || remove
                        if ($entry->generate) {
                            $incomeTransaction = IncomeTransaction::findOne(['income_id' => $entry->income_id, 'date' => $entry->date]);
                            if (!$incomeTransaction) {
                                $incomeTransaction = new IncomeTransaction();
                            }
                            $incomeTransaction->setAttributes($entry->getAttributes());
                            $incomeTransaction->account_id = $incomes[$entry->income_id]->account_id;
                            $incomeTransaction->save();
                            if (!$incomeTransaction->save()) {
                                echo JsonTools::formErrorMessage($incomeTransaction);
                                die();
                            }
                        }
                        else {
                            IncomeTransaction::deleteAll(['income_id' => $entry->income_id, 'date' => $entry->date]);
                        }
                    }
                    else {
                        echo JsonTools::formErrorMessage($entry);
                        die();
                    }
                }
            }

            // save expense transactions
            if (isset($post['TransactionEntry']['generate']) && count($post['TransactionEntry']['generate'])) {
                foreach ($post['TransactionEntry']['generate'] as $index => $generate) {
                    $entry = new TransactionEntry();
                    $entry->setAttributes([
                        'generate' => $generate,
                        'expense_id' => $post['TransactionEntry']['expense_id'][$index],
                        'amount' => $post['TransactionEntry']['amount'][$index],
                        'date' => $post['TransactionEntry']['date'][$index],
                        'budget_id' => $this->activeBudgetId
                    ]);

                    if ($entry->validate()) {
                        // add or edit || remove
                        if ($entry->generate) {
                            $transaction = Transaction::findOne(['expense_id' => $entry->expense_id, 'date' => $entry->date]);
                            if (!$transaction) {
                                $transaction = new Transaction();
                            }
                            $transaction->setAttributes($entry->getAttributes());
                            $transaction->account_id = $expenses[$entry->expense_id]->account_id;
                            if (!$transaction->save()) {
                                echo JsonTools::formErrorMessage($transaction);
                                die();
                            }
                        }
                        else {
                            Transaction::deleteAll(['expense_id' => $entry->expense_id, 'date' => $entry->date]);
                        }
                    }
                    else {
                        echo JsonTools::formErrorMessage($entry);
                        die();
                    }
                }
            }
            echo JsonTools::successMessage('Transactions successfully inserted.');
            die();
        }


        // generate potential income transactions
        // dates will be calculated transactions dates, that would have taken place,
        // if the new account was created on the 1st of the current month
        $date = date('Y-m-d', strtotime('-1 DAY'));
        $day = date('d', strtotime('-1 DAY'));
        $monthStartDate = date('Y-m-1');

        $newIncomeTransDates = [];
        foreach ($incomes as $i) {
            $newIncomeTransDates[$i->id] = [];
            $firstTransDate = Calculator::prevDueDate($i->date, $i->frequency, date('Y-m-d', strtotime('-1 day', strtotime($monthStartDate))));

            if ((strtotime($firstTransDate) >= strtotime($monthStartDate)) && (strtotime($firstTransDate) <= strtotime($date))) {
                $newIncomeTransDates[$i->id][] = $firstTransDate;
                $startDay = date('d', strtotime($firstTransDate));
            }
            else {
                $startDay = 1;
            }

            for ($j = $startDay; $j<= $day; $j++) {
                $nextTransDate = Calculator::nextDueDate($firstTransDate, $i->frequency, date('Y-m-' . $j));
                if ((strtotime($nextTransDate) >= strtotime($monthStartDate)) && (strtotime($nextTransDate) <= strtotime($date)) && (!in_array($nextTransDate, $newIncomeTransDates[$i->id]))) {
                    $newIncomeTransDates[$i->id][] = $nextTransDate;
                }
            }
        }

        // get existing income transactions
        $incomeTransactions = IncomeTransaction::find()
            ->leftJoin(Income::tableName(), IncomeTransaction::tableName() . '.income_id = ' . Income::tableName() . '.id')
            ->where([
                'and',
                [Income::tableName() . '.budget_id' => $this->activeBudgetId],
                [Income::tableName() . '.archived' => 0]
            ])
            ->all();

        // generate potential expense transactions
        // dates will be calculated transactions dates, that would have taken place,
        // if the new account was created on the 1st of the current month
        $newExpenseTransDates = [];
        foreach ($expenses as $e) {
            $newExpenseTransDates[$e->id] = [];
            $firstTransDate = Calculator::prevDueDate($e->date, $e->frequency, date('Y-m-d', strtotime('-1 day', strtotime($monthStartDate))));
            $firstTransTime = strtotime($firstTransDate);

            if ((strtotime($firstTransDate) >= strtotime($monthStartDate)) && (strtotime($firstTransDate) <= strtotime($date))) {
                $newExpenseTransDates[$e->id][] = $firstTransDate;
                $startDay = date('d', strtotime($firstTransDate));
            }
            else {
                $startDay = 1;
            }

            for ($j = $startDay; $j<= $day; $j++) {
                $nextTransDate = Calculator::nextDueDate($firstTransDate, $e->frequency, date('Y-m-' . $j));
                if ((strtotime($nextTransDate) >= strtotime($monthStartDate)) && (strtotime($nextTransDate) <= strtotime($date)) && (!in_array($nextTransDate, $newExpenseTransDates[$e->id]))) {
                    $newExpenseTransDates[$e->id][] = $nextTransDate;
                }
            }
        }

        // get existing expense transactions
        $transactions = Transaction::find()
            ->leftJoin(Expense::tableName(), Transaction::tableName() . '.expense_id = ' . Expense::tableName() . '.id')
            ->leftJoin(Jar::tableName(), Expense::tableName() . '.jar_id = ' . Jar::tableName() . '.id')
            ->where([
                'and',
                ['IN', Expense::tableName() . '.jar_id', $jarIds],
                [Expense::tableName() . '.archived' => 0],
                [Jar::tableName() . '.archived' => 0]
            ])
            ->all();

        // income transactions form
        $itForm = new IncomeTransactionEntry();
        foreach ($incomeTransactions as $i) {
            $itForm->generate[$i->income_id . '_' . $i->date] = 1;
            $itForm->amount[$i->income_id . '_' . $i->date] = $i->amount;
        }

        // expense transactions form
        $tForm = new TransactionEntry();
        foreach ($transactions as $e) {
            $tForm->generate[$e->expense_id . '_' . $e->date] = 1;
            $tForm->amount[$e->expense_id . '_' . $e->date] = $e->amount;
        }

        $this->view->title = 'Step 7 - First Time Setup';
        return $this->render('step-layout', [
            'steps' => $this->steps,
            'currentStepId' => 7,
            'stepProgress' => $stepId,
            'stepProgressFull' => $this->userMeta['fts_step'],
            'stepValues' => [
                'incomes' => $incomes,
                'newIncomeTransDates' => $newIncomeTransDates,
                'expenses' => $expenses,
                'newExpenseTransDates' => $newExpenseTransDates,
                'itForm' => $itForm,
                'tForm' => $tForm
            ],
            'showNext' => true
        ]);*/
    }

    /**
     * Finishing FTS
     */
    public function actionComplete()
    {
        $stepId = strtr($this->userMeta['fts_step'], ['d' => '']);

        if ($this->userMeta['fts_step'] == '5' || $stepId > 5) {
            $userMeta = UserMeta::findOne(['key' => 'fts_step', 'user_id' => $this->mainUserId]);
            $userMeta->value = 'completed';
            $userMeta->save();

            $this->userMeta['fts_step'] = 'completed';
            Yii::$app->session->set('userMeta', $this->userMeta);
            Yii::$app->session->set('fts-completed', true);

            DripHelper::sendCustomEvent('FTS completed', $this->user->email);
        }
        elseif ($stepId < 5) {
            if ($stepId == 0) {
                return $this->redirect(['/first-time-setup/welcome']);
            }
            else {
                return $this->redirect(['/first-time-setup/step-' . $stepId]);
            }
        }
        return $this->redirect(['/']);
    }

    private function _getExpensesVsIncomesBarData(array $jars = [])
    {
        // get total expense
        // new formula for getting the expenses so the amounts are equal in jar details an in the balance bar
        $totalExpensesAmount = 0;
        /** @var Jar $jar */
        foreach ($jars as $jar) {
            $totalExpensesAmount += $jar->getThisMonthsExpensesAmount();
        }

        // get total income amount
        $incomes = Income::findAll(['budget_id' => $this->activeBudgetId, 'archived' => 0]);
        $totalIncomesAmount = 0;
        foreach ($incomes as $a) {
            $totalIncomesAmount += $a->getAverageMonthlyAmount();
        }

        // get percentages
        $totalMoney = $totalIncomesAmount + $totalExpensesAmount;

        if ($totalMoney != 0) {
            $incomesPctg = round($totalIncomesAmount / $totalMoney * 100);
            $expensesPctg = 100 - $incomesPctg;
        }
        else {
            $incomesPctg = 0;
            $expensesPctg = 0;
        }
        return [
            'totalIncomesAmount' => $totalIncomesAmount,
            'totalExpensesAmount' => $totalExpensesAmount,
            'incomesPctg' => $incomesPctg,
            'expensesPctg' => $expensesPctg
        ];
    }

    /**
     * Get a list of accounts to display.
     * @return array List of accounts.
     */
    private function _getAccountsList()
    {
        return Account::find()
            ->where(['budget_id' => $this->activeBudgetId, 'archived' => 0])
            ->orderBy('name ASC')
            ->all();
    }

    /**
     * Get a list of debts to display.
     * @return array List of debts.
     */
    private function _getDebtsList()
    {
        return Debt::find()
            ->where(['budget_id' => $this->activeBudgetId, 'archived' => 0])
            ->orderBy('name ASC')
            ->all();
    }

    /**
     * Get a list of incomes to display.
     * @return array List of incomes.
     */
    private function _getIncomesList()
    {
        return Income::find()
            ->where(['budget_id' => $this->activeBudgetId, 'archived' => 0])
            ->orderBy('name ASC')
            ->all();
    }

    /**
     * Get a list of expenses to display.
     * @return array List of expenses.
     */
    private function _getExpensesList()
    {
        $jars = $this->_getJarsMenuItems();
        $expenses = [];
        foreach ($jars as $j) {
            $expenses += $j->getActiveExpenses();
        }
        return $expenses;
    }

    /**
     * Get all menu items.
     * @return array Menu items.
     */
    private function _getJarsMenuItems()
    {
        return Jar::find()
            ->where(['budget_id' => $this->activeBudgetId, 'archived' => 0])
            ->orderBy('order')
            ->all();
    }

}
