<?php

namespace backend\controllers;

use backend\models\db\Eoms;
use Yii;

use yii\filters\AccessControl;

use yii\helpers\ArrayHelper;
use yii\helpers\Url;

use backend\components\CustomController;

use backend\models\db\Account;
use backend\models\db\Income;
use backend\models\db\IncomeTransaction;

use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;

/**
 * Income payments controller
 */
class IncomeTransactionsController extends CustomController {

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'income-transactions-table',
                            'income-transactions-table-info',
                            'income-transactions-by-year',
                            'income-transactions-by-month',
                            'account-income-transactions-table',
                            'account-income-transactions-table-info'
                        ],
                        'allow' => true,
                        'roles' => ['@']
                    ],
                ],
            ]
        ];

    }


    /**
     * Show income transactions
     */
    public function actionIndex() {

        $incomeId = Yii::$app->getRequest()->getQueryParam('jar_id', false);

        if (!is_numeric($incomeId)) {

            return $this->accessError();

        }

        $income = Income::findOne($incomeId);

        if ($income->budget_id != $this->activeBudgetId) {

            return $this->accessError();

        }

        $transactions = IncomeTransaction::findAll(['income_id' => $income->id]);

        $incomes = Income::findAll(['budget_id' => $this->activeBudgetId, 'archived' => 0]);

        return $this->render('_income-transactions-table.php', [
            'transactions' => $transactions,
            'incomeNames' => ArrayHelper::map($incomes, 'id', 'name'),
        ]);

    }

    /**
     * Show income transactions
     */
    public function actionIncomeTransactionsTable()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        if ($budgetId != $this->activeBudgetId) {
            $this->mockBudgetId = $budgetId;
        }
        $incomeId = Yii::$app->getRequest()->getQueryParam('income_id', false);
        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'date',
                'select' => IncomeTransaction::tableName() . '.date'
            ],
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => Income::tableName() . '.name'
            ],
            [
                'type' => 'normal',
                'as' => 'description',
                'select' => IncomeTransaction::tableName() . '.description'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => IncomeTransaction::tableName() . '.amount'
            ],
            [
                'type' => 'normal',
                'as' => 'account_name',
                'select' => Account::tableName() . '.name'
            ]
        ];
        if (!$incomeId) {
            // get all incomes to show the transactions for deleted incomes as well
            $incomes = Income::findAll(['budget_id' => $this->activeBudgetId]);
            $incomeId = ArrayHelper::map($incomes, 'id', 'id');
        }

        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));

        $countFiltered = IncomeTransaction::getCountFilteredQuery($incomeId, $this->activeBudgetId, $month, $year);
        $selectFiltered = IncomeTransaction::getSelectFilteredQuery($incomeId, $this->activeBudgetId, $month, $year);
        $responseData = DataTables::getResponseData($tableColumns, $countFiltered, $selectFiltered);

        /** @var IncomeTransaction $i */
        foreach ($responseData['items'] as $i) {
            $actionClass = get_class($i);
            $isEditable = Eoms::isTransactionChangeable($this->activeBudgetId, $i->date);

            $row = [
                'isEditable' => $isEditable
            ];
            if ($isEditable) {
                $row['id'] = $i->id;
                $row['type'] = $actionClass;
                $row['editUrl'] = '/money-operations/edit-transaction';
                $row['deleteUrl'] = Url::toRoute(['/money-operations/delete-transaction', 'transaction_id' => $i->id, 'transaction_type' => $actionClass, 'js_callback' => 'reloadAccountsAfterIncomeTransactionDelete']);
                $row['callback'] = 'reloadIncomesAfterIncomeTransactionDelete';
            }

            $columns = [
                [
                    'key' => 'date',
                    'type' => 'date',
                    'value' => $i->date,
                    'isEditable' => true
                ], [
                    'key' => 'name',
                    'type' => 'text',
                    'value' => $i->income->name,
                    'isEditable' => true
                ], [
                    'key' => 'description',
                    'type' => 'text',
                    'value' => $i->description,
                    'isEditable' => true
                ], [
                    'key' => 'amount',
                    'type' => 'currency',
                    'value' => $i->amount,
                    'isEditable' => true
                ], [
                    'key' => 'account',
                    'type' => 'text',
                    'value' => $i->account_name
                ]
            ];
            $responseData['response']['data'][] = DataTables::getRow($row, $columns);
        }
        echo json_encode($responseData['response']);
        die();
    }

    /**
     * Info for transactions table footer
     */
    public function actionIncomeTransactionsTableInfo()
    {
        $incomeId = Yii::$app->getRequest()->getQueryParam('income_id', false);
        if (!is_numeric($incomeId)) {
            return $this->accessError();
        }

        $income = Income::findOne($incomeId);
        $budget = $income->getBudget()->one();
        if ($budget->is_mock && $budget->user_id != $this->mainUserId || !$budget->is_mock && $income->budget_id != $this->activeBudgetId) {
            return $this->accessError();
        }

        echo json_encode([
            'this-months-income-transactions-amount' => Formatter::currency($income->getThisMonthsIncomeTransactionsAmount())
        ]);
        die();
    }


    /**
     * Show income transactions
     */
    public function actionAccountIncomeTransactionsTable()
    {
        $accountId = Yii::$app->getRequest()->getQueryParam('account_id', false);
        if (!is_numeric($accountId)) {
            return $this->accessError();
        }
        $account = Account::findOne($accountId);
        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $account->budget_id])) {
            return $this->accessError();
        }

        // filtering and getting record
        $tableColumns = [
            /*[
                'type' => 'normal',
                'as' => 'checked',
                'select' => IncomeTransaction::tableName() . '.checked'
            ],*/
            [
                'type' => 'normal',
                'as' => 'date',
                'select' => IncomeTransaction::tableName() . '.date'
            ],
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => Income::tableName() . '.name'
            ],
            [
                'type' => 'normal',
                'as' => 'description',
                'select' => IncomeTransaction::tableName() . '.description'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => IncomeTransaction::tableName() . '.amount'
            ]
        ];
        $countFiltered = IncomeTransaction::getFilteredIncomeTransactionsCount($account->id);
        $selectFiltered = IncomeTransaction::getFilteredIncomeTransactionsSelect($account->id);
        $responseData = DataTables::getResponseData($tableColumns, $countFiltered, $selectFiltered);

        /** @var IncomeTransaction $i */
        foreach ($responseData['items'] as $i) {
            $actionClass = get_class($i);
            $isEditable = Eoms::isTransactionChangeable($this->activeBudgetId, $i->date);

            $row = [
                'isEditable' => $isEditable
            ];
            if ($isEditable) {
                $row['id'] = $i->id;
                $row['type'] = $actionClass;
                $row['editUrl'] = '/money-operations/edit-transaction';
                $row['deleteUrl'] = Url::toRoute(['/money-operations/delete-transaction', 'transaction_id' => $i->id, 'transaction_type' => $actionClass, 'js_callback' => 'reloadAccountsAfterIncomeTransactionDelete']);
                $row['callback'] = 'reloadAccountsAfterIncomeTransactionDelete';
            }
            $bankStatType = "it";
            $bankStatName = $bankStatType . $i->id;

            $columns = [
                /*[
                    'key' => 'checked',
                    'type' => 'checkbox',
                    'value' => $i->checked,
                    'cId' => $i->id,
                    'cType' => $bankStatType,
                    'cName' => $bankStatName,
                    'isEditable' => false
                ],*/ [
                    'key' => 'date',
                    'type' => 'date',
                    'value' => $i->date,
                    'isEditable' => true
                ], [
                    'key' => 'name',
                    'type' => 'text',
                    'value' => $i->name,
                    'isEditable' => true
                ], [
                    'key' => 'description',
                    'type' => 'text',
                    'value' => $i->description,
                    'isEditable' => true
                ], [
                    'key' => 'amount',
                    'type' => 'currency',
                    'value' => $i->amount,
                    'isEditable' => true
                ]
            ];
            $responseData['response']['data'][] = DataTables::getRow($row, $columns);
        }
        echo json_encode($responseData['response']);
        die();
    }


    /**
     * Info for transactions table footer
     */
    public function actionAccountIncomeTransactionsTableInfo() {

        $accountId = Yii::$app->getRequest()->getQueryParam('account_id', false);

        if (!is_numeric($accountId)) {

            return $this->accessError();

        }

        $account = Account::findOne($accountId);

        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $account->budget_id])) {

            return $this->accessError();

        }


        echo json_encode([
            'this-months-income-transactions-amount' => Formatter::currency($account->getMonthsIncomeTransactionsAmount(date('m'), date('Y')))
        ]);
        die();

    }



    /**
     * Graph data for transactions by year
     */
    public function actionIncomeTransactionsByYear()
    {
        $year = Yii::$app->getRequest()->getQueryParam('year', false);
        if (!$year) {
            return $this->accessError();
        }
        $budgetId = $this->getBudgetIdFromUrl();
        if ($budgetId != $this->activeBudgetId) {
            $this->mockBudgetId = $budgetId;
        }

        $initTime = $time = strtotime("$year-1-1");
        $incomes = Income::findAll([
            'budget_id' => $this->activeBudgetId,
            'is_adjustment' => 0
        ]);
        $incomeIds = ArrayHelper::map($incomes, 'id', 'id');

        $transactions = IncomeTransaction::find()
            ->select(IncomeTransaction::tableName() . '.*, ' . Income::tableName() . '.name as income_name')
            ->where([
                'and',
                ['IN', 'income_id', $incomeIds],
                ['Year(' . IncomeTransaction::tableName() . '.date)' => $year],
                ['is_adjustment' => 0]
            ])
            ->joinWith('income')
            ->all();

        // need to add now the future recurring incomes
        $today = date('Y-m-d');
        foreach ($incomes as $income) {
            // for future transactions we consider only active incomes
            if (!$income->archived) {
                for ($month = 1; $month <= 12; $month++) {
                    $transactions = array_merge($transactions, $income->getProjectedTransactions($today, $month, $year));
                }
            }
        }

        $firstTransaction = IncomeTransaction::find()
            ->where(['IN', 'income_id', $incomeIds])
            ->orderBy('date ASC')
            ->one();
        if ($firstTransaction) {
            $firstTrnsYearTime = strtotime(date('Y-1-1', strtotime($firstTransaction->date)));
        }

        if (count($transactions) > 0) {
            $transactionsOrdered = [];
            foreach ($transactions as $t) {
                if (!isset($transactionsOrdered[$t->income_name])) {
                    for ($i = 1; $i <= 12; $i++) {
                        $transactionsOrdered[$t->income_name][$i] = 0;
                    }
                }
                $transactionsOrdered[$t->income_name][date('n',strtotime($t->date))] += $t->amount;
            }

            $series = [];
            foreach ($transactionsOrdered as $name => $t) {
                $series[] = [
                    'name' => $name,
                    'data' => array_values($t)
                ];
            }
            if (!count($series)) {
                $series[] = [];
            }
            $response = [
                'title' => $year,
                'year' => $year,
                'series' => $series,
                'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']
            ];
        }
        else {
            $response = false;
        }
        echo json_encode($response);
        die();
    }


    /**
     * Graph data for transactions by month
     */
    public function actionIncomeTransactionsByMonth()
    {
        $year = Yii::$app->getRequest()->getQueryParam('year', false);
        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        if (!$year || !$month) {
            return $this->accessError();
        }
        $budgetId = $this->getBudgetIdFromUrl();
        if ($budgetId != $this->activeBudgetId) {
            $this->mockBudgetId = $budgetId;
        }

        // labels
        $initTime = $time = strtotime("$year-$month-1");
        $labels = [];
        while (date('n', $time) == $month) {
            $day = date('d', $time);
            $labels[] = $day;
            $time += 86400;
        }

        $incomes = Income::findAll([
            'budget_id' => $this->activeBudgetId,
            'is_adjustment' => 0
        ]);
        $incomeIds = ArrayHelper::map($incomes, 'id', 'id');
        // income transactions
        $transactions = IncomeTransaction::find()
            ->select(IncomeTransaction::tableName() . '.*, ' . Income::tableName() . '.name as income_name')
            ->joinWith('income')
            ->where([
                'and',
                ['IN', 'income_id', $incomeIds],
                ['Year(' . IncomeTransaction::tableName() . '.date)' => $year],
                ['Month(' . IncomeTransaction::tableName() . '.date)' => $month],
                ['is_adjustment' => 0]
            ])
            ->all();

        // need to add now the future recurring incomes
        $today = date('Y-m-d');
        foreach ($incomes as $income) {
            // for future transactions we consider only active incomes
            if (!$income->archived) {
                $transactions = array_merge($transactions, $income->getProjectedTransactions($today, $month, $year));
            }
        }

        $series = [];
        if (count($transactions) > 0) {
            $transactionsOrdered = [];
            foreach ($transactions as $t) {
                if (!isset($transactionsOrdered[$t->income_name])) {
                    for ($i = 1; $i <= count($labels); $i++) {
                        $transactionsOrdered[$t->income_name][$i] = 0;
                    }
                }
                $transactionsOrdered[$t->income_name][date('j',strtotime($t->date))] += $t->amount;
            }
            foreach ($transactionsOrdered as $name => $t) {
                $series[] = [
                    'name' => $name,
                    'data' => array_values($t)
                ];
            }
        }
        $response = [
            'title' => date('F Y', $initTime),
            'year' => $year,
            'month' => $month,
            'series' => $series,
            'labels' => $labels
        ];

        echo json_encode($response);
        die();
    }

}
