<?php

namespace backend\controllers;

use backend\components\helpers\Calculator;
use backend\components\helpers\DripHelper;
use backend\models\db\Eoms;
use backend\models\db\Jar;
use Yii;

use yii\caching\TagDependency;
use yii\filters\AccessControl;

use yii\helpers\ArrayHelper;
use yii\helpers\Url;

use backend\components\CustomController;

use backend\models\db\Account;
use backend\models\db\Budget;
use backend\models\db\Income;
use backend\models\db\IncomeChange;
use backend\models\db\IncomeEnd;
use backend\models\db\IncomeTransaction;

use backend\models\form\AddIncome;

use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;
use backend\components\helpers\Frequencies;
use backend\components\helpers\JsonTools;


/**
 * Incomes controller
 */
class IncomesController extends CustomController
{
    public $enableCsrfValidation = false;

    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'income-detail',
                            'incomes-table',
                            'incomes-with-this-months-archived-table',
                            'incomes-no-archived-table',
                            'incomes-table-info',
                            'income-changes-table',
                            'add-income',
                            'add-onetime-income',
                            'edit-onetime-income',
                            'edit-income',
                            'pause-income',
                            'continue-income',
                            'remove-income'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];
    }

    public function beforeAction($action)
    {
        // for mock budgets navigation
        if ($action->id == 'income-detail') {
            $income = Income::findOne(['id' => Yii::$app->request->getQueryParam('id')]);
            if ($income && $income->budget_id != $this->activeBudgetId) {
                $this->mockBudgetId = $income->budget_id;
                $this->view->params['mockBudgetId'] = $income->budget_id;
            }
        }
        elseif ($budgetId = Yii::$app->request->getQueryParam('budget_id')) {
            $budget = Budget::findOne($budgetId);
            if ($budget->is_mock) {
                $this->mockBudgetId = $budget->id;
                $this->view->params['mockBudgetId'] = $budget->id;
            }
        }
        return parent::beforeAction($action);
    }

    /**
     * Show incomes
     */
    public function actionIndex()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        if ($budgetId == $this->mockBudgetId) {
            $this->mockBudgetId = $budgetId;
            $budget = Budget::findOne($budgetId);
        }
        $currentBudget = Budget::findOne($this->activeBudgetId);

        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));

        $incomes = Income::getActiveIncomesQuery($budgetId)->all();
        $totalAverageIncome = 0;
        foreach ($incomes as $i) {
            $totalAverageIncome += $i->getAverageMonthlyAmount();
        }
        $allTransactions = IncomeTransaction::getSelectFilteredQuery(null, $this->activeBudgetId, $month, $year)->all();
        $incomeTransactionsAmount = 0;
        foreach ($allTransactions as $t) {
            $incomeTransactionsAmount += $t->amount;
        }

        // pass mock budget specific data
        $mockBudgetData = [];
        if (isset($budget)) {
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['budgetId'] = $budget->id;
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['isMockBudget'] = true;
            $firstIncomeDate = $budget->getFirstIncomeDate();
            $lastIncomeDate = $budget->getLastIncomeDate();
        }
        else {
            $firstIncomeDate = $currentBudget->getFirstIncomeDate();
            $lastIncomeDate = $currentBudget->getLastIncomeDate();
        }

        $yearsList = Calculator::getYearsListFromTo(
            ['/incomes/index', 'budget_id' => isset($this->mockBudgetId) ? $this->mockBudgetId : false],
            date('Y', strtotime($firstIncomeDate)),
            date('Y', strtotime($lastIncomeDate)),
            $year
        );
        $monthsList = Calculator::getMonthsListFromTo(
            ['/incomes/index', 'budget_id' => isset($this->mockBudgetId) ? $this->mockBudgetId : false],
            date('m', strtotime($firstIncomeDate)),
            date('Y', strtotime($firstIncomeDate)),
            date('m', strtotime($lastIncomeDate)),
            date('Y', strtotime($lastIncomeDate)),
            $month,
            $year
        );

        $this->view->title = 'My Incomes';
        return $this->render('index', $mockBudgetData + [
            'incomes' => $incomes,
            'incomeTransactionsAmount' => $incomeTransactionsAmount,
            'totalAverageIncome' => $totalAverageIncome,
            'allowEdit' => true,
            'fullLoad' => true,
            'months' => $monthsList,
            'years'  => $yearsList,
            'year'   => $year,
            'month'  => $month,
            'monthName' => date('F', strtotime('2010-' . $month . '-1'))
        ]);
    }


    /**
     * Add a new income
     */
    public function actionAddIncome()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        $this->mockBudgetId = $budgetId;

        $incomeForm = new AddIncome();
        if (Yii::$app->request->isPost && $incomeForm->load(Yii::$app->request->post())) {
            $incomeForm->budget_id = $budgetId;
            // set validation attributes
            if ($incomeForm->validate()) {
                $formData = $incomeForm->getAttributes();
                $this->_saveNewIncome($formData);
                if ($incomeForm->jar_id) {
                    TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $incomeForm->jar_id);
                }

                DripHelper::sendCustomEvent('Income created', $this->user->email, [
                    'name' => $formData['name']
                ]);

                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Income successfully created.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($incomeForm);
                die();
            }
        }
        else {
            $incomeForm['date'] = Formatter::date(time());
            $this->jsCallbackToView();
            $budget = Budget::findOne($budgetId ? $budgetId : $this->activeBudgetId);

            $this->view->title = 'Add Income';
            return $this->render('add-income', [
                'budgetId' => $budgetId,
                'formModel' => $incomeForm,
                'frequencies' => Frequencies::getAll(),
                'accounts' => Account::getAccountsListForSelect($budgetId, true), // no credit card accounts here
                'selectedAccount' => $this->defaultAccountId,
                'jars' => ArrayHelper::map($budget->getActiveJars(), 'id', 'name'),
            ]);
        }
    }


    /**
     * Add a new one-time income
     */
    public function actionAddOnetimeIncome()
    {
        $incomeForm = new AddIncome();
        $incomeForm->budget_id = $this->activeBudgetId;
        $budget = Budget::findOne($incomeForm->budget_id);

        if (Yii::$app->request->isPost && $incomeForm->load(Yii::$app->request->post())) {
            $incomeForm->frequency = 'one-time';
            $incomeForm->name = Income::OTHER_INCOME_NAME;
            $incomeForm->is_unbudgeted = 1;
            // set validation attributes
            if ($incomeForm->validate()) {
                $formData = $incomeForm->getAttributes();
                $income = $this->_saveNewIncome($formData, true);
                DripHelper::sendCustomEvent('Transaction added', $this->user->email, [
                    'type' => 'income',
                    'description' => $formData['description']
                ]);

                if (strtotime($formData['date']) > strtotime(date('Y-m-d'))) {
                    $link = '<a href="' . Url::toRoute(['incomes/income-detail', 'id' => $income->id]) . '" class="system-link">here</a>';
                    echo JsonTools::successMessage("You income has been successfully created. Click $link to view this income's detail page.");
                }
                else {
                    echo JsonTools::successMessage('Income successfully created.');
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($incomeForm);
                die();
            }
        }
        else {
            $incomeForm['date'] = Formatter::date(time());
            $this->view->title = 'Add One-off Income';
            return $this->render('add-onetime-income', [
                'formModel' => $incomeForm,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'selectedAccount' => $this->defaultAccountId,
                'jars' => Jar::getJarsForSelect($this->activeBudgetId),
            ]);
        }
    }

    public function actionEditOnetimeIncome()
    {
        $transactionId = Yii::$app->getRequest()->getQueryParam('id', false);
        $jsCallback = Yii::$app->getRequest()->getQueryParam('js_callback', false);
        if (!is_numeric($transactionId)) {
            return $this->accessError();
        }
        $transaction = IncomeTransaction::findOne($transactionId);
        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $transaction->income->budget_id])) {
            return $this->accessError();
        }
        $incomeForm = new AddIncome(['scenario' => 'update']);
        $incomeForm->budget_id = $this->activeBudgetId;
        $budget = Budget::findOne($incomeForm->budget_id);

        $jar = Jar::findOne($transaction->income->jar_id);
        if ($jar) {
            $this->mockBudgetId = $jar->getMockBudgetId();
            $incomeForm->jar_id = $jar->id;
        }

        if (Yii::$app->request->isPost && $incomeForm->load(Yii::$app->request->post())) {
            $income = $transaction->income;
            $incomeForm->income_id = $income->id;
            $oldJarId = $income->jar_id;

            // set validation attributes
            if ($incomeForm->validate()) {
                $transaction->attributes = $incomeForm->attributes;
                $income->jar_id = $incomeForm->jar_id;
                if ($transaction->save() && $income->save()) {
                    TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $income->jar_id);
                    if ($income->jar_id != $oldJarId) {
                        TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $oldJarId);
                    }
                    echo JsonTools::successMessage('Income successfully updated.');
                }
                else {
                    echo JsonTools::errorMessage('Error updating income.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($incomeForm);
                die();
            }
        }
        else {
            $incomeForm->attributes = $transaction->attributes;

            $this->view->title = 'Edit One-off Income';
            return $this->render('add-onetime-income', [
                'id' => $transactionId,
                'jsCallback' => $jsCallback,
                'formModel' => $incomeForm,
                'accounts' => Account::getAccountsListForSelect($this->activeBudgetId),
                'selectedAccount' => $incomeForm->account_id ? $incomeForm->account_id : $this->defaultAccountId,
                'jars' => ArrayHelper::map($budget->getActiveJars(), 'id', 'name'),
            ]);
        }
    }


    /**
     * Edit a income
     */
    public function actionEditIncome()
    {
        $incomeId = Yii::$app->getRequest()->getQueryParam('id', false);
        if (!is_numeric($incomeId)) {
            return $this->accessError();
        }
        $income = Income::findOne($incomeId);
        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $income->budget_id])) {
            return $this->accessError();
        }

        $incomeForm = new AddIncome();
        if (Yii::$app->request->isPost && $incomeForm->load(Yii::$app->request->post())) {
            $incomeForm->budget_id = $income->budget_id;
            $incomeForm->income_id = $income->id;
            $oldJarId = $income->jar_id;

            // set validation attributes
            if ($incomeForm->validate()) {
                $income->setAttributes($incomeForm->getAttributes());
                $income->save();
                TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $income->jar_id);
                if ($income->jar_id != $oldJarId) {
                    TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $oldJarId);
                }

                if (Yii::$app->request->isAjax) {
                    echo JsonTools::successMessage('Income successfully saved.');
                    die();
                }
            }
            elseif (Yii::$app->request->isAjax) {
                echo JsonTools::formErrorMessage($incomeForm);
                die();
            }
        }
        else {
            $incomeForm->setAttributes($income->getAttributes());
            $this->view->title = 'Edit Income';
            $this->jsCallbackToView();

            return $this->render('edit-income', [
                'formModel' => $incomeForm,
                'frequencies' => Frequencies::getAll(),
                'accounts' => ArrayHelper::map(Account::findAll(['budget_id' => $income->budget_id, 'archived' => '0']), 'id', 'name'),
                'jars' => ArrayHelper::map($income->budget->getActiveJars(), 'id', 'name'),
                'incomeId' => $income->id
            ]);
        }
    }

    /**
     * Detail of income
     */
    public function actionIncomeDetail()
    {
        $incomeId = Yii::$app->getRequest()->getQueryParam('id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content
        if (!is_numeric($incomeId)) {
            return $this->accessError();
        }
        $income = Income::findOne($incomeId);
        if ($income->archived) {
            return $this->redirect(['/incomes/index']);
        }
        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $income->budget_id])) {
            return $this->accessError();
        }

        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));

        $transactions = IncomeTransaction::findAll(['income_id' => $income->id]);

        // pass mock budget specific data
        $mockBudgetData = [];
        if ($income->budget->is_mock) {
            $budget = $income->getBudget()->one();

            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['budgetId'] = $budget->id;
            $mockBudgetData['budgetName'] = $budget->name;
            $mockBudgetData['isMockBudget'] = true;

            $firstIncomeDate = $budget->getFirstIncomeDate();
            $lastIncomeDate = $budget->getLastIncomeDate();
        }
        else {
            $currentBudget = Budget::findOne($this->activeBudgetId);
            $firstIncomeDate = $currentBudget->getFirstIncomeDate();
            $lastIncomeDate = $currentBudget->getLastIncomeDate();
        }

        $budgetId = isset($budget) ? $budget->id : $this->activeBudgetId;
        $incomes = Income::getActiveIncomesQuery($budgetId)->all();

        $this->view->title = $income->name . ' income';
        if ($this->pageLoadRequest || $fullLoad) {
            $totalAverageIncome = 0;
            foreach ($incomes as $i) {
                $totalAverageIncome += $i->getAverageMonthlyAmount();
            }
            $allTransactions = IncomeTransaction::getSelectFilteredQuery(null, $this->activeBudgetId)->all();
            $incomeTransactionsAmount = 0;
            foreach ($allTransactions as $t) {
                $incomeTransactionsAmount += $t->amount;
            }

            $yearsList = Calculator::getYearsListFromTo(
                ['/incomes/index', 'budget_id' => isset($this->mockBudgetId) ? $this->mockBudgetId : false],
                date('Y', strtotime($firstIncomeDate)),
                date('Y', strtotime($lastIncomeDate)),
                $year
            );
            $monthsList = Calculator::getMonthsListFromTo(
                ['/incomes/index', 'budget_id' => isset($this->mockBudgetId) ? $this->mockBudgetId : false],
                date('m', strtotime($firstIncomeDate)),
                date('Y', strtotime($firstIncomeDate)),
                date('m', strtotime($lastIncomeDate)),
                date('Y', strtotime($lastIncomeDate)),
                $month,
                $year
            );

            return $this->render('income-detail', $mockBudgetData + [
                'income' => $income,
                'incomes' => $incomes,
                'transactions' => $transactions,
                'incomeTransactionsAmount' => $incomeTransactionsAmount,
                'totalAverageIncome' => $totalAverageIncome,
                'fullLoad' => $fullLoad,
                'allowEdit' => true,
                'months' => $monthsList,
                'years'  => $yearsList,
                'year'   => $year,
                'month'  => $month,
                'monthName' => date('F', strtotime('2010-' . $month . '-1'))
            ]);
        }
        else {
            return $this->render('income-detail-content', $mockBudgetData + [
                'income' => $income,
                'incomes' => $incomes,
                'transactions' => $transactions,
                'fullLoad' => $fullLoad,
                'allowEdit' => true
            ]);
        }
    }

    /**
     * Table of all incomes
     */
    public function actionIncomesTable()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        if ($budgetId != $this->activeBudgetId) {
            $this->mockBudgetId = $budgetId;
            $budget = Budget::findOne($budgetId);
        }

        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => 'name'
            ],
            [
                'type' => 'normal',
                'as' => 'frequency',
                'select' => 'frequency'
            ],
            [
                'type' => 'method',
                'as' => 'nextPayDay',
                'name' => 'getNextDue'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => 'amount'
            ],
            [
                'type' => 'method',
                'as' => 'monthlyAverage',
                'name' => 'getMinimumMonthlyAmount'
            ],
            [
                'type' => 'method',
                'as' => 'realMonthlyAverage',
                'name' => 'getAverageMonthlyAmount'
            ]
        ];

        $filtered = Income::getActiveIncomesQuery($budgetId);
        $responseData = DataTables::getResponseData($tableColumns, $filtered, $filtered);

        /** @var Income $i */
        foreach ($responseData['items'] as $i) {
            // pause or continue
            $incomeEnd = $i->getIncomeEnd()->one();

            $pauseOrCont = ($incomeEnd) ?
                '<a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['incomes/continue-income', 'id' => $i->id]) . '"><i class="glyphicon glyphicon-play"></i>Resume</a>' :
                '<a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['incomes/pause-income', 'id' => $i->id]) . '"><i class="glyphicon glyphicon-pause"></i>Pause</a>';

            $responseData['response']['data'][] = [
                $i->name . ($incomeEnd ? '<span class="row-bg-red"></span>' : ''),
                Formatter::currency($i->amount),
                Frequencies::getLabel($i->frequency),
                '<span class="hidden">' . $i->getNextDue() . '</span>' . Formatter::date($i->getNextDue()),
                Formatter::currency($i->getMinimumMonthlyAmount()),
                Formatter::currency($i->getAverageMonthlyAmount()),
                $i->is_adjustment || $i->frequency == 'one-time' ? '' :
                '<div class="text-center">
                    <a href="#" id="menu1" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="glyphicon glyphicon-cog gi-2x"></i></a>
                    <ul class="dropdown-menu" aria-labelledby="menu1">
                        <li>
                            ' . $pauseOrCont . '
                        </li>
                    </ul>
                </div>'
            ];
        }

        echo json_encode($responseData['response']);
        die();
    }

    /**
     * Table of all incomes, including those archived this month
     */
    public function actionIncomesWithThisMonthsArchivedTable()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        if ($budgetId != $this->activeBudgetId) {
            $this->mockBudgetId = $budgetId;
            $budget = Budget::findOne($budgetId);
        }
        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => 'name'
            ],
            [
                'type' => 'method',
                'as' => 'realMonthlyAverage',
                'name' => 'getAverageMonthlyAmount'
            ],
            [
                'type' => 'method',
                'as' => 'monthlyAverage',
                'name' => 'getMinimumMonthlyAmount'
            ]

        ];

        $filtered = Income::find()
            ->select([
                Income::tableName() . '.*',

            ])
            ->leftJoin(IncomeChange::tableName(), Income::tableName() . '.id = ' . IncomeChange::tableName() . '.income_id AND ' . IncomeChange::tableName() . '.type = "D"')
            ->where([
                'and',
                ['budget_id' => $budgetId],
                [
                    'or',
                    [
                        'and',
                        ['archived' => '0'],
                        ['!=', 'frequency', 'one-time']
                    ],
                    [
                        'and',
                        ['archived' => 1],
                        ['>=', IncomeChange::tableName() . '.time', date('Y-m-1')]
                    ]
                ]
            ]);

        $responseData = DataTables::getResponseData($tableColumns, $filtered, $filtered);
        foreach ($responseData['items'] as $i) {
            /** @var Income $i */
            $responseData['response']['data'][] = [
                $i->name . (($i->archived || $i->getIncomeEnd()->one())? '<span class="row-bg-red"></span>' : ''),
                Formatter::currency($i->getAverageMonthlyAmount()),
                Formatter::currency($i->getMinimumMonthlyAmount())
            ];
        }

        echo json_encode($responseData['response']);
        die();
    }

    /**
     * Table of all incomes, including those archived this month
     */
    public function actionIncomesNoArchivedTable()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        if ($budgetId != $this->activeBudgetId) {
            $this->mockBudgetId = $budgetId;
            $budget = Budget::findOne($budgetId);
        }
        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => 'name'
            ],
            [
                'type' => 'method',
                'as' => 'realMonthlyAverage',
                'name' => 'getAverageMonthlyAmount'
            ],
            [
                'type' => 'method',
                'as' => 'monthlyAverage',
                'name' => 'getMinimumMonthlyAmount'
            ]

        ];

        $filtered = Income::find()
            ->where([
                'and',
                ['budget_id' => $budgetId],
                [
                    'and',
                    ['archived' => '0'],
                    ['!=', 'frequency', 'one-time']
                ]
            ]);

        $responseData = DataTables::getResponseData($tableColumns, $filtered, $filtered);
        foreach ($responseData['items'] as $i) {
            /** @var Income $i */
            $responseData['response']['data'][] = [
                $i->name . ($i->getIncomeEnd()->one() ? '<span class="row-bg-red"></span>' : ''),
                Formatter::currency($i->getAverageMonthlyAmount()),
                Formatter::currency($i->getMinimumMonthlyAmount())
            ];
        }

        echo json_encode($responseData['response']);
        die();
    }


    /**
     * Table info for incomes table
     */
    public function actionIncomesTableInfo()
    {
        $budgetId = $this->getBudgetIdFromUrl();
        $incomes = Income::getActiveIncomesQuery($budgetId)->all();
        $totalAverageIncome = 0;
        foreach ($incomes as $i) {
            $totalAverageIncome += $i->getAverageMonthlyAmount();
        }
        echo json_encode([
            'total-average-monthly-income' => Formatter::currency($totalAverageIncome)
        ]);
        die();
    }


    /**
     * Table of the changes made to all of the incomes
     */
    public function actionIncomeChangesTable()
    {
        $budget = Budget::findOne($this->activeBudgetId);
        $changesForTable = $budget->getIncomeChangesForTable();
        $columnNames = [
            'time',
            'name',
            'parameter',
            'from',
            'to'
        ];
        $responseData = DataTables::getFilteredData($columnNames, $changesForTable);

        foreach ($responseData['items'] as $i) {
            $responseData['response']['data'][] = [
                Formatter::datetime($i['time']),
                $i['name'],
                $i['parameter'],
                $i['from'],
                $i['to']
            ];
        }

        echo json_encode($responseData['response']);
        die();
    }


    /**
     * Pause a income
     */
    public function actionPauseIncome() {

        $incomeId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        // get the income
        $income = Income::findOne(['id' => $incomeId]);

        if (!is_numeric($incomeId) || !Yii::$app->user->can('accessBudget', ['budgetId' => $income->budget_id])) {

            return $this->accessError();

        }

        if ($confirm) {

            $incomeEnd = new IncomeEnd();
            $incomeEnd->income_id = $income->id;
            $incomeEnd->date = date('Y-m-d');
            $incomeEnd->save();

            echo JsonTools::successMessage('Income successfully paused.');
            die();

        }

        $this->view->title = 'Pause Income';

        return $this->render('pause-income', [
            'incomeId' => $income->id
        ]);

    }


    /**
     * Continue a income
     */
    public function actionContinueIncome() {

        $incomeId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        // get the income
        $income = Income::findOne(['id' => $incomeId]);

        if (!is_numeric($incomeId) || !Yii::$app->user->can('accessBudget', ['budgetId' => $income->budget_id])) {

            return $this->accessError();

        }

        if ($confirm) {

            $incomeEnd = IncomeEnd::findOne(['income_id' => $income->id]);
            $incomeEnd->delete();

            echo JsonTools::successMessage('Income successfully resumed.');
            die();

        }

        $this->view->title = 'Continue Income';

        return $this->render('continue-income', [
            'incomeId' => $income->id
        ]);

    }

    /**
     * Remove an income
     */
    public function actionRemoveIncome()
    {
        $incomeId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);
        $hardDelete = Yii::$app->getRequest()->getQueryParam('hard', false);
        if (!is_numeric($incomeId)) {
            return $this->accessError();
        }
        $income = Income::findOne($incomeId);
        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $income->budget_id])) {
            return $this->accessError();
        }
        $this->mockBudgetId = $income->getMockBudgetId();
        $hardDeletable = Eoms::isEntryHardDeletable($this->activeBudgetId, $income);

        if ($confirm) {
            $jarId = $income->jar_id;
            if (Yii::$app->user->hasFinishedFts()) {
                if ($hardDelete && $hardDeletable) {
                    $income->deleteCompletely();
                }
                else {
                    $income->archived = 1;
                    $income->save();
                    // remove income transactions for one-time income
                    if ($income->frequency == 'one-time') {
                        IncomeTransaction::deleteAll(['income_id' => $income->id]);
                    }
                }
            }
            else {
                // just delete the income when in FTS
                $income->deleteCompletely();
            }
            if ($jarId) {
                TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $jarId);
            }
            echo JsonTools::successMessage('Income successfully removed.');
            die();
        }

        $this->view->title = 'Remove Income';
        $this->jsCallbackToView();
        return $this->render('remove-income', [
            'incomeId' => $income->id,
            'frequency' => $income->frequency,
            'hardDeletable' => $hardDeletable
        ]);
    }

    private function _saveNewIncome($formData, $withDescription = false)
    {
        $income = new Income();
        $income->setAttributes($formData);
        // FIXME: workaround to not create the transaction record automatically so a description can be added
        if ($withDescription && empty($formData['is_adjustment'])) {
            $income->is_adjustment = 1;
        }
        $saved = $income->save();
        if ($saved && $withDescription && empty($formData['is_adjustment'])) {
            $income->is_adjustment = 0;
            $income->save();
            $incomeTransaction = new IncomeTransaction();
            $incomeTransaction->setAttributes([
                'income_id' => $income->id,
                'account_id' => $income->account_id,
                'amount' => $income->amount,
                'date' => $income->date,
                'description' => $formData['description']
            ]);
            $saved = $incomeTransaction->save();
        }
        return ($saved) ? $income : false;
    }
}
