<?php

namespace backend\controllers;

use backend\components\helpers\DripHelper;
use Yii;

use yii\filters\AccessControl;

use yii\helpers\ArrayHelper;
use yii\helpers\Url;

use backend\components\CustomController;

use backend\models\db\Budget;
use backend\models\db\Income;
use backend\models\db\IncomeTransaction;
use backend\models\db\User;
use backend\models\db\UserMeta;

use backend\models\form\AddBudget;

use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;
use backend\components\helpers\Frequencies;
use backend\components\helpers\JsonTools;


/**
 * Mock Budgets controller
 */
class MockBudgetsController extends CustomController {

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'add-budget',
                            'edit-budget',
                            'budget-overview',
                            'budget-detail',
                            'switch-budget',
                            'submit-budget',
                            'remove-budget',
                            'monthly-income-graph-data',
                            'jar-allocations-graph-data',
                            'compare-graph-data',
                            'incomes-table',
                        ],
                        'allow'   => true,
                        'roles'   => ['@'],
                    ],
                ],
            ]
        ];

    }

    /**
     * Main Mock budgets
     */
    public function actionIndex() {

        $budgets = $this->_getBudgetsMenuList();

        $this->view->title = 'Mock Budgets';

        return $this->render('index', [
            'budgets' => $budgets
        ]);

    }


    public function actionAddBudget() {

        $budgetForm = new AddBudget();

        if ($budgetForm->load(Yii::$app->request->post())) {

            $budgetForm->user_id = $this->mainUserId;

            if ($budgetForm->validate()) {

                $budget = Budget::findOne($budgetForm->cloned_budget_id);
                $result = $budget->cloneBudget($budgetForm->name, true);

                if (Yii::$app->request->isAjax) {

                    if ($result) {

                        echo JsonTools::successMessage('Budget successfully added.');
                        die();

                    }
                    else {

                        echo JsonTools::errorMessage('Error - cloning failed.');
                        die();

                    }

                }

            }
            elseif (Yii::$app->request->isAjax) {

                echo JsonTools::formErrorMessage($budgetForm);
                die();

            }

        }
        else {

            $budgets = $this->_getBudgetsMenuList();
            $this->view->title = 'Mock Budgets';

            return $this->render('add-budget', [
                'budgets' => ArrayHelper::map($budgets, 'id', 'name'),
                'selectedBudget' => $this->activeBudgetId,
                'formModel' => $budgetForm
            ]);

        }

    }


    /**
     * Edit buget
     */
    public function actionEditBudget() {

        $budgetId = Yii::$app->getRequest()->getQueryParam('id', false);

        if (!is_numeric($budgetId)) {

            return $this->accessError();

        }

        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $budgetId])) {

            return $this->accessError();

        }

        $budget = Budget::findOne(['id' => $budgetId]);

        if ($budget->name == 'Default') {

            echo JsonTools::errorMessage('This budget can not be edited.');
            die();

        }

        $budgetForm = new AddBudget();

        if (Yii::$app->request->isPost && $budgetForm->load(Yii::$app->request->post())) {

//            $budgetForm->budget_id = $this->activeBudgetId;

            // set validation attributes
            if ($budgetForm->validate()) {

                $budget->name = $budgetForm->name;
                $budget->save();

                if (Yii::$app->request->isAjax) {

                    echo JsonTools::successMessage('Budget successfully saved.');
                    die();

                }

            }
            elseif (Yii::$app->request->isAjax) {

                echo JsonTools::formErrorMessage($budgetForm);
                die();

            }

        }
        else {

            $budgetForm->setAttributes($budget->getAttributes());

            $this->view->title = 'Edit Budget';
            $this->jsCallbackToView();

            return $this->render('edit-budget', [
                'formModel' => $budgetForm,
                'budgetId' => $budget->id
            ]);

        }

    }

    /**
     * Mock budget detail
     */
    public function actionBudgetDetail()
    {
        $budgetId = Yii::$app->getRequest()->getQueryParam('id', false);
        if (!is_numeric($budgetId)) {
            $budgetId = Yii::$app->getRequest()->getQueryParam('budget_id', false);
            if (!is_numeric($budgetId)) {
                return $this->accessError();
            }
        }
        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $budgetId])) {
            return $this->accessError();
        }

        // get budgets
        $comparedBudget = Budget::findOne($budgetId);
        // we're doing this to make sure a coach can jump between these pages,
        // even when not logged in as a user - for conversation
        $activeBudgetId = UserMeta::find()
            ->select('value')
            ->where(['user_id' => $comparedBudget->user_id, 'key' => 'active_budget_id'])
            ->scalar();
        $activeBudget = Budget::findOne($activeBudgetId);

        $this->view->title =  '"' . $comparedBudget->name . '" Budget Detail';
        return $this->render('budget-detail', [
            'editable' => ($this->userIdentity->id == $comparedBudget->user_id), // whether user should be able to edit or just view
            'comparedBudgetName' => $comparedBudget->name,
            'comparedBudgetId' => $comparedBudget->id,
            'comparedMinMonthlyIncome'  => $comparedBudget->getMinimumMonthlyIncomeAmount(),
            'comparedAvgMonthlyIncome'  => $comparedBudget->getAverageMonthlyIncomeAmount(),
            'comparedTotalMonthlyBudget' => $comparedBudget->getTotalMonthlyBudget(),
            'comparedMinMonthlySurplus' => $comparedBudget->getMinimumMonthlyIncomeAmount() - $comparedBudget->getTotalMonthlyBudget(),
            'comparedAvgMonthlySurplus' => $comparedBudget->getAverageMonthlyIncomeAmount() - $comparedBudget->getTotalMonthlyBudget(),
            'activeBudgetId' => $activeBudget->id,
            'activeMinMonthlyIncome'  => $activeBudget->getMinimumMonthlyIncomeAmount(),
            'activeAvgMonthlyIncome'  => $activeBudget->getAverageMonthlyIncomeAmount(),
            'activeTotalMonthlyBudget'   => $activeBudget->getTotalMonthlyBudget(),
            'activeMinMonthlySurplus' => $activeBudget->getMinimumMonthlyIncomeAmount() - $activeBudget->getTotalMonthlyBudget(),
            'activeAvgMonthlySurplus' => $activeBudget->getAverageMonthlyIncomeAmount() - $activeBudget->getTotalMonthlyBudget(),
        ]);
    }

    /**
     * Adjust mock budget income
     */
    public function actionBudgetOverview()
    {
        $incomeId = Yii::$app->getRequest()->getQueryParam('id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content
        if (!is_numeric($incomeId)) {
            return $this->accessError();
        }
        $income = Income::findOne($incomeId);
        $transactions = IncomeTransaction::findAll(['income_id' => $income->id]);

        $this->view->title = $income->name . ' income';
        if ($this->pageLoadRequest || $fullLoad) {
            $incomes = Income::findAll(['budget_id' => $this->activeBudgetId, 'archived' => '0']);
            $totalAverageIncome = 0;
            foreach ($incomes as $i) {
                $totalAverageIncome += $i->getAverageMonthlyAmount();
            }

            return $this->render('budget-overview', [
                'budget_id'          => $incomeId,
                'income'             => $income,
                'incomes'            => $incomes,
                'transactions'       => $transactions,
                'totalAverageIncome' => $totalAverageIncome,
                'fullLoad'           => $fullLoad,
                'allowEdit'          => true
            ]);
        }
        else {
            return $this->render('budget-overview-content', [
                'budget_id'    => $incomeId,
                'income'       => $income,
                'transactions' => $transactions,
                'fullLoad'     => $fullLoad,
                'allowEdit'    => true
            ]);
        }
    }


    /**
     * Switch budget
     */
    public function actionSwitchBudget() {

        $budgetId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if (!is_numeric($budgetId)) {

            return $this->accessError();

        }

        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $budgetId])) {

            return $this->accessError();

        }

        $budget = Budget::findOne(['id' => $budgetId]);

        if ($confirm) {

            $meta = UserMeta::findOne(['key' => 'active_budget_id', 'user_id' => $this->mainUserId]);
            $meta->value = (string) $budget->id;
            $meta->save();
            Yii::$app->session->remove('userMeta');

            echo JsonTools::successMessage('Budget successfully switched.');
            die();

        }

        $this->view->title = 'Switch Budget';
        $this->jsCallbackToView();

        return $this->render('switch-budget', [
            'budgetId' => $budget->id
        ]);

    }


    /**
     * Submit budget
     */
    public function actionSubmitBudget() {

        $budgetId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if (!is_numeric($budgetId)) {

            return $this->accessError();

        }

        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $budgetId])) {

            return $this->accessError();

        }

        $budget = Budget::findOne(['id' => $budgetId]);

        if ($confirm) {

            if ($this->coach) {

                $budgetUrl = Url::toRoute(['mock-budgets/budget-detail', 'id' => $budget->id], true);
                $mainUser = User::findOne($this->mainUserId);

                // notification
                $this->coach->addNotification('submitted_mock_budget', [
                    'other_user_name' => $mainUser->getFullName(),
                    'url' => $budgetUrl
                ]);

                // email the coach
                DripHelper::sendEmailEvent(
                    'coach-mock-budget-submitted',
                    $this->coach->email,
                    'A mock budget has been created',
                    [
                        'coach_username' => $this->coach->getFullName(),
                        'username' => $mainUser->getFullName(),
                        'budget_url' => $budgetUrl
                    ]
                );
            }

            echo JsonTools::successMessage('Budget submitted to coach.');
            die();

        }

        $this->view->title = 'Submit Budget to Coach';
        $this->jsCallbackToView();

        return $this->render('submit-budget', [
            'budgetId' => $budget->id
        ]);

    }


    /**
     * Remove budget
     */
    public function actionRemoveBudget() {

        $budgetId = Yii::$app->getRequest()->getQueryParam('id', false);
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if (!is_numeric($budgetId)) {

            return $this->accessError();

        }

        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $budgetId])) {

            return $this->accessError();

        }

        $budget = Budget::findOne(['id' => $budgetId]);

        if ($budget->name == 'Default') {

            echo JsonTools::errorMessage('This budget can not be removed.');
            die();

        }
        elseif ($budget->id == $this->activeBudgetId) {

            echo JsonTools::errorMessage('Please activate a different budget before deleting this budget.');
            die();

        }
        elseif ($confirm) {

            $budget->archived = 1;
            $resp = $budget->save();

            echo JsonTools::successMessage('Budget successfully removed.');
            die();

        }

        $this->view->title = 'Remove Budget';
        $this->jsCallbackToView();

        return $this->render('remove-budget', [
            'budgetId' => $budget->id
        ]);

    }

    /**
     * Retrieve monthly incode graph data via json
     */
    public function actionMonthlyIncomeGraphData() {
        $response = [
            'type'      => 'Pie',
            'chartData' => [
                'labels' => ['Another job', 'Investments', 'Job Income'],
                'series' => [850, 800, 2000],
            ]
        ];

        return json_encode($response);
        die();
    }

    /**
     * Retrieve jar allocations graph data via json
     */
    public function actionJarAllocationsGraphData() {
        $response = [
            'type'      => 'Pie',
            'chartData' => [
                'labels' => ['Everyday', 'Utilities', 'Communication', 'Transport', 'Entertainment', 'Hobbies', 'Car'],
                'series' => [850, 1000, 25, 90, 100, 200, 1350],
            ]
        ];

        return json_encode($response);
        die();
    }

    /**
     * Retrieve compare graph data via json
     */
    public function actionCompareGraphData() {
        $response = [
            'type'         => 'Bar',
            'chartData'    => [
                'labels' => ['Everyday', 'Utilities', 'Communication', 'Transport', 'Entertainment', 'Hobbies', 'Car'],
                'series' => [
                    [
                        'name' => 'Incoming',
                        'data' => [10000, 10000, 10000, 10000, 10000, 10000, 10000]
                    ],
                    [
                        'name' => 'Outgoing',
                        'data' => [850, 320, 140, 400, 1000, 230, 1000]
                    ]
                ]
            ],
            'chartOptions' => [
                'axisX' => ['showGrid' => false],
            ]
        ];

        return json_encode($response);
        die();
    }

    /**
     * Table of all mock incomes
     */
    public function  actionIncomesTable()
    {
        $incomes = Income::find()
            ->where(['budget_id' => $this->activeBudgetId, 'archived' => '0'])
            ->orderBy('name ASC')
            ->all();
        $totalAverageIncome = 0;
        foreach ($incomes as $i) {
            $totalAverageIncome += $i->getAverageMonthlyAmount();
        }

        // filtering and getting record
        $tableColumns = [
            [
                'type'   => 'normal',
                'as'     => 'name',
                'select' => 'name'
            ],
            [
                'type'   => 'normal',
                'as'     => 'frequency',
                'select' => 'frequency'
            ],
            [
                'type' => 'method',
                'as'   => 'nextPayDay',
                'name' => 'getNextDue'
            ],
            [
                'type'   => 'normal',
                'as'     => 'amount',
                'select' => 'amount'
            ],
            [
                'type' => 'method',
                'as'   => 'monthlyAverage',
                'name' => 'getMinimumMonthlyAmount'
            ],
            [
                'type' => 'method',
                'as'   => 'realMonthlyAverage',
                'name' => 'getAverageMonthlyAmount'
            ]
        ];

        $filtered = Income::find()
            ->where(['budget_id' => $this->activeBudgetId, 'archived' => '0']);

        $responseData = DataTables::getResponseData($tableColumns, $filtered, $filtered);

        /** @var Income $i */
        foreach ($responseData['items'] as $i) {
            // pause or continue
            $incomeEnd = $i->getIncomeEnd()->one();

            $pauseOrCont = ($incomeEnd)
                ?
                '<a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['incomes/continue-income', 'id' => $i->id]) . '"><i class="glyphicon glyphicon-refresh"></i>Continue</a>'
                :
                '<a href="#" class="open-modal" data-modal-url="' . Url::toRoute(['incomes/pause-income', 'id' => $i->id]) . '"><i class="glyphicon glyphicon-trash"></i>Pause</a>';

            $responseData['response']['data'][] = [
                $i->name,
                Frequencies::getLabel($i->frequency),
                '<span class="hidden">' . $i->getNextDue() . '</span>' . Formatter::date($i->getNextDue()),
                Formatter::currency($i->amount),
                Formatter::currency($i->getMinimumMonthlyAmount()),
                Formatter::currency($i->getAverageMonthlyAmount()),
                '<div class="text-center">
                    <a href="#" id="menu1" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="glyphicon glyphicon-cog gi-2x"></i></a>
                    <ul class="dropdown-menu" aria-labelledby="menu1">
                        <li>
                            ' . $pauseOrCont . '
                        </li>
                    </ul>
                </div>'
            ];
        }

        echo json_encode($responseData['response']);
        die();
    }


    /**
     * Get a list of budgets to show in the menu.
     * @return array The list of selected budgets.
     */
    private function _getBudgetsMenuList() {

        return Budget::find()
            ->leftJoin(UserMeta::tableName(), UserMeta::tableName() . '.user_id = ' . Budget::tableName() . '.user_id AND ' . UserMeta::tableName() . '.key = \'active_budget_id\' AND ' . UserMeta::tableName() . '.value = ' . Budget::tableName() . '.id')
            ->where([
                'and',
                [Budget::tableName() . '.user_id' => $this->mainUserId],
                [Budget::tableName() . '.archived' => 0]
            ])
            ->orderBy('value desc, name asc')
            // ->createCommand()->rawSql;die($test);
            ->all();

    }
}
