<?php

namespace backend\controllers;

use backend\components\ActiveForm;
use backend\components\helpers\DripHelper;
use backend\components\helpers\Stripe;
use backend\models\db\IpTracking;
use backend\models\db\UserPlan;
use backend\models\db\UserPlanType;
use backend\models\db\UserPlanVariation;
use backend\models\db\UserRole;
use common\models\db\UserPasswordResetRequest;
use Stripe\Coupon;
use Yii;
use yii\filters\AccessControl;
use yii\helpers\Html;
use yii\web\Controller;
use yii\helpers\Url;
use backend\models\db\User;
use backend\models\db\UserRegistration;
use common\models\form\Register;
use common\models\form\RegisterResend;
use yii\web\Response;

/**
 * Site controller
 */
class RegisterController extends Controller
{
    public $defaultAction = 'free';

    public function init()
    {
        // set layout for this whole controller
        $this->layout = 'pre-login';
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'bootcamp',
                            'bootcamp-access',
                            'discount-conquer-your-cc',
                            'conquer-your-cc',
                            'free',
                            'free-twelve-months',
                            'app',
                            'membership',
                            'maintenance',
                            'coaching-maintenance',
                            'new-maintenance',
                            'mosaic',
                            'coaching',
                            'financial-coaching-program',
                            'new-coaching',
                            'mastering-money-coaching-program',
                            'kickstart',
                            'financial-kickstarter',
                            'new-kickstart',
                            'confirm',
                            'resend',
                            'coupon',
                            'click-funnels'
                        ],
                        'allow' => true,
                    ]
                ],
            ],
        ];
    }

    public function beforeAction($action)
    {
        if ($action->id == 'click-funnels') {
            $this->enableCsrfValidation = false;
        }
        return parent::beforeAction($action);
    }

    public function actionIndex()
    {
        $this->view->title = 'Register';
        return $this->render('index', [
            'title' => $this->view->title
        ]);
    }

    public function actionClickFunnels()
    {
        if (Yii::$app->request->isPost) {
            $inputJSON = file_get_contents('php://input');
            if (!$inputJSON) {
                die('ERROR - no input data');
            }
            $input = json_decode($inputJSON, true);
            if (empty($input)) {
                die('ERROR - no input data');
            }

            // new user registered
            if (!empty($input['products']) && !empty($input['products'][0]['stripe_plan']) && !empty($input['contact']['email']) &&
                !empty($input['subscription_id']) && isset($input['event']) &&
                $input['event'] == 'created') {

                /** @var UserPlanVariation $stripePlan */
                $stripePlan = UserPlanVariation::find()->where(['stripe_plan_id' => $input['products'][0]['stripe_plan']])->one();
                if (!$stripePlan) {
                    Yii::$app->response->statusCode = 422;
                    die('ERROR - incorrect stripe plan provided: ' . Html::encode($input['products'][0]['stripe_plan']));
                }
                $email = $input['contact']['email'];
                if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    Yii::$app->response->statusCode = 422;
                    die('ERROR - incorrect email provided: ' . Html::encode($email));
                }
                $user = User::find()->where(['email' => $email, 'user_role_id' => UserRole::CLIENT])->one();
                if ($user) {
                    Yii::$app->response->statusCode = 422;
                    die('ERROR - user already exists: ' . Html::encode($email));
                }
                $user = new User();
                $user->user_role_id = UserRole::CLIENT;
                $user->email = $email;
                $user->click_funnels_data = json_encode($input);
                // fake data for now
                $user->username = Yii::$app->security->generateRandomString();
                $user->password = 'no-password';
                $user->password_salt = 'no-password';
                $user->save();

                $userPlan = new UserPlan();
                $userPlan->user_id = $user->id;
                $userPlan->user_plan_type_id = $stripePlan->user_plan_type_id;
                $userPlan->user_plan_variation_id = $stripePlan->id;
                $userPlan->stripe_subscription_id = $input['subscription_id'];
                $userPlan->start_time = date('Y-m-d 00:00:00');
                $userPlan->valid_until = date('Y-m-d 23:59:59', strtotime('+1 day'));
                $userPlan->save();

                // send email to the user
                $authToken = Yii::$app->security->generateRandomString(32) . time();
                $request = new UserPasswordResetRequest();
                $request->user_id = $user->id;
                $request->auth_token = $authToken;
                $request->save();

                // first make sure the user exists in Drip
                DripHelper::addSubscriber($user->email, $user->id);
                // send email
                DripHelper::sendEmailEvent(
                    'setup',
                    $user->email,
                    'Action Required: Set Up Your Account',
                    [
                        'email' => $user->email,
                        'auth_url' => $request->getPasswordResetUrl()
                    ]
                );
            }
            // hardcoded tagging subscribers in Drip (see issue #377)
            if (!empty($input['products']) && is_array($input['products']) && !empty($input['contact']['email'])) {
                $subscriberEmail = $input['contact']['email'];
                $result = 'Matching tag not found';
                foreach ($input['products'] as $product) {
                    switch ($product['name']) {
                        case 'Grandma\'s Jars Application - Monthly':
                            $result = DripHelper::applyTagToSubscriber($subscriberEmail, 'AF - GMJs Monthly');
                            break;
                        case 'Grandma\'s Jars Application - Yearly':
                            $result = DripHelper::applyTagToSubscriber($subscriberEmail, 'AF - GMJs Yearly');
                            break;
                        case 'Set Up Call':
                            $result = DripHelper::applyTagToSubscriber($subscriberEmail, 'AF - Set Up Call');
                            break;
                        case 'Mastering Your Money Course':
                            $result = DripHelper::applyTagToSubscriber($subscriberEmail, 'AF - Master Your Money Course');
                            break;
                    }
                }
                die($result === true ? 'OK' : ($result === false ? 'FAIL' : $result));
            }
            die('IGNORED');
        }
        Yii::$app->response->statusCode = 422;
        die('ERROR - missing parameters');
    }

    public function actionBootcamp()
    {
        if (!\Yii::$app->user->isGuest) {
            return $this->redirect('/site/index');
        }
        $scenario = UserPlanType::PLAN_TYPE_BOOTCAMP;
        $registerForm = new Register(['scenario' => $scenario]);
        if (Yii::$app->request->isAjax && $registerForm->load(Yii::$app->request->post())) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($registerForm);
        }
        if ($registerForm->load(Yii::$app->request->post()) && $registerForm->validate() && $registerForm->submit()) {
            //Yii::$app->session->setFlash('registration', [['success', 'Registration successful! Please check your email to confirm your registration.']]);
            //return $this->redirect('bootcamp');
            return $this->redirect('https://grandmasjars.com/registration-success/' . $scenario);
        }

        $plan = UserPlanType::getPlanTypeObject($scenario);
        $this->view->title = 'Sign up for free access to the members boot camp';
        $subtitle = 'You can browse the lessons on the members site.<br/>By signing up for the boot camp you\'ll' .
            ' automatically get access to a free 40-day trial of the web app as well.';
        return $this->render('free', [
            'title' => $this->view->title,
            'subtitle' => $subtitle,
            'formModel' => $registerForm,
            'plan' => $plan
        ]);
    }

    public function actionBootcampAccess()
    {
        if (!\Yii::$app->user->isGuest) {
            return $this->redirect('/site/index');
        }
        $scenario = UserPlanType::PLAN_TYPE_BOOTCAMP_ACCESS;
        $registerForm = new Register(['scenario' => $scenario]);
        if (Yii::$app->request->isAjax && $registerForm->load(Yii::$app->request->post())) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($registerForm);
        }
        if ($registerForm->load(Yii::$app->request->post()) && $registerForm->validate() && $registerForm->submit()) {
            //Yii::$app->session->setFlash('registration', [['success', 'Registration successful! Please check your email to confirm your registration.']]);
            //return $this->redirect('bootcamp');
            return $this->redirect('https://grandmasjars.com/registration-success/' . $scenario);
        }

        $plan = UserPlanType::getPlanTypeObject($scenario);
        $this->view->title = 'Sign up for access to the members boot camp';
        $subtitle = 'You can browse the lessons on the members site.<br/>By signing up for the boot camp you\'ll' .
            ' automatically get access to a free 40-day trial of the web app as well.';
        return $this->render('paid-new', [
            'title' => $this->view->title,
            'subtitle' => $subtitle,
            'formModel' => $registerForm,
            'plan' => $plan,
            'noCoupon' => true,
            'hasTrial' => false
        ]);
    }

    public function actionDiscountConquerYourCc()
    {
        return $this->redirect('/register');
        return $this->registerConquerPlan(
            true,
            'Register for the Conquer Your Credit Card Course'
        );
    }

    public function actionConquerYourCc()
    {
        return $this->redirect('/register');
        return $this->registerConquerPlan(
            false,
            'Register for the Conquer Your Credit Card Course'
        );
    }

    public function actionFree()
    {
        if (!\Yii::$app->user->isGuest) {
            return $this->redirect('/site/index');
        }
        $scenario = UserPlanType::PLAN_TYPE_FREE;
        $title = 'Subscribing to the GMJS budgeting app';

        $registerForm = new Register(['scenario' => $scenario]);
        if (Yii::$app->request->isAjax && $registerForm->load(Yii::$app->request->post())) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($registerForm);
        }
        if ($registerForm->load(Yii::$app->request->post()) && $registerForm->validate() && $registerForm->submit()) {
            return $this->redirect('https://grandmasjars.com/registration-success/' . $scenario);
        }

        $plan = UserPlanType::getPlanTypeObject($scenario);
        $this->view->title = $title;
        return $this->render('free', [
            'title' => $this->view->title,
            'formModel' => $registerForm,
            'plan' => $plan
        ]);
    }

    public function actionFreeTwelveMonths()
    {
        return $this->redirect('/register');
        return $this->registerPaidPlan(
            UserPlanType::PLAN_TYPE_FREE_12_MONTHS,
            'Subscribing to the GMJS budgeting app',
            true
        );
    }

    public function actionApp()
    {
        return $this->redirect('/register');
        return $this->registerPaidPlanNew(
            UserPlanType::PLAN_TYPE_APP,
            'You\'re Signing Up to Try Grandma\'s Jars for 30 days.',
            true
        );
    }

    public function actionMembership()
    {
        return $this->redirect('/register');
        return $this->registerPaidPlanNew(
            UserPlanType::PLAN_TYPE_MEMBERSHIP,
            'You\'re Signing Up to Try Grandma\'s Jars for 30 days.',
            true
        );
    }

    public function actionCoaching()
    {
        return $this->redirect('/register');
        return $this->registerPaidPlanNew(
            UserPlanType::PLAN_TYPE_COACHING,
            'Subscribing to the GMJS budgeting app with Personal Coaching',
            false
        );
    }

    public function actionFinancialCoachingProgram()
    {
        return $this->registerPaidPlanNew(
            UserPlanType::PLAN_TYPE_FINANCIAL_COACHING_PROGRAM,
            'Subscribing to the GMJS budgeting app with Financial Coaching Program',
            false
        );
    }

    public function actionNewCoaching()
    {
        return $this->redirect('/register/financial-coaching-program');
    }

    public function actionMasteringMoneyCoachingProgram()
    {
        return $this->redirect('/register');
        return $this->registerPaidPlanNew(
            UserPlanType::PLAN_TYPE_MASTERING_MONEY_COACHING_PROGRAM,
            'Subscribing to the GMJS budgeting app with Mastering Money Coaching Program',
            false
        );
    }

    public function actionKickstart()
    {
        return $this->redirect('/register');
        return $this->registerPaidPlanNew(
            UserPlanType::PLAN_TYPE_KICKSTART,
            'Subscribing to the GMJS Kickstart Package',
            false
        );
    }

    public function actionFinancialKickstarter()
    {
        return $this->registerPaidPlanNew(
            UserPlanType::PLAN_TYPE_FINANCIAL_KICKSTARTER,
            'Subscribing to the GMJS Financial Kickstarter Package',
            false
        );
    }

    public function actionNewKickstart()
    {
        return $this->redirect('/register/financial-kickstarter');
    }

    public function actionMaintenance()
    {
        return $this->registerPaidPlanNew(
            UserPlanType::PLAN_TYPE_MAINTENANCE,
            'Subscribing to the GMJS Maintenance Package',
            false
        );
    }

    public function actionMaintenanceOld()
    {
        return $this->redirect('/register');
        return $this->registerPaidPlanNew(
            UserPlanType::PLAN_TYPE_MAINTENANCE_OLD,
            'Subscribing to the GMJS Maintenance Package',
            false
        );
    }

    public function actionCoachingMaintenance()
    {
        return $this->redirect('/register');
        return $this->registerPaidPlanNew(
            UserPlanType::PLAN_TYPE_COACHING_MAINTENANCE,
            'Subscribing to the GMJS Coaching Maintenance Package',
            false
        );
    }

    public function actionNewMaintenance()
    {
        return $this->redirect('/register/coaching-maintenance');
    }

    public function actionMosaic()
    {
        return $this->redirect('/register');
        return $this->registerPaidPlanNew(
            UserPlanType::PLAN_TYPE_MOSAIC,
            'Subscribing to the GMJS Mosaic Plan',
            false
        );
    }

    protected function registerPaidPlanNew($scenario, $title, $hasTrial)
    {
        if (!\Yii::$app->user->isGuest) {
            return $this->redirect('/site/index');
        }
        $registerForm = new Register(['scenario' => $scenario]);
        if (Yii::$app->request->isAjax && $registerForm->load(Yii::$app->request->post())) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($registerForm);
        }
        if ($registerForm->load(Yii::$app->request->post()) && $registerForm->validate() && $registerForm->submit()) {
            //Yii::$app->session->setFlash('login', [['success', 'Registration successful! Please check your email to confirm your registration and log in.']]);
            //return $this->redirect('/site/login');
            return $this->redirect('https://grandmasjars.com/registration-success/' . $scenario);
        }

        $plan = UserPlanType::getPlanTypeObject($scenario);
        $this->view->title = $title;
        return $this->render('paid-new', [
            'title' => $this->view->title,
            'formModel' => $registerForm,
            'plan' => $plan,
            'hasTrial' => $hasTrial
        ]);
    }

    protected function registerPaidPlan($scenario, $title, $noCoupon = false)
    {
        if (!\Yii::$app->user->isGuest) {
            return $this->redirect('/site/index');
        }
        $registerForm = new Register(['scenario' => $scenario]);
        if (Yii::$app->request->isAjax && $registerForm->load(Yii::$app->request->post())) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($registerForm);
        }
        if ($registerForm->load(Yii::$app->request->post()) && $registerForm->validate() && $registerForm->submit()) {
            //Yii::$app->session->setFlash('login', [['success', 'Registration successful! Please check your email to confirm your registration and log in.']]);
            //return $this->redirect('/site/login');
            return $this->redirect('https://grandmasjars.com/registration-success/' . $scenario);
        }

        $plan = UserPlanType::getPlanTypeObject($scenario);
        $this->view->title = $title;
        return $this->render('paid', [
            'title' => $this->view->title,
            'formModel' => $registerForm,
            'plan' => $plan,
            'noCoupon' => $noCoupon
        ]);
    }

    protected function registerConquerPlan($discount, $title)
    {
        if (!\Yii::$app->user->isGuest) {
            return $this->redirect('/site/index');
        }
        $plan1 = UserPlanType::getPlanTypeObject($discount ?
            UserPlanType::PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_1_YEAR : UserPlanType::PLAN_TYPE_CONQUER_YOUR_CC_1_YEAR);
        $plan2 = UserPlanType::getPlanTypeObject($discount ?
            UserPlanType::PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_3_MONTHS : UserPlanType::PLAN_TYPE_CONQUER_YOUR_CC_3_MONTHS);
        if ($discount) {
            if (!IpTracking::trackAndCheck('conquer') && (!$plan1->access_token || $plan1->access_token != Yii::$app->request->get('token'))) {
                Yii::$app->session->setFlash('warning', 'The discount for this plan is no longer available.');
                return $this->redirect('/register/conquer-your-cc');
            }
        }
        $params = [];
        if (Yii::$app->request->isPost && !empty(Yii::$app->request->post('Register')['plan_variation_id'])) {
            $planVariation = UserPlanVariation::findOne(Yii::$app->request->post('Register')['plan_variation_id']);
            $params['scenario'] = $planVariation->userPlanType->getPlanType();
        }
        else {
            $params['scenario'] = $discount ?
                UserPlanType::PLAN_TYPE_DISCOUNT_CONQUER_YOUR_CC_1_YEAR : UserPlanType::PLAN_TYPE_CONQUER_YOUR_CC_1_YEAR;
        }
        $registerForm = new Register($params);
        if (Yii::$app->request->isAjax && $registerForm->load(Yii::$app->request->post())) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($registerForm);
        }
        if ($registerForm->load(Yii::$app->request->post()) && $registerForm->validate() && $registerForm->submit()) {
            //Yii::$app->session->setFlash('login', [['success', 'Registration successful! Please check your email to confirm your registration and log in.']]);
            //return $this->redirect('/site/login');
            return $this->redirect(
                'https://grandmasjars.com/registration-success/' .
                ($discount ? 'discount-conquer-your-cc' : 'conquer-your-cc')
            );
        }

        $this->view->title = $title;
        return $this->render('conquer', [
            'title' => $this->view->title,
            'formModel' => $registerForm,
            'discount' => $discount,
            'plan1' => $plan1,
            'plan2' => $plan2
        ]);
    }

    /**
     * Confirm your registration from URL provided in email.
     */
    public function actionConfirm()
    {
        if (!\Yii::$app->user->isGuest) {
            return $this->redirect('/site/index');
        }
        // check if authorization
        $authToken = Yii::$app->getRequest()->getQueryParam('auth_token', false);

        if (!$authToken) {
            Yii::$app->session->setFlash('registration', [['danger', 'Missing authorization code.']]);
            return $this->redirect(['index']);
        }
        else {
            $registration = UserRegistration::findOne([
                'auth_token' => $authToken,
                'auth_time' => null
            ]);
            if ($registration) {
                // check whether time allowed
                if ((time() - strtotime($registration->time)) < Yii::$app->params['registrationTokenValidTime']) {

                    $registration->auth_time = date('Y-m-d H:i:s');
                    $registration->save();

                    /** @var User $user */
                    $user = $registration->getUser()->one();
                    $user->archived = 0;
                    $user->save();

                    // email the user
                    $loginUrl = Url::toRoute(Yii::$app->user->loginUrl, true);

                    DripHelper::sendEmailEvent(
                        'registration-confirm',
                        $user->email,
                        'Registration Confirmation',
                        [
                            'username' => $user->getFullName(),
                            'login_url' => $loginUrl
                        ]
                    );

                    $userPlan = UserPlan::getLatestPlan($user->id);
                    if ($userPlan) {
                        //$user->sharpSpringVerified($userPlan);
                        Yii::$app->session->set('signed-up', ['email' => $user->email, 'plan' => $userPlan->userPlanType->name]);
                        Yii::$app->session->setFlash('login', [['success', 'Your registration was successfully confirmed. Please log in.']]);
                        return $this->redirect('/site/login');
                    }
                    else {
                        // free plan, need to redirect to the bootcamp site
                        $bootcampUrl = Yii::$app->params['membershipSiteUrl'];
                        Yii::$app->session->setFlash('registration', [
                            ['success', 'Your registration was successfully confirmed. <a href="' . $bootcampUrl . '">You can now start using the Bootcamp site</a>.']
                        ]);
                        return $this->redirect('index');
                    }
                }
                else {
                    $resendUrl = Url::toRoute(['register/resend']);
                    Yii::$app->session->setFlash('registration', [['danger', 'This authorization code has expired. <a href="' . $resendUrl . '">Resend?</a>']]);
                    return $this->redirect('index');
                }
            }
            else {
                Yii::$app->session->setFlash('registration', [['danger', 'This authorization code is incorrect.']]);
                return $this->redirect(['index']);
            }
        }
    }

    /**
     * Send new account authentication email.
     */
    public function actionResend()
    {
        if (!\Yii::$app->user->isGuest) {
            return $this->redirect('/site/index');
        }
        $resendForm = new RegisterResend();

        if ($resendForm->load(Yii::$app->request->post()) && $resendForm->validate()) {
            $user = User::findOne(['username' => $resendForm->username, 'archived' => 0]);
            if ($user) {
                $activeUserRegistration = UserRegistration::find()
                    ->where(['=', 'user_id', $user->id])
                    ->andWhere(['>', 'auth_time', 0])
                    ->one();

                // look whether user already active
                if ($activeUserRegistration){
                    Yii::$app->session->setFlash('login', [['success', 'Your account has already been authorized. Please log in.']]);
                    return $this->redirect('/site/login');
                }
                else {
                    $authToken = Yii::$app->security->generateRandomString(32) . time();

                    UserRegistration::deleteAll([
                        'user_id' => $user->id,
                        'auth_time' => null
                    ]);

                    $registration = new UserRegistration();
                    $registration->setAttributes([
                        'user_id' => $user->id,
                        'auth_token' => $authToken
                    ]);
                    $registration->save();

                    // email the user
                    $authUrl = Url::toRoute(['register/confirm', 'auth_token' => $authToken], true);

                    DripHelper::sendEmailEvent(
                        'registration',
                        $user->email,
                        'Action Required: Activate Your Account',
                        [
                            'username' => $user->getFullName(),
                            'auth_url' => $authUrl
                        ]
                    );

                    Yii::$app->session->setFlash('resend-registration-email', [['success', 'Your authorization code was resent! Please check your email to confirm your registration.']]);
                }
            }
            else {
                Yii::$app->session->setFlash('resend-registration-email', [['danger', 'User not found.']]);
            }
        }
        $this->view->title = 'Resend Registration Email';
        return $this->render('resend-registration-email', [
            'title' => $this->view->title,
            'formModel' => $resendForm
        ]);
    }

    public function actionCoupon()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        if (Yii::$app->request->isPost && !empty(Yii::$app->request->getBodyParam('id'))) {
            $stripe = new Stripe();
            $discount = $stripe->getCouponDiscount(Yii::$app->request->getBodyParam('id'));
            if ($discount) {
                return array_merge(['success' => true], $discount);
            }
        }
        return ['success' => false];
    }

}
