<?php

namespace backend\controllers;

use backend\components\helpers\Calculator;
use backend\components\helpers\JsonTools;
use Yii;

use yii\filters\AccessControl;

use yii\helpers\ArrayHelper;
use yii\helpers\Url;

use backend\models\db\Budget;
use backend\models\db\Eoms;
use backend\models\db\Expense;
use backend\models\db\ExpenseChange;
use backend\models\db\Jar;
use backend\models\db\Transaction;
use backend\models\db\UserLogin;

use backend\components\CustomController;

/**
 * Reports controller
 */
class ReportsController extends CustomController {

    public $enableCsrfValidation = false;

    public function behaviors() {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'index',
                            'transactions-by-month-report',
                            'monthly-transaction-totals-by-year-report',
                            'monthly-transaction-totals-by-jar-and-year-report',
                            'incomes-and-expenses-changes-report',
                            'tax-report',
                            'end-of-month-summary-report',
                            'end-of-month-bank-account-balances-report',
                            'end-of-month-debt-amounts-report',
                            'total-budgeted-vs-total-spent-report',
                            'transactions-chart-report',
                            'usage-history-report'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];

    }


    public function actionIndex() {

        $this->view->title = 'Reports';

        return $this->render('index', [
            'reports' => $this->_getReportsListForMenu()
        ]);

    }


    /**
     * Transactions by month page
     */
    public function actionTransactionsByMonthReport()
    {
        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        $expenseId = Yii::$app->getRequest()->getQueryParam('expense_id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        $entireYearName = '[Entire Year]';
        $entireYearLink = ['/reports/transactions-by-month-report'];
        if ($expenseId) {
            $expense = Expense::findOne($expenseId);
            $jar = $expense->jar;
            // check access
            if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
                echo JsonTools::errorMessage('Invalid expense.');
                die();
            }
            $entireYearLink['year'] = $year;
            $entireYearLink['expense_id'] = $expenseId;
        }

        $budget = Budget::findOne($this->activeBudgetId);

        // get jar IDs
        $jars = $budget->jars;
        $jarIds = ArrayHelper::map($jars, 'id', 'id');

        // get expense IDs
        $allExpenses = Expense::find()
            ->where([
                'and',
                ['jar_id' => $jarIds],
                ['!=', 'frequency', 'one-time']
            ])
            ->orderBy('name asc')
            ->all();
        $allExpenseIds = ArrayHelper::map($allExpenses, 'id', 'id');

        $yearsList = Calculator::getYearsList(
            '/reports/transactions-by-month-report',
            $year,
            null,
            $allExpenseIds,
            $expenseId,
            'date'
        );
        $monthsList = Calculator::getMonthsList(
            '/reports/transactions-by-month-report',
            $year,
            $month ? $month : false,
            null,
            $allExpenseIds,
            $expenseId,
            'date'
        );
        array_unshift($monthsList, [
            'id' => '',
            'label' => $entireYearName,
            'url' => Url::toRoute($entireYearLink),
            'options' => ['class' => $month ? false : 'active'],
            'linkOptions' => ['class' => 'system-link']
        ]);

        $allExpensesList = []; // for dropdown
        foreach ($allExpenses as $e) {
            $active = ($e->id ==  $expenseId) ? ' active' : '';
            $expenseLink = ['/reports/transactions-by-month-report', 'year' => $year, 'expense_id' => $e->id];
            if ($month) {
                $expenseLink['month'] = $month;
            }
            $allExpensesList[] = [
                'id' => $e->id,
                'label' => strval($e->name),
                'url' => Url::toRoute($expenseLink),
                'options' => ['class' => $active],
                'linkOptions' => ['class' => 'system-link']
            ];
        }

        // data
        if (isset($expense)) {
            if ($month) {
                $transactions = Transaction::find()
                    ->where(['and',
                        ['MONTH(date)' => $month],
                        ['YEAR(date)' => $year],
                        ['expense_id' => $expenseId]
                    ])
                    ->all();
            }
            else {
                $transactions = Transaction::find()
                    ->where(['and',
                        ['BETWEEN', 'MONTH(date)', '1', '12'],
                        ['YEAR(date)' => $year],
                        ['expense_id' => $expenseId]
                    ])
                    ->all();
            }

            $totalAmount = 0;
            foreach ($transactions as $t) {
                $totalAmount += $t->amount;
            }
        }

        $this->view->title = 'Transactions by month';
        if ($this->pageLoadRequest || $fullLoad) {
            return $this->render('report-detail', [
                'contentTemplate' => 'transactions-by-month-report',
                'reports' => $this->_getReportsListForMenu(),
                'reportId' => 1,
                'reportData' => [
                    'expenseId' => $expenseId,
                    'reports' => $this->_getReportsListForMenu(),
                    'years' => $yearsList,
                    'months' => $monthsList,
                    'allExpenses' => $allExpensesList,
                    'year' => $year,
                    'month' => $month,
                    'monthName' => $month ? date('F', strtotime('2010-' . $month . '-1')) : $entireYearName,
                    'expenseName' => $expenseId ? $expense->name : '[Select expense]',
                    'totalAmount' => isset($totalAmount) ? $totalAmount : false
                ]
            ]);
        }
        else {
            return $this->render('transactions-by-month-report', [
                'reportId' => 1,
                'expenseId' => $expenseId,
                'reports' => $this->_getReportsListForMenu(),
                'years' => $yearsList,
                'months' => $monthsList,
                'allExpenses' => $allExpensesList,
                'year' => $year,
                'month' => $month,
                'monthName' => $month ? date('F', strtotime('2010-' . $month . '-1')) : $entireYearName,
                'expenseName' => $expenseId ? $expense->name : '[Select expense]',
                'totalAmount' => isset($totalAmount) ? $totalAmount : false
            ]);
        }
    }


    public function actionMonthlyTransactionTotalsByYearReport() {

        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $expenseId = Yii::$app->getRequest()->getQueryParam('expense_id', false);
        $optionId = Yii::$app->getRequest()->getQueryParam('option_id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        if ($expenseId) {

            $expense = Expense::findOne($expenseId);
            $jar = $expense->jar;

            // check access
            if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {

                echo JsonTools::errorMessage('Invalid expense.');
                die();

            }

        }

        $budget = Budget::findOne($this->activeBudgetId);

        // get jar IDs
        $jars = $budget->jars;
        $jarIds = ArrayHelper::map($jars, 'id', 'id');

        // get expense IDs
        $allExpenses = Expense::find()
            ->where([
                'and',
                ['jar_id' => $jarIds],
                ['!=', 'frequency', 'one-time']
            ])
            ->orderBy('name asc')
            ->all();
        $allExpenseIds = ArrayHelper::map($allExpenses, 'id', 'id');
        $allExpensesList = []; // for dropdown
        foreach ($allExpenses as $e) {

            $active = ($e->id ==  $expenseId) ? ' active' : '';

            $allExpensesList[] = [
                'id' => $e->id,
                'label' => strval($e->name),
                'url' => Url::toRoute(['/reports/monthly-transaction-totals-by-year-report', 'year' => $year, 'expense_id' => $e->id]),
                'options' => ['class' => $active],
                'linkOptions' => ['class' => 'system-link']
            ];

        }

        $yearsList = Calculator::getYearsList(
            '/reports/monthly-transaction-totals-by-year-report',
            $year,
            null,
            $allExpenseIds,
            $expenseId,
            'date'
        );

        // options
        $optionNames = [
            1 => 'Show previous years comparison',
            2 => 'Show planned/spent comparison'
        ];

        $optionsList = [
            [
                'id' => 1,
                'label' => $optionNames[1],
                'url' => Url::toRoute(['/reports/monthly-transaction-totals-by-year-report', 'year' => $year, 'expense_id' => $expenseId, 'option_id' => 1]),
                'options' => ['class' => ($optionId == 1) ? ' active' : ''],
                'linkOptions' => ['class' => 'system-link']
            ],
            [
                'id' => 2,
                'label' => $optionNames[2],
                'url' => Url::toRoute(['/reports/monthly-transaction-totals-by-year-report', 'year' => $year, 'expense_id' => $expenseId, 'option_id' => 2]),
                'options' => ['class' => ($optionId == 2) ? ' active' : ''],
                'linkOptions' => ['class' => 'system-link']
            ]
        ];

        // data
        if (isset($expense)) {

            $transactions = Transaction::find()
                ->where([
                    'and',
                    ['YEAR(date)' => $year],
                    ['expense_id' => $expenseId]
                ])
                ->all();

            $totalAmount = 0;
            foreach ($transactions as $t) {

                $totalAmount += $t->amount;

            }

        }

        $this->view->title = 'Transactions by Expense for Year';

        if ($this->pageLoadRequest || $fullLoad) {

            return $this->render('report-detail', [
                'contentTemplate' => 'monthly-transaction-totals-by-year-report',
                'reports' => $this->_getReportsListForMenu(),
                'reportId' => 2,
                'reportData' => [
                    'expenseId' => $expenseId,
                    'optionId' => $optionId,
                    'reports' => $this->_getReportsListForMenu(),
                    'years' => $yearsList,
                    'allExpenses' => $allExpensesList,
                    'options' => $optionsList,
                    'year' => $year,
                    'expenseName' => $expenseId ? $expense->name : '[Select expense]',
                    'optionName' => $optionId ? $optionNames[$optionId] : '[Additional options]',
                    'totalAmount' => isset($totalAmount) ? $totalAmount : false
                ]
            ]);

        }
        else {

            return $this->render('monthly-transaction-totals-by-year-report', [
                'reportId' => 2,
                'expenseId' => $expenseId,
                'optionId' => $optionId,
                'reports' => $this->_getReportsListForMenu(),
                'years' => $yearsList,
                'allExpenses' => $allExpensesList,
                'options' => $optionsList,
                'year' => $year,
                'expenseName' => $expenseId ? $expense->name : '[Select expense]',
                'optionName' => $optionId ? $optionNames[$optionId] : '[Additional options]',
                'totalAmount' => isset($totalAmount) ? $totalAmount : false
            ]);

        }

    }


    public function actionMonthlyTransactionTotalsByJarAndYearReport()
    {
        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        $optionId = Yii::$app->getRequest()->getQueryParam('option_id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        if ($jarId) {
            $jar = Jar::findOne($jarId);
            // check access
            if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
                echo JsonTools::errorMessage('Invalid jar.');
                die();
            }
        }

        // get jars
        $budget = Budget::findOne($this->activeBudgetId);
        $jars = $budget
            ->getJars()
            ->where(['archived' => 0])
            ->orderBy('order')
            ->all();

        $allJarsList = []; // for dropdown
        foreach ($jars as $j) {
            $active = ($j->id ==  $jarId) ? ' active' : '';
            $allJarsList[] = [
                'id' => $j->id,
                'label' => strval($j->name),
                'url' => Url::toRoute(['/reports/monthly-transaction-totals-by-jar-and-year-report', 'year' => $year, 'jar_id' => $j->id]),
                'options' => ['class' => $active],
                'linkOptions' => ['class' => 'system-link']
            ];
        }

        $yearsList = Calculator::getYearsList(
            '/reports/monthly-transaction-totals-by-jar-and-year-report',
            $year,
            $jarId
        );

        // options
        $optionNames = [
            1 => 'Show previous years comparison',
            2 => 'Show planned/spent comparison',
            3 => 'Show breakdown chart'
        ];

        $optionsList = [
            [
                'id' => 1,
                'label' => $optionNames[1],
                'url' => Url::toRoute(['/reports/monthly-transaction-totals-by-jar-and-year-report', 'year' => $year, 'jar_id' => $jarId, 'option_id' => 1]),
                'options' => ['class' => ($optionId == 1) ? ' active' : ''],
                'linkOptions' => ['class' => 'system-link']
            ],
            [
                'id' => 2,
                'label' => $optionNames[2],
                'url' => Url::toRoute(['/reports/monthly-transaction-totals-by-jar-and-year-report', 'year' => $year, 'jar_id' => $jarId, 'option_id' => 2]),
                'options' => ['class' => ($optionId == 2) ? ' active' : ''],
                'linkOptions' => ['class' => 'system-link']
            ],
            [
                'id' => 3,
                'label' => $optionNames[3],
                'url' => Url::toRoute(['/reports/monthly-transaction-totals-by-jar-and-year-report', 'year' => $year, 'jar_id' => $jarId, 'option_id' => 3]),
                'options' => ['class' => ($optionId == 3) ? ' active' : ''],
                'linkOptions' => ['class' => 'system-link']
            ]
        ];

        // data
        if (isset($jar)) {
            $transactions = Transaction::find()
                ->joinWith('expense')
                ->where([
                    'and',
                    ['YEAR(' . Transaction::tableName() . '.date)' => $year],
                    ['jar_id' => $jarId]
                ])
                ->all();
            $totalAmount = 0;
            foreach ($transactions as $t) {
                $totalAmount += $t->amount;
            }
            $averageAmount = $totalAmount / date('m');
        }

        $this->view->title = 'Transactions by Jar for Year';
        if ($this->pageLoadRequest || $fullLoad) {
            return $this->render('report-detail', [
                'contentTemplate' => 'monthly-transaction-totals-by-jar-and-year-report',
                'reports' => $this->_getReportsListForMenu(),
                'reportId' => 3,
                'reportData' => [
                    'jarId' => $jarId,
                    'optionId' => $optionId,
                    'reports' => $this->_getReportsListForMenu(),
                    'years' => $yearsList,
                    'allJars' => $allJarsList,
                    'options' => $optionsList,
                    'year' => $year,
                    'jarName' => $jarId ? $jar->name : '[Select jar]',
                    'optionName' => $optionId ? $optionNames[$optionId] : '[Additional options]',
                    'totalAmount' => isset($totalAmount) ? $totalAmount : false,
                    'averageAmount' => isset($averageAmount) ? $averageAmount : false
                ]
            ]);
        }
        else {
            return $this->render('monthly-transaction-totals-by-jar-and-year-report', [
                'reportId' => 3,
                'jarId' => $jarId,
                'optionId' => $optionId,
                'reports' => $this->_getReportsListForMenu(),
                'years' => $yearsList,
                'allJars' => $allJarsList,
                'options' => $optionsList,
                'year' => $year,
                'jarName' => $jarId ? $jar->name : '[Select jar]',
                'optionName' => $optionId ? $optionNames[$optionId] : '[Additional options]',
                'totalAmount' => isset($totalAmount) ? $totalAmount : false,
                'averageAmount' => isset($averageAmount) ? $averageAmount : false
            ]);
        }
    }

    public function actionTaxReport() {

        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        $budget = Budget::findOne($this->activeBudgetId);

        // get jar IDs
        $jars = $budget->jars;
        $jarIds = ArrayHelper::map($jars, 'id', 'id');

        $yearsList = Calculator::getYearsList(
            '/reports/tax-report',
            $year,
            $jarIds
        );
        $monthsList = Calculator::getMonthsList(
            '/reports/tax-report',
            $year,
            $month,
            $jarIds
        );

        // data
        $totalAmount = Transaction::find()
            ->select('SUM(' . Transaction::tableName() . '.amount)')
            ->joinWith('expense')
            ->where([
                'and',
                ['MONTH(' . Transaction::tableName() . '.date)' => $month],
                ['YEAR(' . Transaction::tableName() . '.date)' => $year],
                ['jar_id' => $jarIds],
                [Transaction::tableName() . '.tax_deductible' => 1]
            ])
            ->scalar();

        $this->view->title = 'Tax Report';

        if ($this->pageLoadRequest || $fullLoad) {

            return $this->render('report-detail', [
                'contentTemplate' => 'tax-report',
                'reports' => $this->_getReportsListForMenu(),
                'reportId' => 5,
                'reportData' => [
                    'reports' => $this->_getReportsListForMenu(),
                    'years' => $yearsList,
                    'months' => $monthsList,
                    'year' => $year,
                    'month' => $month,
                    'monthName' => date('F', strtotime('2010-' . $month . '-1')),
                    'totalAmount' => isset($totalAmount) ? $totalAmount : false
                ]
            ]);

        }
        else {

            return $this->render('tax-report', [
                'reportId' => 5,
                'reports' => $this->_getReportsListForMenu(),
                'years' => $yearsList,
                'months' => $monthsList,
                'year' => $year,
                'month' => $month,
                'monthName' => date('F', strtotime('2010-' . $month . '-1')),
                'totalAmount' => isset($totalAmount) ? $totalAmount : false
            ]);

        }

    }


    public function actionEndOfMonthSummaryReport() {

        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $optionId = Yii::$app->getRequest()->getQueryParam('option_id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        $budget = Budget::findOne($this->activeBudgetId);

        // get jar IDs
        $jars = $budget->jars;
        $jarIds = ArrayHelper::map($jars, 'id', 'id');

        $firstEoms = Eoms::find()
            ->where([
                'AND',
                ['budget_id' => $this->activeBudgetId]
            ])
            ->orderBy('id asc')
            ->one();

        $lastEoms = Eoms::find()
            ->where([
                'AND',
                ['budget_id' => $this->activeBudgetId]
            ])
            ->orderBy('id desc')
            ->one();

        $yearsList = [];
        $optionsList = [];
        $totalAmount = 0;
        if ($lastEoms) {

            $eoms = Eoms::find()
                ->where([
                    'AND',
                    ['budget_id' => $this->activeBudgetId],
                    ['year' => $lastEoms->year]
                ])
                ->orderBy('id asc')
                ->all();



            $firstYear = $firstEoms->year;
            $lastYear = $lastEoms->year;

            // get years
            for ($i = $firstYear; $i<= $lastYear; $i++) {

                $active = ($i == $year) ? ' active' : '';

                $yearsList[] = [
                    'id' => $i,
                    'label' => strval($i),
                    'url' => Url::toRoute(['/reports/end-of-month-summary-report', 'year' => $i]),
                    'options' => ['class' => $active],
                    'linkOptions' => ['class' => 'system-link']
                ];

            }

            // options
            $optionNames = [
                1 => 'Show previous years comparison'
            ];

            $optionsList = [
                [
                    'id' => 1,
                    'label' => $optionNames[1],
                    'url' => Url::toRoute(['/reports/end-of-month-summary-report', 'year' => $year, 'option_id' => 1]),
                    'options' => ['class' => ($optionId == 1) ? ' active' : ''],
                    'linkOptions' => ['class' => 'system-link']
                ]
            ];

            // data
            foreach ($eoms as $e) {

                $totalAmount += $e->getSummaryAmount();

            }

        }
        else {

            $yearsList[] = [
                'id' => $year,
                'label' => strval($year),
                'url' => Url::toRoute(['/reports/end-of-month-summary-report', 'year' => $year]),
                'options' => ['class' => ' active'],
                'linkOptions' => ['class' => 'system-link']
            ];

        }

        $this->view->title = 'End of Month Summary Report';

        if ($this->pageLoadRequest || $fullLoad) {

            return $this->render('report-detail', [
                'contentTemplate' => 'end-of-month-summary-report',
                'reports' => $this->_getReportsListForMenu(),
                'reportId' => 9,
                'reportData' => [
                    'reports' => $this->_getReportsListForMenu(),
                    'years' => $yearsList,
                    'year' => $year,
                    'totalAmount' => isset($totalAmount) ? $totalAmount : false,
                    'optionId' => $optionId,
                    'options' => $optionsList,
                    'optionName' => $optionId ? $optionNames[$optionId] : '[Additional options]',
                    'lastEoms' => $lastEoms
                ]
            ]);

        }
        else {

            return $this->render('end-of-month-summary-report', [
                'reportId' => 9,
                'reports' => $this->_getReportsListForMenu(),
                'years' => $yearsList,
                'year' => $year,
                'totalAmount' => isset($totalAmount) ? $totalAmount : false,
                'optionId' => $optionId,
                'options' => $optionsList,
                'optionName' => $optionId ? $optionNames[$optionId] : '[Additional options]',
                'lastEoms' => $lastEoms
            ]);

        }

    }


    public function actionIncomesAndExpensesChangesReport()
    {
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content
        $this->view->title = 'Adjustments Made to Incomes &amp; Expenses';

        if ($this->pageLoadRequest || $fullLoad) {
            return $this->render('report-detail', [
                'contentTemplate' => 'incomes-and-expenses-changes-report',
                'reports' => $this->_getReportsListForMenu(),
                'reportId' => 7,
                'reportData' => []
            ]);
        }
        else {
            return $this->render('incomes-and-expenses-changes-report', [
                'reportId' => 7,
                'reports' => $this->_getReportsListForMenu()
            ]);
        }
    }

    public function actionTotalBudgetedVsTotalSpentReport() {

        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $optionId = Yii::$app->getRequest()->getQueryParam('option_id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        // get years
        $budget = Budget::findOne($this->activeBudgetId);
        $firstYear = date('Y', strtotime($budget->created_time));
        $lastYear = date('Y');
        $yearsList = [];
        for ($i = $firstYear; $i<= $lastYear; $i++) {

            $active = ($i == $year) ? ' active' : '';

            $yearsList[] = [
                'id' => $i,
                'label' => strval($i),
                'url' => Url::toRoute(['/reports/total-budgeted-vs-total-spent-report', 'year' => $i]),
                'options' => ['class' => $active],
                'linkOptions' => ['class' => 'system-link']
            ];

        }

        // options
        $optionNames = [
            1 => 'Show breakdown by jar chart'
        ];

        $optionsList = [
            [
                'id' => 1,
                'label' => $optionNames[1],
                'url' => Url::toRoute(['/reports/total-budgeted-vs-total-spent-report', 'year' => $year, 'option_id' => 1]),
                'options' => ['class' => ($optionId == 1) ? ' active' : ''],
                'linkOptions' => ['class' => 'system-link']
            ]
        ];

        // data
        $totalSpent = Transaction::find()
            ->select('SUM(' . Transaction::tableName() . '.amount)')
            ->joinWith('expense')
            ->leftJoin(Jar::tableName(), Jar::tableName() . '.id = jar_id')
            ->where([
                'AND',
                ['budget_id' => $this->activeBudgetId],
                ['YEAR(' . Transaction::tableName() . '.date)' => $year]
            ])
            ->scalar();

        $budgetStartDate = date('Y-m-d', strtotime($budget->created_time));
        $budgetEndDate = date('Y-m-d');

        $totalBudgeted = 0;
        for ($i = 1; $i <= 12; $i++) {

            $plannedStartTime = strtotime(date("$year-$i-t"));
            $plannedEndTime = strtotime(date("$year-$i-1"));

            if (($plannedStartTime >= strtotime($budgetStartDate)) && ($plannedEndTime <= strtotime($budgetEndDate))) {

                $totalBudgeted += $budget->getMonthsExpensesAmount($i, $year);

            }

        }

        $this->view->title = 'Total budgeted vs. Total spent';

        if ($this->pageLoadRequest || $fullLoad) {

            return $this->render('report-detail', [
                'contentTemplate' => 'total-budgeted-vs-total-spent-report',
                'reports' => $this->_getReportsListForMenu(),
                'reportId' => 6,
                'reportData' => [
                    'years' => $yearsList,
                    'year' => $year,
                    'totalBudgeted' => $totalBudgeted,
                    'totalSpent' => $totalSpent,
                    'balance' => $totalBudgeted - $totalSpent,
                    'optionId' => $optionId,
                    'options' => $optionsList,
                    'optionName' => $optionId ? $optionNames[$optionId] : '[Additional options]'
                ]
            ]);

        }
        else {

            return $this->render('total-budgeted-vs-total-spent-report', [
                'reportId' => 6,
                'reports' => $this->_getReportsListForMenu(),
                'years' => $yearsList,
                'year' => $year,
                'totalBudgeted' => $totalBudgeted,
                'totalSpent' => $totalSpent,
                'balance' => $totalBudgeted - $totalSpent,
                'optionId' => $optionId,
                'options' => $optionsList,
                'optionName' => $optionId ? $optionNames[$optionId] : '[Additional options]'
            ]);

        }

    }


    public function actionEndOfMonthBankAccountBalancesReport() {

        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        $optionId = Yii::$app->getRequest()->getQueryParam('option_id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        $firstEoms = Eoms::find()
            ->where([
                'AND',
                ['budget_id' => $this->activeBudgetId]
            ])
            ->orderBy('id asc')
            ->one();

        $lastEoms = Eoms::find()
            ->where([
                'AND',
                ['budget_id' => $this->activeBudgetId]
            ])
            ->orderBy('id desc')
            ->one();

        $monthsList = [];
        $yearsList = [];
        $optionsList = [];
        if ($lastEoms) {

            $eoms = Eoms::find()
                ->where([
                    'AND',
                    ['budget_id' => $this->activeBudgetId],
                    ['year' => $lastEoms->year]
                ])
                ->orderBy('id asc')
                ->all();

            $firstYear = $firstEoms->year;
            $lastYear = $lastEoms->year;

            // get months
            if ($year == $firstYear) {


                $firstMonth = $firstEoms->month;

            }
            else {

                $firstMonth = 1;

            }

            if ($year == date('Y')) {

                $lastMonth = $lastEoms->month;

            }
            else {

                $lastMonth = 12;

            }

            if (!$month && !$optionId) {

                $month = ($year == $lastYear) ? $lastMonth : $firstMonth;

            }

            // get years
            for ($i = $firstYear; $i<= $lastYear; $i++) {

                $active = ($i == $year) ? ' active' : '';

                $yearsList[] = [
                    'id' => $i,
                    'label' => strval($i),
                    'url' => Url::toRoute(['/reports/end-of-month-bank-account-balances-report', 'year' => $i]),
                    'options' => ['class' => $active],
                    'linkOptions' => ['class' => 'system-link']
                ];

            }

            for ($i = $firstMonth; $i<= $lastMonth; $i++) {

                $active = ($i == $month) ? ' active' : '';

                $monthsList[] = [
                    'id' => $i,
                    'label' => date('F', strtotime('2010-' . $i . '-01')),
                    'url' => Url::toRoute(['/reports/end-of-month-bank-account-balances-report', 'year' => $year, 'month' => $i]),
                    'options' => ['class' => $active],
                    'linkOptions' => ['class' => 'system-link']
                ];

            }

            $optionNames = [
                1 => 'Show all years'
            ];

            $optionsList = [
                [
                    'id' => 1,
                    'label' => $optionNames[1],
                    'url' => Url::toRoute(['/reports/end-of-month-bank-account-balances-report', 'option_id' => 1]),
                    'options' => ['class' => ($optionId == 1) ? ' active' : ''],
                    'linkOptions' => ['class' => 'system-link']
                ]
            ];

        }
        else {

            $yearsList[] = [
                'id' => $year,
                'label' => strval($year),
                'url' => Url::toRoute(['/reports/end-of-month-bank-account-balances-report', 'year' => $year]),
                'options' => ['class' => 'active'],
                'linkOptions' => ['class' => 'system-link']
            ];

            $monthsList[] = [
                'id' => $month,
                'label' => date('F', strtotime('2010-' . $month . '-01')),
                'url' => Url::toRoute(['/reports/end-of-month-bank-account-balances-report', 'year' => $year, 'month' => $month]),
                'options' => ['class' => 'active'],
                'linkOptions' => ['class' => 'system-link']
            ];

        }

        $this->view->title = 'End of Month Bank Account Balances';

        if ($this->pageLoadRequest || $fullLoad) {

            return $this->render('report-detail', [
                'contentTemplate' => 'end-of-month-bank-account-balances-report',
                'reports' => $this->_getReportsListForMenu(),
                'reportId' => 10,
                'reportData' => [
                    'years' => $yearsList,
                    'months' => $monthsList,
                    'year' => $year,
                    'month' => $month,
                    'monthName' => date('F', strtotime('2010-' . $month . '-1')),
                    'optionId' => $optionId,
                    'options' => $optionsList,
                    'optionName' => $optionId ? $optionNames[$optionId] : '[Additional options]'
                ]
            ]);

        }
        else {

            return $this->render('end-of-month-bank-account-balances-report', [
                'reportId' => 10,
                'reports' => $this->_getReportsListForMenu(),
                'years' => $yearsList,
                'months' => $monthsList,
                'year' => $year,
                'month' => $month,
                'monthName' => date('F', strtotime('2010-' . $month . '-1')),
                'optionId' => $optionId,
                'options' => $optionsList,
                'optionName' => $optionId ? $optionNames[$optionId] : '[Additional options]'
            ]);

        }

    }


    public function actionEndOfMonthDebtAmountsReport() {

        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        $optionId = Yii::$app->getRequest()->getQueryParam('option_id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        $firstEoms = Eoms::find()
            ->where([
                'AND',
                ['budget_id' => $this->activeBudgetId]
            ])
            ->orderBy('id asc')
            ->one();

        $lastEoms = Eoms::find()
            ->where([
                'AND',
                ['budget_id' => $this->activeBudgetId]
            ])
            ->orderBy('id desc')
            ->one();

        $monthsList = [];
        $yearsList = [];
        $optionsList = [];
        if ($lastEoms) {

            $eoms = Eoms::find()
                ->where([
                    'AND',
                    ['budget_id' => $this->activeBudgetId],
                    ['year' => $year]
                ])
                ->orderBy('id asc')
                ->all();

            $firstYear = $firstEoms->year;
            $lastYear = $lastEoms->year;

            // get months
            if ($year == $firstYear) {


                $firstMonth = $firstEoms->month;

            }
            else {

                $firstMonth = 1;

            }

            if ($year == date('Y')) {

                $lastMonth = $lastEoms->month;

            }
            else {

                $lastMonth = 12;

            }

            // get years
            for ($i = $firstYear; $i<= $lastYear; $i++) {

                $active = ($i == $year) ? ' active' : '';

                $yearsList[] = [
                    'id' => $i,
                    'label' => strval($i),
                    'url' => Url::toRoute(['/reports/end-of-month-debt-amounts-report', 'year' => $i]),
                    'options' => ['class' => $active],
                    'linkOptions' => ['class' => 'system-link']
                ];

            }

            for ($i = $firstMonth; $i<= $lastMonth; $i++) {

                $active = ($i == $month) ? ' active' : '';

                $monthsList[] = [
                    'id' => $i,
                    'label' => date('F', strtotime('2010-' . $i . '-01')),
                    'url' => Url::toRoute(['/reports/end-of-month-debt-amounts-report', 'year' => $year, 'month' => $i]),
                    'options' => ['class' => $active],
                    'linkOptions' => ['class' => 'system-link']
                ];

            }

            $optionNames = [
                1 => 'Show all years'
            ];

            $optionsList = [
                [
                    'id' => 1,
                    'label' => $optionNames[1],
                    'url' => Url::toRoute(['/reports/end-of-month-debt-amounts-report', 'option_id' => 1]),
                    'options' => ['class' => ($optionId == 1) ? ' active' : ''],
                    'linkOptions' => ['class' => 'system-link']
                ]
            ];

        }
        else {

            $yearsList[] = [
                'id' => $year,
                'label' => strval($year),
                'url' => Url::toRoute(['/reports/end-of-month-bank-account-balances-report', 'year' => $year]),
                'options' => ['class' => 'active'],
                'linkOptions' => ['class' => 'system-link']
            ];

            $monthsList[] = [
                'id' => $month,
                'label' => date('F', strtotime('2010-' . $month . '-01')),
                'url' => Url::toRoute(['/reports/end-of-month-bank-account-balances-report', 'year' => $year, 'month' => $month]),
                'options' => ['class' => 'active'],
                'linkOptions' => ['class' => 'system-link']
            ];

        }

        $this->view->title = 'End of Month Debt Amounts';

        if ($this->pageLoadRequest || $fullLoad) {

            return $this->render('report-detail', [
                'contentTemplate' => 'end-of-month-debt-amounts-report',
                'reports' => $this->_getReportsListForMenu(),
                'reportId' => 11,
                'reportData' => [
                    'years' => $yearsList,
                    'months' => $monthsList,
                    'year' => $year,
                    'month' => $month,
                    'monthName' => date('F', strtotime('2010-' . $month . '-1')),
                    'optionId' => $optionId,
                    'options' => $optionsList,
                    'optionName' => $optionId ? $optionNames[$optionId] : '[Additional options]'
                ]
            ]);

        }
        else {

            return $this->render('end-of-month-debt-amounts-report', [
                'reportId' => 11,
                'reports' => $this->_getReportsListForMenu(),
                'years' => $yearsList,
                'months' => $monthsList,
                'year' => $year,
                'month' => $month,
                'monthName' => date('F', strtotime('2010-' . $month . '-1')),
                'optionId' => $optionId,
                'options' => $optionsList,
                'optionName' => $optionId ? $optionNames[$optionId] : '[Additional options]'
            ]);

        }

    }


    public function actionTransactionsChartReport()
    {
        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $jarId = Yii::$app->getRequest()->getQueryParam('jar_id', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        $budget = Budget::findOne($this->activeBudgetId);

        if ($jarId) {
            $jar = Jar::findOne($jarId);
            // check access
            if (!Yii::$app->user->can('editJar', ['jarId' => $jar->id])) {
                echo JsonTools::errorMessage('Invalid jar.');
                die();
            }
        }

        $jars = $budget
            ->getJars()
            ->where(['archived' => 0])
            ->orderBy('order')
            ->all();
        $allJarsList = []; // for dropdown
        foreach ($jars as $j) {
            $active = ($j->id ==  $jarId) ? ' active' : '';
            $allJarsList[] = [
                'id' => $j->id,
                'label' => strval($j->name),
                'url' => Url::toRoute(['/reports/transactions-chart-report', 'year' => $year, 'jar_id' => $j->id]),
                'options' => ['class' => $active],
                'linkOptions' => ['class' => 'system-link']
            ];
        }
        $jarIds = ArrayHelper::map($allJarsList, 'id', 'id');

        $firstExpenseChange = ExpenseChange::find()
            ->joinWith('expense')
            ->where([
                'AND',
                ['type' => 'C'],
                ['jar_id' => $jarId ? $jarId : $jarIds]
            ])
            ->orderBy('time ASC')
            ->one();

        $firstYear = date('Y', strtotime($firstExpenseChange->time));
        $lastYear = date('Y');

        // get years
        $yearsList = [];
        for ($i = $firstYear; $i<= $lastYear; $i++) {
            $active = ($i == $year) ? ' active' : '';
            $yearsList[] = [
                'id' => $i,
                'label' => strval($i),
                'url' => Url::toRoute(['/reports/transactions-chart-report', 'year' => $i]),
                'options' => ['class' => $active],
                'linkOptions' => ['class' => 'system-link']
            ];
        }

        $jarName = $jarId ? $jar->name : '[Select jar]';

        $this->view->title = 'Expenses Breakdown Chart';
        if ($this->pageLoadRequest || $fullLoad) {
            return $this->render('report-detail', [
                'contentTemplate' => 'transactions-chart-report',
                'reports' => $this->_getReportsListForMenu(),
                'reportId' => 12,
                'reportData' => [
                    'years' => $yearsList,
                    'year' => $year,
                    'allJars' => $allJarsList,
                    'jarName' => $jarName,
                    'jarId' => $jarId
                ]
            ]);
        }
        else {
            return $this->render('transactions-chart-report', [
                'reportId' => 12,
                'reports' => $this->_getReportsListForMenu(),
                'years' => $yearsList,
                'year' => $year,
                'allJars' => $allJarsList,
                'jarName' => $jarName,
                'jarId' => $jarId
            ]);
        }
    }

    public function actionUsageHistoryReport() {

        $year = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month = Yii::$app->getRequest()->getQueryParam('month', false);
        $fullLoad = Yii::$app->getRequest()->getQueryParam('full_load', false); // whether to load all page content

        $firstLogin = UserLogin::find()
            ->where(['user_id' => $this->userIdentity->id])
            ->orderBy('id asc')
            ->one();

        $lastLogin = UserLogin::find()
            ->where(['user_id' => $this->userIdentity->id])
            ->orderBy('id desc')
            ->one();

        if ($firstLogin) {

            $firstYear = date('Y', strtotime($firstLogin->time));
            $lastYear = date('Y', strtotime($lastLogin->time));
        }
        else {

            $firstYear = $lastYear = date('Y');

        }

        // get months
        $monthsList = [];
        if ($year == $firstYear) {

            if ($firstLogin) {

                $firstMonth = date('m', strtotime($firstLogin->time));

            }
            else {

                $firstMonth = date('m');

            }

        }
        else {

            $firstMonth = 1;

        }

        if ($year == date('Y')) {

            $lastMonth = date('m');

        }
        else {

            $lastMonth = 12;

        }

        if (!$month) {

            $month = ($year == $lastYear) ? $lastMonth : $firstMonth;

        }

        // get years
        $yearsList = [];
        for ($i = $firstYear; $i<= $lastYear; $i++) {

            $active = ($i == $year) ? ' active' : '';

            $yearsList[] = [
                'id' => $i,
                'label' => strval($i),
                'url' => Url::toRoute(['/reports/usage-history-report', 'year' => $i]),
                'options' => ['class' => $active],
                'linkOptions' => ['class' => 'system-link']
            ];

        }

        // get months
        for ($i = $firstMonth; $i<= $lastMonth; $i++) {

            $active = ($i == $month) ? ' active' : '';

            $monthsList[] = [
                'id' => $i,
                'label' => date('F', strtotime('2010-' . $i . '-01')),
                'url' => Url::toRoute(['/reports/usage-history-report', 'year' => $year, 'month' => $i]),
                'options' => ['class' => $active],
                'linkOptions' => ['class' => 'system-link']
            ];

        }

        // data
        $totalDuration = UserLogin::find()
            ->select('SUM(duration)')
            ->where([
                'and',
                ['user_id' => $this->userIdentity->id],
                ['MONTH(time)' => $month],
                ['YEAR(time)' => $year]
            ])
            ->scalar();

        $hours = floor($totalDuration / 3600);
        $minutes = floor(($totalDuration - $hours * 3600) / 60);
        $seconds = $totalDuration % 60;
        $averageTime = "$hours hours $minutes minutes $seconds seconds";

        $this->view->title = 'Usage History';

        if ($this->pageLoadRequest || $fullLoad) {

            return $this->render('report-detail', [
                'contentTemplate' => 'usage-history-report',
                'reports' => $this->_getReportsListForMenu(),
                'reportId' => 8,
                'reportData' => [
                    'years' => $yearsList,
                    'months' => $monthsList,
                    'year' => $year,
                    'month' => $month,
                    'monthName' => date('F', strtotime('2010-' . $month . '-1')),
                    'averageTime' => $averageTime
                ]
            ]);

        }
        else {

            return $this->render('usage-history-report', [
                'reportId' => 8,
                'reports' => $this->_getReportsListForMenu(),
                'years' => $yearsList,
                'months' => $monthsList,
                'year' => $year,
                'month' => $month,
                'monthName' => date('F', strtotime('2010-' . $month . '-1')),
                'averageTime' => $averageTime
            ]);

        }

    }


    /**
     * Get a list of reports for the reports left-side menu.
     * @return array List of reports.
     */
    private function _getReportsListForMenu() {

        return [
            (object)[
                'id'    => 1,
                'name'  => 'Transactions by Expense for Month',
                'desc'  => 'View all your transactions for each expense.',
                'link' => Url::toRoute(['/reports/transactions-by-month-report']),
                'icon' => 'align-justify'
            ],
            (object)[
                'id'    => 2,
                'name'  => 'Transactions by Expense for Year',
                'desc'  => 'All your tax deductable transactions made during this fiscal year.',
                'link' => Url::toRoute(['/reports/monthly-transaction-totals-by-year-report']),
                'icon' => 'share-alt'
            ],
            (object)[
                'id'    => 3,
                'name'  => 'Transaction by Jar for Year',
                'desc'  => 'View all your transactions for each jar as well as a summary of the last 12 months for each.',
                'link' => Url::toRoute(['/reports/monthly-transaction-totals-by-jar-and-year-report']),
                'icon' => 'list'
            ],
            (object)[
                'id'    => 5,
                'name'  => 'Tax Report',
                'desc'  => 'View your tax deductible transactions sorted by year and month.',
                'link' => Url::toRoute(['/reports/tax-report']),
                'icon' => 'usd'
            ],
            (object)[
                'id'    => 6,
                'name'  => 'Total Budgeted vs. Total Spent',
                'desc'  => 'Comparison between expenses and budget as well as a summary of the last 12 months for each.',
                'link' => Url::toRoute(['/reports/total-budgeted-vs-total-spent-report']),
                'icon' => 'random'
            ],
            (object)[
                'id'    => 9,
                'name'  => 'End Of Month Summary Graph',
                'desc'  => 'Comparison between expenses and budget as well as a summary of the last 12 months for each.',
                'link' => Url::toRoute(['/reports/end-of-month-summary-report']),
                'icon' => 'stats'
            ],
            (object)[
                'id'    => 10,
                'name'  => 'End of Month Bank Account Balances',
                'desc'  => 'Comparison between expenses and budget as well as a summary of the last 12 months for each.',
                'link' => Url::toRoute(['/reports/end-of-month-bank-account-balances-report']),
                'icon' => 'book'
            ],
            (object)[
                'id'    => 11,
                'name'  => 'End of Month Debts Report',
                'desc'  => 'Comparison between expenses and budget as well as a summary of the last 12 months for each.',
                'link' => Url::toRoute(['/reports/end-of-month-debt-amounts-report']),
                'icon' => 'book'
            ],
            (object)[
                'id'    => 12,
                'name'  => 'Expenses Breakdown Chart',
                'desc'  => 'Comparison between expenses and budget as well as a summary of the last 12 months for each.',
                'link' => Url::toRoute(['/reports/transactions-chart-report']),
                'icon' => 'unchecked'
            ],
            (object)[
                'id'    => 7,
                'name'  => 'Adjustments Made to Incomes &amp; Expenses',
                'desc'  => 'View a list of all adjustments that have been made separated by Income and Expense.',
                'link' => Url::toRoute(['/reports/incomes-and-expenses-changes-report']),
                'icon' => 'calendar'
            ],
            (object)[
                'id'    => 8,
                'name'  => 'Usage History',
                'desc'  => 'Review the time spent in the application and the times someone has logged into your account.',
                'link' => Url::toRoute(['/reports/usage-history-report']),
                'icon' => 'time'
            ]

        ];

    }

}
