<?php

namespace backend\controllers;


use backend\components\helpers\DripHelper;
use backend\components\helpers\Formatter;
use backend\components\helpers\GoogleAnalytics;
use backend\components\helpers\Stripe;
use backend\models\db\User;
use backend\models\db\UserMeta;
use backend\models\db\UserPlanVariation;
use backend\models\db\UserRole;
use common\models\db\UserPasswordResetRequest;
use common\models\form\Register;
use Yii;
use backend\models\db\Invoice;
use backend\models\db\UserPlan;
use backend\models\db\UserPlanFeature;
use backend\models\db\UserPlanType;

use yii\filters\AccessControl;

use backend\components\helpers\JsonTools;
use backend\components\CustomController;
use yii\helpers\Url;
use yii\web\NotFoundHttpException;
use yii\web\Response;


/**
 * Subscription controller
 */
class SubscriptionController extends CustomController
{
    public $enableCsrfValidation = false;
    public $fileActions = [
        'print-invoice',
        'print-receipt'
    ];

    public function behaviors()
    {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => ['webhook', 'gravity'],
                        'allow' => true,
                        'roles' => ['?']
                    ],
                    /*
                     * As the coach and superadmin have all right that client has
                     * we need to explicitly block access for them to this controller
                     */
                    [
                        'allow' => false,
                        'roles' => ['coach', 'superadmin'],
                    ],
                    [
                        'allow' => true,
                        'roles' => ['client'],
                    ]
                ]
            ]
        ];

    }

    public function beforeAction($action)
    {
        if (Yii::$app->session->get('invalidSubscription')) {
            $this->layout = 'minimal';
        }
        return parent::beforeAction($action);
    }

    public function actionIndex()
    {
        $userPlan = UserPlan::getLatestPlan();
        if (Yii::$app->session->get('invalidSubscription') && !$userPlan->getIsExpired()) {
            Yii::$app->session->remove('invalidSubscription');
            $this->redirect('/subscription/index');
        }

        $planFeatures = UserPlanFeature::find()
            ->where(['archived' => 0])
            ->orderBy('order')
            ->asArray()
            ->all();

        $maintenanceAllowedPlans = [
            UserPlanType::PLAN_TYPE_COACHING,
            UserPlanType::PLAN_TYPE_KICKSTART,
            UserPlanType::PLAN_TYPE_MAINTENANCE
        ];
        $plansQuery = UserPlanType::find()
            ->with('userPlanTypeDuration', 'userPlanDetails')
            ->where(['is_visible' => 1])
            ->orderBy('order ASC');
        // exclude maintenance plan
        /*if (!$userPlan || !in_array($userPlan->userPlanType->getPlanType(), $maintenanceAllowedPlans)) {
            $plansQuery->andWhere([
                'not like', 'name', 'maintenance'
            ]);
        }*/
        $plansDetails = $plansQuery->all();

        $invoices = Invoice::find()
            ->where(['user_id' => Yii::$app->user->id])
            ->orderBy(['bill_date' => SORT_DESC, 'id' => SORT_DESC])
            ->all();

        $card = null;
        $userMeta = UserMeta::findAllSorted(['user_id' => $userPlan->user_id]);
        if (!empty($userMeta['stripe_customer_id'])) {
            $stripe = new Stripe();
            $customer = $stripe->getCustomer($userMeta['stripe_customer_id']);
            if ($customer) {
                $card = $stripe->getDefaultCard($customer);
            }
        }

        $this->view->title = 'Subscriptions';
        return $this->render('manage-subscription', [
            'title' => $this->view->title,
            'planFeatures' => $planFeatures,
            'plansDetails' => $plansDetails,
            'userPlan' => $userPlan,
            'invoices' => $invoices,
            'card' => $card,
            'updateCardUrl' => Url::to(['subscription/update-card'])
        ]);
    }

    public function actionUpdateCard($token)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;
        if (!$token) {
            return [
                'status' => 'error',
                'message' => 'Token parameter missing.'
            ];
        }
        $userMeta = UserMeta::findAllSorted(['user_id' => Yii::$app->user->identity->id]);
        if (empty($userMeta['stripe_customer_id'])) {
            return [
                'status' => 'error',
                'message' => 'User data not found in Stripe.'
            ];
        }
        $stripe = new Stripe();
        $customer = $stripe->getCustomer($userMeta['stripe_customer_id']);
        if (!$customer) {
            return [
                'status' => 'error',
                'message' => 'User data not found in Stripe.'
            ];
        }
        $customer->source = $token;
        $customer->save();

        return ['status' => 'success'];
    }

    public function actionCancel()
    {
        $confirm = Yii::$app->getRequest()->getQueryParam('confirm', false);

        if ($confirm) {
            try {
                $userPlan = UserPlan::getActivePlan();
                if (!empty($userPlan->stripe_subscription_id)) {
                    $stripe = new Stripe();
                    $stripe->cancelSubscription($userPlan);
                }
                $userPlan->is_cancelled = 1;
                $userPlan->save();
                echo JsonTools::successMessage('Subscription successfully cancelled.');
                die();
            }
            catch (\Exception $e) {
                echo JsonTools::errorMessage('Could not cancel subscription.');
                die();
            }
        }

        $this->view->title = 'Cancel Subscription';
        return $this->render('cancel-subscription');
    }

    public function actionChange($id, $token)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        $currentPlan = UserPlan::getLatestPlan();
        $planVariation = UserPlanVariation::findOne($id);
        if (!$planVariation) {
            JsonTools::errorMessage('Incorrect plan selected.');
            return ['result' => false];
        }
        $userPlan = new UserPlan();
        $userPlan->setAttributes([
            'user_id' => Yii::$app->user->identity->id,
            'user_plan_type_id' => $planVariation->getUserPlanType()->one()->id,
            'user_plan_variation_id' => $planVariation->id,
            'stripe_subscription_id' => $currentPlan->stripe_subscription_id,
            'stripe_coupon_id' => $currentPlan->stripe_coupon_id,
            'start_time' => Formatter::localDatetimeToUtcDatetime(time(), 'Y-m-d H:i:s')
        ]);
        $userPlan->save();

        // change the current subscription plan in Stripe
        $stripe = new Stripe();
        $stripeResult = $stripe->changeCustomerPlan($userPlan, $planVariation->stripe_plan_id, $token);
        if (!$stripeResult || is_string($stripeResult)) {
            JsonTools::errorMessage(
                'Could not subscribe to the selected plan.' .
                (is_string($stripeResult) ? ' ' . $stripeResult : '')
            );
            return ['result' => false];
        }
        return ['result' => true];
    }

    public function actionPrintInvoice($invoice)
    {
        $invoiceObject = Invoice::find()
            ->where(['id' => $invoice, 'user_id' => Yii::$app->user->identity->id])
            ->one();
        if (!$invoiceObject) {
            throw new NotFoundHttpException();
        }

        Yii::$app->response->format = Response::FORMAT_RAW;
        Yii::$app->response->headers->add('Content-Type', 'application/pdf; charset=utf-8');
        return $invoiceObject->renderPdfInvoice();
    }

    public function actionPrintReceipt($invoice)
    {
        $invoiceObject = Invoice::find()
            ->where(['id' => $invoice, 'user_id' => Yii::$app->user->identity->id])
            ->one();
        if (!$invoiceObject) {
            throw new NotFoundHttpException();
        }

        Yii::$app->response->format = Response::FORMAT_RAW;
        Yii::$app->response->headers->add('Content-Type', 'application/pdf; charset=utf-8');
        return $invoiceObject->renderPdfReceipt();
    }

    /**
     * Stripe Webhook
     */
    public function actionWebhook()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        $input = Yii::$app->request->getBodyParams();
        if (empty($input)) {
            return ['result' => 'error - no input'];
        }

        $result = 'not handled general';
        $eventData = $input['data']['object'];
        $stripe = new Stripe();
        switch ($input['type']) {
            case 'customer.subscription.created':
                $userPlan = UserPlan::getByStripeSubscriptionID($eventData['id']);
                if ($userPlan) {
                    if (substr($userPlan->valid_until, 0, 10) != date('Y-m-d', $eventData['current_period_end'])) {
                        $userPlan->updateAttributes([
                            'valid_until' => date('Y-m-d 23:59:59', strtotime('+' . Stripe::GRACE_PERIOD_DAYS . ' days', $eventData['current_period_end']))
                        ]);
                        //$userPlan->user->sharpSpringChangedPlan($userPlan);
                        $userPlan->user->updateMembershipLevel($userPlan);
                    }
                    $result = 'ok';
                }
                else {
                    $userPlan = UserPlan::getByStripeCustomerID($eventData['customer'], $eventData['items']['data'], $eventData['id']);
                    if ($userPlan) {
                        //$userPlan->user->sharpSpringChangedPlan($userPlan);
                        $userPlan->user->updateMembershipLevel($userPlan);
                        $result = 'ok';
                    }
                    else {
                        Yii::$app->response->statusCode = 500;
                        $result = 'error';
                    }
                }
                if ($userPlan && $userPlan->stripe_subscription_id && !$userPlan->user_plan_type_id && !$userPlan->user_plan_variation_id) {
                    $userPlan->updatePlanFromStripe($eventData);
                }
                break;
            case 'customer.subscription.updated':
                $userPlan = UserPlan::getByStripeSubscriptionID($eventData['id']);
                if ($userPlan && $userPlan->updatePlanFromStripe($eventData)) {
                    //$userPlan->user->sharpSpringChangedPlan($userPlan);
                    $userPlan->user->updateMembershipLevel($userPlan);
                    $result = 'ok';
                }
                else {
                    Yii::$app->response->statusCode = 500;
                    $result = 'error';
                }
                break;
            case 'customer.subscription.deleted':
                if ($eventData['cancel_at_period_end']) {
                    $result = 'this type of cancellation is not handled';
                }
                else {
                    $userPlan = UserPlan::getByStripeSubscriptionID($eventData['id']);
                    if ($userPlan && substr($userPlan->valid_until, 0, 10) >= date('Y-m-d')) {
                        $userPlan->updateAttributes([
                            'valid_until' => date('Y-m-d H:i:s', $eventData['canceled_at'])
                        ]);
                        $userPlan->user->removeMembershipLevel($userPlan);
                        $result = 'ok';
                    }
                    else {
                        Yii::$app->response->statusCode = 500;
                        $result = 'error';
                    }
                }
                break;
            case 'charge.succeeded':
                $userPlan = UserPlan::getByStripeChargeID($eventData['id']);
                if ($userPlan && $userPlan->userPlanVariation) {
                    if ($eventData['paid']) {
                        // upfront payment should be always coupled with a subscription with an optional trial period
                        if (empty($userPlan->userPlanVariation->upfront_payment)) {
                            $planLength = $userPlan->userPlanVariation->plan_length;
                            if ($planLength > 0) {
                                $userPlan->updateAttributes([
                                    'valid_until' => Formatter::localDatetimeToUtcDatetime(
                                        strtotime(UserPlanVariation::getDaysMonths($planLength, $userPlan->userPlanVariation->plan_length_in_weeks)), 'Y-m-d 23:59:59'
                                    )
                                ]);
                            }
                        }
                        $invoice = Invoice::generateChargeInvoice($userPlan, $eventData);
                        if ($invoice) {
                            // send the receipt
                            $invoice->sendEmail(false);
                            $result = 'ok';
                        }
                        else {
                            Yii::$app->response->statusCode = 500;
                            $result = 'error - invoice not created';
                        }
                    }
                    else {
                        Yii::$app->response->statusCode = 500;
                        $result = 'error - not paid';
                    }
                }
                else {
                    Yii::$app->response->statusCode = 500;
                    $result = 'error - charge ID not found';
                }
                break;
            case 'invoice.upcoming':
                if (!empty($eventData['lines']['data']['id'])) {
                    $userPlan = UserPlan::getByStripeSubscriptionID($eventData['lines']['data']['id']);
                    if ($userPlan) {
                        // send the upcoming invoice event
                        $userPlan->sendUpcomingEmail($eventData['amount_due']);
                        $result = 'ok';
                    }
                    else {
                        Yii::$app->response->statusCode = 500;
                        $result = 'error - user plan not found';
                    }
                }
                else {
                    Yii::$app->response->statusCode = 500;
                    $result = 'error - subscription ID not found';
                }
                break;
            case 'invoice.created':
                if ($stripe->createInvoice($eventData)) {
                    $result = 'ok';
                }
                else {
                    Yii::$app->response->statusCode = 500;
                    $result = 'error';
                }
                break;
            case 'invoice.payment_succeeded':
                $invoice = Invoice::findOne(['stripe_invoice_id' => $eventData['id']]);
                if ($invoice && $stripe->invoicePaid($eventData, $invoice)) {
                    if (count($invoice->invoiceItems) > 0) {
                        $userPlan = $invoice->invoiceItems[0]->userPlan;
                        if ($userPlan->userPlanVariation) {
                            GoogleAnalytics::collectGAEcommerceData(
                                $invoice->id,
                                $userPlan->userPlanVariation->name,
                                $userPlan->userPlanVariation->price,
                                $invoice->user_id
                            );
                            $result = 'ok';
                        }
                        else {
                            Yii::$app->response->statusCode = 500;
                            $result = 'error - no user plan';
                        }
                    }
                }
                else {
                    Yii::$app->response->statusCode = 500;
                    $result = 'error - no invoice';
                }
                break;
            case 'invoiceitem.created':
                // handle only items with positive amount
                if ($eventData['amount'] > 0) {
                    $userPlan = UserPlan::getByStripeSubscriptionID($eventData['subscription']);
                    if (!$userPlan || !$userPlan->userPlanVariation) {
                        Yii::$app->response->statusCode = 500;
                        $result = 'error';
                    }
                    else {
                        $invoice = $userPlan->getLatestInvoice(false);
                        if (!$invoice) {
                            $invoice = $stripe->createInvoice($eventData);
                        }
                        if ($stripe->invoicePaid($eventData, $invoice)) {
                            GoogleAnalytics::collectGAEcommerceData(
                                $invoice->id,
                                $userPlan->userPlanVariation->name,
                                $userPlan->userPlanVariation->price,
                                $invoice->user_id
                            );
                            $result = 'ok';
                        }
                        else {
                            Yii::$app->response->statusCode = 500;
                            $result = 'error';
                        }
                    }
                }
                else {
                    $result = 'error - negative amount';
                }
                break;
            default:
                $result = 'not handled';
                break;
        }
        return [
            'result' => $result
        ];
    }

    /**
     * Gravity Forms webhook
     *
     * {
        "id":"115",
        "form_id":"8",
        "post_id":null,
        "date_created":"2018-06-27 14:07:18",
        "date_updated":null,
        "is_starred":"0",
        "is_read":"0",
        "ip":"46.101.250.164",
        "source_url":"https:\/\/www.grandmasjars.com\/?gf_page=preview&id=8",
        "user_agent":"Mozilla\/5.0 (X11; Linux x86_64) AppleWebKit\/537.36 (KHTML, like Gecko) Chrome\/67.0.3396.99 Safari\/537.36",
        "currency":"AUD",
        "payment_status":"Active",
        "payment_date":"2018-06-27 14:07:18",
        "payment_amount":"10.00",
        "payment_method":"",
        "transaction_id":"sub_D7u6d1Ze7yOTZ8",
        "is_fulfilled":"1",
        "created_by":"1",
        "transaction_type":"2",
        "status":"active",
        "6.2":"",
        "6.3":"Test",
        "6.4":"",
        "6.6":"Customer",
        "6.8":"",
        "5":"bartek@procreative.eu",
        "7":"Third Choice|10",
        "9.1":"XXXXXXXXXXXX4242",
        "9.4":"Visa"
        }
     */
    public function actionGravity()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        $input = Yii::$app->request->getBodyParams();
        if (empty($input)) {
            return ['result' => 'error - no input'];
        }

        $email = null;
        foreach ($input as $key => $val) {
            if (filter_var($val, FILTER_VALIDATE_EMAIL)) {
                $email = $val;
                break;
            }
        }
        $transactionId = !empty($input['transaction_id']) ? $input['transaction_id'] : null;
        $paymentAmount = !empty($input['payment_amount']) ? $input['payment_amount'] : null;

        if (!$email || !$transactionId || !$paymentAmount) {
            return ['result' => 'error - missing obligatory fields'];
        }

        $oneOffPayment = strpos($transactionId, 'ch_') === 0;

        if ($oneOffPayment) {
            // scenario is based on the amount paid
            $paymentAmount = (int)($paymentAmount * 100);
            $scenario = 'gravity-' . $paymentAmount;
            $planVariation = UserPlanType::getPlanVariationObject($scenario);
            if (!$planVariation) {
                return ['result' => 'error - cannot find matching user plan for scenario: ' . $scenario];
            }
        }

        $user = new User();
        $user->user_role_id = UserRole::CLIENT;
        $user->email = $email;
        $user->click_funnels_data = json_encode($input);
        // fake data for now
        $user->username = Yii::$app->security->generateRandomString();
        $user->password = 'no-password';
        $user->password_salt = 'no-password';
        $user->save();

        $userPlan = new UserPlan();
        $userPlan->user_id = $user->id;
        if ($oneOffPayment) {
            $userPlan->user_plan_type_id = $planVariation->user_plan_type_id;
            $userPlan->user_plan_variation_id = $planVariation->id;
            $userPlan->stripe_charge_id = $transactionId;
        }
        else {
            $userPlan->stripe_subscription_id = $transactionId;
        }
        $userPlan->start_time = date('Y-m-d 00:00:00');
        $userPlan->valid_until = date('Y-m-d 23:59:59', strtotime('+1 day'));
        $userPlan->save();

        // send email to the user
        $authToken = Yii::$app->security->generateRandomString(32) . time();
        $request = new UserPasswordResetRequest();
        $request->user_id = $user->id;
        $request->auth_token = $authToken;
        $request->source = 'gravity-forms';
        $request->save();

        // first make sure the user exists in Drip
        DripHelper::addSubscriber($user->email, $user->id);
        // send email
        DripHelper::sendEmailEvent(
            'setup',
            $user->email,
            'Action Required: Set Up Your Account',
            [
                'email' => $user->email,
                'auth_url' => $request->getPasswordResetUrl()
            ]
        );
        return ['result' => 'success'];
    }

}
