<?php

namespace backend\controllers;

use backend\models\db\Eoms;
use Yii;

use yii\filters\AccessControl;

use yii\helpers\Url;

use backend\components\CustomController;

use backend\models\db\Account;
use backend\models\db\Transfer;
use backend\models\db\TransferEvent;

use backend\components\helpers\DataTables;
use backend\components\helpers\Formatter;
use backend\components\helpers\JsonTools;


/**
 * Transactions controller
 */
class TransferEventsController extends CustomController {

    public $enableCsrfValidation = false;

    public function behaviors() {

        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => [
                            'transfer-events-table',
                            'account-transfer-events-table',
                            'account-transfer-events-table-checked'
                        ],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ]
        ];

    }

    /**
     * Data for transfers table on account page
     */
    public function actionAccountTransferEventsTable() {

        $accountId = Yii::$app->getRequest()->getQueryParam('account_id', false);
        $account = Account::findOne($accountId);

        // check if user can edit jar
        if (!Yii::$app->user->can('accessBudget', ['budgetId' => $account->budget_id])) {

            return $this->accessError();

        }

        // filtering and getting record
        $tableColumns = [
            /*[
                'type' => 'normal',
                'as' => 'checked',
                'select' => Transfer::tableName() . '.checked'
            ],*/
            [
                'type' => 'normal',
                'as' => 'date',
                'select' => Transfer::tableName() . '.date'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => Transfer::tableName() . '.amount'
            ],
            [
                'type' => 'normal',
                'as' => 'account_from_name',
                'select' => 'a_from.name'
            ],
            [
                'type' => 'normal',
                'as' => 'account_to_name',
                'select' => 'a_to.name'
            ]
        ];

        $countFiltered = TransferEvent::getFilteredTransferEventsCount($account->id);
        $selectFiltered = TransferEvent::getFilteredTransferEventsSelect($account->id);
        $responseData = DataTables::getResponseData($tableColumns, $countFiltered, $selectFiltered);

        foreach ($responseData['items'] as $i) {
            $actionClass = get_class($i);
            $isEditable = Eoms::isTransactionChangeable($this->activeBudgetId, $i->date);

            $row = [
                'isEditable' => $isEditable
            ];
            if ($isEditable) {
                $row['id'] = $i->id;
                $row['type'] = $actionClass;
                $row['editUrl'] = '/money-operations/edit-transaction';
                $row['deleteUrl'] = Url::toRoute(['/money-operations/delete-transaction', 'transaction_id' => $i->id, 'transaction_type' => $actionClass, 'js_callback' => 'reloadAccountsAfterTransferEventDelete']);
                $row['callback'] = 'reloadAccountsAfterTransferEventDelete';
            }
            $bankStatType = "te";
            $bankStatName = $bankStatType . $i->id;

            $columns = [
                /*[
                    'key' => 'checked',
                    'type' => 'checkbox',
                    'value' => $i->checked,
                    'cId' => $i->id,
                    'cType' => $bankStatType,
                    'cName' => $bankStatName,
                    'isEditable' => false
                ],*/ [
                    'key' => 'date',
                    'type' => 'date',
                    'value' => $i->date,
                    'isEditable' => true
                ], [
                    'key' => 'amount',
                    'type' => 'currency',
                    'value' => $i->amount,
                    'isEditable' => true
                ], [
                    'key' => 'account_from_name',
                    'type' => 'text',
                    'value' => $i->account_from_name
                ], [
                    'key' => 'account_to_name',
                    'type' => 'text',
                    'value' => $i->account_to_name
                ]
            ];

            $responseData['response']['data'][] = DataTables::getRow($row, $columns);
        }

        echo json_encode($responseData['response']);
        die();

    }


    /**
     * Data for transfer events table on transfers page
     */
    public function actionTransferEventsTable()
    {
        // filtering and getting record
        $tableColumns = [
            [
                'type' => 'normal',
                'as' => 'date',
                'select' => TransferEvent::tableName() . '.date'
            ],
            [
                'type' => 'normal',
                'as' => 'name',
                'select' => Transfer::tableName() . '.name'
            ],
            [
                'type' => 'normal',
                'as' => 'amount',
                'select' => TransferEvent::tableName() . '.amount'
            ],
            [
                'type' => 'normal',
                'as' => 'account_from_name',
                'select' => 'fa.name'
            ],
            [
                'type' => 'normal',
                'as' => 'account_to_name',
                'select' => 'ta.name'
            ]
        ];

        $year   = Yii::$app->getRequest()->getQueryParam('year', date('Y'));
        $month  = Yii::$app->getRequest()->getQueryParam('month', date('m'));

        $countFiltered = TransferEvent::find()
            ->select([
                Transfer::tableName() . '.name',
                TransferEvent::tableName() . '.*',
                'fa.name AS account_from_name',
                'ta.name AS account_to_name'
            ])
            ->joinWith('transfer')
            ->joinWith('transfer.accountFrom fa')
            ->joinWith('transfer.accountTo ta')
            ->where([
                'and',
                ['Month(' . TransferEvent::tableName() . '.date)' => $month],
                ['Year(' . TransferEvent::tableName() . '.date)' => $year],
                [Transfer::tableName() . '.budget_id' => $this->activeBudgetId],
                ['is_adjustment' => 0]
            ]);
        $selectFiltered = TransferEvent::find()
            ->select([
                Transfer::tableName() . '.name',
                TransferEvent::tableName() . '.*',
                'fa.name AS account_from_name',
                'ta.name AS account_to_name'
            ])
            ->joinWith('transfer')
            ->joinWith('transfer.accountFrom fa')
            ->joinWith('transfer.accountTo ta')
            ->where([
                'and',
                ['Month(' . TransferEvent::tableName() . '.date)' => $month],
                ['Year(' . TransferEvent::tableName() . '.date)' => $year],
                [Transfer::tableName() . '.budget_id' => $this->activeBudgetId],
                ['is_adjustment' => 0]
            ]);
        $responseData = DataTables::getResponseData($tableColumns, $countFiltered, $selectFiltered);

        foreach ($responseData['items'] as $i) {
            $actionClass = get_class($i);
            $isEditable = Eoms::isTransactionChangeable($this->activeBudgetId, $i->date);

            $row = [
                'isEditable' => $isEditable
            ];
            if ($isEditable) {
                $row['id'] = $i->id;
                $row['type'] = $actionClass;
                $row['editUrl'] = '/money-operations/edit-transaction';
                $row['deleteUrl'] = Url::toRoute(['/money-operations/delete-transaction', 'transaction_id' => $i->id, 'transaction_type' => $actionClass, 'js_callback' => 'reloadTransfersAfterTransferEventDeleted']);
                $row['callback'] = 'reloadTransfersAfterTransferEventDeleted';
            }

            $columns = [
                [
                    'key' => 'date',
                    'type' => 'date',
                    'value' => $i->date,
                    'isEditable' => true
                ], [
                    'key' => 'name',
                    'type' => 'text',
                    'value' => $i->transfer->name,
                    'isEditable' => false
                ], [
                    'key' => 'amount',
                    'type' => 'currency',
                    'value' => $i->amount,
                    'isEditable' => true
                ], [
                    'key' => 'from_account',
                    'type' => 'text',
                    'value' => $i->transfer->accountFrom->name,
                    'isEditable' => false
                ], [
                    'key' => 'to_account',
                    'type' => 'text',
                    'value' => $i->transfer->accountTo->name,
                    'isEditable' => false
                ]
            ];
            $responseData['response']['data'][] = DataTables::getRow($row, $columns);
        }
        echo json_encode($responseData['response']);
        die();
    }

}
