<?php

namespace backend\models\db;

use backend\components\helpers\Formatter;
use yii\helpers\Html;


/**
 * This is the model class for table "eoms".
 *
 * @property integer $id
 * @property integer $budget_id
 * @property integer $month
 * @property integer $year
 * @property integer $submitted
 * @property string $submitted_at
 * @property integer $cc_pay_off_option
 *
 * @property AccountBalance[] $accountBalances
 * @property AccountEomsBalance[] $accountEomsBalances
 * @property DebtAmount[] $debtAmounts
 * @property Budget $budget
 */
class Eoms extends \yii\db\ActiveRecord
{
    const EOMS_OVERDUE_TIME = 1814400; // 3 weeks in seconds
    const LOCK_WARNING_TIME = 345600; // 4 days in seconds

    const CC_PAY_OFF_NOT_EVERY_MONTH = 1;
    const CC_PAY_OFF_EVERY_MONTH = 2;

    public $user_id;


    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'eoms';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['budget_id', 'month', 'year'], 'required'],
            [['budget_id', 'month', 'year'], 'integer']
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'budget_id' => 'Budget ID',
            'month' => 'Month',
            'year' => 'Year',
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccountBalances()
    {
        return $this->hasMany(AccountBalance::className(), ['eoms_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccountEomsBalances()
    {
        return $this->hasMany(AccountEomsBalance::className(), ['eoms_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getDebtAmounts()
    {
        return $this->hasMany(DebtAmount::className(), ['eoms_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getJarAmounts()
    {
        return $this->hasMany(JarAmount::className(), ['eoms_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getBudget()
    {
        return $this->hasOne(Budget::className(), ['id' => 'budget_id']);
    }


    public function getSubmittedAccountBalances($ccAccounts = false, $accountId = null)
    {
        $query = AccountBalance::find()
            ->joinWith('account')
            ->joinWith('account.budget')
            ->where([
                'and',
                ['eoms_id' => $this->id],
                [$ccAccounts ? '=' : '!=', 'account_type_id', 2],
                ['is_mock' => 0]
            ])
            ->orderBy('id asc');
        if ($accountId) {
            return $query
                ->andWhere([
                    'account_id' => $accountId
                ])
                ->one();
        }
        return $query->all();
    }

    public function getSubmittedDebtAmounts($ccDebts = false, $debtId = null)
    {
        $query = DebtAmount::find()
            ->with('debt')
            ->joinWith('debt')
            ->joinWith('debt.budget')
            ->where([
                'and',
                // [DebtAmount::tableName() . '.budget_id' => $this->activeBudgetId],
                ['eoms_id' => $this->id],
                [$ccDebts ? '=' : '!=', Debt::tableName() . '.debt_type_id', 1],
                ['is_mock' => 0]
            ])
            ->orderBy(DebtAmount::tableName() . '.id asc');
        if ($debtId) {
            return $query
                ->andWhere([
                    'debt_id' => $debtId
                ])
                ->one();
        }
        return $query->all();
    }

    public function getSummaryData()
    {
        /** @var Budget $budget */
        $budget = $this->getBudget()->one();

        // get the accounts balance
        $accountEomsBalances = $this->getAccountEomsBalances()->all();
        $accountsEomsBalance = 0;
        foreach ($accountEomsBalances as $a) {
            $accountsEomsBalance += $a->balance;
        }

        // get CC debts amount for this month
        $timeNow = strtotime("{$this->year}-{$this->month}-20");
        $prevEomsMonth = date('m', strtotime('-1 month', $timeNow));
        $prevEomsYear  = date('Y', strtotime('-1 month', $timeNow));
        $ccDebtTransactionsAmount = 0; $ccActualBalanceTotal = 0;
        $ccActiveDebts = $budget->getMonthsActiveCcDebts($this->month, $this->year);
        /** @var Account $a */
        foreach ($ccActiveDebts as $a) {
            // cc spending = actual balance - starting balance
            // (should be positive if money was spent)
            $submittedAmount = $this->getSubmittedAccountBalances(true, $a->id);
            $startingBalance = Formatter::float($a->getCurrentBalance($prevEomsMonth, $prevEomsYear));
            $actualBalance   = Formatter::float($a->getCurrentBalance($this->month, $this->year));
            $spending        = Formatter::float($actualBalance - $startingBalance);

            // active CCs has debt represented as a negative value
            $ccActualBalanceTotal -= $submittedAmount ? $submittedAmount->balance : $actualBalance;
            // skip CCs which either had positive spending (money paid off) but the balance is still negative
            // or the actual balance is 0 (CC fully paid off)
            if ($spending > 0 && $actualBalance < 0 || $actualBalance == 0) {
                continue;
            }
            $ccDebtTransactionsAmount -= $spending;
        }

        // get everyday spendings for next month
        $jar = Jar::findOne(['type' => Jar::EVERYDAY_JAR_TYPE, 'budget_id' => $this->budget_id]);
        $month = $this->month + 1;
        if ($month > 12) {
            $month = 1;
            $year = $this->year + 1;
        }
        else {
            $year = $this->year;
        }
        $jarAmount = JarAmount::find()
            ->where([
                'and',
                ['month' => (int)$month],
                ['year' => (int)$year],
                ['jar_id' => $jar->id],
                ['is not', 'monthly_budget_adjustment', null]
            ])
            ->one();
        // check if there's override (used only for the EoMS calculations)
        $nextMonthsEverydayAmount = $jarAmount ?
            $jarAmount->monthly_budget_adjustment : $jar->getMonthlyBudget($month, $year);

        // get jar funds remaining
        $remainingJarAmount = $budget->getMonthsFundsRemainingAmount($this->month, $this->year);

        if ($this->cc_pay_off_option == self::CC_PAY_OFF_NOT_EVERY_MONTH) {
            $equationResult = $accountsEomsBalance - $ccDebtTransactionsAmount -
                $nextMonthsEverydayAmount - $remainingJarAmount;
        }
        else {
            $equationResult = $accountsEomsBalance - $ccActualBalanceTotal -
                $nextMonthsEverydayAmount - $remainingJarAmount;
        }

        return [
            'accountsEomsBalance' => $accountsEomsBalance,
            'ccDebtTransactionsAmount' => $ccDebtTransactionsAmount,
            'ccActualBalanceTotal' => $ccActualBalanceTotal,
            'nextMonthsEverydayAmount' => $nextMonthsEverydayAmount,
            'remainingJarAmount' => $remainingJarAmount,
            'equationResult' => $equationResult
        ];
    }

    public function calculateSummaryData()
    {
        $eomsData = $this->getSummaryData();
        $eomsData['ccAmount'] = $this->cc_pay_off_option == self::CC_PAY_OFF_NOT_EVERY_MONTH ?
            $eomsData['ccDebtTransactionsAmount'] : $eomsData['ccActualBalanceTotal'];
        $eomsData['equationResult'] = $eomsData['accountsEomsBalance'] - $eomsData['ccAmount']
            - $eomsData['nextMonthsEverydayAmount'] - $eomsData['remainingJarAmount'];
        return $eomsData;
    }

    public function getSummaryAmount()
    {
        $data = $this->calculateSummaryData();
        return $data['equationResult'];
    }

    public function runMonthlyBudgetUpdate()
    {
        // update account balances
        $accountBalances = $this->getAccountBalances()
            ->joinWith('account.budget')
            ->where(['is_mock' => 0])
            ->all();
        foreach ($accountBalances as $a) {
            $account = $a->getAccount()
                ->one();
            $account->balance = $a->balance;
            $account->save();
        }

        // update debts
        $debtAmounts = $this->getDebtAmounts()
            ->joinWith('debt.budget')
            ->where(['is_mock' => 0])
            ->all();
        foreach ($debtAmounts as $d) {
            $debt = $d->getDebt()
                ->one();
            $debt->amount = $d->amount;
            $debt->save();
        }

        // update jar amounts
        $jarAmounts = $this->getJarAmounts()
            ->joinWith('jar.budget')
            ->where(['is_mock' => 0])
            ->all();
        foreach ($jarAmounts as $j) {
            $jar = $j->getJar()
                ->one();
            $jar->amount = $j->funds_brought_forward;
            $jar->save();
        }
        return true;
    }

    /**
     * Checks if the EoMS is overdue (more than 3 weeks)
     * so we should force the submission
     * @return bool
     */
    public function isOverdue()
    {
        if (time() - strtotime(date('Y-m-t 23:59:59', strtotime("{$this->year}-{$this->month}-20"))) >= self::EOMS_OVERDUE_TIME) {
            return true;
        }
        return false;
    }

    public function showLockWarning()
    {
        $eomsDueTime = strtotime(date('Y-m-t 23:59:59', strtotime("{$this->year}-{$this->month}-20")));
        $eomsOverdueTime = $eomsDueTime + self::EOMS_OVERDUE_TIME;
        $eomsWarningTime = $eomsOverdueTime - self::LOCK_WARNING_TIME;
        return time() >= $eomsWarningTime && time() <= $eomsOverdueTime ?
            ceil(($eomsOverdueTime - time()) / 86400) :
            false;
    }

    public function getWarningText()
    {
        $daysLeft = $this->showLockWarning();
        return
            'Your ' . Html::a('EoMS for ' . date('F Y', strtotime("{$this->year}-{$this->month}-20")), ['/budget/end-of-month-summary']) .
            ' has not been set yet! It should be completed as soon as possible.' .
            ($daysLeft ? '<br/><br/>' .
                '<strong>The previous month\'s records will lock in ' . $daysLeft . ' days and you will not be able to add any further transactions to it.<br/>' .
                'Please make sure your previous month has been finalized.</strong>' : '');
    }

    /**
     * Get the most relevant unsubmitted EoMS record (if any)
     * @param int $activeBudgetId
     * @param bool $unsubmitted
     * @return Eoms|null
     */
    public static function getLastEoms($activeBudgetId, $unsubmitted = true)
    {
        return $unsubmitted ?
            Eoms::find()
                ->where([
                    'budget_id' => $activeBudgetId,
                    'submitted' => 0
                ])
                ->orderBy('year asc, month asc')
                ->one() :
            Eoms::find()
                ->where([
                    'budget_id' => $activeBudgetId,
                    'submitted' => 1
                ])
                ->orderBy('year desc, month desc')
                ->one();
    }

    /**
     * Checks if the transaction can be edited/deleted or not
     * @param $budgetId
     * @param $transactionDate
     * @return bool
     */
    public static function isTransactionChangeable($budgetId, $transactionDate)
    {
        $lastSubmittedEoms = Eoms::getLastEoms($budgetId, false);
        return !$lastSubmittedEoms ||
            $transactionDate > date('Y-m-t', strtotime("{$lastSubmittedEoms->year}-{$lastSubmittedEoms->month}-20"));
    }

    /**
     * Checks if the entry (expense/income/transfer/debt/account) can be hard deleted or not
     * @param int $budgetId
     * @param $entry
     * @return bool
     */
    public static function isEntryHardDeletable($budgetId, $entry)
    {
        $changeRecord = null;
        switch (get_class($entry)) {
            case 'backend\models\db\Expense':
                $changeRecord = ExpenseChange::find()
                    ->where([
                        'expense_id' => $entry->id,
                        'type' => 'C'
                    ])
                    ->one();
                break;
            case 'backend\models\db\Income':
                $changeRecord = IncomeChange::find()
                    ->where([
                        'income_id' => $entry->id,
                        'type' => 'C'
                    ])
                    ->one();
                break;
            case 'backend\models\db\Transfer':
                $changeRecord = TransferChange::find()
                    ->where([
                        'transfer_id' => $entry->id,
                        'type' => 'C'
                    ])
                    ->one();
                break;
            case 'backend\models\db\Account':
                $changeRecord = AccountChange::find()
                    ->where([
                        'account_id' => $entry->id,
                        'type' => 'C'
                    ])
                    ->one();
                break;
            case 'backend\models\db\Debt':
                $changeRecord = DebtChange::find()
                    ->where([
                        'debt_id' => $entry->id,
                        'type' => 'C'
                    ])
                    ->one();
                break;
        }
        if ($changeRecord) {
            $entryDate = substr($changeRecord->time, 0, 10);
        }
        else {
            $entryDate = date('Y-m-d');
        }
        return self::isTransactionChangeable($budgetId, $entryDate);
    }

    /**
     * Checks if EoMS records for all previous months
     * are created and if not, creates them
     * @param $budgetId
     * @param $todayDate
     * @return bool|array
     */
    public static function fixMissingEoms($budgetId, $todayDate)
    {
        $budget = Budget::findOne($budgetId);
        if (!$budget || $budget->is_mock || $budget->user->user_role_id != UserRole::CLIENT) {
            return [];
        }
        $budgetTime         = strtotime($budget->created_time);
        $moneyOperationTime = strtotime($budget->getFirstMoneyOperationDate() . ' 00:00:00');
        $lastEoms           = Eoms::getLastEoms($budgetId, false);
        if ($lastEoms) {
            $eomsTime    = strtotime(date("{$lastEoms->year}-{$lastEoms->month}-t 23:59:59"));
            $currentTime = $eomsTime > $moneyOperationTime ? $eomsTime : $moneyOperationTime;
        }
        // make sure it's not earlier than budget creation time
        elseif ($budgetTime > $moneyOperationTime) {
            $currentTime = $budgetTime;
        }
        else {
            $currentTime = $moneyOperationTime;
        }
        $eomsAdded = [];
        $currentMonth = date('m', $currentTime);
        $currentYear  = date('Y', $currentTime);
        $today        = date('Y-m', strtotime($todayDate));
        while ("$currentYear-$currentMonth" < $today) {
            if (!Eoms::find()->where([
                'budget_id' => $budgetId,
                'month' => (int)$currentMonth,
                'year' => $currentYear
            ])->exists()) {
                $eoms = new Eoms();
                $eoms->setAttributes([
                    'budget_id' => $budgetId,
                    'month' => (int)$currentMonth,
                    'year' => $currentYear
                ]);
                $eoms->save();
                $eomsAdded[] = $eoms->attributes;
            }
            $currentTime  = strtotime('+1 month', strtotime(date('Y-m-20', $currentTime)));
            $currentMonth = date('m', $currentTime);
            $currentYear  = date('Y', $currentTime);
        }
        return $eomsAdded;
    }

    public function getIsLastEoMS()
    {
        $eoms = Eoms::getLastEoms($this->budget_id, false);
        return $eoms->id == $this->id;
    }

    /**
     * Returns true if we need to force the EoMS
     * @param integer $activeBudgetId
     * @param integer $mainUserId
     * @return bool|null
     */
    public static function checkForceEoms($activeBudgetId, $mainUserId)
    {
        $budget = Budget::findOne($activeBudgetId);
        if (!$budget || $budget->user->user_role_id != UserRole::CLIENT) {
            return null;
        }
        $moneyOperationTime = strtotime($budget->getFirstMoneyOperationDate() . ' 00:00:00');
        $isEndOfMonth = (date('t') - date('d')) <= 0;
        $eoms = Eoms::find()
            ->where(['budget_id' => $activeBudgetId])
            ->orderBy('id desc')
            ->one();

        // check for last EoMS
        if (!$isEndOfMonth) {
            // find previous user login and see whether there was an EOM completed;
            $prevMonthLogin = UserLogin::find()
                ->joinWith('user.userParent')
                ->where([
                    'AND',
                    [
                        'OR',
                        ['!=', 'MONTH(' . UserLogin::tableName() . '.time)', date('m')],
                        ['!=' ,'YEAR(' . UserLogin::tableName() . '.time)', date('Y')]
                    ],
                    [
                        'OR',
                        [UserLogin::tableName() . '.user_id' => $mainUserId],
                        ['user_parent_id' => $mainUserId]
                    ]
                ])
                ->orderBy('time desc')
                ->one();

            if ($prevMonthLogin) {
                $prevLoginYear = date('Y', strtotime($prevMonthLogin->time));
                $prevLoginMonth = date('m', strtotime($prevMonthLogin->time));
                $prevMonthEoms = Eoms::find()
                    ->where([
                        'AND',
                        ['budget_id' => $activeBudgetId],
                        ['year'      => $prevLoginYear],
                        ['month'     => $prevLoginMonth]
                    ])
                    ->orderBy('id desc')
                    ->one();
                // create EoMS only if there were transactions during the month
                if (!$prevMonthEoms && $moneyOperationTime <= strtotime(date("{$prevLoginYear}-{$prevLoginMonth}-t 23:59:59"))) {
                    $eoms = new Eoms();
                    $eoms->setAttributes([
                        'budget_id' => $activeBudgetId,
                        'year'      => $prevLoginYear,
                        'month'     => $prevLoginMonth
                    ]);
                    $eoms->save();
                }
            }
        }

        // check for current EoMS
        $isThisMonthsEoms = $eoms ? ($eoms->year == date('Y')) && ($eoms->month == date('m')) : false;
        // create EoMS only if there were transactions during the month
        if ($isEndOfMonth && !$isThisMonthsEoms && $moneyOperationTime <= strtotime(date("Y-m-t 23:59:59"))) {
            if (!Eoms::find()->where([
                'budget_id' => $activeBudgetId,
                'month' => date('m'),
                'year' => date('Y')
            ])->exists()) {
                $eoms = new Eoms();
                $eoms->setAttributes([
                    'budget_id' => $activeBudgetId,
                    'month' => date('m'),
                    'year' => date('Y')
                ]);
                $eoms->save();
            }
        }

        if ($eoms && $eoms->submitted != 1) {
            return true;
        }
        return null;
    }

}
