<?php

namespace backend\models\db;
use Yii;
use yii\caching\TagDependency;


/**
 * This is the model class for table "income_transactions".
 *
 * @property integer $id
 * @property integer $income_id
 * @property integer $account_id
 * @property double $amount
 * @property string $date
 * @property string $description
 *
 * @property Income $income
 * @property Account $account
 */
class IncomeTransaction extends \yii\db\ActiveRecord
{
    public $name;
    public $account_name;
    public $income_name;
    public $parent_name;
    public $jar_id;
    public $jar_name;
    public $is_adjustment;

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'income_transactions';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['income_id', 'account_id', 'checked'], 'integer'],
            ['amount', 'number'],
            ['date', 'safe'],
            ['description', 'string', 'max' => 255],

            [['account_id', 'amount', 'date'], 'required', 'on' => 'api'],
            [['description', 'jar_id', 'income_id', 'checked'], 'safe', 'on' => 'api']
        ];
    }

    public function fields()
    {
        $fields = parent::fields();

        $fields['income_name'] = function($model) { return $model->income_name ? $model->income_name : $model->name; };
        $fields['account_name'] = function($model) { return $model->account_name; };
        $fields['jar_id'] = function($model) {
            if ($model->jar_id) {
                return $model->jar_id;
            }
            $jar = $model->income->jar;
            if ($jar) {
                return $jar->id;
            }
            return null;
        };
        $fields['jar_name'] = function($model) {
            if ($model->jar_name) {
                return $model->jar_name;
            }
            $jar = $model->income->jar;
            if ($jar) {
                return $jar->name;
            }
            return null;
        };

        return $fields;
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'income_id' => 'Income ID',
            'account_id' => 'Account ID',
            'amount' => 'Amount',
            'date' => 'Date',
            'checked' => 'Checked'
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getIncome()
    {
        return $this->hasOne(Income::className(), ['id' => 'income_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAccount()
    {
        return $this->hasOne(Account::className(), ['id' => 'account_id']);
    }

    /**
     * @inheritdoc
     */
    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);

        if (!empty($this->income->jar_id)) {
            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $this->income->jar_id);
        }
    }

    /**
     * @inheritdoc
     */
    public function afterDelete()
    {
        if ($this->income->jar_id) {
            TagDependency::invalidate(Yii::$app->cache, 'jarFundsRemaining-' . $this->income->jar_id);
        }
        parent::afterDelete();
    }


    public static function getFilteredIncomeTransactionsCount($accountId, $includeAdjustments = true)
    {
        $query = IncomeTransaction::find()
            ->joinWith('income')
            ->where([
                'and',
                [IncomeTransaction::tableName() . '.account_id' => $accountId],
                ['MONTH(' . IncomeTransaction::tableName() . '.date)' => date('m')],
                ['YEAR(' . IncomeTransaction::tableName() . '.date)' => date('Y')]
            ]);
        if ($includeAdjustments) {
            $query->andWhere([
                'or',
                ['is_adjustment' => 0],
                ['is_adjustment' => Income::INCOME_ADJUSTMENT]
            ]);
        }
        else {
            $query->andWhere([
                'is_adjustment' => 0
            ]);
        }

        return $query;
    }

    public static function getFilteredIncomeTransactionsSelect($accountId, $includeAdjustments = true)
    {
        $query = IncomeTransaction::find()
            ->select(IncomeTransaction::tableName() . '.*, ' . Income::tableName() . '.name')
            ->joinWith('income')
            ->where([
                'and',
                [IncomeTransaction::tableName() . '.account_id' => $accountId],
                ['MONTH(' . IncomeTransaction::tableName() . '.date)' => date('m')],
                ['YEAR(' . IncomeTransaction::tableName() . '.date)' => date('Y')]
            ]);
        if ($includeAdjustments) {
            $query->andWhere([
                'or',
                ['is_adjustment' => 0],
                ['is_adjustment' => Income::INCOME_ADJUSTMENT]
            ]);
        }
        else {
            $query->andWhere([
                'is_adjustment' => 0
            ]);
        }

        return $query;
    }

    public static function getCountFilteredQuery($incomeId = null, $budgetId = null, $month = null, $year = null)
    {
        if (!$month) {
            $month = date('m');
        }
        if (!$year) {
            $year = date('Y');
        }
        $query = IncomeTransaction::find()
            ->joinWith('income')
            ->where([
                'and',
                ['Month(' . IncomeTransaction::tableName() . '.date)' => $month],
                ['Year(' . IncomeTransaction::tableName() . '.date)' => $year],
                ['is_adjustment' => 0]
            ]);
        if ($incomeId) {
            $query->andWhere([IncomeTransaction::tableName() . '.income_id' => $incomeId]);
        }
        elseif ($budgetId) {
            $query->andWhere([Income::tableName() . '.budget_id' => $budgetId]);
        }
        return $query;
    }

    public static function getSelectFilteredQuery($incomeId = null, $budgetId = null, $month = null, $year = null)
    {
        if (!$month) {
            $month = date('m');
        }
        if (!$year) {
            $year = date('Y');
        }
        $query = IncomeTransaction::find()
            ->select(IncomeTransaction::tableName() . '.*, ' . Account::tableName() . '.name as account_name')
            ->joinWith('income')
            ->joinWith('account')
            ->where([
                'and',
                ['MONTH(' . IncomeTransaction::tableName() . '.date)' => $month],
                ['YEAR(' . IncomeTransaction::tableName() . '.date)' => $year],
                ['is_adjustment' => 0]
            ]);
        if ($incomeId) {
            $query->andWhere([IncomeTransaction::tableName() . '.income_id' => $incomeId]);
        }
        elseif ($budgetId) {
            $query->andWhere([Income::tableName() . '.budget_id' => $budgetId]);
        }
        return $query;
    }

}
